/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2007 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* tuna: tun interface creation and deletion                                  */
/*============================================================================*/

/*
 * tunhandler
 *
 * input:  smode  - 0=create interface, 1=delete interface
 *	   anchor - pointer to vpnd global data
 *
 * return: in interface creation mode
 *
 *	   0 - success
 *	   1 - no tun device available
 *	   2 - tun configuration failed
 *	   3 - ifconfig failed
 *	   4 - creating route to partner failed
 *	   5 - creating priority route to peer failed
 *	   6 - creation of additional routes failed
 *	   7 - no icmp socket (with keepalive only)
 *
 *	   in interface deletion mode:
 *
 *	   0 - success
 *	   1 - tun device close error
 *	   3 - ifconfig down failed
 *	   4 - deleting route to partner failed
 *	   5 - deletion of priority route or additional routes failed
 *
 * This procedure creates or deletes the proxy tun interface.
 */

int tunhandler(int smode,VPN *anchor)
{
#if defined(SunOS) || defined(OSNetBSD) || defined(OSOpenBSD)
	return 1;
#else
	int i;			/* general purpose	*/
	int j;			/* general purpose	*/
	int k;			/* general purpose	*/
	int err;		/* return code		*/
	int quiet;		/* >0 for quiet errors	*/
	struct ifreq ifr;	/* used to set up if	*/
#if defined(FreeBSD)
	char bfr[16];		/* string creation	*/
	struct ifaliasreq ifa;	/* used to set up if	*/
#endif


	/* debug message */

	ENTER("tunhandler");

	/* reset return code */

	err=0;

	/* handle deletion requests in error handler tree */

	if(smode)goto delete;

	/* silence error handling */

	quiet=1;

	/* get tun device, handle errors */

#ifdef LINUX
	if((anchor->pty=open("/dev/net/tun",O_RDWR))==-1)JUMP("open",err1);
#elif defined(FreeBSD)
	for(anchor->proxy=0;anchor->proxy<256;anchor->proxy++)
	{
		sprintf(bfr,"/dev/tun%d",anchor->proxy++);
		if((anchor->pty=open("",O_RDWR))!=-1)break;
	}
	if(anchor->pty==-1)JUMP("open",err1);
#endif

	/* set tun modes */

#ifdef LINUX
	memset(&ifr,0,sizeof(ifr));
	ifr.ifr_flags=IFF_TUN|IFF_NO_PI;
	if(ioctl(anchor->pty,TUNSETIFF,(void *)&ifr)<0)
		JUMP("ioctl(TUNSETIFF)",err2);
	anchor->proxy=atoi(ifr.ifr_name+3);
#elif defined(FreeBSD)
	i=0;
	if(ioctl(anchor->pty,TUNSLMODE,&i)==-1)JUMP("ioctl(TUNSLMODE)",err2);
	if(ioctl(anchor->pty,TUNSIFHEAD,&i)==-1)JUMP("ioctl(TUNSIFHEAD)",err2);
#endif

	/* get control socket, react to errors */

	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)JUMP("socket",err3);

	/* reset error flag */

	j=0;

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"tun%u",anchor->proxy);
#if defined(FreeBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"tun%u",anchor->proxy);
#endif

	/* set up interface address, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
	if(ioctl(i,SIOCSIFADDR,&ifr)<0)j++;
#elif defined(FreeBSD)
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
#endif

	/* set up interface peer address, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
	if(!j)if(ioctl(i,SIOCSIFDSTADDR,&ifr)<0)j++;
#elif defined(FreeBSD)
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
#endif

	/* set up interface netmask, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCSIFNETMASK,&ifr)<0)j++;
#elif defined(FreeBSD)
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCAIFADDR,&ifa)<0)j++;
#endif

	/* set up interface mtu, react to errors */

	ifr.ifr_mtu=anchor->mtu;
	if(!j)if(ioctl(i,SIOCSIFMTU,&ifr)<0)j++;

	/* activate interface, react to errors */

#ifdef LINUX
	ifr.ifr_flags=IFF_UP|IFF_RUNNING|IFF_POINTOPOINT|IFF_NOARP|
		IFF_MULTICAST;
#elif defined(FreeBSD)
	ifr.ifr_flags=IFF_UP|IFF_POINTOPOINT|IFF_NOARP|IFF_MULTICAST;
	if(anchor->ctun)ifr.ifr_flags|=SC_COMPRESS;
#endif
	if(!j)if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)j++;

	/* close control socket */
 
	if(close(i))ERRNO("close");

	/* signal any errors encountered */

#ifdef LINUX
	if(j)JUMP("ioctl(SIOCSIFADDR,SIOCSIFDSTADDR,SIOCSIFNETMASK," \
		"SIOCSIFMTU,SIOCSIFFLAGS)",err3);
#elif defined(FreeBSD)
	if(j)JUMP("ioctl(SIOCAIFADDR,SIOCSIFMTU,SIOCSIFFLAGS)",err3);
#endif

	/* set up route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(1,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				GOTO("autoroute",err4);
	}
#endif

	/* set up priority route to peer, react to errors */

	if(anchor->peerroute)
		if(route(1,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				GOTO("peerroute",err5);

	/* set up all additional routes */

	for(k=0;k<anchor->rmax;k++)
		if(route(1,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))
		GOTO("routex",err6);

	/* in case of keepalive packages when link idle open icmp socket */

	if(anchor->keepalive)if(icmphandler(anchor))GOTO("icmphandler",err7);

	/* mark tun link as established */

	anchor->slip=1;

	/* execute tun up process, if there is any */

	doexec(anchor->slipup,ifr.ifr_name);

	/* signal success */

	goto err0;

	/* done, if no tun link */

delete:	if(!anchor->slip)GOTO("TUN link not up",err0);

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"tun%u",anchor->proxy);
#if defined(FreeBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"tun%u",anchor->proxy);
#endif

	/* execute tun down process, if there is any */

	doexec(anchor->slipdown,ifr.ifr_name);

	/* enable verbose mode */

	quiet=0;

	/* in case of keepalive packages when link idle close icmp socket */

	if(anchor->keepalive)icmphandler(anchor);

	/* error handling */

err7:	if(quiet)err++;
err6:	if(quiet)err++;

	/* delete additional routes */

	for(k=anchor->rmax-1;k>=0;k--)
		if(route(0,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))if(!quiet++)DEBUG1(3,"routex");

	/* delete priority route to peer */

	if(anchor->peerroute)
		if(route(0,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				if(!quiet++)DEBUG1(3,"peerroute");

err5:	if(quiet)err++;

	/* delete route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(0,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				if(!quiet++)DEBUG1(3,"autoroute");
	}

err4:	if(quiet)err++;
#elif defined(FreeBSD)
	if(quiet)err++;
#endif

	/* get control socket, react to errors */

	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)
	{
		if(!quiet++)ERRNO("socket");
	}

	/* if control socket could be created */

	else
	{
		/* deactivate interface, react to errors */

#if defined(FreeBSD)
		((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_addr.s_addr=getaddr(anchor->local_ip);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_addr.s_addr=getaddr(anchor->remote_ip);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))
			->sin_addr.s_addr=htonl(0xffffffff);
		if(ioctl(i,SIOCDIFADDR,&ifa)<0)
			ERRNO("ioctl(SIOCDIFADDR)");
#endif

		ifr.ifr_flags=0;
		if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFFLAGS)");

		/* reset interface address, react to errors */

#ifdef LINUX
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=0;
		if(ioctl(i,SIOCSIFADDR,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFADDR)");
#endif

		/* close control socket */
 
		if(close(i))if(!quiet++)ERRNO("close");
	}

err3:	if(quiet)err++;
err2:	if(quiet)err++;

	/* close tun device */

	if(close(anchor->pty))if(!quiet++)ERRNO("tun close");

err1:	if(quiet)err++;

	/*  mark tun link as down */

	anchor->slip=0;

	/* debug message */

err0:	LEAVE("tunhandler");

	/* signal success or error */

	return err;
#endif
}
