/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2001 Andreas Steinmetz, ast@domdv.de
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */
 
#include "vpnd.h"
 
/*============================================================================*/
/* humble servant: control socket creation and deletion                       */
/*============================================================================*/
 
/*
 * controlhandler
 *
 * input:  mode   - 0=create socket, 1=delete socket
 *
 * return: in socket creation mode
 *
 *	   0 - success
 *	   1 - creating the socket failed
 *	   2 - binding the socket failed
 *	   3 - listening on the socket failed
 *
 *	   in socket deletion mode:
 *
 *	   0 - success
 *	   1 - closing the socket failed
 *	   2 - deleting the socket failed
 *
 * This procedure creates or deletes the control socket.
 */

int controlhandler(int mode)
{
	int s;			/* socket handle	*/
	int err;		/* return code		*/
	int quiet;		/* >0 for quiet errors	*/
	mode_t mask;		/* saved umask		*/
	struct sockaddr_un a;	/* socket address	*/
	struct stat stb;	/* deletion check	*/


	/* debug message */

	ENTER("controlhandler");

	/* reset return code */

	err=0;

	/* quiet error cleanup */

	quiet=1;

	/* done if no control socket has to be used */

	if(!control)GOTO("no control socket",err0);

	/* handle delete request */

	if(mode)goto delete;

	/* create socket, handle errors */

	if((s=socket(PF_UNIX,SOCK_STREAM,0))==-1)JUMP("socket",err1);

	/* prepare socket address */

	memset(&a,0,sizeof(a));
	a.sun_family=AF_UNIX;
	strcpy(a.sun_path,control);

	/* remove any other socket of the same name - now here's
	   a little race condition but that can't be helped */

	if(!lstat(control,&stb))
	{
		if(!S_ISSOCK(stb.st_mode))GOTO("file exists",err2);
		if(unlink(control))JUMP("unlink socket",err2);
	}

	/* set up a reasonable umask for the socket */

	mask=umask(077);

	/* bind the socket, handle errors, restore old umask */

	if(bind(s,(struct sockaddr *)(&a),sizeof(a)))
	{
		umask(mask);
		GOTO("bind",err2);
	}
	umask(mask);

	/* another race condition on systems where the umask
	   is not honored during bind (socket creation) */

#ifdef OSNetBSD
	if(chmod(control,0700))JUMP("chmod",err3);
#endif

	/* make socket listen, handle errors */

	if(listen(s,5))JUMP("listen",err3);

	/* store socket handle */

	ctrl=s;

	/* signal success */

	goto err0;

	/* socket deletion */

delete:	if(ctrl==-1)GOTO("no control socket",err0);
	s=ctrl;

	/* prepare silent error handling */

	quiet=0;

	/* unlink socket, handle errors */

err3:	if(quiet)err++;
	if(unlink(control))if(!quiet++)ERRNO("unlink");

	/* close socket, handle errors */

err2:	if(quiet)err++;
	if(close(s))if(!quiet++)ERRNO("close");

	/* mark control socket closed */

err1:	if(quiet)err++;
	ctrl=-1;

	/* debug message */

err0:	LEAVE("controlhandler");

	/* signal success or error */

	return err;
}

/*
 * docontrol
 *
 * UNDER DEVELOPMENT!!!
 *
 */

int docontrol(int status)
{
	int s;
	socklen_t l;
	int r;
	struct sockaddr a;
	struct timeval t;
	fd_set f;
	WORD08 b;


	/* reset return code */

	r=0;

	/* accept pending request, signal no action in case
	   of error */

	l=sizeof(a);
	if((s=accept(ctrl,&a,&l))==-1)JUMP("accept",err1);

	/* wait for request (50ms timeout), signal no action in case
	   of request error */

	t.tv_sec=0;
	t.tv_usec=50000;
	FD_ZERO(&f);
	FD_SET(s,&f);
	if(select(s+1,&f,NULL,NULL,&t)<=0)JUMP("select(read)",err2);
	if(read(s,&b,1)!=1)JUMP("read",err2);

	/* proceed according to request */

	switch(b)
	{
	/* peer link status request */

	case CTRL_STATUS:
		b=(WORD08)(status);
		break;

	/* peer link (forced) down request */

	case CTRL_DOWN:
	case CTRL_FORCEDOWN:
		r=b;
		b=CTRL_OK;
		break;

	/* unknown request */

	default:b=CTRL_FAIL;
		break;
	}

	/* reply must be immediately written, otherwise signal no
	   action */

	t.tv_sec=0;
	t.tv_usec=0;
	FD_ZERO(&f);
	FD_SET(s,&f);
	if(select(s+1,NULL,&f,NULL,&t)<=0)JUMP("select(write)",err2);
	if(write(s,&b,1)!=1)JUMP("write",err2);

	/* close socket */

err2:	close(s);

	/* return action to be taken */

err1:	return r;
}
