/*
 * SHA1 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * SHA1 algorithm: see FIPS PUB 180-1
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact ast@domdv.de
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the SHA1 routines.
 *
 * Data sizes:
 *
 * data block for SHA1 transformation	WORD08[64]
 * resulting SHA1 hash			WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * SHA1_FULL defined:
 *
 * 1. Call sha1init
 * 2. For all data to be hashed call sha1next
 * 3. To retrieve the hash call sha1end
 *
 * SHA1_FAST defined:
 *
 * Call sha1 to get the hash for the specified data
 *
 * SHA1_HMAC_FULL defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hminit
 * 3. For all data to be hashed call sha1hmnext
 * 4. To retrieve the hash call sha1hmend
 * 5. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 *
 * SHA1_HMAC_FAST defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 */

/* sanity settings */

#if !defined(SHA1_FULL) && !defined(SHA1_FAST) && !defined(SHA1_HMAC_FULL) && !defined(SHA1_HMAC_FAST)
#define SHA1_FULL
#define SHA1_FAST
#define SHA1_HMAC_FULL
#define SHA1_HMAC_FAST
#define SHA1_PARANOID
#endif

#if defined(SHA1_HMAC_FULL) || defined(SHA1_HMAC_FAST)
#ifndef SHA1_HMAC
#define SHA1_HMAC
#endif
#ifndef SHA1_PARANOID
#define SHA1_PARANOID
#endif
#endif

/* definitions */

#define SHA1_SIZE 20
#define SHA1(a) WORD08 a[SHA1_SIZE]

/* required types */

typedef struct
{
	WORD32 sha1[5];	/* current sha1 result				*/
	WORD32 total;	/* total amount of bytes processed		*/
	union
	{
	  WORD32 l[16];	/* longword buffer for paranoia memory reset	*/
	  WORD08 b[64];	/* buffer for incomplete 64 byte message block	*/
	}bfr;
	WORD08 size;	/* size of data in buffer			*/
} SHA1DATA;

#ifdef SHA1_HMAC
typedef struct
{
	WORD32 isha1[5];	/* preprocessed HMAC key (inner)	*/
	WORD32 osha1[5];	/* preprocessed HMAC key (outer)	*/
} SHA1HMDATA;
#endif

/* function prototypes */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)
extern void CRYPTOCALL sha1init(SHA1DATA *ptr);
extern void CRYPTOCALL sha1next(WORD08 *data,WORD32 length,SHA1DATA *ptr);
extern void CRYPTOCALL sha1end(WORD08 *result,SHA1DATA *ptr);
#endif
#ifdef SHA1_FAST
extern void CRYPTOCALL sha1(WORD08 *data,WORD32 length,WORD08 *result);
#endif
#if defined(SHA1_HMAC_FULL) || defined(SHA1_HMAC_FAST)
extern void CRYPTOCALL sha1hmkey(WORD08 *key,WORD32 keylength,SHA1HMDATA *ptr);
#endif
#ifdef SHA1_HMAC_FULL
extern void CRYPTOCALL sha1hminit(SHA1DATA *ptr,SHA1HMDATA *key);
#define sha1hmnext(a,b,c) sha1next(a,b,c)
extern void CRYPTOCALL sha1hmend(WORD08 *result,SHA1DATA *ptr,SHA1HMDATA *key);
#endif
#ifdef SHA1_HMAC_FAST
extern void CRYPTOCALL sha1hmac(WORD08 *data,WORD32 length,WORD08 *result,SHA1HMDATA *key);
#endif

