#include <Balloons.h>
#include <LCheckBox.h>
#include <LRadioButton.h>
#include <LPeriodical.h>
#include <LWindow.h>
#include <LStream.h>
#include <TArrayIterator.h>
#include "ABalloon.h"

/*	---------------------------------------------------------------------------------------------
	ABalloon		PowerPlant attachment class for balloon help
	
	Author: James W. Walker <mailto:jwwalker@kagi.com>
	
	Creation date: 2/26/98
	
	Last Modified: 4/11/98
	
	Purpose:	To assign help balloons to panes in a PowerPlant
				window.  You can use different messages depending
				on whether the item is on or off, enabled or disabled.
				
	Usage:		Attach an ABalloon to each pane (usually a control)
				for which you want to provide balloon help.
				Specify a STR# ID and a string index for each state.
				And remember to register ABalloon at startup.
				You must compile with RTTI for proper function.
	
	Overriding:	The most likely methods to override are ComputeTip
				(to provide more appropriate placement of the
				balloon tip) and ComputeStringIndex (if you have
				more than the 4 states I provided for.)
			
	Restrictions:	ABalloon is (C)1998 by James W. Walker.
				Permission is granted for use of ABalloon
				free of charge, other than acknowledgement of
				James W. Walker in any program using ABalloon
				(perhaps in an About box or in accompanying
				documentation). 
	
	Change history:
		1.0  2/28/98		First release.
		1.1  4/11/98
				- Don't show balloons for invisible panes
				  (Thanks, Daniel Chiaramello, for the correction)
				- Rolled ABalloonHelper::CreateABalloonHelper into
				  ABalloonHelper::AddABalloon.
				- sBalloonAttachments is now a TArray<ABalloon*>
				  instead of TArray<LAttachment*>.
				- In the HMShowBalloon call, use kHMRegularWindow
				  instead of kHMSaveBitsNoWindow, because IM says
				  that the former should normally be used.
	---------------------------------------------------------------------------------------------
*/
LAttachable*	ABalloon::sLastBalloonedItem = nil;
Int16			ABalloon::sLastIndex = 0;
ResIDT			ABalloon::sLastStringList = 0;

#if !__option(RTTI)
	#error you must compile with RTTI for this to work right
#endif

#pragma mark --------- ABalloonHelper declaration

/*	---------------------------------------------------------------------------------------------
	class ABalloonHelper
	
	This periodical is in charge of sending DoBalloon messages to
	appropriate ABalloon attachments.
	
	This is a singleton, created the first time an ABalloon is created.
	---------------------------------------------------------------------------------------------
*/
class ABalloonHelper  : public LPeriodical
{
public:
	static void		AddABalloon( ABalloon* inABalloon );
	static void		RemoveABalloon( ABalloon* inABalloon );
	
protected:
	
					ABalloonHelper();
					
					~ABalloonHelper();
					
	virtual	void	SpendTime(
							const EventRecord		&inMacEvent);

	static 	ABalloonHelper*	sTheHelper;		// the single object of this class
	
	static	TArray<ABalloon*>*	sBalloonAttachments;
};

ABalloonHelper*			ABalloonHelper::sTheHelper = nil;
TArray<ABalloon*>*		ABalloonHelper::sBalloonAttachments = nil;

#pragma mark --------- ABalloonHelper code

void	ABalloonHelper::AddABalloon( ABalloon* inABalloon )
{
	if (sTheHelper == nil)
	{
		sTheHelper = new ABalloonHelper;
		sBalloonAttachments = new TArray<ABalloon*>;
	}
	sBalloonAttachments->InsertItemsAt( 1, LArray::index_Last, inABalloon );
}

void	ABalloonHelper::RemoveABalloon( ABalloon* inABalloon )
{
	sBalloonAttachments->Remove( inABalloon );
}

ABalloonHelper::ABalloonHelper()
{
	StartRepeating();
}

ABalloonHelper::~ABalloonHelper()
{
	sTheHelper = nil;
	delete sBalloonAttachments;
	sBalloonAttachments = nil;
}

/*	---------------------------------------------------------------------------------------------
	ABalloonHelper::SpendTime
	
	This function determines which ABalloon attachment, if any, should
	show its balloon.
	---------------------------------------------------------------------------------------------
*/
void	ABalloonHelper::SpendTime( const EventRecord	&inMacEvent )
{
	if (::HMGetBalloons())
	{
		Point	globalPt = inMacEvent.where;
		WindowPtr	macWindowP;
		::FindWindow(globalPt, &macWindowP);
		
		if (macWindowP != nil)
		{
			LWindow	*theWindow = LWindow::FetchWindowObject(macWindowP);
			if ((theWindow != nil) &&
				theWindow->IsActive() &&
				theWindow->IsEnabled())
			{
				Point	portMouse = globalPt;
				theWindow->GlobalToPortPoint(portMouse);
				
				// We have an array of all active ABalloons. Of those
				// attached to subpanes of this window, we want to find
				// the deepest one containing the point.
				ABalloon*	theBestSoFar = nil;
				Rect		theBestFrameSoFar;
				TArrayIterator<ABalloon*>	iterator(*sBalloonAttachments);
				ABalloon*	theAttachment;
				
				while (iterator.Next(theAttachment))
				{
					LPane* ownerPane = dynamic_cast<LPane*>(
						theAttachment->GetOwnerHost() );
					Rect	ownerFrame;
					if ( (ownerPane != nil) &&
						ownerPane->IsVisible() &&		// 4/10/98
						(ownerPane->GetMacPort() == macWindowP) &&
						ownerPane->CalcPortFrameRect(ownerFrame) &&
						::PtInRect( portMouse, &ownerFrame ) )
					{
						// The current frame is completely enclosed in
						// theBestFrameSoFar just in case their union
						// equals theBestFrameSoFar.
						Rect	theUnion;
						::UnionRect( &theBestFrameSoFar, &ownerFrame, &theUnion );
						
						if ( (theBestSoFar == nil) ||
							::EqualRect( &theBestFrameSoFar, &theUnion )
						)
						{
							theBestSoFar = theAttachment;
							theBestFrameSoFar = ownerFrame;
						}
					}
				} // end of loop over ABalloons
				
				if (theBestSoFar != nil)
				{
					theBestSoFar->DoBalloon();
				}
			}
		}
	}
}



#pragma mark --------- ABalloon code

// ABalloon::ABalloon	stream constructor.
ABalloon::ABalloon(
		LStream			*inStream)
	: LAttachment( inStream )
{
	// I don't want to have to bother filling this out in
	// Constructor every time.
	mMessage = msg_Nothing;
	
	*inStream >> mStringListID;
	*inStream >> mOffDisabledIndex;
	*inStream >> mOffEnabledIndex;
	*inStream >> mOnDisabledIndex;
	*inStream >> mOnEnabledIndex;
	
	ABalloonHelper::AddABalloon( this );
}

// ABalloon::~ABalloon	destructor.
ABalloon::~ABalloon()
{
	ABalloonHelper::RemoveABalloon( this );
}

/*	---------------------------------------------------------------------------------------------
	ABalloon::ShowBalloon
	
	Show a balloon, given a string index and pane frame.
	We must be sure not to display the same balloon that is already
	showing, to avoid flicker.
	---------------------------------------------------------------------------------------------
*/
void	ABalloon::ShowBalloon( Int16 inStringIndex, Rect& inAltRect )
{
	if (::HMIsBalloon() && (sLastBalloonedItem == mOwnerHost) &&
		(sLastIndex == inStringIndex) && (sLastStringList == mStringListID) )
	{
		// already showing it, nothing to do
	}
	else
	{
		HMMessageRecord		theMessageRec;
		
		theMessageRec.hmmHelpType = khmmString;
		::GetIndString( theMessageRec.u.hmmString, mStringListID,
			inStringIndex );
		
		sLastBalloonedItem = mOwnerHost;
		sLastIndex = inStringIndex;
		sLastStringList = mStringListID;
		
		::HMShowBalloon( &theMessageRec, ComputeTip( inAltRect ),
			&inAltRect, nil, 0, 0, kHMRegularWindow );
	}
}


/*	---------------------------------------------------------------------------------------------
	ABalloon::ComputeTip
	
	Compute the location for the balloon's tip, given the pane's frame.
	---------------------------------------------------------------------------------------------
*/
Point	ABalloon::ComputeTip( const Rect& inAltRect )
{
	Point	theTip;
	theTip.v = (inAltRect.top + inAltRect.bottom) / 2;
	theTip.h = inAltRect.right - 4;
	
	if ( (nil != dynamic_cast<LRadioButton*>(mOwnerHost)) ||
		(nil != dynamic_cast<LCheckBox*>(mOwnerHost))
	)
	{
		theTip.h = inAltRect.left + 4;
	}
	return theTip;
}


/*	---------------------------------------------------------------------------------------------
	ABalloon::DoBalloon
	
	Compute the pane's local frame and call ShowBalloon.
	---------------------------------------------------------------------------------------------
*/
void	ABalloon::DoBalloon()
{
	LPane*	thePane = dynamic_cast<LPane*>( mOwnerHost );
	if (thePane != nil)
	{
		Rect	theFrame;
		
		thePane->CalcPortFrameRect( theFrame );
		thePane->PortToGlobalPoint( topLeft(theFrame) );
		thePane->PortToGlobalPoint( botRight(theFrame) );
		
		ShowBalloon( ComputeStringIndex(thePane), theFrame );
	}
}

/*	---------------------------------------------------------------------------------------------
	ABalloon::ComputeStringIndex
	
	Examine the state of the pane and return the appropriate string index.
	---------------------------------------------------------------------------------------------
*/
Int16	ABalloon::ComputeStringIndex( LPane* inPane )
{
	Int16	whichString;
	if (inPane->GetValue() == 0)
	{
		if (inPane->IsEnabled())
		{
			whichString = mOffEnabledIndex;
		}
		else
		{
			whichString = mOffDisabledIndex;
		}
	}
	else
	{
		if (inPane->IsEnabled())
		{
			whichString = mOnEnabledIndex;
		}
		else
		{
			whichString = mOnDisabledIndex;
		}
	}
	return whichString;
}
