/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	
	
	$Id: CPGPEncoderDecoder.cp,v 1.59.2.2.2.1 1998/11/12 03:10:02 heller Exp $
____________________________________________________________________________*/

#include <string.h>

#include "PGPKeys.h"
#include "PGPUtilities.h"
#include "MacFiles.h"
#include "pgpMacMemory.h"
#include "MacStrings.h"
#include "MacErrors.h"
#include "PGPSharedEncryptDecrypt.h"
#include "PGPUserInterface.h"
#include "WarningAlert.h"
#include "pflPrefTypes.h"
#include "pgpOpenPrefs.h"
#include "pgpClientPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpWordWrap.h"
#include "pgpMemoryIO.h"
#include "pgpVersionHeader.h"
#include "CSecureMemory.h"
#include "pgpClientErrors.h"

#include "CTempFile.h"

#include "CPGPEncoderDecoder.h"


struct SDecodeData {
	Boolean				useCache;
	Boolean				FYEO;
	AnimatedCursorRef	cursor;
	Handle				handle;
	PGPKeySetRef		defaultKeys;
};


// Statics
Boolean					CPGPEncoderDecoder::sCheckForMissingKeys = true;
CSignPassphraseCache *	CPGPEncoderDecoder::sSigningPassphraseCache = nil;
CSignPassphraseCache *	CPGPEncoderDecoder::sDecryptPassphraseCache = nil;
PGPtlsContextRef		CPGPEncoderDecoder::sTLSContext
												= kInvalidPGPtlsContextRef;



// Constants
const ResIDT	STRx_PGPEncoderDecoder				=	12121;
const SInt16	kNoPublicKeysID						=	1;
const SInt16	kNoPrivateKeysID					=	2;
const SInt16	kAddKeysID							=	3;

const ResIDT	STRx_PGPEncoderDecoderErrorStrings	=	12122;
const SInt16	kErrorStringID						=	1;
const SInt16	kErrorStringAdminPrefsNotFoundID	=	2;

	void
CPGPEncoderDecoder::Init(
	PGPContextRef	inContext)
{
	PGPError	pgpError;
	
	sSigningPassphraseCache = new CSignPassphraseCache(inContext);
	sDecryptPassphraseCache = new CSignPassphraseCache(inContext);
	pgpError = PGPNewTLSContext(inContext, &sTLSContext);
	PGPThrowIfPGPErr_(pgpError);
	
#if PGP_BUSINESS_SECURITY
	pgpError = PGPCheckAutoUpdateKeysFromServer(
						PGPGetContextMemoryMgr(inContext),
						true,
						nil,
						nil);
	PGPThrowIfPGPErr_(pgpError);
#endif
}



	void
CPGPEncoderDecoder::Cleanup()
{
	delete sSigningPassphraseCache;
	delete sDecryptPassphraseCache;
	if (PGPtlsContextRefIsValid(sTLSContext)) {
		PGPFreeTLSContext(sTLSContext);
	}
}



	Boolean
CPGPEncoderDecoder::EncodeHandle(
	PGPContextRef				inContext,
	Handle						ioDataH,
	EEncodeOptions				inOptions,
	UInt32						inNumRecipients,
	const PGPRecipientSpec *	inRecipientList,
	Boolean						inUseCache)
{
	Boolean				result = false;
	OSErr				err;
	PGPError			pgpErr;
	AnimatedCursorRef	cursorRef = nil;
	PGPKeySetRef		defaultKeySet = kInvalidPGPKeySetRef;
	PGPKeySetRef		recipientSet = kInvalidPGPKeySetRef;
	PGPKeySetRef		newKeys = kInvalidPGPKeySetRef;
	PGPKeySetRef		selectedSet = kInvalidPGPKeySetRef;
	PGPKeyRef			signingKey = kInvalidPGPKeyRef;
	PGPOptionListRef	optionList = kInvalidPGPOptionListRef;
	PGPPrefRef			prefRef = kInvalidPGPPrefRef;
	PGPIORef			wrapInput = kInvalidPGPIORef;
	PGPIORef			wrapOutput = kInvalidPGPIORef;
	
	try {
		if (::GetHandleSize(ioDataH) > 0) {
			CSecureCString256	convEncryptPassphrase;
			CSecureMemory		signingBuffer(
									PGPGetContextMemoryMgr(inContext),
									256);
			PGPSize				signingPasskeySize;
			UInt32				numPublicKeys = 1;
			UInt32				numPrivateKeys = 1;
			Boolean				usePassphrase = true;
			Boolean				fyeo = false;
	
			// Check for missing keys
			if (sCheckForMissingKeys) {
				err = CountKeysInDefaultKeyring(	inContext,
													&numPublicKeys,
													&numPrivateKeys);
				PGPThrowIfOSErr_(err);
				
				if ((numPublicKeys != 0) && (numPrivateKeys != 0)) {
					sCheckForMissingKeys = false;
				}
			}
			
			// Open default keyset
			pgpErr = PGPOpenDefaultKeyRings(	inContext,
												0,
												&defaultKeySet);
			PGPThrowIfPGPErr_(pgpErr);

			// If encrypting, get the recipients
			if ((inOptions & encodeOptions_Encrypt)
			&& (! (inOptions & encodeOptions_ConvEncrypt))) {
				PGPRecipientOptions		options = kPGPRecipientOptionsHideFileOptions
											| kPGPRecipientOptionsShowFYEO;
				PGPRecipientSettings	settings =
											kPGPRecipientSettingsDefault;
				
				if (numPublicKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPublicKeysID);
				}

				if (! inUseCache) {
					options |= kPGPRecipientOptionsAlwaysShowDialog;
				}
				pgpErr = PGPClientRecipientDialog(	inContext,
													sTLSContext,
													defaultKeySet,
													inNumRecipients,
													inRecipientList,
													options,
													settings,
													&settings,
													&recipientSet,
													&newKeys);
				PGPThrowIfPGPErr_(pgpErr);

				if (PGPRefIsValid(newKeys)) {
					pgpErr = PGPSelectKeysDialog(
								inContext,
								kPGPSelectKeysImportVariation,
								CString(	STRx_PGPEncoderDecoder,
											kAddKeysID),							
								newKeys,							
								defaultKeySet,
								&selectedSet);
					if (pgpErr != kPGPError_UserAbort) {
						PGPThrowIfPGPErr_(pgpErr);
						pgpErr = PGPSharedAddKeysToDefaultKeyring(
									selectedSet);
						PGPThrowIfPGPErr_(pgpErr);
					}
				}
				
				if (settings & kPGPRecipientSettingsConvEncrypt) {
					inOptions &= ~encodeOptions_Encrypt;
					inOptions |= encodeOptions_ConvEncrypt;
				} else {
					PGPUInt32		numKeys;
					pgpErr	= PGPCountKeys( recipientSet, &numKeys);
					PGPThrowIfPGPErr_(pgpErr);
				
					if ( numKeys == 0) {
						inOptions &= ~encodeOptions_Encrypt;
						if (! (inOptions & encodeOptions_Sign)) {
							PGPThrowOSErr_(userCanceledErr);
						}
					}
				}
				
				if (settings & kPGPRecipientSettingsFYEO) {
					fyeo = true;
				}
			}
			
			// If conventionally encrypting, get passphrase
			if (inOptions & encodeOptions_ConvEncrypt) {
				char *	conventionalPassphrase;
				
				pgpErr = PGPConventionalEncryptionPassphraseDialog(
									inContext,
									PGPOUIOutputPassphrase(	inContext,
															&conventionalPassphrase),
									PGPOLastOption(inContext));
				PGPThrowIfPGPErr_(pgpErr);
				strcpy(convEncryptPassphrase, conventionalPassphrase);
				PGPFreeData(conventionalPassphrase);
			}
			
			// If signing, get passphrase
			if (inOptions & encodeOptions_Sign) {
				PGPGetPassphraseSettings	userSettings;
				
				// Make sure we have a key to sign to
				if (numPrivateKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPrivateKeysID);
				}
				
				// Get passphrase
				if (! inUseCache) {
					sSigningPassphraseCache->Forget();
				}
				if (! sSigningPassphraseCache->GetPassphraseOrPasskey(defaultKeySet,
				signingBuffer.mMemory, &usePassphrase, &signingPasskeySize,
				&signingKey)) {
					usePassphrase = true;
					pgpErr = PGPClientSigningPassphraseDialog(
								inContext,
								defaultKeySet,
								nil,
								kPGPGetPassphraseOptionsHideFileOptions,
								userSettings,
								signingKey,
								(char *) signingBuffer.mMemory,
								&userSettings,
								&signingKey);
					if (pgpErr != kPGPError_KeyUnusableForSignature) {
						PGPThrowIfPGPErr_(pgpErr);
					} else {
						PGPByte *	passkey = nil;
						
						pgpErr = PGPReconstitutionDialog(
										signingKey,
										defaultKeySet,
										kInvalidPGPtlsContextRef,
										&passkey,
										&signingPasskeySize);
						PGPThrowIfPGPErr_(pgpErr);
						pgpCopyMemory(	passkey,
										signingBuffer.mMemory,
										signingPasskeySize);
						PGPFreeData(passkey);
						usePassphrase = false;
					}
					SetPassbufferCache(	true,
										signingKey,
										usePassphrase,
										signingBuffer.mMemory,
										signingPasskeySize);
				}
				
				// If we are only signing, wrap the input
				if ((! (inOptions & encodeOptions_Encrypt))
				&& (! (inOptions &encodeOptions_ConvEncrypt))) {
					PGPBoolean	wordWrapEnabled;
					
					pgpErr = PGPOpenClientPrefs(
								PGPGetContextMemoryMgr(inContext),
								&prefRef);
					PGPThrowIfPGPErr_(pgpErr);
					pgpErr = PGPGetPrefBoolean(	prefRef,
												kPGPPrefWordWrapEnable,
												&wordWrapEnabled);
					PGPThrowIfPGPErr_(pgpErr);
					
					if (wordWrapEnabled) {
						PGPUInt32		wordWrapWidth;
						PGPFileOffset	bufSize;
						PGPSize			unused;	
						
						{
							StHandleLocker	theLock(ioDataH);
							
							pgpErr = PGPGetPrefNumber(	prefRef,
														kPGPPrefWordWrapWidth,
														&wordWrapWidth);
							PGPThrowIfPGPErr_(pgpErr);
							pgpErr = PGPNewMemoryIOFixedBuffer(
											PGPGetContextMemoryMgr(inContext),
											*ioDataH,
											::GetHandleSize(ioDataH),
											(PGPMemoryIORef*) &wrapInput);
							PGPThrowIfPGPErr_(pgpErr);
							pgpErr = PGPNewMemoryIO(
										PGPGetContextMemoryMgr(inContext),
										(PGPMemoryIORef*) &wrapOutput);
							PGPThrowIfPGPErr_(pgpErr);
							pgpErr = pgpWordWrapIO(	wrapInput,
													wrapOutput,
													wordWrapWidth,
													"\r");
							PGPThrowIfPGPErr_(pgpErr);
							PGPFreeIO(wrapInput);
							wrapInput = kInvalidPGPIORef;
						}
						
						pgpErr = PGPIOSetPos(	wrapOutput,
												0);
						PGPThrowIfPGPErr_(pgpErr);
						pgpErr = PGPIOGetEOF(	wrapOutput,
												&bufSize);
						PGPThrowIfPGPErr_(pgpErr);
						::SetHandleSize(ioDataH, bufSize);
						PGPThrowIfMemError_();
						::HLock(ioDataH);
						pgpErr = PGPIORead(	wrapOutput,
											bufSize,
											*ioDataH,
											&unused);
						::HUnlock(ioDataH);
						PGPThrowIfPGPErr_(pgpErr);
						PGPFreeIO(wrapOutput);
						wrapOutput = kInvalidPGPIORef;
					}
				}
			}

			// Load the cursor
			err = Get1AnimatedCursor(acur_BeachBall, &cursorRef);
			PGPThrowIfOSErr_(err);
			
			// Do the encrypting
			PGPSize				bufSize;
			void *				buf;

			pgpErr = PGPGuaranteeMinimumEntropy(inContext);
			PGPThrowIfPGPErr_(pgpErr);
			optionList = CreateOptionList(	inContext,
											inOptions,
											recipientSet,
											signingKey,
											convEncryptPassphrase,
											signingBuffer.mMemory,
											signingPasskeySize,
											usePassphrase);
			::HLock(ioDataH);
			pgpErr = PGPEncode(	inContext,
								PGPOInputBuffer(	inContext,
													*ioDataH,
													::GetHandleSize(ioDataH)),
								PGPOAllocatedOutputBuffer(	inContext,
															&buf,
															MAX_PGPSize,
															&bufSize),
								PGPOEventHandler(	inContext,
													PGPEncodeEventHandler,
													cursorRef),
								PGPOSendNullEvents(	inContext,
													500),
								PGPOForYourEyesOnly(inContext, fyeo),
								optionList,
								PGPOLastOption(inContext));
			::HUnlock(ioDataH);
			PGPThrowIfPGPErr_(pgpErr);
			
			::PtrToXHand(buf, ioDataH, bufSize);
			PGPFreeData(buf);
								
			result = true;
		}
	}
	
	catch (CComboError & comboError) {
		if (! comboError.IsCancelError()) {
			if (comboError.HavePGPError()) {
				ShowError(comboError.pgpErr);
			} else {
				ShowError(MacErrorToPGPError(comboError.err));
			}
		}
	}

	catch(...) {
		ShowError(kPGPError_UnknownError);
	}
	
	// Cleanup
	if (cursorRef != nil) {
		DisposeAnimatedCursor(cursorRef);
	}
	if (PGPOptionListRefIsValid(optionList)) {
		PGPFreeOptionList(optionList);
	}
	if (PGPKeySetRefIsValid(selectedSet)) {
		PGPFreeKeySet(selectedSet);
	}
	if (PGPKeySetRefIsValid(newKeys)) {
		PGPFreeKeySet(newKeys);
	}
	if (PGPKeySetRefIsValid(recipientSet)) {
		PGPFreeKeySet(recipientSet);
	}
	if (PGPKeySetRefIsValid(defaultKeySet)) {
		PGPFreeKeySet(defaultKeySet);
	}
	if (PGPPrefRefIsValid(prefRef)) {
		PGPClosePrefFile(prefRef);
	}
	if (PGPIORefIsValid(wrapInput)) {
		PGPFreeIO(wrapInput);
	}
	if (PGPIORefIsValid(wrapOutput)) {
		PGPFreeIO(wrapOutput);
	}

	return result;
}



	Boolean
CPGPEncoderDecoder::DecodeHandle(
	PGPContextRef			inContext,
	Handle					inDataH,
	PGPKeySetRef			inKeySet,
	SDecodeSectionHandle &	outDecodeSectionsHandle,
	Boolean *				outFYEO,
	Boolean					inUseCache)
{
	Boolean				result = false;

	if (::GetHandleSize(inDataH) > 0) {
		OSErr				err;
		PGPError			pgpErr;
		UInt32				numPublicKeys = 1;
		UInt32				numPrivateKeys = 1;
		SDecodeData			decodeData = {inUseCache, false, nil, nil, inKeySet};

		// Check for missing keys
		try {
			if (sCheckForMissingKeys) {
				err = CountKeysInDefaultKeyring(	inContext,
													&numPublicKeys,
													&numPrivateKeys);
				PGPThrowIfOSErr_(err);
				
				if (numPublicKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPublicKeysID);
				} else if (numPrivateKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPrivateKeysID);
				} else {
					sCheckForMissingKeys = false;
				}
			}
			
			// Get the cursor
			err = Get1AnimatedCursor(acur_BeachBall, &decodeData.cursor);
			PGPThrowIfOSErr_(err);
			
			// Create Handle for storing decoded data
			decodeData.handle = ::NewHandle(0);
			PGPThrowIfMemFail_(decodeData.handle);
			
			::HLock(inDataH);
			::HLock(decodeData.handle);
			pgpErr = PGPDecode(	inContext,
								PGPOInputBuffer(	inContext,
													*inDataH,
													::GetHandleSize(inDataH)),
								PGPOEventHandler(	inContext,
													PGPDecodeEventHandler,
													&decodeData),
								PGPOSendNullEvents(	inContext,
													500),
								PGPOKeySetRef(	inContext,
												inKeySet),
								PGPOSendEventIfKeyFound(inContext, true),
								PGPOPassThroughIfUnrecognized(	inContext,
																true),
								PGPOLastOption(inContext));
			::HUnlock(decodeData.handle);
			::HUnlock(inDataH);
			PGPThrowIfPGPErr_(pgpErr);
			
			outDecodeSectionsHandle =
					(SDecodeSectionHandle) decodeData.handle;
			*outFYEO = decodeData.FYEO;
			decodeData.handle = nil;
			result = true;
		}
		
		catch (CComboError & comboError) {
			if (! comboError.IsCancelError()) {
				if (comboError.HavePGPError()) {
					ShowError(comboError.pgpErr);
				} else {
					ShowError(MacErrorToPGPError(comboError.err));
				}
			}
		}

		catch(...) {
			ShowError(kPGPError_UnknownError);
		}
		
		// Cleanup
		if (decodeData.cursor != nil) {
			DisposeAnimatedCursor(decodeData.cursor);
		}
		if (decodeData.handle != nil) {
			::DisposeHandle(decodeData.handle);
		}
	}
	
	return result;
}



	void
CPGPEncoderDecoder::FreeDecodeSection(
	SDecodeSection *	inSection)
{
	if (inSection->buf != nil) {
		PGPFreeData(inSection->buf);
	}
	if (PGPKeySetRefIsValid(inSection->addKeysSet)) {
		PGPFreeKeySet(inSection->addKeysSet);
	}
}



	PGPOptionListRef
CPGPEncoderDecoder::CreateOptionList(
	PGPContextRef	inContext,
	EEncodeOptions	inOptions,
	PGPKeySetRef	inRecipients,
	PGPKeyRef		inSigningKey,
	char *			inConvEncryptPassphrase,
	void *			inPassBuffer,
	PGPSize			inPassKeySize,
	Boolean			inUsePassphrase)
{
	PGPOptionListRef	optionList;
	CString				commentString;
	PGPPrefRef			prefRef = kInvalidPGPPrefRef;
	PGPError			pgpErr;
	
	try {
		// Add admin comment
#if PGP_BUSINESS_SECURITY
		pgpErr = PGPOpenAdminPrefs(	PGPGetContextMemoryMgr(inContext),
									&prefRef );

		if (pgpErr != kPGPError_NoErr) {
			WarningAlert(	kWAStopAlertType,
							kWAOKStyle,
							STRx_PGPEncoderDecoderErrorStrings,
							kErrorStringID,
							CString(pgpErr),
							CString(	STRx_PGPEncoderDecoderErrorStrings,
										kErrorStringAdminPrefsNotFoundID));
			PGPThrowOSErr_(userCanceledErr);
		}
		PGPGetPrefStringBuffer(	prefRef,
								kPGPPrefComments,
								commentString.GetBufferSize(),
								commentString);
		PGPClosePrefFile(prefRef);
#endif
		// We need the client prefs for the rest of the options
		pgpErr = PGPOpenClientPrefs(	PGPGetContextMemoryMgr(inContext),
										&prefRef);
		PGPThrowIfPGPErr_(pgpErr);
		
		// If there was no admin comment, add the client comment
		if (commentString.GetLength() == 0) {
			PGPGetPrefStringBuffer(	prefRef,
									kPGPPrefComment,
									commentString.GetBufferSize(),
									commentString);
		}

		pgpErr = PGPNewOptionList(inContext, & optionList);
		PGPThrowIfPGPErr_(pgpErr);
		if (commentString.GetLength() != 0) {
			PGPAppendOptionList(	optionList,
									PGPOCommentString(	inContext,
														commentString),
									PGPOLastOption(inContext));
		}
								
		// Add version string
		PGPAppendOptionList(	optionList,
								PGPOVersionString(	inContext,
													pgpVersionHeaderString),
								PGPOLastOption(inContext));
		
		// Add rest of options
		if ((inOptions & encodeOptions_Encrypt)
		&& (! (inOptions & encodeOptions_ConvEncrypt))) {
			PGPAppendOptionList(	optionList,
									PGPOEncryptToKeySet(	inContext,
															inRecipients),
									PGPOLastOption(inContext));

			PGPCipherAlgorithm *	allowedAlgorithms = nil;
			PGPSize					bufSize;
			
			PGPGetPrefData(	prefRef,
							kPGPPrefAllowedAlgorithmsList,
							&bufSize,
							&allowedAlgorithms);
			if (allowedAlgorithms != nil) {
				PGPAppendOptionList(
					optionList,
					PGPOPreferredAlgorithms(	inContext,
												allowedAlgorithms,
					bufSize / sizeof( PGPCipherAlgorithm ) ),
					PGPOLastOption(inContext));
				PGPDisposePrefData(	prefRef,
									allowedAlgorithms);
			}
		}
		if (inOptions & encodeOptions_ConvEncrypt) {
			PGPUInt32	preferredAlgorithm;
			
			pgpErr = PGPGetPrefNumber(	prefRef,
										kPGPPrefPreferredAlgorithm,
										&preferredAlgorithm);
			if (pgpErr != kPGPError_NoErr) {
				preferredAlgorithm = kPGPCipherAlgorithm_CAST5;
			}
			PGPAppendOptionList(	
							optionList,
							PGPOConventionalEncrypt(inContext, 
								PGPOPassphrase(inContext,
										inConvEncryptPassphrase),
								PGPOLastOption(inContext)),
							PGPOCipherAlgorithm(	inContext,
								(PGPCipherAlgorithm) preferredAlgorithm),
							PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_Sign) {
			PGPAppendOptionList(
				optionList,
				PGPOSignWithKey(	inContext,
									inSigningKey,
									(inUsePassphrase) ? 
										PGPOPassphrase(
											inContext,
											(char *) inPassBuffer) :
										PGPOPasskeyBuffer(	inContext,
															inPassBuffer,
															inPassKeySize),
									PGPOLastOption(inContext)),
				PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_TextOutput) {
			PGPAppendOptionList(	optionList,
									PGPOArmorOutput(inContext, true),
									PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_TreatInputAsText) {
			PGPAppendOptionList(	optionList,
									PGPODataIsASCII(inContext, true),
									PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_ClearSign) {
			PGPAppendOptionList(	optionList,
									PGPOClearSign(inContext, true),
									PGPOLastOption(inContext));
		}
		PGPClosePrefFile(prefRef);
	}
	
	catch (...) {
		if (PGPPrefRefIsValid(prefRef)) {
			PGPClosePrefFile(prefRef);
		}
		throw;
	}
	
	return optionList;
}



	void
CPGPEncoderDecoder::SetPassbufferCache(
	Boolean			inSigning,
	PGPKeyRef	 	inKey,
	PGPBoolean		inUsePassphrase,
	void *			inBuffer,
	PGPSize			inPasskeySize)
{
	PGPPrefRef				prefRef = kInvalidPGPPrefRef;
	
	try {
		CSignPassphraseCache *	theCache;
		PGPBoolean				cacheEnabled;
		PGPUInt32				cacheDuration = 0;
		PGPError				pgpErr;
		
		// Get the cache values
		pgpErr = PGPOpenClientPrefs(PGPGetContextMemoryMgr(PGPGetKeyContext(inKey)),
									&prefRef);
		PGPThrowIfPGPErr_(pgpErr);
		
		if (inSigning) {
			theCache = sSigningPassphraseCache;
			pgpErr = PGPGetPrefBoolean(	prefRef,
										kPGPPrefSignCacheEnable,
										&cacheEnabled);
			PGPThrowIfPGPErr_(pgpErr);
			if (cacheEnabled) {
				pgpErr = PGPGetPrefNumber(	prefRef,
											kPGPPrefSignCacheSeconds,
											&cacheDuration);
				PGPThrowIfPGPErr_(pgpErr);
			}
		} else {
			theCache = sDecryptPassphraseCache;
			pgpErr = PGPGetPrefBoolean(	prefRef,
										kPGPPrefDecryptCacheEnable,
										&cacheEnabled);
			PGPThrowIfPGPErr_(pgpErr);
			if (cacheEnabled) {
				pgpErr = PGPGetPrefNumber(	prefRef,
											kPGPPrefDecryptCacheSeconds,
											&cacheDuration);
				PGPThrowIfPGPErr_(pgpErr);
			}
		}
		
		theCache->SetCacheSeconds(cacheDuration);
		if (inUsePassphrase) {
			theCache->RememberPassphrase((char *) inBuffer, inKey);
		} else {
			theCache->RememberPasskey(inBuffer, inPasskeySize, inKey);
		}
	}
	
	catch (...) {
	}

	if (PGPPrefRefIsValid(prefRef)) {
		PGPClosePrefFile(prefRef);
	}
}


	void
CPGPEncoderDecoder::ShowError(
	PGPError	inErr)
{
	CString		theErrorString;
	
	PGPGetClientErrorString(	inErr,
								theErrorString.GetBufferSize() + 1,
								theErrorString);
	WarningAlert(	kWAStopAlertType,
					kWAOKStyle,
					STRx_PGPEncoderDecoderErrorStrings,
					kErrorStringID,
					CString(inErr),
					theErrorString);
}



	PGPError
CPGPEncoderDecoder::PGPDecodeEventHandler(
	PGPContextRef	inContext,
	PGPEvent *		inEventP,
	PGPUserValue	inUserValue)
{
	PGPError				result = kPGPError_NoErr;
	static PGPInt32			sSection;
	static Boolean			sAllocated;
	static PGPUInt32		sKeyIDCount = 0;
	static PGPKeySetRef		sRecipientKeySet = kInvalidPGPKeySetRef;
	static PGPKeyID *		sKeyIDList = nil;
	char *					tempPassphrase = nil;
	SDecodeSectionHandle	decodeSectionH = 
							(SDecodeSectionHandle) ((SDecodeData *)
									inUserValue)->handle;
	
	try {
		AnimateCursor(((SDecodeData *) inUserValue)->cursor);
		
		switch (inEventP->type) {
			case kPGPEvent_InitialEvent:
			{
				sSection = -1;
				sAllocated = false;
			}
			break;
			
			
			case kPGPEvent_EndLexEvent:
			{
				sAllocated = false;
				if (PGPKeySetRefIsValid(sRecipientKeySet)) {
					PGPFreeKeySet(sRecipientKeySet);
					sRecipientKeySet = kInvalidPGPKeySetRef;
				}
				if (IsntNull(sKeyIDList)) {
					delete[] sKeyIDList;
					sKeyIDList = nil;
				}
			}
			break;
			
			
			case kPGPEvent_AnalyzeEvent:
			{
				if (inEventP->data.analyzeData.sectionType ==
				kPGPAnalyze_Encrypted) {
					if (! sAllocated) {
						sSection++;
						// Unlock long enough to grow the handle
						::HUnlock((Handle) decodeSectionH);
						::SetHandleSize(	(Handle) decodeSectionH,
											(sSection + 1)
												* sizeof(SDecodeSection));
						::HLock((Handle) decodeSectionH);
						(*decodeSectionH)[sSection].buf = nil;
						(*decodeSectionH)[sSection].addKeysSet =
							kInvalidPGPKeySetRef;
						(*decodeSectionH)[sSection].sigDataValid = false;
						sAllocated = true;
					}
					(*decodeSectionH)[sSection].encrypted = true;
				}
			}
			break;
			
			
			case kPGPEvent_RecipientsEvent:
			{
				PGPUInt32	numFoundKeys = 0;
				
				if (PGPKeySetRefIsValid(inEventP->
				data.recipientsData.recipientSet)) {
					result = PGPIncKeySetRefCount(inEventP->
									data.recipientsData.recipientSet);
					if (IsntPGPError(result)) {
						sRecipientKeySet = inEventP->
							data.recipientsData.recipientSet;
						if (IsntNull(inEventP->
						data.recipientsData.keyIDArray)) {
							sKeyIDCount = inEventP->
								data.recipientsData.keyCount;
							sKeyIDList = new PGPKeyID[sKeyIDCount];
							if (sKeyIDList == nil) {
								result = kPGPError_OutOfMemory;
							} else {
								pgpCopyMemory(
									inEventP->data.recipientsData.keyIDArray,
									sKeyIDList,
									sKeyIDCount * sizeof(PGPKeyID));
							}
						}
					}
				}
			}
			break;
			
			
			case kPGPEvent_PassphraseEvent:
			{
				PGPKeyRef			keyRef = kInvalidPGPKeyRef;
				CSecureMemory		passbuffer(PGPGetContextMemoryMgr(
										inContext), 256);
				PGPSize				passkeySize;
				PGPBoolean			usePassphrase;

				if (! ((SDecodeData *) inUserValue)->useCache) {
					sDecryptPassphraseCache->Forget();
				}
				
				if (inEventP->data.passphraseData.fConventional) {
					usePassphrase = true;
					result = PGPConventionalDecryptionPassphraseDialog(
									inContext,
									PGPOUIOutputPassphrase(
										inContext,
										&tempPassphrase),
									PGPOLastOption(inContext));
					PGPThrowIfPGPErr_(result);
					strcpy(	(char *) passbuffer.mMemory,
							tempPassphrase);
					PGPFreeData(tempPassphrase);
					tempPassphrase = nil;
				} else {
					// Get passphrase
					if (! sDecryptPassphraseCache->GetPassphraseOrPasskey(
						inEventP->data.passphraseData.keyset,
						passbuffer.mMemory, &usePassphrase, &passkeySize,
						&keyRef)) {
						PGPKeySetRef	newKeys = kInvalidPGPKeySetRef;
						
						usePassphrase = true;
						result = PGPClientDecryptionPassphraseDialog(
										inContext,
										sTLSContext,
										nil,
										sRecipientKeySet,
										sKeyIDList,
										sKeyIDCount,
										kInvalidPGPOptionListRef,
										&tempPassphrase,
										&keyRef,
										&newKeys);
						if (result != kPGPError_KeyUnusableForDecryption) {
							PGPThrowIfPGPErr_(result);
							if (PGPKeySetRefIsValid(newKeys)) {
								if (! sAllocated) {
									sSection++;
									// Unlock long enough to grow the handle
									::HUnlock((Handle) decodeSectionH);
									::SetHandleSize(
										(Handle) decodeSectionH,
										(sSection + 1) * sizeof(
											SDecodeSection));
									::HLock((Handle) decodeSectionH);
									(*decodeSectionH)[sSection].buf = nil;
									(*decodeSectionH)[sSection].addKeysSet =
										kInvalidPGPKeySetRef;
									(*decodeSectionH)[sSection].sigDataValid =
										false;
									(*decodeSectionH)[sSection].encrypted =
										false;
									sAllocated = true;
								}
								if (PGPKeySetRefIsValid(
								(*decodeSectionH)[sSection].addKeysSet)) {
									result = PGPAddKeys(
												(*decodeSectionH)[sSection].
													addKeysSet,
												newKeys);
									PGPThrowIfPGPErr_(result);
									result = PGPCommitKeyRingChanges(
												newKeys);
									PGPThrowIfPGPErr_(result);
									PGPFreeKeySet((*decodeSectionH)[sSection].
										addKeysSet);
								}
								(*decodeSectionH)[sSection].addKeysSet = 
									newKeys;
							}
							strcpy(	(char *) passbuffer.mMemory,
									tempPassphrase);
							PGPFreeData(tempPassphrase);
							tempPassphrase = nil;
						} else {
							usePassphrase = false;
							result = PGPReconstitutionDialog(
								keyRef,
								((SDecodeData *) inUserValue)->defaultKeys,
								kInvalidPGPtlsContextRef,
								(PGPByte **) &tempPassphrase,
								&passkeySize);
							PGPThrowIfPGPErr_(result);
							pgpCopyMemory(	tempPassphrase,
											passbuffer.mMemory,
											passkeySize);
							PGPFreeData(tempPassphrase);
							tempPassphrase = nil;
						}
						SetPassbufferCache(	false,
											keyRef,
											usePassphrase,
											passbuffer.mMemory,
											passkeySize);
					}
				}
					
				// Add the passphrase to the job
				result = PGPAddJobOptions(
							inEventP->job,
							(usePassphrase) ?
								PGPOPassphrase( inContext,
												(char *) passbuffer.mMemory) :
								PGPOPasskeyBuffer(	inContext,
													passbuffer.mMemory,
													passkeySize),
							PGPOLastOption(inContext));
				PGPThrowIfPGPErr_(result);
			}
			break;
			
			
			case kPGPEvent_SignatureEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1)
											* sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}

				PGPBoolean		addedKeySetToJob;
				PGPKeySetRef	newKeySet;
				
				result = PGPSignerKeyLookup(	inContext,
												inEventP,
												kInvalidPGPKeySetRef,
												&addedKeySetToJob,
												&newKeySet);
				PGPThrowIfPGPErr_(result);
				if (addedKeySetToJob) {
					if (PGPKeySetRefIsValid((*decodeSectionH)[sSection].
					addKeysSet)) {
						result = PGPAddKeys(
									(*decodeSectionH)[sSection].addKeysSet,
									newKeySet);
						PGPThrowIfPGPErr_(result);
						result = PGPCommitKeyRingChanges(newKeySet);
						PGPThrowIfPGPErr_(result);
						PGPFreeKeySet((*decodeSectionH)[sSection].addKeysSet);
					}
					(*decodeSectionH)[sSection].addKeysSet = newKeySet;
				} else {
					(*decodeSectionH)[sSection].sigData = inEventP->
							data.signatureData;
					(*decodeSectionH)[sSection].sigDataValid = true;
				}
			}
			break;
			
			
			case kPGPEvent_OutputEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1)
											* sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}
				if (inEventP->data.outputData.forYourEyesOnly) {
					((SDecodeData *) inUserValue)->FYEO = true;
				}
				result = PGPAddJobOptions(	inEventP->job,
											PGPOAllocatedOutputBuffer(
												inContext,
												&(*decodeSectionH)[sSection].
														buf,
												MAX_PGPSize,
												&(*decodeSectionH)[sSection].
														size),
											PGPOLastOption(inContext));
			}
			break;
			
			
			case kPGPEvent_KeyFoundEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1) * sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}

				if (PGPKeySetRefIsValid(
				(*decodeSectionH)[sSection].addKeysSet)) {
					result = PGPAddKeys(
								inEventP->data.keyFoundData.keySet,
								(*decodeSectionH)[sSection].addKeysSet);
					PGPThrowIfPGPErr_(result);
					result = PGPCommitKeyRingChanges(
								(*decodeSectionH)[sSection].addKeysSet);
					PGPThrowIfPGPErr_(result);
				} else {
					result = PGPIncKeySetRefCount(
								inEventP->data.keyFoundData.keySet);
					PGPThrowIfPGPErr_(result);
					(*decodeSectionH)[sSection].addKeysSet = inEventP->
						data.keyFoundData.keySet;
				}
			}
			break;
		}
	}
	
	catch (...) {
		sAllocated = false;
		if (tempPassphrase != nil) {
			PGPFreeData(tempPassphrase);
		}
		if (PGPKeySetRefIsValid(sRecipientKeySet)) {
			PGPFreeKeySet(sRecipientKeySet);
			sRecipientKeySet = kInvalidPGPKeySetRef;
		}
		if (IsntNull(sKeyIDList)) {
			delete[] sKeyIDList;
			sKeyIDList = nil;
		}
	}
	
	return result;
}
