//////////////////////////////////////////////////////////////////////////////
// PGPdiskRequestQueue.cpp
//
// Definition of class PGPdiskRequestQueue.
//////////////////////////////////////////////////////////////////////////////

// $Id: PGPdiskRequestQueue.cpp,v 1.4.10.1 1999/08/19 08:35:16 nryan Exp $

// Copyright (C) 1998 by Network Associates, Inc.
// All rights reserved.

#define	__w64
#include <vdw.h>

#include "Required.h"

#include "Globals.h"
#include "PGPdiskRequestQueue.h"
#include "PGPdiskWorkerThread.h"
#include "PGPdisk.h"


////////////
// Constants
////////////

const PGPUInt16 kPGPdiskDefaultPRQDepth = 32;


////////////////////////////////////////////////////
// Class PGPdiskRequestQueue public member functions
////////////////////////////////////////////////////

// The PGPdiskRequestQueue default constructor.

PGPdiskRequestQueue::PGPdiskRequestQueue() 
	: mPRQAllocator(kPGPdiskDefaultPRQDepth)
{
	mInitErr = mPRQAllocator.mInitErr;
}

// The PGPdiskRequestQueue destructor.

PGPdiskRequestQueue::~PGPdiskRequestQueue()
{
}

// Count returns the number of request in the queue.

PGPUInt32 
PGPdiskRequestQueue::Count()
{
	return mPRQList.Count();
}

// QueueIrpForProcessing queues an IRP for processing.

DualErr 
PGPdiskRequestQueue::QueueIrpForProcessing(PIRP pIrp)
{
	DualErr		derr;
	KIrp		I(pIrp);
	PRQRequest	*pReq;

	pgpAssertAddrValid(pIrp, IRP);

	// Get memory for request structure.
	derr = mPRQAllocator.Allocate(&pReq);

	// Initialize the request and place it in the queue.
	if (derr.IsntError())
	{
		pReq->op	= kPRQOp_ProcessIrp;
		pReq->pIrp	= pIrp;

		I.MarkPending();
		Push(pReq);
	}

	return derr;
}

// QueueAsyncCallback queues an async callback request.

DualErr 
PGPdiskRequestQueue::QueueAsyncCallback(
	PEZCALLBACK	callback, 
	PGPUInt32	refData)
{
	DualErr		derr;
	PRQRequest	*pReq;

	pgpAssertAddrValid(callback, VoidAlign);

	// Get memory for request structure.
	derr = mPRQAllocator.Allocate(&pReq);

	// Initialize the request and place it in the queue.
	if (derr.IsntError())
	{
		pReq->op		= kPRQOp_AsyncCallback;
		pReq->callback	= callback;
		pReq->refData	= refData;

		Push(pReq);
	}

	return derr;
}

// QueueSyncCallback queues a ssync callback request.

DualErr 
PGPdiskRequestQueue::QueueSyncCallback(
	PEZCALLBACK	callback, 
	PGPUInt32	refData, 
	PKEVENT		pEvent)
{
	DualErr		derr;
	PRQRequest	*pReq;

	pgpAssertAddrValid(callback, VoidAlign);
	pgpAssertAddrValid(pEvent, KEVENT);

	// Get memory for request structure.
	derr = mPRQAllocator.Allocate(&pReq);

	// Initialize the request and place it in the queue.
	if (derr.IsntError())
	{
		pReq->op			= kPRQOp_SyncCallback;
		pReq->callback		= callback;
		pReq->refData		= refData;
		pReq->pSyncEvent	= pEvent;

		Push(pReq);
	}

	return derr;
}

// ReturnPRQ is called to return a processed PRQRequest to the memory pool.

void 
PGPdiskRequestQueue::ReturnPRQ(PRQRequest *pReq)
{
	pgpAssertAddrValid(pReq, PRQRequest);

	mPRQAllocator.Free(pReq);
}

// Pop pops a request from the bottom of the queue. TRUE if successful, FALSE
// otherwise.
 
PRQRequest * 
PGPdiskRequestQueue::Pop()
{
	PRQRequest *pReq;

	// If list is empty, bail.
	if (mPRQList.IsEmpty())
		return NULL;

	// Pop an item off the queue.
	mPRQList.PublicLock();
	pReq = mPRQList.RemoveTail();
	mPRQList.PublicUnlock();

	return pReq;
}


/////////////////////////////////////////////////////
// Class PGPdiskRequestQueue private member functions
/////////////////////////////////////////////////////

// Push pushes a new request on top of the queue. TRUE if successful, FALSE
// otherwise.

void 
PGPdiskRequestQueue::Push(PRQRequest *pReq)
{
	pgpAssertAddrValid(pReq, PRQRequest);

	// Insert the request into the queue.
	mPRQList.PublicLock();
	mPRQList.InsertHead(pReq);
	mPRQList.PublicUnlock();
}
