/*____________________________________________________________________________
	Copyright (C) 2000 Pretty Good Privacy, Inc.
	All rights reserved.
	

	$Id: RichTextHandling.c,v 1.15.8.2.2.8 2000/08/09 01:23:09 build Exp $
____________________________________________________________________________*/

/*::: MODULE OVERVIEW :::::::::::::
Purpose is to encapsulate low-level functionality for handling Notes 
rich-text. To use this functionality, the developer should include the 
LibRichText.h header file in her own module.

--- suggested enhancements ---
11/10/98 PR
+ Decouple the rich-text item information members in the ItemInfo structure 
  from the linked-list implementation of stringing such items together. 
  My suggested approach is to create a new structure called something like 
  ItemNode that has two members: an ItemInfo structure and a pointer to 
  the next ItemNode in the list.
+ A currently un-addressed problem in this implementation is that the content 
  of rich-text items become locked in memory for access purposes. However 
  the items are never then unlocked until the rich-text field is discarded. 
  This is wasteful of memory resources. My suggested solution is to 
  implement a mechanism for saving and restoring state information about 
  the memory locks on rich-text content associated with a given rich-text 
  context. An idea of the direction I would take can be gleaned from the 
  sidelined commentary in the f_TestStringStart() function and in the 
  corresponding, currently unused MemLockStateInfo structure defined in the 
  RichTextHandling.h header file. Such a mechanism would provide internal, 
  and possibly external, logic a simple means with which to conserve memory 
  resources.

11/26/98 PR
Perhaps some reference-count locking mechanism should be incorporated into 
the ItemInfo structure so that items can be locked and "unlocked" freely with 
no risk of invalidating unknown pointers which rely on the formerly locked 
item information. The current set-up poses some risks. See the in-line 
documentation in us_VirtualAppend() for a real example a risk being taken 
which could be avoided if we had some reference-count mechanism.

I wonder... Would OSLockBlock() and OSUnlockBlock() already provide us with 
a refrence-count mechanism that we could utilize?

11/30/98 PR
The current implementation of "absolute offsets," including ActualFrontier, 
include the length of the TYPE_COMPOSITE beginning to rich-text items. Better 
to make absolute offsets completely independent of the item structure against 
which the offset is to be measured.

11/30/98 PR
Based on the logic specified in eus_ReplaceRtfSpanText(), we may be able to 
simplify matters by getting rid of semi-virtuality altogether. Since 
virtuality can always be adjusted with little performance hit if the 
adjustments are made only at the end of an item, there seems to be little 
need for a semi-virtuality that is little more than a mechanism for adjusting 
virtualized content at the end of virtuality.
	12/8/98 PR: comment
	But semi-virtuality may still be useful for streaming CD records together 
	without having to worry about item limits -- ad-hoc buffers, for example? 
	I don't really like allocating a full segment if all I need is some space 
	to string a few CD records together. But then again, I want to isolate 
	this module and allow it to be accessible from a higher level, meaning 
	that I have to give up some efficiency & fine-tuning possibilities.

2/23/00 PR: The NSFItemConvertToText and NSFItemConvertValueToText APIs might 
	be a much simpler way to extract text from rich-text items than what's 
	currently implemented here. Much testing is required to find out: 
	end-of-line handling and hotspot-text handling, for prime examples.

--- revision history ---------
2/23/00 Version 1.1.2 Paul Ryan
+ enhancements to handling of the rich-text-field information structure
+ improved initial-context checking in us_CompileActuality() such that 
  procedure fails if specified item is not in fact a rich-text field
+ Extended us_VirtualizeRestOfRtf() such that the target virtuality does not 
  have to belong to the rich-text context structure holding the source 
  actuality. The extenstion caused the signature to eus_InsertSubformTop() 
  to change as well.
+ documentation adjustments

9/12/99 Version 1.1 Paul Ryan
+ fixed bugs in eus_ReplaceRtfWithStandardText() so that messages exceeding 
  64K are handled properly
+ Enhanced eus_ReplaceRtfSpanText() to handle translation of PGP ASCII-Armor 
  blocks greaer than 64K in size. Process led to creation of 
  us_AppendTextToVirtuality() and concomitant adjustment of 
  eus_ReplaceRtfWithStandardText().

1/16/99 Version 1.0 Paul Ryan
::::::::::::::::::::::::::::::::::::*/

#include "RichTextHandling.h"


const BOOL  ei_FOREVER = TRUE;
const DWORD  eul_ERR_FAILURE = 0xFFFFFFFF;	//function failure code for 
				//	eul_GetTextSpanText(), etc.

#define muc_SIGTYPE_LSIG  '\0'
#define muc_SIGTYPE_WSIG  (BYTE) '\xFF'	//for some reason '\xFF' by itself 
				//	produces a maxed-out 'int', all F's
static const char mc_LINEFEED_RICHTEXT = '\0';
static const DWORD  mul_MAXLEN_RICH_TEXT_ITEM = MAXONESEGSIZE, 
					mul_COMFY_LIMIT = 0x0400 * 40, 
					mul_STRETCH_LIMIT = 0x0400 * 55, 
					mul_FRONTIER_NO_MORE = 0xFFFFFFFF;
static const CDHOTSPOTEND  mt_HOTSPOT_END = {{SIG_CD_HOTSPOTEND, 
													sizeof( CDHOTSPOTEND)}};
static const CDPARAGRAPH  mt_PARAGRAPH = {{SIG_CD_PARAGRAPH, 
													sizeof( CDPARAGRAPH)}};
static const WORD  mus_MAX_STYLE_ID = 0xFFFF, 
					mus_LENLIMIT_CDTEXT_REC = 20 * 0x400;


/** eus_InitializeRtfContext( ***
Purpose is to initialize a rich-text-field context structure the caller needs 
to use to employ the functionality of this module for handling Lotus Notes 
rich text. The caller chooses whether to engage a complete initialization 
based on an existing field or to simply initialize a context with which 
free-form rich-text content may be constructed. The caller is responsible for 
disposal of the resources tracked by the rich-text context structure, via a 
call to ef_FreeRtfContext().

--- parameters & return ----
h_NOTE: handle to the open note containing the rich-text field which should 
	be described by this procedure
pc_ITMNM: pointer to the string telling the name of the rich-text field to be 
	described by this procedure
pt_cursor: Optional Output. Pointer to the variable in which to store a 
	rich-text cursor pointing to the first CD record in the rich-text field 
	to be manipulated. Ignored if passed in null.
pt_context: Output. Pointer to the context structure to be initialized to 
	describe the specified field. This context structure is required input to 
	most of the public interfaces to this module and so must be maintained by 
	the calling procedure.
RETURN: eus_SUCCESS if no error occured. !eus_SUCCESS if any parameters are 
	invalid. The Notes API error code otherwise. The procedure will always 
	be successful if the context structure is not being initialized to 
	describe an existing rich-text field.

--- revision history -------
2/23/00 PR: documentation improvement
2/11/99 PR: created			*/
STATUS eus_InitializeRtfContext( NOTEHANDLE  h_NOTE, 
									char  pc_ITMNM[], 
									CdRecordCursor *const  pt_cursor, 
									RtfTrackingInfo *const  pt_context)	{
	ItemInfo *  pt_item, * pt_itm;
	UINT  ui_blankItems = NULL;
	STATUS  us_err;
	BOOL  f_failure = FALSE;

	if (!( h_NOTE && pc_ITMNM && pt_context))
		return !eus_SUCCESS;

	memset( pt_context, NULL, sizeof( RtfTrackingInfo));
	if (pt_cursor)
		memset( pt_cursor, NULL, sizeof( CdRecordCursor));

	//Fill out the context's Actuality list. If the specified rich-text field 
	//	doesn't exist, short-circuit with success.
	if (us_err = us_CompileActuality( h_NOTE, pc_ITMNM, &pt_item))
		return us_err;
	if (!pt_item)
		return eus_SUCCESS;

	//initialize the cursor to point to the beginning of the first rich-text 
	//	item's content
	if (pt_cursor)
		if (!f_ConstructStartCursor( NULL, pt_item, pt_cursor, NULL))
			goto errJump;

	pt_itm = pt_item;
	while (pt_itm->ul_length == sizeof( WORD))	{
		ui_blankItems++;
		if (!( pt_itm = pt_itm->pt_next))
			goto errJump;
	}

	pt_context->pt_Actuality = pt_item;
	pt_context->ul_ActualFrontier = sizeof( WORD) * (ui_blankItems + 1);

	return eus_SUCCESS;

errJump:
	ClearItemList( &pt_item);

	return !eus_SUCCESS;
} //eus_InitializeRtfContext(


/** ef_CursorToAttachmentHotspot( ***
Purpose is to find and provide information about the next file-attachment in 
the rich-text field being navigated by the passed-in cursor.

--- parameters & return ----
pt_cursor: Input & Output. Pointer to the rich-text cursor positioned to the 
	point at which the search for the next file-attachment hotspot should 
	commence. This cursor is then used when advancing through CD records. If 
	the procedure is successful, the cursor will finish and the end-hotspot 
	CD record if a file-attachment hotspot is found, else it will be 
	nullified because it has travelled through the entire rich-text field if 
	no file-attachment hotspot was found.
pt_CONTEXT: Optional. Pointer to environment information about the rich-text 
	field being navigated. If passed in null, the rich-text cursor being used 
	will be unable to move from virtuality to actuality.
f_BUMP_FORWARD_FIRST: Flag telling whether the passed-in starting cursor 
	should be moved to the next CD record before initiating the search for 
	file-attachment hotspots. Usually when starting at the beginning of the 
	rich-text field, the flag will be set to FALSE so that the first CD 
	record may be checked as to whether it's a file-attachment hotspot. 
	Ensuing calls would then set the flag to TRUE to move to the next CD 
	record since the current CD record (an end-hotspot record) is already 
	known.
pt_span: Optional Output. Pointer to the structure to hold information about 
	where a found file-attachment hotspot begins and ends within the 
	rich-text field. Feature is ignored if the pointer passed in is null.
ppc_FileNmAttachment: Optional Output. Pointer to the string pointer that 
	should be set to the name Notes is using for the file attachment 
	associated with the hotspot. Feature is ignored if the pointer passed in 
	is null. The pointed-to memory is owned by the note; the caller should 
	not free the pointer.
ppc_FileNmOriginal: Optional Output. Pointer to the string pointer that 
	should be set to the name of the file the user attached to the note. 
	Feature is ignored if the pointer passed in is null. The pointed-to 
	memory is owned by the note; the caller should not free the pointer.
RETURN: TRUE if no error situation was encountered; FALSE otherwise.

--- side effect --------
The rich-text environment information pointed to through pt_cursor or 
pt_CONTEXT may be adjusted due to cursor advancement. If the enhancement 
suggested below is implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/10/98 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution.

--- revision history -------
12/11/98 PR: created		*/
BOOL ef_CursorToAttachmentHotspot( 
								CdRecordCursor *const  pt_cursor, 
								const RtfTrackingInfo *const  pt_CONTEXT, 
								const BOOL  f_BUMP_FORWARD_FIRST, 
								CdRecordSpan *const  pt_span, 
								const char * *const  ppc_FileNmAttachment, 
								const char * *const  ppc_FileNmOriginal)	{
	const WORD  us_FILE_HOTSPOT_WEIRD = 10;

	WORD  us;

	if (!( pt_cursor->puc_location && pt_cursor->pt_item && 
									pt_cursor->us_recLength && (pt_CONTEXT ? 
									(BOOL) pt_CONTEXT->pt_Actuality : TRUE)))
		return FALSE;

	//as applicable, default the passed-in span structure to all null and the 
	//	file-name pointers to null
	if (pt_span)
		memset( pt_span, (char) NULL, sizeof( CdRecordSpan));
	if (ppc_FileNmAttachment)
		*ppc_FileNmAttachment = NULL;
	if (ppc_FileNmOriginal)
		*ppc_FileNmOriginal = NULL;

	//If requested, do an intial advance of the cursor to the next CD record. 
	//	If the rich-text field has been exhausted, return that the 
	//	procedure has run error free, but flag that no file-attachment 
	//	hotspot was found.
	if (f_BUMP_FORWARD_FIRST)	{
		AdvanceCdCursor( pt_cursor, pt_CONTEXT, NULL, NULL);
		if (!pt_cursor->pt_item)	{
			pt_cursor->puc_location = NULL;
			return TRUE;
		}
	}

	//Loop record-by-record until the beginning of a file-attachment hotspot 
	//	is found. The "weird" file-hotspot types occurs when an attachment 
	//	is copied to a new document via the "inherit entire selected document 
	//	into rich text field" form property [and maybe via 
	//	@Command( [MailForward]) too?].
	while (!( LOBYTE( SIG_CD_HOTSPOTBEGIN) == *pt_cursor->puc_location && 
										((us = ((CDHOTSPOTBEGIN *) 
										pt_cursor->puc_location)->Type) == 
										HOTSPOTREC_TYPE_FILE || us == 
										us_FILE_HOTSPOT_WEIRD)))	{
		//Advance the cursor to the next CD record. If the rich-text field 
		//	has been exhausted, return that the procedure has run error free, 
		//	but flag that no file-attachment hotspot was found.
		AdvanceCdCursor( pt_cursor, pt_CONTEXT, NULL, NULL);
		if (!pt_cursor->pt_item)	{
			pt_cursor->puc_location = NULL;
			return TRUE;
		}
	}

	//as applicable, set the span's beginning-coordinate information and set 
	//	the filename pointers
	if (pt_span)
		pt_span->t_cursorBegin = *pt_cursor;
	if (ppc_FileNmAttachment)
		*ppc_FileNmAttachment = pt_cursor->puc_location + 
													sizeof( CDHOTSPOTBEGIN);
	if (ppc_FileNmOriginal)
		*ppc_FileNmOriginal = pt_cursor->puc_location + 
											sizeof( CDHOTSPOTBEGIN) + 
											strlen( pt_cursor->puc_location + 
											sizeof( CDHOTSPOTBEGIN)) + 1;

	//at least to make sure we have a bona-fide hotspot, loop 
	//	record-by-record until the end of the hotspot is found
	do	{
		//advance the cursor to the next CD record
		AdvanceCdCursor( pt_cursor, pt_CONTEXT, NULL, NULL);

		//If no more CD records are available in the rich-text field or if a 
		//	begin-hotspot CD record is encountered, we have a weird situation 
		//	where another begin-hotspot CD record is found before a 
		//	corresponding end-hotspot CD record. This can be a valid 
		//	configuration, but we aren't sophisticated enough yet to handle 
		//	the situation.
		if (!pt_cursor->pt_item || LOBYTE( SIG_CD_HOTSPOTBEGIN) == 
													*pt_cursor->puc_location)
			return FALSE;
	} while (LOBYTE( SIG_CD_HOTSPOTEND) != *pt_cursor->puc_location);

	//set the span's ending-coordinate information
	if (pt_span)
		pt_span->t_cursorEnd = *pt_cursor;

	//return that the procedure executed successfully
	return TRUE;
} //ef_CursorToAttachmentHotspot(


/** eus_ResetAttachHotspotNames( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/15/98 PR: created		*/
//DOC!!
STATUS eus_ResetAttachHotspotNames( const char  pc_ATTACHNM[], 
									const char  pc_FILENM[], 
									const CdRecordCursor  t_CURSOR, 
									RtfTrackingInfo *const  pt_context)	{
	WORD  us_lenOrigRec, us;
	char * pc_AttachNmOrig, * pc_FileNmOrig;
	UINT  ui_lenAttachNm;
	BYTE * puc;
	CDHOTSPOTBEGIN * pt;
	STATUS  us_error;

	if (!( (pc_ATTACHNM || pc_FILENM) && t_CURSOR.puc_location && 
								t_CURSOR.pt_item && t_CURSOR.us_recLength && 
								*t_CURSOR.puc_location == LOBYTE( 
								SIG_CD_HOTSPOTBEGIN) && pt_context))
		return !eus_SUCCESS;

//PGP development shortcut: skip all the complexity of doing this with a 
//	virtual start point
if (t_CURSOR.pt_item->i_type != mi_ACTUAL || pt_context->ul_ActualFrontier == 
														mul_FRONTIER_NO_MORE)
	return !eus_SUCCESS;

	if (us_error = us_VirtualizeThruActualLocation( t_CURSOR.puc_location, 
														pt_context, NULL))
		return us_error;

	us_lenOrigRec = ((CDHOTSPOTBEGIN *) t_CURSOR.puc_location)->Header.Length;

	//get a pointer to where we can append an adjusted begin-hotspot record
	pc_AttachNmOrig = t_CURSOR.puc_location + sizeof( CDHOTSPOTBEGIN);
	us = sizeof( CDHOTSPOTBEGIN) + (ui_lenAttachNm = (pc_ATTACHNM ? strlen( 
										pc_ATTACHNM) : strlen( 
										pc_AttachNmOrig)) + 1) + (pc_FILENM ? 
										strlen( pc_FILENM) : strlen( 
										pc_FileNmOrig = pc_AttachNmOrig + 
										strlen( pc_AttachNmOrig) + 1)) + 1;
	if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, us, &puc))
		return us_error;

	memcpy( puc, t_CURSOR.puc_location, sizeof( CDHOTSPOTBEGIN));

	(pt = (CDHOTSPOTBEGIN *) puc)->Header.Length = us;
	pt->DataLength = us - sizeof( CDHOTSPOTBEGIN);
	pt->Flags &= 0x0000FFFF;		//relativize!!

	strcpy( puc += sizeof( CDHOTSPOTBEGIN), pc_ATTACHNM ? pc_ATTACHNM : 
															pc_AttachNmOrig);
	strcpy( puc + ui_lenAttachNm, pc_FILENM ? pc_FILENM : pc_FileNmOrig);

	pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + us;
	pt_context->ul_ActualFrontier += us_lenOrigRec + us_lenOrigRec % 2;

	return eus_SUCCESS;
} //eus_ResetAttachHotspotNames(


/** eus_InsertTextAtHotspotEnd( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/15/98 PR: created		*/
//DOC!!
STATUS eus_InsertTextAtHotspotEnd( const char *const  PC, 
									const CdRecordCursor  t_CURSOR, 
									RtfTrackingInfo *const  pt_context)	{
	DWORD  ul;
	BYTE * puc;
	CDTEXT  t = {{SIG_CD_TEXT}};
	STATUS  us_error;

	if (!( PC && t_CURSOR.puc_location && t_CURSOR.pt_item && 
												t_CURSOR.us_recLength && 
												*t_CURSOR.puc_location == 
												LOBYTE( SIG_CD_HOTSPOTEND)))
		return !eus_SUCCESS;

//PGP development shortcut: skip all the complexity of doing this with a 
//	virtual start point
if (t_CURSOR.pt_item->i_type != mi_ACTUAL || pt_context->ul_ActualFrontier == 
														mul_FRONTIER_NO_MORE)
	return !eus_SUCCESS;

	if (us_error = us_VirtualizeThruActualLocation( t_CURSOR.puc_location, 
														pt_context, NULL))
		return us_error;

	//get a pointer at which we can virtualize the insertion CDTEXT
	if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, (ul = 
										sizeof( CDTEXT) + strlen( PC)), &puc))
		return us_error;

t.FontID = FontSetColor( DEFAULT_SMALL_FONT_ID, NOTES_COLOR_DKRED);	//should 
					//	make adjustable by caller
	t.Header.Length = (WORD) ul;

	//virtualize the CDTEXT
	memcpy( puc, &t, sizeof( CDTEXT));
memcpy( puc + sizeof( CDTEXT), PC, strlen( PC));	//not accounting for 
													//	linefeed translation

	//done with the virtualization of the CDTEXT, update the variable that 
	//	tracks the length of the virtual item being written to
	pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + ul;

	return eus_SUCCESS;
} //eus_InsertTextAtHotspotEnd(


/** eus_RemoveRtSpan( ***


--- parameters & return ----
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/13/98 PR: created		*/
//DOC!!
STATUS eus_RemoveRtSpan( const CdRecordSpan *const  pt_SPAN, 
						  RtfTrackingInfo *const  pt_context)	{
	DWORD  ul_begin, ul_end;
	STATUS  us_error;

	if (!( pt_SPAN && pt_SPAN->t_cursorBegin.puc_location && 
								pt_SPAN->t_cursorBegin.us_recLength && 
								pt_SPAN->t_cursorBegin.pt_item && pt_context))
		return !eus_SUCCESS;

//PGP development shortcuts: no offsets, only actuality involved
_ASSERTE( !(pt_SPAN->us_offsetBegin || pt_SPAN->us_offsetEnd));

	ul_begin = ul_GetAbsoluteOffset( pt_SPAN->t_cursorBegin.puc_location, 
											pt_SPAN->t_cursorBegin.pt_item, 
											pt_context->pt_Actuality);
	ul_end = pt_SPAN->t_cursorEnd.pt_item ? ul_GetAbsoluteOffset( 
										pt_SPAN->t_cursorEnd.puc_location, 
										pt_SPAN->t_cursorEnd.pt_item, 
										pt_context->pt_Actuality) : 
										eul_ERR_FAILURE;
_ASSERTE( ul_begin != eul_ERR_FAILURE);

	if (us_error = us_VirtualizeThruActualLocation( 
										pt_SPAN->t_cursorBegin.puc_location, 
										pt_context, NULL))
		return us_error;

	pt_context->ul_ActualFrontier = ul_end != eul_ERR_FAILURE ? ul_end : 
														mul_FRONTIER_NO_MORE;

	return eus_SUCCESS;
} //eus_RemoveRtSpan(


/** eus_ReplaceRtfWithStandardText( ***
Purpose is to adjust the passed-in rich-text context so that the current 
actual rich-text items will be replaced with the virtual rich-text items 
to be written by this fuction using the text content passed in.

--- parameters & return ----
PC: pointer to the null-terminated text content to be streamed into the 
	virtual rich-text items to be created by this function
f_TRANSLATE_LINEFEEDS: flag specifying whether CRLFs should be translated to 
	rich-text linefeeds when copying content (TRUE) or not (FALSE)
pt_context: Input & Output. Pointer to environment information about the 
	rich-text field being operated on. Its virtuality information will be 
	updated by this function.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
9/12/99 PR
+ fixed bug that manifested itself when multiple CDTEXTs had to be written
+ fixed bug that manifested itself when PGP ASCII Armor exceeds 64K in length
+ added documentation

12/16/98 PR: created		*/
STATUS eus_ReplaceRtfWithStandardText( const char  PC[], 
										const BOOL  f_TRANSLATE_LINEFEEDS, 
										RtfTrackingInfo *const  pt_context)	{
	ItemInfo * pt_item = pt_context->pt_Actuality;
	DWORD  ul = 0;

	if (!( PC && pt_context && pt_item))
		return !eus_SUCCESS;

	//set the actual frontier to the end of actuality
	do
		ul += pt_item->ul_length + pt_item->ul_length % 2;
	while (pt_item = pt_item->pt_next);
	pt_context->ul_ActualFrontier = ul;

	//if any virtuality obtains, clear any items beyond a first item and 
	//	reset the length of the first item to the beginning of the item
	if (pt_item = pt_context->pt_Virtuality)	{
		if (pt_item->pt_next)	{
			ClearItemList( &pt_item->pt_next);
			pt_context->pt_endVirtual = pt_item;
		}
		pt_item->ul_length = sizeof( WORD);
	} //if (pt_item = pt_context->pt_Virtuality)

	return us_AppendTextToVirtuality( PC, NULL, f_TRANSLATE_LINEFEEDS, NULL, 
													NULL, NULL, pt_context);
} //eus_ReplaceRtfWithStandardText(


/** ul_getNearLinebreakOffset( ***
Purpose is to return the offset to the first character following the nearest, 
best linebreak candidate in an ASCII text run, accounting if necessary for 
translation of CRLFs to rich-text linefeed characters. If CRLF translation is 
specified, the nearest CRLF will be sought as the linebreak point. If a CRLF 
is not found or if CRLF translation is not specified, the nearest word break 
will be sought as the linebreak point. If all else fails, a linebreak point 
corresponding to the specified limit point will be returned.

--- parameters & return ----
PC: pointer to the null-terminated ASCII text run to be considered
ul_LIMIT: the maximum length at which a linebreak point may be returned
f_TRANSLATE_LINEFEEDS: flag specifying whether CRLFs should be translated to 
	rich-text linefeeds when copying content (TRUE) or not (FALSE)
RETURN: The offset to the first character following the nearest, best 
	linebreak candidate. If the linebreak is at a CRLF, the offset will be to 
	the beginning of the CRLF.

--- revision history -------
9/12/99 PR: created			*/
static unsigned long ul_getNearLinebreakOffset( 
										const char *const  PC, 
										unsigned long  ul_LIMIT, 
										const BOOL  f_TRANSLATE_LINEFEEDS)	{
	const char * pc, * pc_limit = NULL;
	DWORD  ul;

	_ASSERTE( PC && ul_LIMIT);

	//if we're translating linefeeds, the nearest linebreak is the nearest 
	//	linefeed, presuming one is available
	if (f_TRANSLATE_LINEFEEDS)	{
		//determine the pointer where the ultimate limit occurs in the 
		//	untranslated text
		TranslateCrlfText( PC, ul_LIMIT, &ul, NULL, &pc_limit);

		//Step back until a CRLF is encountered. If one is encountered, 
		//	determine and return the offset in terms of the *translated* 
		//	text. The "1" represents the length of a rich-text linefeed.
		pc = pc_limit;
		do
			if (memcmp( pc, epc_CRLF, ei_LEN_CRLF) == ei_SAME)
				return ul - (pc_limit - pc - ei_LEN_CRLF + 1);
		while (--pc > PC);
	} //if (f_TRANSLATE_LINEFEEDS)

	//the nearest linebreak is at the beginning of the nearest word, 
	//	presuming one is available
	if (!pc_limit)	{
		_ASSERTE( ul_LIMIT <= strlen( PC));

		pc_limit = PC + (ul = ul_LIMIT);
	}
	pc = pc_limit;
	while (--pc > PC)
		if (*pc == ec_SPACE)
			return ul - (pc_limit - pc + 1);

	//since there isn't a good linebreak, we'll just say the break is right 
	//	at the limit, probably intra-word
	return ul;
} //ul_getNearLinebreakOffset(


/** eus_ReplaceRtfSpanText( ***
Purpose is to replace an identified span of rich text with the contents of  
the provided, null-terminated text buffer.

--- parameters & return ----
pt_span: Input & Output. Pointer to a rich-text span structure which holds 
	the beginning cordinates of the span to be replaced, and optionally, the 
	ending cordinates of the span. If the ul_LEN_TEXT_TO_REPLACE parameter 
	is provided, passed-in ending cordinates are ignored and will be reset 
	according to the dimension of that parameter.
	If not null, the ending cordinates will be reset by the procedure to 
	point to the last CDTEXT written and containing the last of the 
	replacement text (NOTE: this output not yet enabled!!)
ul_LEN_TEXT_TO_REPLACE: Optional. The length of the text to in the rich-text 
	field to be replaced. Parameter required if null ending cordinates are 
	provided in the pt_span parameter.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be translated to CRLFs when assessing content 
	(TRUE) or not (FALSE)
PC: the replacement text content, null-terminated
pt_context: Input & Output. Pointer to tracking information about the 
	operative rich-text field.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
9/12/99 PR
+ added support for replacement content in excess of MAXONESEGSIZE
+ documentation adjustment

12/8/98 PR: created			*/
STATUS eus_ReplaceRtfSpanText( CdRecordSpan *const  pt_span, 
								const DWORD  ul_LEN_TEXT_TO_REPLACE, 
								const BOOL  f_TRANSLATE_LINEFEEDS, 
								const char  PC[], 
								RtfTrackingInfo *const  pt_context)	{
	CdRecordCursor  t_cursorBegin, t_cursorEnd, t_cursor;
	WORD  us_offsetEnd, us_lenEndPreserve;
	BYTE * puc, * puc_beginSpanCdRec;
	DWORD  ul_lenReplacement, ul;
	STATUS  us_error;
	BOOL  f_OneCdRecordSpan, f_NullifyEndCursor, f_ItemJustLocked, 
			f_result = FALSE;

	//If any of the parameters are obviously invalid, short-circuit with 
	//	general failure. A valid span requires either a content length or a 
	//	valid end cursor.
	if (!( pt_span && PC && pt_context && 
								pt_span->t_cursorBegin.pt_item && 
								pt_span->t_cursorBegin.puc_location && 
								(pt_span->t_cursorEnd.puc_location && 
								pt_span->t_cursorEnd.pt_item || 
								ul_LEN_TEXT_TO_REPLACE) && 
								(pt_span->t_cursorBegin.pt_item == 
								pt_span->t_cursorEnd.pt_item ? 
								pt_span->t_cursorBegin.puc_location < 
								pt_span->t_cursorEnd.puc_location : TRUE)))
		return !eus_SUCCESS;

	//now that the input seems okay, it's safe to initialize certain local 
	//	variables
	t_cursorBegin = pt_span->t_cursorBegin;
	puc_beginSpanCdRec = t_cursorBegin.puc_location;
	t_cursorEnd = pt_span->t_cursorEnd;
	us_offsetEnd = pt_span->us_offsetEnd;

	//determine the length of the replacement content, accounting if 
	//	needed for CRLF to rich-text linefeed conversion
	if (f_TRANSLATE_LINEFEEDS)
		TranslateCrlfText( PC, NULL, &ul_lenReplacement, NULL, NULL);
	else
		ul_lenReplacement = strlen( PC);

	//if the passed-in span is marked by offset instead of end cursor, 
	//	determine the appropriate end-cursor configuration
	if (ul_LEN_TEXT_TO_REPLACE)
		if (!( f_result = f_ProcessKnownLengthRtfText( t_cursorBegin, 
										pt_span->us_offsetBegin, 
										ul_LEN_TEXT_TO_REPLACE, 
										f_TRANSLATE_LINEFEEDS, pt_context, 
										&t_cursorEnd, &us_offsetEnd, NULL)))
			return !eus_SUCCESS;

	ul = sizeof( CDTEXT) + ul_lenReplacement;

	//set a flag telling whether we're dealing with a span that involves only 
	//	one CD record
	us_lenEndPreserve = t_cursorEnd.us_recLength - (sizeof( CDTEXT) + 
																us_offsetEnd);
	f_OneCdRecordSpan = puc_beginSpanCdRec == t_cursorEnd.puc_location && 
								ul + pt_span->us_offsetBegin + 
								us_lenEndPreserve <= mus_LENLIMIT_CDTEXT_REC;

	//if the starting CD record of the span is not virtualized...
	if (t_cursorBegin.pt_item->i_type != mi_VIRTUAL)	{
		//Virtualize up to and including the CD record preceding the starting 
		//	CD record involved in the span. The actual frontier absolute 
		//	offset will then point to the starting CD record.
		if (us_error = us_VirtualizeThruActualLocation( puc_beginSpanCdRec, 
														pt_context, NULL))
			return us_error;
	//Else we have to deal with already virtualized content. And that's a lot 
	//	of complicated work.
	}else	{
return !eus_SUCCESS;	//PGP development shortcut
		//if the CD record following the span's ending CD record resides on 
		//	an item different from the item containing the starting CD 
		//	record...
			//if the item containing the starting CD record is not the last 
			//	virtual item, set the virtuality member of a proxy 
			//	item-tracking structure to a copy of the item structure 
			//	associated with the starting CD record's item but with a null 
			//	next-item pointer

			//if the starting CD record is not the first record in the item...
				//reset the length of the item to point to the end of the 
				//	CD record preceding the starting CD record
			//else reset the length of the item to point to the beginning of 
			//	the item (i.e. no item content at this point)
		//else we need to recast the item bounding the rich-text span
			//in a proxy item-tracking structure, create an initial virtual 
			//	item and initialize it with the content on the item 
			//	containing the starting CD record that precedes the starting 
			//	record

		//set an internal flag telling this procedure to nullify the 
		//	beginning cursor in the passed-in span structure to indicate its 
		//	invalidity to the caller
	} //if (t_cursorBegin.pt_item->i_type != mi_VIRTUAL)

	//if the starting cordinates of the span to be replaced fall within, not 
	//	at the beginning of, a CDTEXT...
	if (*puc_beginSpanCdRec == LOBYTE( SIG_CD_TEXT) && 
												pt_span->us_offsetBegin)	{
		WORD  us;
		CdRecordCursor  t_newStart;

		//determine the length of the replacement CDTEXT as the sum of the 
		//	length of a CDTEXT structure, the length of the content which 
		//	precedes the the beginning of the span in the initial CDTEXT, the 
		//	length of our replacement text, and if the ending cordinates of 
		//	the span fall within the same CDTEXT that the starting 
		//	cordinates do, the length of the content following the end of 
		//	the span
		ul += pt_span->us_offsetBegin;
//		ul += pt_span->us_offsetBegin + 1;
		if (f_OneCdRecordSpan)
			ul += us_lenEndPreserve;
//			ul += (us_lenEndPreserve = t_cursorEnd.us_recLength - 
//										(sizeof( CDTEXT) + us_offsetEnd) - 1);

		//get a pointer at which we can virtualize a CDTEXT CD record of the 
		//	length just determined
		us = ul < mus_LENLIMIT_CDTEXT_REC ? (WORD) ul : 
													mus_LENLIMIT_CDTEXT_REC;
		if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, us, &puc))
			return us_error;

		//Virtualize the initial CDTEXT structure that accounts for the 
		//	ultimate length of the CDTEXT, content included. Utilize the 
		//	CDTEXT associated with the beginning of the rich-text span to 
		//	preserve the font signature.
		memcpy( puc, puc_beginSpanCdRec, sizeof( CDTEXT));
		((WSIG *) puc)->Length = us;

		//virtualize the content that precedes the beginning of the span in 
		//	the initial CDTEXT
		memcpy( puc + sizeof( CDTEXT), puc_beginSpanCdRec + 
									sizeof( CDTEXT), pt_span->us_offsetBegin);

		t_newStart.puc_location = puc;
		t_newStart.pt_item = pt_context->pt_endVirtual;
		t_newStart.us_recLength = us;
		if (us_error = us_AppendTextToVirtuality( PC, ul_lenReplacement, 
										f_TRANSLATE_LINEFEEDS, &t_newStart, 
										pt_span->us_offsetBegin, 
										NULL, pt_context))
			return us_error;

		//if the ending cordinates of the span fall within the same CDTEXT 
		//	that the starting cordinates do...
		if (f_OneCdRecordSpan)	{
			//tack on the content following the end of the span
			memcpy( puc + ul - us_lenEndPreserve, puc_beginSpanCdRec + 
											sizeof( CDTEXT) + us_offsetEnd + 
											1, us_lenEndPreserve);

			//done with the virtualization of the CDTEXT, update the variable 
			//	that tracks the length of the virtual item being written to
			pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + ul;
		}
	//else we construct our own CDTEXTs to hold the replacement content
	}else	{
//PGP development shortcut: assuming that the first record in the span is a 
//	CDTEXT. If it's not, return general failure.
if (*puc_beginSpanCdRec != LOBYTE( SIG_CD_TEXT))
	return !eus_SUCCESS;

		//Determine a FONTID that we can use in the CDTEXT we need to write. 
		//	Basically, we want the closest CDTEXT before (and including) the 
		//	beginning of the span. If one isn't available, we want the 
		//	closest one after the beginning CD record of the span (even if 
		//	it's in the span). If no CDTEXTs are available at all, we will 
		//	use the Notes default FONTID.

//PGP development shortcut: using the beginning, source CDTEXT's FONTID
		if (us_error = us_AppendTextToVirtuality( PC, ul_lenReplacement, 
									f_TRANSLATE_LINEFEEDS, NULL, NULL, 
									((CDTEXT *) puc_beginSpanCdRec)->FontID, 
									pt_context))
			return us_error;
	} //if (*t_cursorBegin.puc_location == LOBYTE(

	//Set an internal flag telling whether this procedure should nullify the 
	//	ending cursor to indicate its invalidity to the caller because none 
	//	of the ending CD record's content was preserved. Some of the ending 
	//	content is to be preserved if the ending cordinates of the span are 
	//	not located at the same CD record as the beginning cordinates and if 
	//	they fall within, not at the end of, a CDTEXT. If such preservation 
	//	(i.e. no nullification) is required...
	if (!( f_NullifyEndCursor = !(!f_OneCdRecordSpan && 
									*t_cursorEnd.puc_location == LOBYTE( 
									SIG_CD_TEXT) && us_lenEndPreserve)))	{
		//get a pointer at which we can virtualize a CDTEXT containing only 
		//	the content to be preserved
		if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, sizeof( 
										CDTEXT) + us_lenEndPreserve, &puc))
			return us_error;

		//virtualize a CDTEXT based on the ending CDTEXT's format but 
		//	containing just the content not included in the specified span
		memcpy( puc, t_cursorEnd.puc_location, sizeof( CDTEXT));
		((WSIG *) puc)->Length = sizeof( CDTEXT) + us_lenEndPreserve;
		memcpy( puc + sizeof( CDTEXT), t_cursorEnd.puc_location + sizeof( 
												CDTEXT) + us_offsetEnd + 1, 
												us_lenEndPreserve);

		//if included in the passed-in span structure, set an internal flag 
		//	telling this procedure to reset the ending cordinates to point 
		//	to the beginning of the CDTEXT just virtualized (NOT YET DONE)

		//done with the virtualization of the CDTEXT, update the variable 
		//	that tracks the length of the virtual item being written to
		pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + ul;
	} //if (!( f_NullifyEndCursor = 

	//if the CD record that follows the original ending CD record is 
	//	virtualized...
	t_cursor = t_cursorEnd;
	AdvanceCdCursor( &t_cursor, pt_context, NULL, &f_ItemJustLocked);
	if (t_cursor.pt_item && t_cursor.pt_item->i_type == mi_VIRTUAL)	{
return !eus_SUCCESS;		//PGP development shortcut
		//if that CD record is the first record in its item, restore 
		//	virtuality by...
			//setting the next pointer of the last item in the proxy 
			//	virtuality equal to the item containing this CD record that 
			//	follows the ending CD record
		//else
			//virtualize that CD record following the original ending CD 
			//	record and all ensuing records up to the end of its item

			//set the next pointer of the last proxy item equal to the item 
			//	pointed to by the next pointer of the item we just virtualized
	//else if a following CD record exists (in Actuality)...
	}else if (t_cursor.pt_item)	{
		//reset the actual-frontier absolute offset to the beginning of the 
		//	CD record following the original ending CD record
		pt_context->ul_ActualFrontier = ul_GetAbsoluteOffset( 
									t_cursor.puc_location, t_cursor.pt_item, 
									pt_context->pt_Actuality);

		//if the item containing the following CD record was just locked in 
		//	memory for the sake of this operation, unlock it now
		if (f_ItemJustLocked)	{
			OSUnlockBlock( t_cursor.pt_item->bid_contents);
			t_cursor.pt_item->puc_start = NULL;
		}
	//else reset the actual-frontier absolute offset to the ultimate length 
	//	of the rich-text field
	}else
		pt_context->ul_ActualFrontier = ul_GetAbsoluteOffset( 
							t_cursorEnd.puc_location, 
							t_cursorEnd.pt_item, pt_context->pt_Actuality) + 
							t_cursorEnd.us_recLength + 
							t_cursorEnd.us_recLength % 2;

	//if either the original starting CD record or the CD record following 
	//	the original ending CD record was virtualized...
		//if the first item of the proxy virtuality is the same as the 
		//	original item containing the starting CD record...
			//discard the items that fall between the original item 
			//	containing the starting CD record and the item pointed to by 
			//	the next pointer in the last proxy-virtual item
		//else
			//if the original item containing the starting CD record is the 
			//	first virtual item...
				//set the virtuality member of the main item-tracking 
				//	structure to point to the first proxy virtual item
			//else locate in the original virtuality the item whose next 
			//	pointer specifies this original item and reset its next 
			//	pointer to the first proxy virtual item

			//discard the original item and any that follow it up through 
			//	the one pointed to by the next pointer in the last 
			//	proxy-virtual item

	//if internal flags specify that the cordinates of the passed-in span 
	//	need to be reset, do what's needed 

	return eus_SUCCESS;
} //eus_ReplaceRtfSpanText(


/** us_AppendTextToVirtuality( ***
Purpose is to append a null-terminated buffer of text content to the current 
virtual rich-text stream. The procedue does not concern itself with 
paragraph styles even though it does break up larger content streams into 
separate paragraphs.

--- parameters & return ----
PC: the replacement text content, null-terminated
ul_lenReplacement: Optional. The length of the replacement content to be 
	written. If null, the procedure will write out all of the replacement 
	text content.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be translated to CRLFs when assessing content 
	(TRUE) or not (FALSE)
pt_startCdText: Optional. Input & Output. Pointer to a CdRecordCursor 
	positioned at a CDTEXT which has already been started and to which text 
	should continue to be added up to the length specified in the 
	us_recLength member. If NULL, a new CDTEXT will be fashioned to which the 
	provided text will begin to be written. The CDTEXT must be positioned at 
	the end of the last obtaining virtual item, and the ul_length member of 
	that item must not include this new CDTEXT being written.
	If the length of the starting CDTEXT is shortened from that allocated by 
	the caller (because, for instance, an appropriate linebreak falls before 
	the end of the allocated space) the us_recLength member of the cursor 
	will be updated to reflect the revised length. Note that this adjustment 
	will only occur if more than one CDTEXT (and thus paragraph) needs to be 
	used.
us_lenStartText: Optional. The length of text content already written into 
	the CDTEXT pointed to by t_startCdText.
ul_FONT: Optional. The FONTID that should be used with new CDTEXTs written by 
	this function. If omitted and t_startCdText is provided, procedure will 
	default to the FONTID contained in the starting CDTEXT. If omitted and no 
	t_startCdText is provided, procedure will default to the monospaced 
	FOREIGN_FONT_ID provided by Notes.
pt_context: Input & Output. Pointer to tracking information about the 
	operative rich-text field. The ul_length member of the last virtual item 
	(pointed to by pt_endVirtual) will be updated to reflect CD records fully 
	written. Note that a first CDTEXT described by pt_startCdText will not be 
	fully written if all the replacement content specified fails to fill up 
	the space allocated for the CDTEXT content by the caller.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
9/12/99 PR: created		*/
static STATUS us_AppendTextToVirtuality( 
										const char  PC[], 
										DWORD  ul_lenReplacement, 
										const BOOL  f_TRANSLATE_LINEFEEDS, 
										CdRecordCursor *const pt_startCdText, 
										const WORD  us_lenStartText, 
										const FONTID  ul_FONT, 
										RtfTrackingInfo *const  pt_context)	{
	CDTEXT  t = {{SIG_CD_TEXT}};
	BYTE * puc;
	const char * pc = PC;
	STATUS  us_error;

	_ASSERTE( PC && pt_context && (pt_startCdText ? 
				pt_startCdText->pt_item && pt_startCdText->puc_location && 
				*pt_startCdText->puc_location == LOBYTE( SIG_CD_TEXT) && 
				pt_startCdText->us_recLength == 
				((WSIG *) pt_startCdText->puc_location)->Length && 
				pt_startCdText->pt_item == pt_context->pt_endVirtual && 
				pt_startCdText->pt_item->puc_start + 
				pt_startCdText->pt_item->ul_length + 
				pt_startCdText->pt_item->ul_length % 2 == 
				pt_startCdText->puc_location : TRUE) && 
				(ul_lenReplacement && !f_TRANSLATE_LINEFEEDS ? 
				ul_lenReplacement <= strlen( PC) : TRUE));

	//if the length of the replacement text to be written was not specified 
	//	by the caller, default it to the length implied by the length of the 
	//	specified null-terminated text buffer
	if (!ul_lenReplacement)
		if (f_TRANSLATE_LINEFEEDS)
			TranslateCrlfText( PC, NULL, &ul_lenReplacement, NULL, NULL);
		else
			ul_lenReplacement = strlen( PC);

	//if a starting CDTEXT has been provided...
	if (pt_startCdText)	{
		const WORD  us_LEN_REMAIN = pt_startCdText->us_recLength - 
											sizeof( CDTEXT) - us_lenStartText;

		DWORD  ul_len = ul_lenReplacement;

		//Unless the length of the content to write has been specified and 
		//	the length will fit into the remaining content space available in 
		//	the CDTEXT, determine the offset to the best place to break the 
		//	text stream such that we conform to the length limit we have on 
		//	the CDTEXT. If the offset differs from the length limit, adjust 
		//	the cursor's and CDTEXT's length specifiers.
		if (!ul_lenReplacement || ul_lenReplacement > us_LEN_REMAIN)
			ul_len = ul_getNearLinebreakOffset( PC, us_LEN_REMAIN, 
													f_TRANSLATE_LINEFEEDS);

		//if the length of the starting CDTEXT is to be shortened, reflect 
		//	that in the starting cursor and in the CDTEXT itself
		if (ul_len < ul_lenReplacement)
			((WSIG *) pt_startCdText->puc_location)->Length = 
								pt_startCdText->us_recLength = (WORD) ul_len;

		//add the length of content to the current CDTEXT
		puc = pt_startCdText->puc_location + sizeof( CDTEXT) + 
															us_lenStartText;
		if (f_TRANSLATE_LINEFEEDS)	{
			DWORD  ul;

			TranslateCrlfText( PC, ul_len, &ul, puc, &pc);
			_ASSERTE( ul == ul_len);
		}else	{
			memcpy( puc, PC, ul_len);
			pc += ul_len;
		} //if (f_TRANSLATE_LINEFEEDS)

		//if there's no excess content space in the CDTEXT, update the length 
		//	member of the end virtual item to reflect the full CD record added
		if (ul_lenReplacement >= us_LEN_REMAIN)
			pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDTEXT) + us_lenStartText + 
									ul_len;

		//if all text content has been written, short-circuit with success
		if (ul_len = ul_lenReplacement)
			return eus_SUCCESS;

		//adjust the length of the content remaining to be written by the 
		//	amount just written, accounting for a CRLF linebreak if necessary 
		//	(ul_getNearLinebreakOffset returns offsets to the *beginning* 
		//	of CRLF linebreaks)
		if (f_TRANSLATE_LINEFEEDS && memcmp( pc, epc_CRLF, ei_LEN_CRLF) == 
																ei_SAME)	{
			pc += ei_LEN_CRLF;
			ul_lenReplacement -= 1;	//i.e. the length of a rich-text linefeed
		}
		ul_lenReplacement -= ul_len;

		//Since we know now that more than just the starting CDTEXT is needed 
		//	to accommodate the replacement content, virtualize a CDPARAGRAPH 
		//	record, continuing to use the paragraph style currently in 
		//	effect, and update the length member of the end virtual item to 
		//	reflect the addition. We do this to ensure that we will comply 
		//	with the 64K limit on paragraphs.
		if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, 
												sizeof( CDPARAGRAPH), &puc))
			return us_error;
		memcpy( puc, &mt_PARAGRAPH, sizeof( CDPARAGRAPH));
		pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDPARAGRAPH);
	} //if (pt_startCdText->pt_item)

	//set the FONTID of ensuing CDTEXTS to be written
	if (ul_FONT)
		t.FontID = ul_FONT;
	else if (pt_startCdText)
		t.FontID = ((CDTEXT *) pt_startCdText->puc_location)->FontID;
	else
		t.FontID = FOREIGN_FONT_ID;

	//As necessary, break up the replacement content into multiple 
	//	paragraphs. We can't simply break it up into multiple CDTEXTs because 
	//	of the 64K limit on paragraphs.
	while (ul_lenReplacement + sizeof( CDTEXT) > mus_LENLIMIT_CDTEXT_REC)	{
		//determine offset to the best place to break the text stream such 
		//	that we conform to the length limit we have on the CDTEXT
		const WORD  US = (WORD) ul_getNearLinebreakOffset( pc, 
											mus_LENLIMIT_CDTEXT_REC - sizeof( 
											CDTEXT), f_TRANSLATE_LINEFEEDS), 
					us_LEN_REC = sizeof( CDTEXT) + US;

		//get a pointer at which we can virtualize the CDTEXT
		if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, 
												us_LEN_REC + us_LEN_REC % 2 + 
												sizeof( CDPARAGRAPH), &puc))
			return us_error;

		//finish off the CDTEXT structure to be used with the replacement text
		t.Header.Length = us_LEN_REC;
		memcpy( puc, &t, sizeof( CDTEXT));
		if (f_TRANSLATE_LINEFEEDS)
			TranslateCrlfText( pc, US, NULL, puc += sizeof( CDTEXT), &pc);
		else
			memcpy( puc += sizeof( CDTEXT), pc, US);

		//virtualize a CDPARAGRAPH record, continuing to use the paragraph 
		//	style currently in effect
		memcpy( puc + US + us_LEN_REC % 2, &mt_PARAGRAPH, 
														sizeof( CDPARAGRAPH));

		//done with the virtualization of the CDTEXT, update the variable 
		//	that tracks the length of the virtual item being written to
		pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + us_LEN_REC + us_LEN_REC % 2 + 
									sizeof( CDPARAGRAPH);

		//adjust the length of the content remaining to be written by the 
		//	amount just written, accounting for a CRLF linebreak if necessary 
		//	(ul_getNearLinebreakOffset returns offsets to the *beginning* 
		//	of CRLF linebreaks)
		if (f_TRANSLATE_LINEFEEDS && memcmp( pc, epc_CRLF, ei_LEN_CRLF) == 
																ei_SAME)	{
			pc += ei_LEN_CRLF;
			ul_lenReplacement -= 1;	//i.e. the length of a rich-text linefeed
		}
		ul_lenReplacement -= US;
	} //while (ul_lenReplacement + sizeof( CDTEXT)

	//get a pointer at which we can virtualize the last replacement CDTEXT
	if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, sizeof( 
										CDTEXT) + ul_lenReplacement, &puc))
		return us_error;

	//finish off the CDTEXT structure to be used
	t.Header.Length = (WORD) (ul_lenReplacement + sizeof( CDTEXT));

	//virtualize a separated CDTEXT record involving the controlling CDTEXT 
	//	format and the rest of the output text
	memcpy( puc, &t, sizeof( CDTEXT));
	if (f_TRANSLATE_LINEFEEDS)
		TranslateCrlfText( pc, ul_lenReplacement, NULL, puc + 
													sizeof( CDTEXT), NULL);
	else
		memcpy( puc + sizeof( CDTEXT), pc, ul_lenReplacement);

	//done with the virtualization of the CDTEXT, update the variable that 
	//	tracks the length of the virtual item being written to
	pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDTEXT) + ul_lenReplacement;

	return eus_SUCCESS;
} //us_AppendTextToVirtuality(


/** ul_GetAbsoluteOffset( ***


--- parameters & return ----


--- revision history -------
11/30/98 PR: created		*/
//DOC!!
static __inline DWORD ul_GetAbsoluteOffset( 
									const BYTE *const  PUC, 
									const ItemInfo *const  pt_ITEM, 
									const ItemInfo *const  pt_ITEM_LIST)	{
	const ItemInfo * pt_item = pt_ITEM_LIST;
	DWORD  ul_lengthCount = (DWORD) NULL;

	while (pt_item != pt_ITEM || !pt_item)	{
		//remember, "<item>.length" includes a TYPE_COMPOSITE header (2 bytes)
		ul_lengthCount += pt_item->ul_length + pt_item->ul_length % 2;

		pt_item = pt_item->pt_next;
	}

	return pt_item ? ul_lengthCount + PUC - pt_ITEM->puc_start : 
															eul_ERR_FAILURE;
} //ul_GetAbsoluteOffset(


/** TranslateCrlfText( ***
Purpose is to translate ASCII text with embedded CRLF linefeeds to equivalent 
rich-text ready text. All this means is that CRLFs are converted to rich-text 
linefees (the null character).

--- parameters ---------
PC: pointer to the ASCII text to be translated
ul_LIMIT: Optional. Limit on how much translated text to accumulate. If 
	omitted, it is assumed that the ASCII text should be translated in full.
pul_length: Optional. Pointer to the variable in which to return the length 
	of the text translated. Will not exceed a provided ul_LIMIT.
puc_write: Optional. Pointer to a buffer in which to write translated text.
ppc_ending: Optional. Pointer to a pointer variable in which to return the 
	position in the original ASCII of the character immediately following the 
	last character translated.

--- revision history ---
9/12/99 PR: added documentation
12/16/98 PR: created	*/
static void TranslateCrlfText( const char *const  PC, 
								DWORD  ul_LIMIT, 
								DWORD *const  pul_length, 
								BYTE *const  puc_write, 
								const char * *const  ppc_ending)	{
	char * pc;
	DWORD  ul_traversed = NULL, ul_lenTranslated = NULL, ul = NULL;
	BOOL  f_LimitHit = FALSE;

	_ASSERTE( PC && (pul_length || puc_write));

	if (pul_length)
		*pul_length = NULL;
	if (puc_write)
		*puc_write = NULL;
	if (ppc_ending)
		*ppc_ending = NULL;
	
	//loop through the passed-in content, from CRLF to CRLF, processing 
	//	as requested
	while (!f_LimitHit && (pc = strstr( PC + ul_traversed, epc_CRLF)))	{
		//determine how much input content we just traversed in moving to 
		//	the current CRLF
		ul = pc - (PC + ul_traversed);

		if (f_LimitHit = ul_LIMIT && ul_lenTranslated + ul >= ul_LIMIT)
			ul = ul_LIMIT - ul_lenTranslated;

		//if we're suppossed to copy translated content using the passed-in 
		//	pointer, copy the section of text we just traversed plus the CRLF 
		//	translated to a rich-text linefeed
		if (puc_write)	{
			memcpy( puc_write + ul_lenTranslated, PC + ul_traversed, ul);
			if (!f_LimitHit)
				puc_write[ ul_lenTranslated + ul] = mc_LINEFEED_RICHTEXT;
		}

		//increment the translated length based on the offset from where we 
		//	started searching plus the translated length of the CRLF we found
		ul_lenTranslated += ul + (!f_LimitHit ? 1 : 0);

		//increment the offset into the input content so our next pass will 
		//	be ready to go
		ul_traversed += ul + (!f_LimitHit ? ei_LEN_CRLF : 0);
	} //while (!f_LimitHit && pc = strstr( PC + ul_traversed,

	//if requested and input content remains, copy any trailing input content 
	//	containing no CRLFs using the passed-in pointer
	if (!f_LimitHit && (ul = strlen( PC) - ul_traversed) && puc_write)
		memcpy( puc_write + ul_lenTranslated, PC + ul_traversed, ul);

	//if requested, set the caller's length variable with the size of the 
	//	input content with CRLFs translated to rich-text linefeeds
	if (pul_length)
		*pul_length = ul_lenTranslated + (!f_LimitHit ? ul : 0);
	if (ppc_ending)
		*ppc_ending = PC + ul_traversed;
} //TranslateCrlfText(


/** us_GetVirtualCdRecAppendPointer( ***
Purpose is to deliver a location pointer that the caller can use to 
append a custom-built CD record of a specified maximum length to the end of 
current virtual content. (CD records cannot be split between items.)

--- parameters & return ----
pt_context: Input & Output. Pointer to tracking information about the 
	operative rich-text field. The pt_endVirtual member may be updated by 
	the procedure.
ul_LEN_TO_ACCOMMODATE: the length of _item_ content that must be available 
	beginning at the returned location pointer
ppuc: Output. Memory address of the caller's pointer variable that this 
	procedure should set.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- note -------------------
If a follow-on virtual item must be created, the memory associated with the 
former ending virtual item will not be unlocked.

--- revision history -------
2/23/00 PR: accommodated the signature change to us_StartVirtualItemList()
12/8/98 PR: created			*/
static STATUS us_GetVirtualCdRecAppendPointer( 
										RtfTrackingInfo *const  pt_context, 
										const DWORD  ul_LEN_TO_ACCOMMODATE, 
										BYTE * *const  ppuc)	{
	ItemInfo * pt_item;
	int  i_limit, i;
	STATUS  us_error;

	_ASSERTE( pt_context && ppuc);

	//if the length to accommodate exceeds the maximum size of a rich-text 
	//	item, return general failure
	if (ul_LEN_TO_ACCOMMODATE > mul_MAXLEN_RICH_TEXT_ITEM)
		return !eus_SUCCESS;

	//if virtuality hasn't been started yet, start it now
	if (!pt_context->pt_Virtuality)
		if (us_error = us_StartVirtualItemList( NULL, pt_context))
			return us_error;

	//determine the operative maximum length in effect for the last virtual 
	//	item currently underway
	i_limit = (pt_item = pt_context->pt_endVirtual)->ul_length <= 
										mul_COMFY_LIMIT ? mul_COMFY_LIMIT : 
										mul_STRETCH_LIMIT;

	//if the length to accommodate fits within the space still available 
	//	within the last virtual item...
	if (i_limit - (i = pt_item->ul_length + pt_item->ul_length % 2) > 
												(int) ul_LEN_TO_ACCOMMODATE)
		//set the location pointer to where the caller's custom writing 
		//	(virtualization) should begin
		*ppuc = pt_item->puc_start + i;
	//else we need to string on an additional virtual item so that the 
	//	requested length can be accommodated
	else	{
		//create a new end virtual item and update the tracking information 
		//	accordingly
		if (us_error = us_InsertNextVirtualItem( pt_item, mi_REGULAR_VIRTUAL, 
																(WORD) NULL))
			return us_error;
		pt_item = pt_context->pt_endVirtual = pt_item->pt_next;

		//set the location pointer to where the caller's custom writing 
		//	(virtualization) should begin
		*ppuc = pt_item->puc_start + sizeof( WORD);
	} //if (ul_limit - (ul = pt_item->ul_length +

	return eus_SUCCESS;
} //us_GetVirtualCdRecAppendPointer(


/** eus_CommitChangedRtf( ***
Purpose is to commit a rich-text field that has undergone changes to its 
underlying note, thus converting all virtuality to actuality.

--- parameters & return ----
h_NOTE: handle to the note on which the specified rich-text field exists
pc_ITMNM: pointer to a string telling the name of the rich-text field to be 
	replaced
pt_context: Input & Output. Pointer to the rich-text context structure that 
	describes the rich-text field being manipulated. Structure will be reset 
	to reflect the committal (i.e. virtual content becomes actual content).
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR
+ safety & functionality improvement of having the rich-text field's context 
  structure updated to reflect work done by the procedure
+ logic adjustment to accommodate creation, not replacement, of a field
+ documentation improvement

1/16/99 PR: created			*/
STATUS eus_CommitChangedRtf( HANDLE  h_NOTE, 
								char  pc_ITMNM[], 
								RtfTrackingInfo *const  pt_context)	{
	ItemInfo * pt_item = pt_context->pt_Actuality, * pt_itm, * * ppt_itm, 
				* pt_nxt;
	STATUS  us_err;

	if (!pt_context->pt_Virtuality)
		return eus_SUCCESS;

	if (us_err = us_VirtualizeRestOfRtf( pt_context, NULL))
		return us_err;

	//Remove the current field items on the note. Could use NSFItemDelete, 
	//	but doing it by the BLOCKIDs we already have probably saves cycles.
	if (pt_context->pt_Actuality)	{
		UnlockItems( pt_context->pt_Actuality, NULL);
		do
			if (us_err = NSFItemDeleteByBLOCKID( h_NOTE, ((ItemInfo_Actual *) 
														pt_item)->bid_item))
				return us_err;
		while (pt_item = pt_item->pt_next);
	}

	//For each virtual item, actualize it and update the context structure 
	//	accordingly...
	UnlockItems( pt_context->pt_Virtuality, NULL);
	pt_item = pt_context->pt_Virtuality;
	ppt_itm = &pt_context->pt_Actuality;
	do	{
		//if a follow-on information structure needs to be allocated for the 
		//	soon-to-be actual rich-text item, do so now
		if (!*ppt_itm)	{
			if (!( *ppt_itm = malloc( sizeof( ItemInfo_Actual))))	{
				if (pt_context->pt_Actuality)
					FreeItemList( &pt_context->pt_Actuality);
				return !eus_SUCCESS;
			}
			(*ppt_itm)->pt_next = NULL;
		}

		//copy the virtual item's information structure into the next 
		//	available actual-item information structure and reset properties 
		//	appropriately
		pt_nxt = (pt_itm = *ppt_itm)->pt_next;
		*pt_itm = *pt_item;
		pt_itm->pt_next = pt_nxt;
		pt_itm->i_type = mi_ACTUAL;

		//Actualize the constructed virtual item by appending it to the note. 
		//	As a safety measure, note that the state of the "virtual" item 
		//	has now changed to actual.
		if (us_err = NSFItemAppendByBLOCKID( h_NOTE, ITEM_SIGN, pc_ITMNM, 
									(WORD) strlen( pc_ITMNM), 
									pt_itm->bid_contents, pt_itm->ul_length, 
									&((ItemInfo_Actual *) pt_itm)->bid_item))
			return us_err;
		pt_item->i_type = mi_ACTUAL;

		//in preparation for the next pass, get the address of the pointer to 
		//	the next "actual" item information structure which may be already 
		//	allocated
		ppt_itm = &pt_itm->pt_next;
	} while (pt_item = pt_item->pt_next);

	//do final cleanup, with the result that everything previously virtual is 
	//	reflected now as actual
	if (pt_itm->pt_next)
		FreeItemList( &pt_itm->pt_next);
	FreeItemList( &pt_context->pt_Virtuality);
	pt_context->ul_ActualFrontier = sizeof( WORD);

	return eus_SUCCESS;
} //eus_CommitChangedRtf(


/** us_VirtualizeRestOfRtf( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- side effect ------------

--- revision history -------
2/23/00 PR
+ changed signature to allow wrting to a virtual context that does not belong 
  to the source context
+ added in-line documentation
+ minor logic shortening, token renaming

early 1999 PR: created			*/
//DOC!!
static STATUS us_VirtualizeRestOfRtf( RtfTrackingInfo *const  pt_src, 
										RtfTrackingInfo * pt_trgt)	{
	ItemInfo * pt_item;
	BOOL  f_ItemJustLocked = FALSE;
	DWORD  ul = NULL;
	STATUS  us_err = eus_SUCCESS;

	_ASSERTE( pt_src);

	if (!pt_trgt)
		pt_trgt = pt_src;

	//if there's no more actual content to virtualize, short-circuit with 
	//	success
	if (!pt_src->pt_Actuality || pt_src->ul_ActualFrontier == 
														mul_FRONTIER_NO_MORE)
		return eus_SUCCESS;

	//determine the total physical length of the current (actual) rich-text 
	//	field
	pt_item = pt_src->pt_Actuality;
	while (pt_item->pt_next)	{
		ul += ul % 2 + pt_item->ul_length;
		pt_item = pt_item->pt_next;
	}
	ul += ul % 2 + pt_item->ul_length;

	//if there's more actual content to virtualize...
	if (pt_src->ul_ActualFrontier < ul)	{
		//if necessary, lock in the last actual item
		if (!pt_item->puc_start)	{
			pt_item->puc_start = OSLockBlock( BYTE, pt_item->bid_contents);
			f_ItemJustLocked = TRUE;
		}

		//virtualize up through the end of the last actual item
		us_err = us_VirtualizeThruActualLocation( pt_item->puc_start + 
										pt_item->ul_length, pt_src, pt_trgt);

		//if we locked the last actual item, release it now
		if (f_ItemJustLocked)	{
			OSUnlockBlock( pt_item->bid_contents);
			pt_item->puc_start = NULL;
		}
	} //if (pt_src->ul_ActualFrontier < ul)

	//note that there's no more actual content to virtualize
	if (!us_err)
		pt_src->ul_ActualFrontier = mul_FRONTIER_NO_MORE;

	return us_err;	//probably success
} //us_VirtualizeRestOfRtf(


/** us_VirtualizeThruActualLocation( ***
Purpose is to virtualize _actual_ rich-text content up to a specified point 
and thus advance the "actual frontier" which demarcates virtuality (changed 
rich-text) from actuality (the current rich-text stored in the field). 

--- parameters & return ----
puc_END_POINT: pointer to the byte location which should become the new 
	"actual frontier" and thus before which any actual content between itself 
	and the current actual frontier should be virtualized
pt_src: Input & Output. Address of the information structure describing the 
	rich-text field holding the source content. The actual-frontier member 
	will be updated to reflect the work of this procedure. If a target 
	rich-text context is not specified, procedure will write the content to 
	this source context's virtuality.
pt_trgt: Optional Output. Address of the rich-text context whose virtuality 
	component will receive the rich-text content being copied. If null, 
	procedure will write the content to the source context's virtuality.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR
+ signature change to accommodate writing to virtuality not belonging to the 
  source rich-text context structure
+ minor logic shortening, documentation adjustment

12/8/98 PR: created			*/
static STATUS us_VirtualizeThruActualLocation( 
											const BYTE *const  puc_END_POINT, 
											RtfTrackingInfo *const  pt_src, 
											RtfTrackingInfo * pt_trgt)	{
	ItemInfo * pt_item;
	const BYTE * puc_Appendage;
	DWORD  ul_CopyLength, ul_lengthCount = NULL, 
			ul_TotalActualLength = NULL;
	BOOL  f_ItemJustLocked = FALSE;
	STATUS  us_error;

	_ASSERTE( puc_END_POINT && pt_src && pt_src->pt_Actuality && 
											pt_src->ul_ActualFrontier != 
											mul_FRONTIER_NO_MORE);

	if (!pt_trgt)
		pt_trgt = pt_src;

	//compile the total length of the actual rich-text field
	pt_item = pt_src->pt_Actuality;
	while (pt_item)	{
		ul_TotalActualLength += ul_TotalActualLength % 2 + pt_item->ul_length;
		pt_item = pt_item->pt_next;
	}

	//determine working cordinates of the current "actual frontier" point, 
	//	as well as the absolute offset associated with the end of the item 
	//	containing the actual-frontier point
	if (!f_LocateAbsoluteOffset( pt_src->ul_ActualFrontier, 
											pt_src->pt_Actuality, &pt_item, 
											&puc_Appendage, &ul_lengthCount, 
											&f_ItemJustLocked))
		return !eus_SUCCESS;

	//if the actual frontier appears in the same item as the specified byte 
	//	location to become the new frontier and the specified byte location 
	//	does not follow the actual-frontier location, return general failure
	if (pt_item->puc_start < puc_END_POINT && puc_END_POINT <= 
								pt_item->puc_start + pt_item->ul_length)	{
		if (puc_END_POINT < puc_Appendage)
			return !eus_SUCCESS;
	//else make sure that the specified byte location follows the 
	//	actual-frontier location by testing the rich-text items following the 
	//	actual frontier's item until an item is found that contains the 
	//	specified location to become the new frontier
	}else	{
		ItemInfo * pt_itm = pt_item->pt_next;

		do
			if (pt_itm->puc_start < puc_END_POINT && puc_END_POINT <= 
										pt_itm->puc_start + pt_itm->ul_length)
				break;
		while( pt_itm = pt_itm->pt_next);

		//if no containing item was found, return general failure
		if (!pt_itm)
			return !eus_SUCCESS;
	} //if (pt_item->puc_start <= puc_END_POINT &&

	//if the current actual frontier and the byte location that is to become 
	//	the new "actual frontier" fall on different items, virtualize any 
	//	content falling between the actual frontier and the beginning of the 
	//	item containing the byte location of the new frontier
	while (puc_END_POINT < pt_item->puc_start || puc_END_POINT > 
								pt_item->puc_start + pt_item->ul_length)	{
		//determine the length of content to virtualize from the current item
		ul_CopyLength = ul_lengthCount - pt_src->ul_ActualFrontier;

		//Virtualize the length of content after fist advancing the actual 
		//	frontier by that length
		pt_src->ul_ActualFrontier += ul_CopyLength + ul_CopyLength % 2 + 
																sizeof( WORD);
		us_error = us_VirtualAppend( puc_Appendage, ul_CopyLength, 
											pt_trgt, ul_TotalActualLength);

		//if the item containing the content just virtualized was locked in 
		//	memory for the sake of this operation, unlock it now
		if (f_ItemJustLocked)	{
			OSUnlockBlock( pt_item->bid_contents);
			pt_item->puc_start = NULL;
			f_ItemJustLocked = FALSE;
		}

		//if any error occurred during the virtualization, return its code 
		//	to the caller
		if (us_error)
			return us_error;

		//move to the next item on our way toward the item containing the 
		//	end, specified byte location
		pt_item = pt_item->pt_next;
		if (!pt_item->puc_start)	{
			pt_item->puc_start = OSLockBlock( BYTE, pt_item->bid_contents);
			f_ItemJustLocked = TRUE;
		}

		//update the variable telling the absolute length up through the 
		//	current item, and set the pointer telling where virtualization 
		//	should start on our new item
		ul_lengthCount += ul_lengthCount % 2 + pt_item->ul_length;
		puc_Appendage = pt_item->puc_start + sizeof( WORD);
	} //puc_END_POINT < pt_item->puc_start ||

	//Get the length of whatever content is left over in the current actual 
	//	item up to the point that will mark the new actual frontier. If 
	//	there is no remaining content, return success.
	if ((ul_CopyLength = puc_END_POINT - puc_Appendage) <= 0)
		return eus_SUCCESS;

	//virtualize the last piece of content before the new actual frontier
	pt_src->ul_ActualFrontier += ul_CopyLength + ul_CopyLength % 2;
	return us_VirtualAppend( puc_Appendage, ul_CopyLength, pt_trgt, 
														ul_TotalActualLength);
} //us_VirtualizeThruActualLocation(


/** us_VirtualAppend( ***
Purpose is to append the specified length of CD-record content to the end of 
any current virtual content, accounting as necessary for the creation of 
rich-text items to accommodate the content being appended.

The function uses two threshold values on the size of virtual items it 
creates: "comfortable" and "stretch." The comfortable limit specifies a 
standard size that leaves ample room for ad-hoc growth if needed and that 
accommodates the fact that Notes handles rich-text items better when they're 
not maxed out at 64K. The stretch limit is used principally on what will 
likely be the last virtual item, allowing the length of that item to be 
somewhat larger so that another, likely small, item doesn't have to be 
created -- an efficiency measure, in other words. When this procedure is used 
against "proxy" virtuality, the stretch limit can regularly come into play 
because the proxy virtuality usually consists of enlarging particular virtual 
items that are about to be replaced.

--- parameters & return ----
puc_Appendage: pointer to the beginning of the content to be appended to any 
	current virtual content
ul_AppendLength: the byte-length of the content to be appended
pt_context: Input & Output. Pointer to environment information about the 
	rich-text field being operated on. Its pointer to virtuality may be 
	initialized by this function.
ul_LEN_TOTAL_ACTUAL: Optional. The aggregate absolute length of all the CD 
	records comprising the rich-text field as it actually exists now, no 
	changes having been committed. Used only in conjunction with stretching 
	the length of what seems to be the last virtual item in order that 
	another, possibly small, virtual item does not have to be created. If 
	passed in null, this "stretching" functionality will be ignored.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- side effects ------------
One somewhat risky behavior of this function is that if the virtual item we 
are initially appending to is filled, its memory will be unlocked by this 
procedure after it creates a follow-on virtual item. If any active pointers 
outside this procedure point to this virtual item, they will be invalidated 
with no way to know it. If the suggested enhancement immediately below is 
implemented, this issue should pose no more risk.

--- suggested enhancement ---
11/26/98 PR
Perhaps some reference-count locking mechanism should be incorporated into 
the ItemInfo structure so that items can be locked and "unlocked" freely with 
no risk of invalidating unknown pointers relying on formerly locked item 
information.

--- revision history --------
2/23/00 PR: accommodated the signature change to us_StartVirtualItemList()
9/12/99 PR: documentation adjustment
12/8/98 PR: created			*/
static STATUS us_VirtualAppend( const BYTE * puc_Appendage, 
								DWORD  ul_AppendLength, 
								RtfTrackingInfo *const  pt_context, 
								const DWORD  ul_LEN_TOTAL_ACTUAL)	{
	const DWORD  ul_REMAINING_FRONTIER = ul_LEN_TOTAL_ACTUAL && 
								pt_context->ul_ActualFrontier != 
								mul_FRONTIER_NO_MORE ? ul_LEN_TOTAL_ACTUAL - 
								pt_context->ul_ActualFrontier : 0;

	DWORD  ul_currLimit;
	ItemInfo * pt_item;
	STATUS  us_error;

	_ASSERTE( puc_Appendage && pt_context);

	//if the caller passed a null length, short-circuit with success
	if (!ul_AppendLength)
		return eus_SUCCESS;

	//if virtuality hasn't been started yet, start it now
	if (!pt_context->pt_Virtuality)
		if (us_error = us_StartVirtualItemList( NULL, pt_context))
			return us_error;

	//Set the initial length limit we will use against the size of the 
	//	current virtual item. If we're already over the "comfortable" limit, 
	//	default to the "stretch" limit size.
	pt_item = pt_context->pt_endVirtual;
	ul_currLimit = pt_item && pt_item->ul_length > mul_COMFY_LIMIT ? 
										mul_STRETCH_LIMIT : mul_COMFY_LIMIT;

	//if adding the specified length of content exceeds the limit we have on 
	//	the length of the current virtual item, run a loop that will fill 
	//	up as many virtual items as necessary until the remaining length 
	//	will fit within an item with space left over for later virtualizing
	while (pt_item->ul_length + pt_item->ul_length % 2 + ul_AppendLength > 
															ul_currLimit)	{
		//if we're using the "comfortable" limit and it seems that the 
		//	current virtual item will be the last and the length to be 
		//	virtualized fits within the "stretch" limit, switch to using the 
		//	stretch limit as the size threshold and break out of this 
		//	"fill full items" loop
		if (ul_currLimit == mul_COMFY_LIMIT && ul_LEN_TOTAL_ACTUAL && 
									pt_item->ul_length + pt_item->ul_length % 
									2 + ul_AppendLength + ul_AppendLength % 
									2 + ul_REMAINING_FRONTIER <= 
									mul_STRETCH_LIMIT)	{
			ul_currLimit = mul_STRETCH_LIMIT;
			break;
		//if we are dealing with special insertions into already written 
		//	virtuality via a proxy virtuality, use the stretch limit to 
		//	squash any memcpying domino effect by trying to fit a few bytes 
		//	into an already full space
		}else if (!pt_context->ul_ActualFrontier)	{
			ul_currLimit = mul_STRETCH_LIMIT;

			//if the switch to a stretch limit now accommodates the length 
			//	to be virtualized within the curent item, break out of this 
			//	"fill full items" loop
			if (pt_item->ul_length + pt_item->ul_length % 2 + 
											ul_AppendLength <= ul_currLimit)
				break;
		} //if (ul_currLimit == mul_COMFY_LIMIT &&

		//Fill up the current virtual item with as much of the content to be 
		//	virtualized as our limit will allow, get ready with the next 
		//	virtual item to fill, and unlock the memory associated with the 
		//	virtual item just filled. In the case of the last step, we are 
		//	assuming that there are no other active pointers into the item 
		//	filled. This is only risky for the virtual item initially passed 
		//	into this procedure, since we create any follow-on items 
		//	ourselves in this loop and so know all there is to know about 
		//	those items.
		if (us_error = us_FillVirtualItemAndStartNext( ul_currLimit, 
											&pt_context->pt_endVirtual, 
											&puc_Appendage, &ul_AppendLength))
			return us_error;
		OSUnlockBlock( pt_item->bid_contents);
		pt_item->puc_start = NULL;

		pt_item = pt_context->pt_endVirtual;
	} //while (pt_context->pt_endVirtual->ul_length +

	//copy whatever content wasn't processed above into the latest curent 
	//	virtual item
	memcpy( pt_item->puc_start + pt_item->ul_length + pt_item->ul_length % 
									2, puc_Appendage, ul_AppendLength);
	pt_item->ul_length += pt_item->ul_length % 2 + ul_AppendLength;

	return eus_SUCCESS;
} //us_VirtualAppend(


/** us_StartVirtualItemList( ***
Purpose is to lay an infrastructure to support the changing of the current 
field's rich-text content in a normalized, flexible way. This infrastructure 
is a set of "virtual" rich-text items similar to the actual items currently 
in use by the note, the only difference being that the actual items have 
been committed (saved) to the note, and therefore owned by the note, whereas 
the virtual items are owned by our process until we decide to commit the 
virtual items and so replace the current actual items.

--- parameter & return ----
ul_LEN_PARTICULAR: Optional. The content length the caller has requested to 
	be allocated to the first virtual item. The procedure will allocate a 
	shorter length if the requested length exceeds the maximum allowed 
	(mul_MAXLEN_RICH_TEXT_ITEM). If null, the item will be set to a default 
	length.
pt_context: Output. Pointer to a structure of item-tracking information about 
	the rich-text field being operated on. Members of this structure related 
	to virtuality will be adjusted by this function.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history ------
2/23/00 PR: extended to allow creation of specific-length items
11/25/98 PR: created		*/
static __inline STATUS us_StartVirtualItemList( 
										const DWORD  ul_LEN_PARTICULAR, 
										RtfTrackingInfo *const  pt_context)	{
	STATUS  us_err;

	_ASSERT( pt_context && !pt_context->pt_Virtuality);

	//create the first virtual item associated with this rich-text field and 
	//	set the current-virtual member of the item-tracking structure being 
	//	used by the rich-text field
	if (us_err = us_CreateVirtualItem( mi_REGULAR_VIRTUAL, 
												ul_LEN_PARTICULAR, 
												&pt_context->pt_Virtuality))
		return us_err;
	pt_context->pt_endVirtual = pt_context->pt_Virtuality;

	return eus_SUCCESS;
} //us_StartVirtualItemList(


/** us_CreateVirtualItem( ***
Purpose is to create a structure for tracking and describing a virtual 
rich-text item.

--- parameters & return ----
i_TYPE: enumerated value specifying the type of virtual item to create: 
	mi_REGULAR_VIRTUAL or mi_SEMI_VIRTUAL
ul_LEN_PARTICULAR: Optional. The content length the caller has requested to 
	be allocated to the item. The procedure will allocate the requested 
	length if the length complys with the constraints associated with the 
	item type specified. If null, the item will be set to a default length.
ppt_item: Output. Memory address of the pointer variable to set to the 
	item structure created.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: extended to allow creation of a specific-length item 
	irrespective of item type
11/28/98 PR: created		*/
static STATUS us_CreateVirtualItem( const VirtualType  i_TYPE, 
									const DWORD  ul_LEN_PARTICULAR, 
									ItemInfo * *const  ppt_item)	{
	const DWORD  ul_SEMI_VIRTUAL_LIMIT = 0x0400 * 5;

	ItemInfo * pt_item;
	STATUS  us_err;

	_ASSERTE( ppt_item && i_TYPE);

	//allocate memory to hold the structure that will track and describe the 
	//	item we're creating, and initialize it's linked-list "next" member to 
	//	null
	if (!( pt_item = malloc( sizeof( ItemInfo))))
		return !eus_SUCCESS;
	pt_item->pt_next = NULL;

	//Allocate memory to hold the content we will be putting into the item. 
	//	If a particular allocation length is requested, comply as long as 
	//	the request is less than the maximum length allowed if a regular 
	//	vitual item (i.e. one appendable to a note) or greater than the 
	//	minimum allowed if the item is simply to hold a stream of CD records. 
	//	In the latter case no limits are set because such items are not 
	//	committed as such to a note; their content must be fully virtualized 
	//	first. We want however to ensure that such items have a minimum 
	//	length so that we won't have the overhead of creating lots of little 
	//	ones linked together.
	if (us_err = OSMemAlloc( NULL, i_TYPE == mi_REGULAR_VIRTUAL ? 
							ul_LEN_PARTICULAR && ul_LEN_PARTICULAR < 
							mul_MAXLEN_RICH_TEXT_ITEM ? ul_LEN_PARTICULAR : 
							mul_MAXLEN_RICH_TEXT_ITEM : ul_LEN_PARTICULAR && 
							ul_LEN_PARTICULAR > ul_SEMI_VIRTUAL_LIMIT ? 
							ul_LEN_PARTICULAR : ul_SEMI_VIRTUAL_LIMIT, 
							&pt_item->bid_contents.pool))	{
		free( pt_item);
		*ppt_item = NULL;
		return us_err;
	}

	//nullify the item block so Notes won't think it owns the block (although 
	//	this is a bit strange since sometimes Notes returns a null block 
	//	member for blocks it owns -- so far I've seen this only on the 
	//	rich-text Body field during the mail-send event), set the item member 
	//	that indicates that the item is virtual and prepare the item so that 
	//	it can be written to
	pt_item->bid_contents.block = NULLBLOCK;
	pt_item->i_type = mi_VIRTUAL;
	pt_item->puc_start = OSLockBlock( BYTE, pt_item->bid_contents);

	//initialize item members according to item type
	if (i_TYPE == mi_REGULAR_VIRTUAL)	{
		*(WORD *) pt_item->puc_start = TYPE_COMPOSITE;
		pt_item->ul_length = sizeof( WORD);
	}else
		pt_item->ul_length = NULL;

	*ppt_item = pt_item;

	return eus_SUCCESS;
} //us_CreateVirtualItem(


/** us_FillVirtualItemAndStartNext( ***
Purpose is to fill the virtual rich-text item currently being populated with 
the specified stream of CD-record content.

--- parameters & return ----
ul_CURR_LIMIT: the maximum allowed size that the current virtual item may 
	become
ppt_item: Input & Output. Memory address of the pointer to the item to which 
	the specified content should be appended, as much as possible. If a 
	follow-on virtual item is created because the appendage fills up the 
	item initially pointed to by this parameter, the pointer will be moved to 
	the beginning of the follow-on item just before the procedure ends. Thus 
	the pointer is always "at-the-ready" for another round of virtualization.
ppuc_Appendage: Input & Output. Memory address of the pointer positioned to 
	the content to be virtualized in this procedure. The pointer will be 
	advanced by this procedure by the number of bytes actually virtualized. 
	So if the content to be appended cannot all fit within the specified 
	virtual item, another call to this procedure will pick up where the last 
	call left off.
pul_AppendLength: Input & Output. Pointer to the variable in which the length 
	of the content to be virtualized is stored. The variable will be 
	decremented by the amount of content actually virtualized by this 
	procedure. So if the content to be appended cannot all fit within the 
	specified virtual item, the caller already has the length it should use 
	in the next call to this procedure so it may pick up where it left off.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: logic shortened slightly
11/28/98 PR: created		*/
static STATUS us_FillVirtualItemAndStartNext( 
										const DWORD  ul_CURR_LIMIT, 
										ItemInfo * *const  ppt_item, 
										const  BYTE * *const  ppuc_Appendage, 
										DWORD *const  pul_AppendLength)	{
	ItemInfo * pt_item = ppt_item ? *ppt_item : NULL;
	DWORD  ul_nextRecLength, ul_copyLength = NULL;
	STATUS  us_error;

	//determine the space available in the virtual item to be appended to
	const DWORD  ul_SPACE_AVAILABLE = pt_item ? ul_CURR_LIMIT - 
												pt_item->ul_length + 
												pt_item->ul_length % 2 : NULL;

	_ASSERTE( ul_CURR_LIMIT && pt_item && ppuc_Appendage && pul_AppendLength);

	//measure out the length of the content that can fit within the space 
	//	available by looping over the succession of CD records that make up 
	//	the content
	ul_nextRecLength = us_getCdRecLength( *ppuc_Appendage);
	while (ul_copyLength + ul_copyLength % 2 + ul_nextRecLength < 
													ul_SPACE_AVAILABLE)	{
		ul_copyLength += ul_copyLength % 2 + ul_nextRecLength;
		ul_nextRecLength = us_getCdRecLength( *ppuc_Appendage + 
										ul_copyLength + ul_copyLength % 2);
	}

	//if the virtual item doesn't have any content in it yet and the first 
	//	CD record is so large that it can't fit within the specified limit...
	if (ul_SPACE_AVAILABLE == ul_CURR_LIMIT && !ul_copyLength)	{
		//if the record is larger than the maximum size that a rich-text 
		//	item can hold, return general failure
		if (ul_nextRecLength >= mul_MAXLEN_RICH_TEXT_ITEM - sizeof( WORD))
			return !eus_SUCCESS;

		//go ahead and allow the item length to be stretched to accommodate 
		//	the very large record.
		ul_copyLength = ul_nextRecLength;
	}

	//copy into the virtual item the content that can fit and adjust the 
	//	item's length accordingly
	memcpy( pt_item->puc_start + pt_item->ul_length + pt_item->ul_length % 2, 
											*ppuc_Appendage, ul_copyLength);
	pt_item->ul_length += pt_item->ul_length % 2 + ul_copyLength;

	//create and move to a follow-on virtual item so we're all ready to 
	//	append the next time this function is called
	if (us_error = us_InsertNextVirtualItem( pt_item, mi_REGULAR_VIRTUAL, 
																		NULL))
		return us_error;
	*ppt_item = pt_item->pt_next;

	//Advance the content pointer by the amout of content we virtualized, 
	//	and decrease the byte length of the content to be appended by a 
	//	corresponding amount. These outputs allow the caller to continuously 
	//	call this procedure until an entire length of content is virtualized.
	*ppuc_Appendage += ul_copyLength + ul_copyLength % 2;
	*pul_AppendLength -= ul_copyLength + ul_copyLength % 2;

	return eus_SUCCESS;
} //us_FillVirtualItemAndStartNext(


/** us_InsertNextVirtualItem( ***
Purpose is to append a new rich-text item to the list of virtual items 
currently in play.

--- parameters & return ----
pt_item: Input & Output. Pointer to information about the virtual rich-text 
	item that precedes the virtual item to be created. The "pt_next" member 
	of the specified item will be reset to reflect the insertion.
i_TYPE: enumerated value specifying the type of virtual item to create: 
	mi_REGULAR_VIRTUAL or mi_SEMI_VIRTUAL
ul_LEN_PARTICULAR: Optional. The content length the caller has requested to 
	be allocated to the item. The procedure will allocate the requested 
	length if the length complys with the constraints associated with the 
	item type specified. If null, the item will be set to a default length.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: documentation & token-name adjustment associated with general 
	enhancement to allow creation of a specific-length item irrespective of 
	item type
11/28/98 PR: created		*/
static __inline STATUS us_InsertNextVirtualItem( 
											ItemInfo *const  pt_item, 
											const VirtualType  i_TYPE, 
											const DWORD  ul_LEN_PARTICULAR)	{
	ItemInfo * pt_newItem;
	STATUS  us_err;

	_ASSERTE( pt_item && i_TYPE);

	//create a virtual item
	if (us_err = us_CreateVirtualItem( i_TYPE, ul_LEN_PARTICULAR, 
																&pt_newItem))
		return us_err;

	//put the new item in place after the passed-in item
	pt_newItem->pt_next = pt_item->pt_next;
	pt_item->pt_next = pt_newItem;

	return eus_SUCCESS;
} //us_InsertNextVirtualItem(


/** eul_GetRtfSpanText( ***
Purpose is to copy a given span of rich-text textual content into a buffer, 
translating rich-text linefeed and paragraph markers to CRLFs if required. 
A text span is delimited by a beginning cursor and either an end cursor or a 
maximum buffer length. If both an end cursor and maximum buffer length are 
provided, the first of the two to be encountered/fulfilled relative to the 
beginning cursor (in terms of the succession of rich-text) will be recognized 
as the span's endpoint.

This function allocates the buffer into which the content is copied, the 
caller is responsible for freeing the memory pointer to the buffer!!

--- parameters & return ----
pt_SPAN: Pointer to a structure describing the span of rich-text whose 
	textual content should be copied The end cursor member's location pointer 
	may be set to null if a maximum content length is specified.
ul_MAXLEN_BUFFER: Optional. 
			provided that an end cursor has been specified in 
			the span-structure parameter. 
	The maximum-desired length of the buffer to 
	be created to contain the copied content. A null value indicates that no 
	maximum limit should be enforced.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be translated to CRLFs when copying content 
	(TRUE) or not (FALSE)
pt_CONTEXT: Optional. Pointer to state information about the rich-text field. 
	If passed in null, no transfer between virtuality and actuality can occur.
ppc: Output. Memory address of the pointer to set to the buffer that this 
	function will create in which to store the rich-text content to be 
	copied. The caller is responsible for freeing the memory associated with 
	this buffer.
RETURN: The length of the buffer allocated by this function to contain the 
	copied rich-text content. eul_ERR_FAILURE if an invalid parameter 
	configuration was passed.

--- side effect -------------
The rich-text environment information pointed to through pt_SPAN or 
pt_CONTEXT may be adjusted due to cursor advancement. If the enhancement 
suggested below is implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/10/98 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution.

--- revision history --------
2/23/00 PR: minor logic shortening
11/23/98 PR: created		*/
DWORD eul_GetRtfSpanText( const CdRecordSpan *const  pt_SPAN, 
							const DWORD  ul_MAXLEN_BUFFER, 
							const BOOL  f_TRANSLATE_LINEFEEDS, 
							const RtfTrackingInfo *const  pt_CONTEXT, 
							char * *const  ppc)	{
	//Determine the maximum-desired content (not buffer) length, considering 
	//	that if rich-text linefeed and paragraph markers are to be translated 
	//	with CRLFs, a non-content terminating null character will be 
	//	necessary. Also determine the content length needed to accommodate 
	//	the specified text span. If a length cannot be determined, an invalid  
	//	rich-text span must have been specified.
	const DWORD  ul_MAXLEN_CONTENT = ul_MAXLEN_BUFFER - 
													(f_TRANSLATE_LINEFEEDS && 
													ul_MAXLEN_BUFFER ? 1 : 0);
	const DWORD  ul_LEN_CONTENT = ul_TextSpanLength( pt_SPAN, 
										ul_MAXLEN_CONTENT, 
										f_TRANSLATE_LINEFEEDS, pt_CONTEXT);

	WORD  us;
	DWORD  ul_lenAggregate = NULL;
	CdRecordCursor  t_cursor;

	//if invalid parameters were passed, short-circuit with general failure
	if (!ppc || ul_LEN_CONTENT == eul_ERR_FAILURE)
		return eul_ERR_FAILURE;

	//Allocate a buffer to the size of the ultimate buffer length, taking 
	//	into account that if rich-text linefeed and paragraph markers are to 
	//	be translated to CRLFs, an extra byte is needed to null-terminate the 
	//	string. Next, if CRLF-translation is needed, set the last byte of the 
	//	allocated buffer to null.
	if (!( *ppc = malloc( ul_LEN_CONTENT + (f_TRANSLATE_LINEFEEDS ? 1 : 0))))
		return eul_ERR_FAILURE;
	if (f_TRANSLATE_LINEFEEDS)
		(*ppc)[ ul_LEN_CONTENT] = NULL;

	//if there's no length to the text span, short-circuit with a return of 
	//	the ultimate content length plus one for null-termination if rich-
	//	text linefeed and paragraph markers were to be translated to CRLFs
	if (!ul_LEN_CONTENT)
		return f_TRANSLATE_LINEFEEDS ? 1 : 0;

	//copy the rich-text span's textual content into the buffer just allocated
	if (!f_ProcessKnownLengthRtfText( pt_SPAN->t_cursorBegin, 
									pt_SPAN->us_offsetBegin, ul_LEN_CONTENT, 
									f_TRANSLATE_LINEFEEDS, pt_CONTEXT, 
									&t_cursor, &us, *ppc))
		goto errJump;

	//If applicable, verify that the ending coordinates of the text copying 
	//	match the end coordinates specified. If they don't, return failure.
	if (pt_SPAN->t_cursorEnd.puc_location && 
									!(pt_SPAN->t_cursorEnd.puc_location == 
									t_cursor.puc_location && 
									pt_SPAN->us_offsetEnd == us))
		goto errJump;

	//return the ultimate text length, incremented by one if rich-text 
	//	linefeed and paragraph markers were translated to CRLFs
	return ul_LEN_CONTENT + (f_TRANSLATE_LINEFEEDS ? 1 : 0);

errJump:
	free( *ppc);
	*ppc = NULL;

	return eul_ERR_FAILURE;
} //eul_GetRtfSpanText(


/** f_ProcessKnownLengthRtfText( ***
Purpose is to serve as a means by which to conduct basic operations vis--vis 
a known length of rich-text textual content. Two basic operations are 
supported: provide the coordinates of the end of the specified length in 
terms of the rich-text constructs of a cursor and offset or copy the textual 
content into a memory buffer. Both operations allow rich-text linefeed and 
paragraph markers to be treated as CRLFs.

Typically the "known length" is derived from a prior call to 
ul_TextSpanLength().

--- parameters & return ----
t_CURSOR: rich-text cursor pointing to the first CD record at which 
	processing should commence on the rich-text field
us_OFFSET: Number of bytes from content-start in the first CD record after 
	which processing should commence. Only meaningful if the first CD record 
	is a CDTEXT.
ul_LEN_TEXT: The ultimate length of the textual content to be processed. This 
	value is crucial and must be already known by the caller, most likely via 
	a prior call to ul_TextSpanLength().
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be translated to CRLFs when processing rich-text 
	textual content (TRUE) or not translated (FALSE)
pt_CONTEXT: Optional. Pointer to state information about the rich-text field. 
	If passed in null, no transfer between virtuality and actuality can occur.
pt_cursorEnd: Optional Output. Pointer to the variable in which to store the 
	cursor pointing to the CD record containing the content that marks the 
	end of the specified length of rich text. If passed in null, this 
	functionality will be ignored.
pus_offsetEnd: Optional Output. Pointer to the variable in which to store the 
	number of content bytes processed in the final CD record. Guaranteed to 
	be null if the final CD record is a CDPARAGRAPH. If passed in null, this 
	functionality will be ignored.
pc: Optional Output. Pointer to the buffer in which to store an unformatted 
	copy of the rich-text content being traversed. If passed in null, the 
	copying functionality will be ignored.
RETURN: TRUE if no errors occurred; FALSE otherwise

--- side effect -------------
The rich-text environment information pointed to through pt_SPAN or 
pt_CONTEXT may be adjusted due to cursor advancement. If the enhancement 
suggested below is implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/10/98 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution.

--- revision history -------
9/12/99 PR
+ minor logic shortening
+ documentation adjustment

11/23/98 PR: created		*/
static BOOL f_ProcessKnownLengthRtfText( 
									const CdRecordCursor  t_CURSOR, 
									const WORD  us_OFFSET, 
									const DWORD  ul_LEN_TEXT, 
									const BOOL  f_TRANSLATE_LINEFEEDS, 
									const RtfTrackingInfo *const  pt_CONTEXT, 
									CdRecordCursor *const  pt_cursorEnd, 
									WORD *const  pus_offsetEnd, 
									char *const  pc)	{
	BYTE  uc;
	const BYTE * puc;
	WORD  us;
	DWORD  ul_length = NULL;
	int  i;
	BOOL  f_MaximumReached = NULL;
	CdRecordCursor  t_cursor;

	_ASSERTE( t_CURSOR.pt_item && t_CURSOR.puc_location && ul_LEN_TEXT);

	//default the outputs to null values
	if (pt_cursorEnd)
		memset( pt_cursorEnd, NULL, sizeof( CdRecordCursor));
	if (pus_offsetEnd)
		*pus_offsetEnd = NULL;
	if (pc)
		*pc = NULL;

	//if the starting cursor is on a CDTEXT or CDPARAGRAPH...
	puc = t_CURSOR.puc_location;
	uc = *puc;
	if (LOBYTE( SIG_CD_TEXT) == uc || LOBYTE( SIG_CD_PARAGRAPH) == uc)	{
		//if the starting cursor is on a CDTEXT
		if (LOBYTE( SIG_CD_TEXT) == uc)	{
			//determine the length of the record's available actual content, 
			//	taking into account any beginning offset
			us = t_CURSOR.us_recLength - sizeof( CDTEXT) - us_OFFSET;

			//set a pointer to the beginning of the record's available actual 
			//	content, taking into account any beginning offset
			puc += sizeof( CDTEXT) + us_OFFSET;
		//else the record is a CDPARAGRAPH, so indicate this by nullifying 
		//	the content pointer
		}else
			puc = NULL;

		//start off with this record's contribution to the rich-text span's 
		//	textual content
		i = i_ProcessRtfText( puc, us, ul_LEN_TEXT, f_TRANSLATE_LINEFEEDS, 
										&ul_length, pc, &f_MaximumReached);

		//if all of the content in the text span has been processed, return 
		//	success after setting the required outputs
		if (ul_length == ul_LEN_TEXT)	{
			if (pt_cursorEnd)
				*pt_cursorEnd = t_CURSOR;
			if (pus_offsetEnd)
				*pus_offsetEnd = us_OFFSET + i;
			return TRUE;
		}

		//if no more data can fit within the specified length, and since 
		//	we haven't matched the specified length exactly, the specified 
		//	length must be invalid, so return failure
		if (f_MaximumReached)
			return FALSE;
	} //if (LOBYTE( SIG_CD_TEXT) == us ||

	//loop forward record-by-record until the specified length of textual 
	//	content has been processed
	t_cursor = t_CURSOR;
	do	{
		//advance to the next CD record, if no further records exist, some 
		//	weird error occurred, so return failure
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, NULL);
		if (!t_cursor.pt_item)
			return FALSE;

		//the current CD record is of interest only if it's a CDTEXT or 
		//	CDPARAGRAH
		uc = *t_cursor.puc_location;
		if (LOBYTE( SIG_CD_TEXT) == uc || LOBYTE( SIG_CD_PARAGRAPH) == uc)	{
			//if the record is a CDTEXT, determine the pointer to the 
			//	beginning of its available content and the content's length
			if (LOBYTE( SIG_CD_TEXT) == uc)	{
				puc = t_cursor.puc_location + sizeof( CDTEXT);
				us = t_cursor.us_recLength - sizeof( CDTEXT);
			//else the record is a CDPARAGRAPH, so indicate this by 
			//	nullifying the content pointer
			}else
				puc = NULL;

			//process this record's contribution to the specified length of 
			//	rich-text textual content
			i = i_ProcessRtfText( puc, us, ul_LEN_TEXT, 
										f_TRANSLATE_LINEFEEDS, 
										&ul_length, pc, &f_MaximumReached);
		} //if (LOBYTE( SIG_CD_TEXT) == uc ||
	} while (!f_MaximumReached);

	//if we haven't matched the specified length exactly, the specified 
	//	length must be invalid, so return failure
	if (ul_length != ul_LEN_TEXT)
		return FALSE;

	//set the required outputs
	if (pt_cursorEnd)
		*pt_cursorEnd = t_cursor;
	if (pus_offsetEnd)
		*pus_offsetEnd = i;

	return TRUE;
} //f_ProcessKnownLengthRtfText(


/** eul_GetRtfText( ***
Purpose is to return the textual content of the rich-text field starting from 
its beginning. If any of the rich-text field has been virtualized, "the 
beginning" is the beginning of virtuality.

--- parameters & return -----
pt_CONTEXT: pointer to state information about the rich-text field
ul_MAXLEN_BUFFER: Optional. The maximum-desired length of the buffer to be 
	created to contain the copied content. A null value indicates that no 
	maximum limit should be enforced.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be translated to CRLFs when copying content 
	(TRUE) or not (FALSE)
ppc_output: Output. Pointer to the pointer to the buffer specifying the 
	buffer that this function will create in which to store the rich-text 
	content to be copied. The caller is responsible for freeing the memory 
	associated with this buffer.
RETURN: The length of the buffer allocated by this function to contain the 
	copied rich-text content. eul_ERR_FAILURE if an invalid parameter 
	configuration was passed.

::: TO DO ::::::::::::
Check out NSFItemConvertToText() to see how that handles CRLFs, tables, etc. 
Consider whether it may be useful here to enhance or replace some of its 
functionality.

--- side effect -------------
The rich-text environment information pointed to through pt_CONTEXT may be 
adjusted due to cursor advancement. If the enhancement suggested below is 
implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/13/98 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution.

--- revision history --------
2/23/00 PR
+ logic shortening & rationalization
+ adjusted arbitrary maximum-length constant for a rich-text field to a 
  reasonable length

11/13/98 PR: created		*/
DWORD eul_GetRtfText( const RtfTrackingInfo *const  pt_CONTEXT, 
						const DWORD  ul_MAXLEN_BUFFER, 
						const BOOL  f_TRANSLATE_LINEFEEDS, 
						char * *const  ppc_output)	{
//	const DWORD  ul_MAXLEN_RTF = 0x00FFFFFF;

	CdRecordSpan  t_dmySpan;

	if (!( pt_CONTEXT && pt_CONTEXT->pt_Actuality && ppc_output))
		return eul_ERR_FAILURE;

	//nullify the dummy rich-text span structure
	memset( &t_dmySpan, NULL, sizeof( CdRecordSpan));

	//initialize the beginning cursor to the beginning of the current 
	//	rich-text field
	if (!f_ConstructStartCursor( pt_CONTEXT, NULL, &t_dmySpan.t_cursorBegin, 
																		NULL))
		return eul_ERR_FAILURE;

	//get the rich-text field's text, returning its length
	return eul_GetRtfSpanText( &t_dmySpan, ul_MAXLEN_BUFFER /*? 
ul_MAXLEN_BUFFER : ul_MAXLEN_RTF*/, 
										f_TRANSLATE_LINEFEEDS, pt_CONTEXT, 
										ppc_output);
} //eul_GetRtfText(


/** eus_CopyRtfIntoBuffer( ***
Copies the specified rich-text content into a memory buffer.

--- parameters & return ----
pt_CTXT: pointer to state information about the rich-text field whose content 
	is to be copied
ph: Output. Address of the variable in which to store the Notes handle to the 
	memory in which the copy will be stored. The un/lock state of the handle 
	is controlled by the optional ppuc paramenter, described below.
pul_len: Output. Address of the variable in which to store the length of the 
	content copied.
ppuc: Optional Output. Address to the pointer in which to store the beginning 
	of the memory buffer in which the rich-text field content will be stored. 
	If null pointer provided, this functionality is ignored and the memory 
	handle will be returned unlocked, else it will be returned locked.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR
+ converted to Notes memory handling instead of malloc/free in order to 
  provide good support for holding onto the allocation for an extended 
  period of time (locking/unlocking, etc.)
+ completed standard documentation

12/10/98 PR: created		*/
STATUS eus_CopyRtfIntoBuffer( const RtfTrackingInfo *const  pt_CTXT, 
								HANDLE *const  ph, 
								DWORD *const  pul_len, 
								BYTE * *const  ppuc)	{
	HANDLE  h;
	DWORD  ul_len, ul;
	BYTE * puc_start, * puc, * puc_frontier;
	ItemInfo * pt_item;
	BOOL  f_ItemGotLocked = FALSE;
	STATUS  us_err;

	if (!( pt_CTXT && (pt_CTXT->pt_Virtuality || pt_CTXT->pt_Actuality) && 
															ph && pul_len))
		return !eus_SUCCESS;

	*pul_len = (DWORD) *ph = NULL;
	if (ppuc)
		*ppuc = NULL;

	if (!( ul_len = ul_LogicalRtfLength( pt_CTXT)))
		return !eus_SUCCESS;
	_ASSERTE( ul_len > 1);

	if (us_err = OSMemAlloc( NULL, ul_len += sizeof( WORD), &h))
		return us_err;
	puc = puc_start = OSLockObject( h);

	*(WORD *) puc = TYPE_COMPOSITE;
	puc += sizeof( WORD);

	//if applicable, copy any virtaul content into the buffer
	if (pt_item = pt_CTXT->pt_Virtuality)	{
		do	{
			ul = pt_item->ul_length - sizeof( WORD);

			if (!pt_item->puc_start)	{
				pt_item->puc_start = OSLockBlock( BYTE, 
													pt_item->bid_contents);
				f_ItemGotLocked = TRUE;
			}
			memcpy( puc += (DWORD) puc % 2, pt_item->puc_start + 
														sizeof( WORD), ul);
			if (f_ItemGotLocked)	{
				OSUnlockBlock( pt_item->bid_contents);
				f_ItemGotLocked = FALSE;
			}

			puc += ul;
		} while (pt_item = pt_item->pt_next);
	} //if (pt_item = pt_CTXT->pt_Virtuality

	//if applicable, add on any content between the actual frontier and the 
	//	end of actuality
	if (pt_CTXT->pt_Actuality && pt_CTXT->ul_ActualFrontier != 
									mul_FRONTIER_NO_MORE && 
									f_LocateAbsoluteOffset( 
									pt_CTXT->ul_ActualFrontier, 
									pt_CTXT->pt_Actuality, &pt_item, 
									&puc_frontier, NULL, &f_ItemGotLocked))	{
		ul = pt_item->puc_start + pt_item->ul_length - puc_frontier;
		memcpy( puc += (DWORD) puc % 2, puc_frontier, ul);
		puc += ul;

		if (f_ItemGotLocked)	{
			OSUnlockBlock( pt_item->bid_contents);
			pt_item->puc_start = NULL;
		}

		while (pt_item = pt_item->pt_next)	{
			ul = pt_item->ul_length - sizeof( WORD);

			if (!pt_item->puc_start)	{
				pt_item->puc_start = OSLockBlock( BYTE, 
													pt_item->bid_contents);
				f_ItemGotLocked = TRUE;
			}
			memcpy( puc += (DWORD) puc % 2, pt_item->puc_start + 
														sizeof( WORD), ul);
			if (f_ItemGotLocked)	{
				OSUnlockBlock( pt_item->bid_contents);
				f_ItemGotLocked = FALSE;
			}

			puc += ul;
		} //while (pt_item = pt_item->pt_next)
	} //if (pt_CTXT->pt_Actuality &&
	
	_ASSERTE( puc == puc_start + ul_len);

	*ph = h;
	*pul_len = ul_len;
	if (ppuc)
		*ppuc = puc_start;
	else
		OSUnlockObject( h);

	return eus_SUCCESS;
} //eus_CopyRtfIntoBuffer(


/** ul_LogicalRtfLength( ***


--- parameter & return ----

RETURN: 

--- revision history ------
2/23/00 PR: minor documentation adjustment
1/16/99 PR: created		*/
//DOC!!
static DWORD ul_LogicalRtfLength( const RtfTrackingInfo *const  pt_CONTEXT)	{
	ItemInfo * pt_item;
	DWORD  ul = NULL;
	BOOL  f_ItemGotLocked;
	BYTE * puc;

	_ASSERTE( pt_CONTEXT && (pt_CONTEXT->pt_Virtuality || 
												pt_CONTEXT->pt_Actuality));

	//if applicable, determine the total length of the content virtualized 
	//	so far
	if (pt_item = pt_CONTEXT->pt_Virtuality)
		do
			ul += pt_item->ul_length - sizeof( WORD);
		while (pt_item = pt_item->pt_next);

	//if applicable, add on the length of any content between the actual 
	//	frontier and the end of actuality
	if (pt_CONTEXT->pt_Actuality && pt_CONTEXT->ul_ActualFrontier != 
										mul_FRONTIER_NO_MORE && 
										f_LocateAbsoluteOffset( 
										pt_CONTEXT->ul_ActualFrontier, 
										pt_CONTEXT->pt_Actuality, &pt_item, 
										&puc, NULL, &f_ItemGotLocked))	{
		ul += ul % 2 + pt_item->puc_start + pt_item->ul_length - puc;
		if (f_ItemGotLocked)	{
			OSUnlockBlock( pt_item->bid_contents);
			pt_item->puc_start = NULL;
		}

		while (pt_item = pt_item->pt_next)
			ul += ul % 2 + pt_item->ul_length - sizeof( WORD);
	} //if (pt_CONTEXT->pt_Actuality &&

	return ul;
} //ul_LogicalRtfLength(


/** ul_TextSpanLength( ***
Purpose is to determine the size of the specified span of rich-text textual 
content, no matter if rich-text linefeed and paragraph markers should be 
translated to CRLFs. A text span is delimited by a beginning cursor and 
either an end cursor or a maximum content length. If both an end cursor and 
maximum content length are provided, the first of the two to be 
encountered/fulfilled relative to the beginning cursor (in terms of the 
succession of rich-text) will be recognized as the span's endpoint.

--- parameters & return ----
pt_SPAN: Pointer to the structure specifying the span of rich-text whose 
	textual content is to be sized. The end cursor member's location pointer 
	may be set to null if a maximum content length is specified.
ul_MAXLEN_CONTENT: Optional.
			provided that an end cursor has been specified in 
			the span-structure parameter. 
	The maximum-desired length of any content 
	to be sized. Acts as a short-circuit on the sizing process. A null value 
	indicates that no maximum constriant is to be applied.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeed and 
	paragraph markers should be sized as CRLFs when determining ultimate size
pt_CONTEXT: Optional. Pointer to state information about the rich-text field. 
	If passed in null, no transfer between virtuality and actuality can occur.
RETURN: If successful, the length of the the span's textual content. 
	eul_ERR_FAILURE if the parameters specify an invalid rich-text span.

--- revision history -------
2/23/00 PR: minor logic shortening
11/24/98 PR: created		*/
static DWORD ul_TextSpanLength( const CdRecordSpan *const  pt_SPAN, 
								const DWORD  ul_MAXLEN_CONTENT, 
								const BOOL  f_TRANSLATE_LINEFEEDS, 
								const RtfTrackingInfo *const  pt_CONTEXT)	{
	const BYTE *const  puc_END_RECORD = pt_SPAN ? 
									pt_SPAN->t_cursorEnd.puc_location : NULL;

	const BYTE * puc = pt_SPAN->t_cursorBegin.puc_location;
	BYTE uc = *puc;
	WORD  us;
	DWORD  ul_length = NULL;
	BOOL  f_MaximumReached;
	CdRecordCursor  t_cursor;

	//if any of the parameters are obviously invalid, short-circuit with 
	//	general failure
//A valid span requires either a maximum-desired 
//	buffer length (not content length) or a valid end cursor. 
	if (!( pt_SPAN && puc && pt_SPAN->t_cursorBegin.pt_item && 
										(puc_END_RECORD ? 
										(BOOL) pt_SPAN->t_cursorEnd.pt_item : 
										TRUE) /*|| ul_MAXLEN_CONTENT)*/ && 
										(pt_SPAN->t_cursorBegin.pt_item == 
										pt_SPAN->t_cursorEnd.pt_item ? 
										puc < puc_END_RECORD : TRUE)))
		return eul_ERR_FAILURE;

	//if the starting cursor is on a CDTEXT or CDPARAGRAPH...
	if (LOBYTE( SIG_CD_TEXT) == uc || LOBYTE( SIG_CD_PARAGRAPH) == uc)	{
		//if the starting cursor is on a CDTEXT...
		if (LOBYTE( SIG_CD_TEXT) == uc)	{
			//if the beginning and ending cursors point to the same CD record 
			//	and the ending offset is less than the beginning offset, 
			//	short-circuit with a null content length
			if (puc == puc_END_RECORD && pt_SPAN->us_offsetEnd < 
													pt_SPAN->us_offsetBegin)
				return NULL;

			//determine the length of the record's available actual content, 
			//	taking into account any beginning offset and, if the starting 
			//	and ending cursor point to the same record, any ending offset
			if (puc != puc_END_RECORD)
				us = pt_SPAN->t_cursorBegin.us_recLength - sizeof( CDTEXT) - 
													pt_SPAN->us_offsetBegin;
			else
				us = pt_SPAN->us_offsetEnd - pt_SPAN->us_offsetBegin;

			//set a pointer to the beginning of the record's available actual 
			//	content, taking into account any beginning offset
			puc += sizeof( CDTEXT) + pt_SPAN->us_offsetBegin;
		//else the record is a CDPARAGRAPH, so indicate this by nullifying 
		//	the content pointer
		}else
			puc = NULL;

		//start off the current-length variable by the amount of the record's 
		//	content that could be included under any specified maximum limit
		i_ProcessRtfText( puc, us, ul_MAXLEN_CONTENT, 
										f_TRANSLATE_LINEFEEDS, &ul_length, 
										NULL, &f_MaximumReached);

		//if applicable and the maximum-desired content length has been met, 
		//	return that the span length is the value of the current-length 
		//	variable
		if (ul_MAXLEN_CONTENT && f_MaximumReached)
			return ul_length;
	//else if the beginning and ending cursors point to the same CD record, 
	//	short-circuit with a null content length
	}else if (puc == puc_END_RECORD)
		return NULL;

	//advance to the next CD record
	t_cursor = pt_SPAN->t_cursorBegin;
	AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, NULL);

	//if no further records exist...
	if (!t_cursor.pt_item)
		//if an end cursor was specified, we failed to make it there, so 
		//	return general failure, else return that the content length 
		//	is the value of the current-length variable
		return puc_END_RECORD ? eul_ERR_FAILURE : ul_length;

	//loop until the CD record the end cursor points to is reached
	while (t_cursor.puc_location != puc_END_RECORD)	{
		//the current CD record is of interest only if it's a CDTEXT or 
		//	CDPARAGRAH
		uc = *t_cursor.puc_location;
		if (LOBYTE( SIG_CD_TEXT) == uc || LOBYTE( SIG_CD_PARAGRAPH) == uc)	{
			//if the record is a CDTEXT, determine the pointer to the 
			//	beginning of its available content and the content's length
			if (LOBYTE( SIG_CD_TEXT) == uc)	{
				puc = t_cursor.puc_location + sizeof( CDTEXT);
				us = t_cursor.us_recLength - sizeof( CDTEXT);
			//else the record is a CDPARAGRAPH, so indicate this by 
			//	nullifying the content pointer
			}else
				puc = NULL;

			//increment the current-length variable by the amount of the 
			//	record's content which could be included under any specified 
			//	maximum limit
			i_ProcessRtfText( puc, us, ul_MAXLEN_CONTENT, 
										f_TRANSLATE_LINEFEEDS, &ul_length, 
										NULL, &f_MaximumReached);

			//if applicable and the maximum-desired content length has been 
			//	met, return that the content length is the value of the 
			//	current-length variable
			if (ul_MAXLEN_CONTENT && f_MaximumReached)
				return ul_length;
		} //if (LOBYTE( SIG_CD_TEXT) == uc ||

		//advance the current cursor to the next CD record
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, NULL);

		//if no further records exist...
		if (!t_cursor.pt_item)
			//if an end cursor was specified, we failed to make it there, so 
			//	return general failure, else return that the content length 
			//	is the value of the current-length variable
			return puc_END_RECORD ? eul_ERR_FAILURE : ul_length;
	} //while (t_cursor.puc_location != puc_END_RECORD)

	//if this end CD record is a CDTEXT or CDPARAGRAH...
	uc = *pt_SPAN->t_cursorBegin.puc_location;
	if (LOBYTE( SIG_CD_TEXT) == uc || LOBYTE( SIG_CD_PARAGRAPH) == uc)	{
		//if the ending cursor is on a CDTEXT, set a pointer to the beginning 
		//	of the record's available actual content, and set the length of 
		//	the record's available actual content equal to the ending offset
		if (LOBYTE( SIG_CD_TEXT) == uc)	{
			puc = puc_END_RECORD + sizeof( CDTEXT);
			us = pt_SPAN->us_offsetEnd;
		//else the record is a CDPARAGRAPH, so indicate this by nullifying 
		//	the content pointer
		}else
			puc = NULL;

		//one last time, increment the current-length variable by the amount 
		//	of the record's content that could be included under any 
		//	specified maximum limit
		i_ProcessRtfText( puc, us, ul_MAXLEN_CONTENT, f_TRANSLATE_LINEFEEDS, 
													&ul_length, NULL, NULL);
	} //if (LOBYTE( SIG_CD_TEXT) == us ||

	//return that the content length is the value of the current-length 
	//	variable 
	return ul_length;
} //ul_TextSpanLength(


/** i_ProcessRtfText( ***
For the given CD record, this function appends sizing information about the 
record's textual content and, if requested, appends the record's content to 
a provided buffer.

--- parameters & return ---
pc_content: Pointer to the rich-text textual content to be sized or copied. 
	A null value indicates that a paragraph equivalent should be apppended.
us_LEN_AVAILABLE: the length of the current textual content available to 
	be sized or copied
ul_MAXLEN_CONTENT: Optional. The maximum-desired length of the rich-text 
	content being sized or copied. If null, this constraint functionality is 
	bypassed.
f_TRANSLATE_LINEFEEDS: flag specifying whether rich-text linefeeds and 
	paragraph markers should be translated to a CRLF for the sake of sizing 
	and copying (TRUE) or not (FALSE)
pul_lenAggregate: Input & Output. Pointer to the variable holding the 
	aggregate length of the rich-text thus far sized or copied. The variable 
	is incremented according to the number of bytes of additional content 
	copied or able-to-be-copied given any constraints.
pc_contentBuf: Pointer to a buffer into which the rich-text content should 
	be copied. A null pointer indicates that the content should only be 
	sized, not copied.
pf_MaximumReached: Optional. Pointer to a flag variable in which to return 
	whether the maximum-desired content length has been met (TRUE) or not 
	(FALSE). Only applicable when ul_MAXLEN_CONTENT has been specified. A 
	null pointer indicates that this functionality should be ignored.
RETURN: If dealing with a CDTEXT, the offset corresponding to the number of 
	content bytes processed. If the first content byte cannot be processed, 
	a return of -1 will be received, meaning that the ultimate last content 
	byte is the preceding CD record's last content byte.

--- revision history -------
2/23/00 PR: documenation adjustment
11/23/98 PR: created		*/
static int i_ProcessRtfText( const char *const  pc_CONTENT, 
								const WORD  us_LEN_AVAILABLE, 
								const DWORD  ul_MAXLEN_CONTENT, 
								const BOOL  f_TRANSLATE_LINEFEEDS, 
								DWORD *const  pul_lenAggregate, 
								BYTE  pc_contentBuf[], 
								BOOL *const  pf_MaximumReached)	{
	//determine whether there's any chance that an applicable 
	//	maximum-desired content length could be exhausted by processing  
	//	the specified rich-text content
	const BOOL f_CHANCE_OF_MAX = ul_MAXLEN_CONTENT && *pul_lenAggregate + 
										us_LEN_AVAILABLE * 
										(f_TRANSLATE_LINEFEEDS ? 
										ei_LEN_CRLF : 1) > ul_MAXLEN_CONTENT;

	WORD  i;

	_ASSERTE( pul_lenAggregate);

	//if applicable, default to FALSE the flag telling whether the maximum 
	//	content length has been reached
	if (pf_MaximumReached)
		*pf_MaximumReached = FALSE;

	//if the curent cursor is on a CDTEXT...
	if (pc_CONTENT)	{
		//if no translation of rich-text linefeed or paragraph markers 
		//	is requested
		if (!f_TRANSLATE_LINEFEEDS)	{
			//determine the length of the content of interest, accounting 
			//	if necessary for the maximum limit on content length
			i = us_LEN_AVAILABLE - (f_CHANCE_OF_MAX ? 
												(WORD) (*pul_lenAggregate + 
												us_LEN_AVAILABLE - 
												ul_MAXLEN_CONTENT) : 0);

			//if applicable, append the content of interest to the 
			//	copied-content buffer
			if (pc_contentBuf)
				memcpy( pc_contentBuf + *pul_lenAggregate, pc_CONTENT, i);

			//increment the current-length variable by the length of the 
			//	content of interest
			*pul_lenAggregate += i;
		//else it's most efficient to work character-by-character, accounting 
		//	for linefeed-to-CRLF translation as linefeeds occur
		}else
			//loop forward from the first to the last character of the 
			//	available content
			for (i = 0; i < us_LEN_AVAILABLE; i++)	{
				//if the pointed-to character is not a rich-text linefeed, 
				//	increment the current-length variable and, if applicable, 
				//	append the character to the copied-content buffer
				if (pc_CONTENT[ i] != mc_LINEFEED_RICHTEXT)	{
					if (pc_contentBuf)
						pc_contentBuf[ *pul_lenAggregate] = pc_CONTENT[ i];
					(*pul_lenAggregate)++;
				//else if no chance exists that the specified maximum limit
				//	will be exceeded, increment the current-length variable 
				//	by the size of a CRLF and, if applicable, append the CRLF 
				//	to the copied-content buffer
				}else if (!f_CHANCE_OF_MAX)	{
					if (pc_contentBuf)
						memcpy( pc_contentBuf + *pul_lenAggregate, epc_CRLF, 
																ei_LEN_CRLF);
					*pul_lenAggregate += ei_LEN_CRLF;
				//Else if applicable and the current-length plus the size 
				//	of the replacement CRLF exceeds the specified maximum 
				//	length, return that the maximum-desired content length 
				//	has been reached at the previous content character. This 
				//	test removes the possibility of a split CRLF occurring at 
				//	the end of the content buffer being filled/sized.
				}else if (ul_MAXLEN_CONTENT && *pul_lenAggregate + 
										ei_LEN_CRLF > ul_MAXLEN_CONTENT)	{
					if (pf_MaximumReached)
						*pf_MaximumReached = TRUE;

					return i - 1;
				//else increment the current-length variable by the size of 
				//	a CRLF and, if applicable, append the CRLF to the 
				//	copied-content buffer
				}else	{
					if (pc_contentBuf)
						memcpy( pc_contentBuf + *pul_lenAggregate, epc_CRLF, 
																ei_LEN_CRLF);
					*pul_lenAggregate += ei_LEN_CRLF;
				} //if (pc_content[ i] != c_LINEFEED_RICHTEXT)
			} //for (i = 0; i < us_LEN_AVAILABLE; i++)
		//END if (!f_TRANSLATE_LINEFEEDS)
	//else the current cursor must be on a CDPARAGRAPH...
	}else	{
		//if rich-text linefeeds and paragraph markers are to be translated 
		//	to CRLFs
		if (f_TRANSLATE_LINEFEEDS)	{
			//If applicable and the current-length plus the size of the 
			//	replacement CRLF exceeds the specified maximum length, return 
			//	that the maximum-desired content length has been reached. 
			//	This test removes the possibility of a split CRLF occurring 
			//	at the end of the content buffer being filled/sized.
			if (ul_MAXLEN_CONTENT && *pul_lenAggregate + ei_LEN_CRLF > 
														ul_MAXLEN_CONTENT)	{
				if (pf_MaximumReached)
					*pf_MaximumReached = TRUE;
				return NULL;
			//else increment the current-length variable by the size of a 
			//	CRLF and, if applicable, append a CRLF to the copied-content 
			//	buffer
			}else	{
				if (pc_contentBuf)
					memcpy( pc_contentBuf + *pul_lenAggregate, epc_CRLF, 
																ei_LEN_CRLF);
				*pul_lenAggregate += ei_LEN_CRLF;
			} //if (ul_MAXLEN_CONTENT && *pul_lenAggregate +
		//else increment the current-length variable and, if applicable, 
		//	append a rich-text linefeed to the copied-content buffer
		}else	{
			if (pc_contentBuf)
				pc_contentBuf[ *pul_lenAggregate] = mc_LINEFEED_RICHTEXT;
			*pul_lenAggregate;
		} //if (f_TRANSLATE_LINEFEEDS)
	} //if (pc_content)

	//if applicable and the maximum-desired content length has been met, 
	//	indicate that the maximum-desired content length has been reached
	if (ul_MAXLEN_CONTENT && pf_MaximumReached && *pul_lenAggregate == 
															ul_MAXLEN_CONTENT)
		*pf_MaximumReached = TRUE;

	//if we're dealing with a CDTEXT, return the offset into the content at 
	//	which processing was stopped (usually it will mark the entire content)
	return pc_CONTENT ? i : NULL;
} //i_ProcessRtfText(


/** eus_CursorToStringStart( ***
Purpose is to case-sensitive search for and point to the next instance of a 
passed-in string within the rich-text field whose context is passed in as 
well. A successful find means that the string has been discovered in, at its 
most complex, a contiguous succession of CDTEXT records, "contiguous" meaning 
uninterrupted by any other type of CD record.

--- parameters & return ----
PC: pointer to the string to be searched for
f_CASE_SENSITIVE: flag telling whether the search should be case-sensitive 
t_cursor: the starting point for the search within the rich-text field
us_OFFSET_AT_START: if the starting point is a CDTEXT record, the offset into 
	the record's text run at which the search should commence
pt_context: Optional. Pointer to contextual information about the rich-text 
	field being navigated. If null, only the stream of CD records following 
	t_cursor may be navigated, meaning that no switching from virtuality to 
	actuality can occur.
pt_foundCursor: Output. Pointer to the rich-text cursor in which to store the 
	cordinates of the CDTEXT containing at least the start of a found 
	instance of the string that was searched for. If no instance was found, 
	this cursor's location pointer is guaranteed to be NULL.
pus_foundOffset: Output. Pointer to the variable in which to store the 
	1-based offset into the CDTEXT's content after which an instance of the 
	sought-for string was found.
RETURN: TRUE if no error occured. FALSE otherwise

--- side effect --------
The rich-text environment information pointed to through t_cursor or 
pt_CONTEXT may be adjusted due to cursor advancement. If the enhancement 
suggested below is implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/10/98 PR
The starting memory-lock state should be restored upon function exit (absent 
the item containing the "found cursor," of course), but no good mechanism is 
yet available with which to accomplish this. See this module's 
suggested-enhancement note for a description of the problem and proposed 
solution.

--- revision history -------
9/12/99 PR: documentation adjustment
11/22/98 PR: created		*/
BOOL ef_CursorToStringStart( const char  PC[], 
								const BOOL  f_CASE_SENSITIVE, 
								CdRecordCursor  t_cursor, 
								const WORD  us_OFFSET_AT_START, 
								const RtfTrackingInfo *const  pt_CONTEXT, 
								CdRecordCursor *const  pt_foundCursor, 
								WORD *const  pus_foundOffset)	{
	char * pc = NULL;
	BOOL  f_found = FALSE, f_result = TRUE;

	//if obvious invalidity among the parameters is present, short-circuit 
	//	with general failure
	if (!( PC && t_cursor.pt_item && t_cursor.puc_location && 
									pt_foundCursor && pus_foundOffset && 
									(pt_CONTEXT ? 
									(BOOL) pt_CONTEXT->pt_Actuality : TRUE)))
		return FALSE;

	//default the found cursor's location to NULL
	pt_foundCursor->puc_location = NULL;

	//if this is a non-case-sensitive search, convert the passed-in string 
	//	to upper-case, as required by the f_TestStringStart() function
	if (!f_CASE_SENSITIVE)	{
		if (!( pc = calloc( strlen( PC) + 1, sizeof( char))))
			return !eus_SUCCESS;
		strupr( strcpy( pc, PC));
	}

	//if the passed-in cursor is at a CDTEXT and the offset-at-start value is 
	//	reasonable, test whether a string match is found beginning in the 
	//	CDTEXT
	if (LOBYTE( SIG_CD_TEXT) == *t_cursor.puc_location && 
									us_OFFSET_AT_START + sizeof( CDTEXT) < 
									t_cursor.us_recLength)
		if (!( f_result = f_TestStringStart( pc ? pc : PC, f_CASE_SENSITIVE, 
									t_cursor, us_OFFSET_AT_START, pt_CONTEXT, 
									&f_found, pus_foundOffset)))
			goto cleanUp;

	//If an instance of the sought-for string wasn't found in the first CDTEXT
	//	from the passed-in offset, search forward through the rest of the 
	//	rich-text field for an instance. Offsets are of no concern anymore.
	while (!f_found)	{
		//Advance the current cursor to the next CD record.  If no 
		//	rich-text is left, break out of the loop.
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, NULL);
		if (!t_cursor.pt_item)
			break;

		//if the cursor is not at a CDTEXT, iterate the loop.
		if (LOBYTE( SIG_CD_TEXT) != *t_cursor.puc_location)
			continue;

		//if a string match is found beginning in the CDTEXT...
		if (!( f_result = f_TestStringStart( pc ? pc : PC, f_CASE_SENSITIVE, 
												t_cursor, 0, pt_CONTEXT, 
												&f_found, pus_foundOffset)))
			goto cleanUp;
	} //while (!f_found)

	//If an instance of the sought-for string was found, set the "found" 
	//	cursor equal to the cursor where the instance was located. The  
	//	"found" offset relative to the start of the sought-for string 
	//	was set during the search.
	if (f_found)
		*pt_foundCursor = t_cursor;
	//else nullify the puc_location member of the "found" cursor to signify 
	//	that no instance was found
	else
		pt_foundCursor->puc_location = NULL;

cleanUp:
	//if memory was allocated to accommodate a non-case-sensitive search, 
	//	free it
	if (pc)
		free( pc);

	return f_result;	//probably TRUE
} //ef_CursorToStringStart(


/** AdvanceCdCursor( ***
Purpose is to advance the given rich-text cursor to its next CD record, 
flexibly taking into account the rich-text field environment.

--- parameters -------------
pt_cursor: Input & Output. Pointer to the cursor to be advanced. Output is 
	either an advanced cursor or a cursor whose item pointer is nullified to 
	indicate that no ensuing CD records are available.
pt_CONTEXT: Optional Input. Pointer to the context of the rich-text 
	information being navigated. If passed in null, the cursor will not 
	travel automatically from a virtual or semi-virtual rich-text item to 
	an ensuing actual rich-text item.
pf_LeftSemiVirtuality: Optional Output. Pointer to the flag in which to store 
	whether the cursor has advanced out of semi-virtual rich-text content. If 
	passed in null, the procedure will ignore this functionality.
pf_ItemGotLocked: Optional. Pointer to the flag in which to store whether 
	a rich-text item was newly locked in the course of this procedure. If 
	passed in null, the procedure will ignore this functionality.

--- side effect ------------
The rich-text environment information pointed to by pt_cursor or pt_context 
may be adjusted due to cursor advancement.

--- suggested enhancment ---
2/23/00 PR: In the case where an item's reported length exceeds the actual 
	length of valid CD content, the length member is not being corrected. It 
	probably could be corrected (carefully!), but an interested caller should 
	be notified if any update occurred.

--- revision history -------
2/23/00 PR
+ improved fault tolerance to handle the case where an item's reported length 
  exceeds the actual length of valid CD content
+ documentation adjustment

11/23/98 PR: created		*/
static void AdvanceCdCursor( CdRecordCursor *const  pt_cursor, 
								const RtfTrackingInfo *const  pt_CONTEXT, 
								BOOL *const  pf_LeftSemiVirtuality, 
								BOOL *const  pf_ItemGotLocked)	{
	BYTE * puc = pt_cursor->puc_location + pt_cursor->us_recLength + 
												pt_cursor->us_recLength % 2;

	_ASSERTE( pt_cursor && pt_cursor->puc_location && pt_cursor->pt_item && 
											(pt_CONTEXT ? (BOOL) 
											pt_CONTEXT->pt_Actuality : TRUE));

	//as applicable, indicate as a default that semi-virtuality has not been 
	//	left and no the memory associated with an item different from the 
	//	current item has been locked
	if (pf_LeftSemiVirtuality)
		*pf_LeftSemiVirtuality = FALSE;
	if (pf_ItemGotLocked)
		*pf_ItemGotLocked = FALSE;

	//If cursor is not on the last CD record of the current item... ("Last 
	//	record" is determined according to the following algorithm which 
	//	Notes itself seems to follow: <1> if the reported item length has 
	//	been exceeded [the '+ 1' is a measure to ensure that the last CD 
	//	record does not end one byte short of the length of the actual item 
	//	stored by Notes {a situation that occurs rarely, perhaps as a result 
	//	of some previous truncation of the item}] or <2> if the ostensible 
	//	next CD record has a null, and thus invalid, signature.)
	if ((DWORD) (pt_cursor->puc_location - pt_cursor->pt_item->puc_start + 
											pt_cursor->us_recLength + 1) < 
											pt_cursor->pt_item->ul_length && 
											LOBYTE( *puc))
		//simply advance forward to the next record
		pt_cursor->puc_location = puc;
	//else if cursor is at the end of anything virtual and the caller has 
	//	chosen this option...
	else if (pt_CONTEXT && !(pt_cursor->pt_item->pt_next || 
								pt_cursor->pt_item->i_type == mi_ACTUAL))	{
		//if the caller has chosen the option, note whether we're leaving 
		//	semi-virtuality
		if (pf_LeftSemiVirtuality && pt_CONTEXT->pt_SemiVirtuality && 
										f_ItemIsInList( pt_cursor->pt_item, 
										pt_CONTEXT->pt_SemiVirtuality))
			*pf_LeftSemiVirtuality = TRUE;

		//Set the cursor to point to the "next" record. If there aren't any 
		//	more records, quit this town; the pt_item member of the cursor 
		//	will be nullified to indicate the condition.
		if (!f_LocateAbsoluteOffset( pt_CONTEXT->ul_ActualFrontier, 
											pt_CONTEXT->pt_Actuality, 
											&pt_cursor->pt_item, 
											&pt_cursor->puc_location, NULL, 
											pf_ItemGotLocked))
			return;
	//else since cursor is on the last CD record of a rich-text item (actual 
	//	or virtual)...
	}else	{
		//If no further items are available, we've reached the end of all 
		//	rich text known to this function, so return. The cursor's null 
		//	item pointer will indicate the condition to the caller.		
		if (!( pt_cursor->pt_item = pt_cursor->pt_item->pt_next))
			return;

		//if the content of the succeeding item hasn't been locked in memory, 
		//	do it now, and if applicable, set to TRUE the flag telling 
		//	whether a rich-text item different from the item containing the 
		//	starting cursor was locked
		if (!pt_cursor->pt_item->puc_start)	{
			pt_cursor->pt_item->puc_start = OSLockBlock( BYTE, 
											pt_cursor->pt_item->bid_contents);
			if (pf_ItemGotLocked)
				*pf_ItemGotLocked = TRUE;
		}

		//Set cursor to the first CD record in the item.  For non 
		//	semi-virtual items, the WORD increment steps over the lead 
		//	TYPE_COMPOSITE item designation
		pt_cursor->puc_location = pt_cursor->pt_item->puc_start + 
											(pt_CONTEXT && f_ItemIsInList( 
											pt_cursor->pt_item, 
											pt_CONTEXT->pt_SemiVirtuality) ? 
											NULL : sizeof( WORD));
	} //if ((DWORD) (pt_cursor->puc_location -

	//store in the cursor structure the length of the just-moved-to CD record
	pt_cursor->us_recLength = us_getCdRecLength( pt_cursor->puc_location);
} //AdvanceCdCursor(


/** f_ItemIsInList( ***
Purpose is to determine whether a given rich-text item appears in a given 
linked-list of rich-text items.

--- parameters & return ----
pt_ITEM: pointer to the rich-text item to test for appearance
pt_HEAD_NODE: pointer to the head node of the list to be searched
RETURN: TRUE if the given item appears in the list; FALSE otherwise

--- revision history -------
10/27/98 PR: created		*/
static __inline BOOL f_ItemIsInList( 
									const ItemInfo *const  pt_ITEM, 
									const ItemInfo *const  pt_HEAD_NODE)	{
	const ItemInfo * pt_item;

	_ASSERTE( pt_ITEM);

	//if the list passed in is empty, short-circuit that the item is not 
	//	in the list
	if (!( pt_item = pt_HEAD_NODE))
		return FALSE;

	//loop through the items in the passed-in list
	do	{
		//if the passed-in item equates to the list item, return that 
		//	the passed-in item does appear in the list
		if (pt_ITEM->bid_contents.pool == pt_item->bid_contents.pool)
			return TRUE;
	} while( pt_item = pt_item->pt_next);

	//since we've made it through the list without a match, return 
	//	that the passed-in item does not appear in the list
	return FALSE;
} //f_ItemIsInList(


/** f_LocateAbsoluteOffset( ***
Purpose is to translate a logical offset into a succession of CD records into 
the physical location of the content pointed to by the offset. The logical 
offset is spedified in terms of bytes and ignores the fact that rich-text 
_items_ are concatenated to comprise a rich-text _field_. Rather the offset 
views the succession of rich-text CD records as uniterrupted. This function 
translates the logical offset into pointers to the rich-text item containing 
the logical offset and to the actual location in that item of the CD record 
"pointed to" by the logical offset.

--- parameters & return ----
ul_OFFSET: the logical offset to be translated into physical location pointers
pt_headNode: pointer to the item from which the logical offset is to be 
	measured
ppt_item: Output. Pointer to the address in which to store the pointer to the 
	rich-text item containing the logical offset. The address is guaranteed 
	to be null if the offset is invalid for some reason.
ppuc_location: Output. Pointer to the address in which to store the pointer 
	to the CD record "pointed to" by the logical offset.
pul_AggregateLength: Optional Output. Pointer to the variable in which to 
	store the aggregate length of the physical rich-text items up to and 
	including the item containing the logical offset. If pointer is null, 
	this functionality will be ignored.
pf_ItemGotLocked: Optional Output. Pointer to the variable in which to store 
	whether the memory containing the content of the rich-text item in which 
	the logical offset falls was locked in order that pointers into the 
	content could be set. If pointer is null, this functionality will be 
	ignored.
RETURN: TRUE if the absolute offset was successfully located; FALSE otherwise

--- revision history -------
11/18/98 PR: created		*/
static BOOL f_LocateAbsoluteOffset( DWORD  ul_OFFSET, 
									ItemInfo *const  pt_headNode, 
									ItemInfo * *const  ppt_item, 
									const BYTE * *const  ppuc_location, 
									DWORD *const  pul_AggregateLength, 
									BOOL *const  pf_ItemGotLocked)	{
	ItemInfo * pt_item = pt_headNode;
	DWORD  ul_lengthCount = (DWORD) NULL;

	_ASSERTE( pt_headNode && ppt_item && ppuc_location);

	//Default to null the variable in which the pointer to the rich-text 
	//	item containing the logical offset will be stored, and if applicable, 
	//	the flag telling whether the content of the rich-text item in which 
	//	the logical offset falls had to be memory-locked.
	*ppt_item = NULL;
	if (pf_ItemGotLocked)
		*pf_ItemGotLocked = FALSE;

	//get the length of all actual items up to and including the one 
	//	containing the logical offset
	while ( (ul_lengthCount += pt_item->ul_length) < ul_OFFSET)	{
		//if needed, add on "even out" byte
		ul_lengthCount += pt_item->ul_length % 2;

		if (!( pt_item = pt_item->pt_next))
			return FALSE;
	}

	//if the aggregate length equals the logical offset, the offset must 
	//	point to invalid territory, so return that the offset could not be 
	//	located
	if (ul_lengthCount == ul_OFFSET)
		return FALSE;

	//If necessary, lock the memory containing the content of the rich-text 
	//	item containing the logical offset. Also, if the caller requests, 
	//	communicate whether a lock was made.
	if (!pt_item->puc_start)	{
		pt_item->puc_start = OSLockBlock( BYTE, pt_item->bid_contents);

		if (pf_ItemGotLocked)
			*pf_ItemGotLocked = TRUE;
	}

	//if the caller requests, communicate the aggregate length of the 
	//	rich-text items involved in determining the physical location of the 
	//	logical offset
	if (pul_AggregateLength)
		*pul_AggregateLength = ul_lengthCount;

	//set the pointers the caller will use to access the physical rich-text 
	//	item and CD record "pointed to" by the logical offset, then return 
	//	that the offset was successfully translated
	*ppuc_location = pt_item->puc_start + pt_item->ul_length - 
												(ul_lengthCount - ul_OFFSET);
	*ppt_item = pt_item;
	return TRUE;
} //f_LocateAbsoluteOffset(


/** f_TestStringStart( ***
Purpose is to test whether an instance of a particular string is found 
starting in the current CDTEXT record, from the passed-in offset on. Note 
that the entire string doesn't need to be in the current CDTEXT; it just 
needs to start there. As the function looks on into follow-on CDTEXTs for 
remaining characters in a string instance that may be a match, if a 
non-CDTEXT record is encountered, the search is aborted.

--- parameters & return -----
PC: The string for which an instance is being sought in the current rich-text 
	field. If the search is to be non-case-sensitive, this string must be in 
	*upper case*.
f_CASE_SENSITIVE: flag telling whether the search should be case-sensitive
t_CURSOR: cursor pointing to the initial CDTEXT to be searched for the start 
	of an instance of the sought-for string
us_OFFSET_AT_START: 1-based offset into the record's text run after which the 
	search should commence
pt_context: Input. Pointer to environment information about the 
	rich-text field being manipulated.
pf_found: Output. Pointer to the flag in which to store whether the start of 
	an instance of the string was found (TRUE) or not (FALSE).
pus_foundOffset: Output. Pointer to the variable in which to store the 
	1-based offset into the CDTEXT after which the start of a found instance 
	begins.
RETURN: TRUE if no error occurred. FALSE if heap memory could not be 
	allocated.

--- side effect --------
The rich-text environment information pointed to through pt_cursor or 
pt_context may be adjusted due to cursor advancement. If the enhancement 
suggested below is implemented, this side effect would be eliminated.

--- suggested enhancement ---
11/10/98 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution. The sidelined commentatry in this function indicates an 
direction by which such a mechanism might be implemented.

On second thought, it would probably be better to do the state saving and 
restoring in a higher-level function, because this function may be called so 
often that it's better just to leave memory locked.

--- revision history --------
9/12/99 PR: documentation adjustment
11/10/98 PR: created		*/
static BOOL f_TestStringStart( const char  PC[], 
								const BOOL  f_CASE_SENSITIVE, 
								const CdRecordCursor  t_CURSOR, 
								const WORD  us_OFFSET_AT_START, 
								const RtfTrackingInfo *const  pt_CONTEXT, 
								BOOL *const  pf_found, 
								WORD *const  pus_foundOffset)	{
	const WORD  us_LEN_STR = strlen( PC), 
				us_LEN_1ST_CD_OFFSET = sizeof( CDTEXT) + us_OFFSET_AT_START, 
				us_BYTES_TO_SEARCH_INITIAL = t_CURSOR.us_recLength - 
														us_LEN_1ST_CD_OFFSET;

	char * pc_initial = NULL, * pc_latest = NULL, * pc;
	WORD  us_BytesToSearch = us_BYTES_TO_SEARCH_INITIAL, 
			us_lenAllocated = 0, us_CharactersMatched, us_BytesToSearchNext;
	CdRecordCursor  t_cursor;
//MemoryLockStateInfo  t_lockState;
	BOOL  f_ItemGotLocked, f_result = TRUE;

	_ASSERTE( PC && t_CURSOR.puc_location && LOBYTE( SIG_CD_TEXT) == 
								*t_CURSOR.puc_location && pf_found && 
								pus_foundOffset && t_CURSOR.us_recLength >= 
								us_LEN_1ST_CD_OFFSET);

	//default the found flag to FALSE
	*pf_found = FALSE;

	//if there isn't any content to search, short-circuit that no 
	//	instance of the sought-for string starts in this CDTEXT
	if (!us_BytesToSearch)
		return TRUE;

	//Set the step-wise pointer to be used in the search to one less than 
	//	where the search will start. Also, if this search is to be 
	//	case-insensitive, create an upper-case copy of the content of 
	//	interest in the initial CDTEXT.
	if (!f_CASE_SENSITIVE)	{
		if (!( pc = pc_initial = calloc( us_BytesToSearch + 1, sizeof( char))))
			return FALSE;
		strupr( memcpy( pc--, t_CURSOR.puc_location + us_LEN_1ST_CD_OFFSET, 
														us_BytesToSearch));
	}else
		pc = t_CURSOR.puc_location + us_LEN_1ST_CD_OFFSET - 1;

//store a copy of the current memory-lock state of the rich-text field's 
//	item content
//EnsuingLockStatePush( t_CURSOR.pt_item, pt_context, &t_lockState);

	//loop until an instance of the sought-for string is found or it is  
	//	proven that the initial CDTEXT starts no instance of the string
	while (pc = memchr( pc + 1, PC[0], us_BytesToSearch--))	{
		//initialize this pass by updating variables telling how many 
		//	characters in the sought-for string have been matched and how 
		//	many characters remain to be searched in the initial CDTEXT
		us_CharactersMatched = 1;
		us_BytesToSearchNext = us_BYTES_TO_SEARCH_INITIAL - (pc - 
												(t_CURSOR.puc_location + 
												us_LEN_1ST_CD_OFFSET)) - 1;
		us_BytesToSearchNext = us_BytesToSearchNext < us_LEN_STR - 1 ? 
										us_BytesToSearchNext : us_LEN_STR - 1;

		//if the next characters to be tested do not match the sought-for 
		//	string, search anew starting with the next character over
		if (memcmp( pc + us_CharactersMatched, PC + us_CharactersMatched, 
											us_BytesToSearchNext) != ei_SAME)
			continue;

		//Update the number of characters matched. If the number equals 
		//	the length of the sought-for string, set the found flag.
		if (( us_CharactersMatched += us_BytesToSearchNext) == us_LEN_STR)
			*pf_found = TRUE;
		//else prepare to search follow-on CDTEXT records by intializing 
		//	a navigation cursor to the current CDTEXT
		else
			t_cursor = t_CURSOR;

		//loop through follow-on CDTEXTs until it has been determined whether 
		//	the current potential match is indeed a match
		while (!*pf_found)	{
			//Advance to the next CD record in the rich-text field. If the 
			//	record is not a CDTEXT, break out of the loop
			AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, &f_ItemGotLocked);
			if (LOBYTE( SIG_CD_TEXT) != *t_cursor.puc_location)
				break;

			//Determine the number of bytes to be searched based on the 
			//	content of the latest CDTEXT record. If the CDTEXT has no 
			//	content, iterate the loop to get the next CDTEXT.
			if (!( us_BytesToSearchNext = t_cursor.us_recLength - 
									(WORD) sizeof( CDTEXT) < us_LEN_STR - 
									us_CharactersMatched ? 
									t_cursor.us_recLength - sizeof( CDTEXT) : 
									us_LEN_STR - us_CharactersMatched))
				continue;

			//If this is a non-case-sensitive search, copy the bytes to be 
			//	searched into a string and convert to upper case. If needed, 
			//	increase (or create) the memory set aside for the string 
			//	if not enough is available already.
			if (!f_CASE_SENSITIVE)	{
				if (us_BytesToSearchNext + 1 > us_lenAllocated)
					if (!( pc_latest = realloc( pc_latest, us_lenAllocated = 
												us_BytesToSearchNext + 1)))	{
						f_result = FALSE;
						goto cleanUp;
					}
				memcpy( pc_latest, t_CURSOR.puc_location + sizeof( CDTEXT), 
														us_BytesToSearchNext);
				pc_latest[ us_BytesToSearchNext] = (char) NULL;
				strupr( pc_latest);
			}

			//if the next characters to be tested do not match the 
			//	sought-for string, break out of the immediate loop
			if (memcmp( pc_latest ? pc_latest : t_cursor.puc_location + 
											sizeof( CDTEXT), 
											PC + us_CharactersMatched, 
											us_BytesToSearchNext) != ei_SAME)
				break;

			//Update the number of characters matched. If the number equals 
			//	the length of the sought-for string, set the found flag.
			if (( us_CharactersMatched += us_BytesToSearchNext) == us_LEN_STR)
				*pf_found = TRUE;
		} //while (!*pf_found)

		//if an instance of the sought-for string was found, set the "found 
		//	offset" parameter to the offset after which the first character 
		//	was found, then break out of the loop
		if (*pf_found)	{
			*pus_foundOffset = pc - (pc_initial ? pc_initial - 
									us_OFFSET_AT_START : 
									t_CURSOR.puc_location + sizeof( CDTEXT));
			break;
		}
	} //while (pc = memchr(( pc_initial

cleanUp:
//if necessary, restore the original memory-lock state of the rich-text 
//	field's item content
//if (f_ItemGotLocked)
//	EnsuingLockStateRestore( t_lockState);

	if (!f_CASE_SENSITIVE)	{
		if (pc_initial)
			free( pc_initial);
		if (pc_latest)
			free( pc_latest);
	}

	return f_result;	//probably TRUE
} //f_TestStringStart(


/** eus_SwapHotspots( ***
Purpose is to replace a given rich-text hotspot with a different rich-text 
hotspot.

--- parameters & return ----
pt_cursor: Input & Output. Pointer to the rich-text cursor at which the 
	hotspot to be replaced is located. If the replacement is successful, the 
	cursor will be reset to point to the beginning of the new, replacement 
	hotspot in the context of the overall rich-text field.
t_replacementCursor: the cursor pointing to the beginning of the CD records 
	comprising the replacement hotspot
pt_context: Input. Pointer to the environment information about the 
	rich-text field being manipulated.
RETURN: eus_SUCCESS if no error occured. !eus_SUCCESS if any parameters or 
	the replacement hotspot is invalid. The Notes API error code otherwise.

--- side effect --------
The rich-text environment information associated with pt_cursor or pt_context 
will likely be adjusted due to writing CD records and to cursor advancement.

--- revision history -------
11/10/98 PR: created		*/
STATUS eus_SwapHotspots( CdRecordCursor *const  pt_cursor, 
							CdRecordCursor  t_replacementCursor, 
							RtfTrackingInfo *const  pt_context)	{
	//if any of the parameters are obviously invalid , short-circuit with 
	//	general failure
	if (!( pt_cursor && pt_cursor->puc_location && pt_cursor->pt_item && 
								t_replacementCursor.puc_location && 
								t_replacementCursor.pt_item && pt_context && 
								pt_context->pt_Actuality))
		return !eus_SUCCESS;

	//if the feedback option was enabled, default that the replacement 
	//	hotspot is not invalid

	//if the last CD record in the replacement hotspot CD records in not 
	//	an end-hotspot record, short-circuit with failure

	//if no virtualization has occurred yet
		//set the actual frontier to the beginning of the current 
		//	actual rich-text item
		//note the first actual item to be virtualized in a local variable
	//virtualize from the current actual frontier up to the start 
	//	of the attachment hotspot
	//write in the replacement rich-text text hotspot
	//reset the actual frontier to the record following the next, actual 
	//	hotspot end record
	//reset the passed-in cursor to point to the beginning of the 
	//	replacement hotspot

	return eus_SUCCESS;
} //eus_SwapHotspots(


/** us_CompileActuality( ***
Purpose is to compile a linked-list complete description of the specified 
rich-text field stored in the specified note. A rich-text field is comprised 
of a succession of same-named TYPE_COMPOSITE items.

--- parameters & return ----
h_NOTE: the note to be inspected for the specifiend rich-text field
pc_ITMNM: the name of the rich-text field whose item information is to be 
	gathered
ppt_Actuality: Output. Pointer to the head node of the linked-list complete 
	description of the specified rich-text field. If the specified rich-text 
	field does not exist, the head node is guaranteed to be NULL.
RETURN:
	eus_SUCCESS if no errors occurred
	ERR_ITEM_DATATYPE if specified item is not rich-text in type
	ERR_MEMORY if a memory allocation for the item-description list failed
	the Notes API error code otherwise

--- revision history -------
2/23/00 PR
+ enhanced context-checking to react properly if specified item is not 
  rich-text
+ improved error-return information & associated documentation
+ minor performance enhancement

10/26/98 PR: created		*/
static STATUS us_CompileActuality( NOTEHANDLE  h_NOTE, 
									char  pc_ITMNM[], 
									ItemInfo * *const  ppt_Actuality)	{
	WORD  us_len, us;
	ItemInfo_Actual  t_actualItem;
	ItemInfo * pt_entry;
	STATUS  us_err;

	_ASSERTE( h_NOTE && pc_ITMNM && ppt_Actuality);

	//get information about the named item
	if (us_err = NSFItemInfo( h_NOTE, pc_ITMNM, us_len = (WORD) strlen( 
									pc_ITMNM), &t_actualItem.bid_item, 
									&us, &t_actualItem.t_item.bid_contents, 
									&t_actualItem.t_item.ul_length))
			//if the specified item doesn't exist, nullify the list to 
			//	indicate that no rich-text needs to be processed
			if (ERR( us_err) == ERR_ITEM_NOT_FOUND)	{
				*ppt_Actuality = NULL;
				return eus_SUCCESS;
			}else
				return us_err;

	//if the specified item isn't rich-text, short-circuit with general 
	//	failure
	if (us != TYPE_COMPOSITE)
{BYTE * puc = OSLockBlock( BYTE, t_actualItem.t_item.bid_contents);
OSUnlockBlock( t_actualItem.t_item.bid_contents);
		return ERR_ITEM_DATATYPE;
}
	//allocate memory to hold the first item's information in the linked-list 
	//	complete description of the rich-text field
	if (!( pt_entry = *ppt_Actuality = malloc( sizeof( ItemInfo_Actual))))
		return ERR_MEMORY;

	//initiate loop to finish construction of a complete description of the 
	//	rich-text field's item information
	for (; ei_FOREVER;)	{
		BLOCKID  bid_prevItem;

		//Initialize the final members of the item-information structure: the 
		//	pointer to the start of the item's content (NULL indicates that 
		//	the content hasn't been memory locked) and the type member to 
		//	signify that this is an actual, not virtual item.
		t_actualItem.t_item.puc_start = NULL;
		t_actualItem.t_item.i_type = mi_ACTUAL;

		//The structure filled, copy its contents into the item's entry in the
		//	complete linked-list description of the rich-text field. The entry
		//	must be cast to an ItemInfo_Actual to avoid a compiler warning.
		*(ItemInfo_Actual *) pt_entry = t_actualItem;

		//Get the item information about the next, same-named rich-text item 
		//	in the note. If no next item exists, close off the linked-list 
		//	description of the rich-text field.
		bid_prevItem = t_actualItem.bid_item;
		if (us_err = NSFItemInfoNext( h_NOTE, bid_prevItem, pc_ITMNM, us_len, 
										&t_actualItem.bid_item, NULL, 
										&t_actualItem.t_item.bid_contents, 
										&t_actualItem.t_item.ul_length))
			if (ERR( us_err) == ERR_ITEM_NOT_FOUND)	{
				pt_entry->pt_next = NULL;

				return eus_SUCCESS;
			//Else an unexpected error has occurred. Free whatever memory 
			//	was allocated within the function before returning the error.
			}else	{
				FreeItemList( ppt_Actuality);
				return us_err;
			}

		//allocate memory to hold this next item's information in the 
		//	linked-list complete description of the rich-text field
		if (!( pt_entry = pt_entry->pt_next = 
										malloc( sizeof( ItemInfo_Actual))))	{
			FreeItemList( ppt_Actuality);
			return ERR_MEMORY;
		}
	} //for (; ei_FOREVER;)

	return 0;	//should never get here, line used to avoid a compiler 
				//	warning in VC++ 6.0
} //us_CompileActuality(


/** eus_AddRtfActualContext( ***
Add "actuality" context about a specified rich-text field to the provided 
rich-text context structure. This exported interface provides a means by 
which an already constructed stream of virtual CD records may be prepared to 
replace an existing rich-text field already attached to a note.

--- NOTE! ------------------
2/23/00 Procedure not yet tested!!

--- parameters & return ----
h_NOTE: the note to be inspected for the specifiend rich-text field
pc_ITMNM: the name of the rich-text field whose item information is to be 
	gathered
pt_context: Output. Structure containing contextual information about the 
	rich-text field being manipulated. This structure is required input to 
	most of the public interfaces to this module and so must be maintained 
	by the calling procedure.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: created		*/
STATUS eus_AddRtfActualContext( NOTEHANDLE  h_NOTE, 
								char  pc_ITMNM[], 
								RtfTrackingInfo *const  pt_context)	{
	if (!( h_NOTE && pc_ITMNM && pt_context))
		return !eus_SUCCESS;

	//fill out the context's Actuality list
	return us_CompileActuality( h_NOTE, pc_ITMNM, &pt_context->pt_Actuality);
} //eus_AddRtfActualContext(


/** ef_CopyRtfContent( ***
Produces a copy of the rich-text content associated with a given rich-text 
context.

--- NOTE! ------------------
2/23/00 Procedure not yet tested!!

--- parameters & return ----
pt_SRC: structure containing contextual information about the rich-text 
	content to be copied
f_CONSERVE_MEMORY: flag telling whether as much as possible only the memory 
	needed for a copy of the source content should be allocated (TRUE) or 
	(FALSE) any "slack" memory currently allocated in the source container 
	(for content growth or insertion) should be preserved
pt_dest: Output. Structure in which to record context information about the 
	copy produced of the source's virtual rich-text content. If procedure is 
	unsuccessful, the structure is guaranteed to reflect that it has no 
	virtual content.
RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: created		*/
STATUS eus_CopyRtfContent( const RtfTrackingInfo *const  pt_SRC, 
							const BOOL  f_CONSERVE_MEMORY, 
							RtfTrackingInfo *const  pt_dest)	{
	ItemInfo * pt_s;
	DWORD  ul;
	BOOL  f_hasActualityToCopy;
	STATUS  us_err;

	if (!( pt_SRC && pt_dest))
		return !eus_SUCCESS;

	pt_dest->pt_Virtuality = pt_dest->pt_endVirtual = NULL;

	//if there's nothing to copy, short-circuit with success
	pt_s = pt_SRC->pt_Virtuality;
	if (!( (f_hasActualityToCopy = pt_SRC->pt_Actuality && 
											pt_SRC->ul_ActualFrontier != 
											mul_FRONTIER_NO_MORE) || pt_s))
		return eus_SUCCESS;

	//if source has virtualized content...
	if (pt_s)	{
		BOOL  f_keepNewItemOpenForMore, f_ItemJustLocked = FALSE;
		ItemInfo * pt_d;

		//start off destination's "virtual" item-information list
		ul = pt_s->ul_length;
		f_keepNewItemOpenForMore = f_hasActualityToCopy && !pt_s->pt_next;
		if (us_err = us_StartVirtualItemList( f_CONSERVE_MEMORY && 
												!f_keepNewItemOpenForMore ? 
												ul : NULL, pt_dest))
			return us_err;

		//for each of source's virtual items...
		pt_d = pt_dest->pt_Virtuality;
		for (; ei_FOREVER; ) {
			//if necessary, lock in the source item
			if (!pt_s->puc_start)	{
				pt_s->puc_start = OSLockBlock( BYTE, pt_s->bid_contents);
				f_ItemJustLocked = TRUE;
			}

			//copy the content from source to destination
			memcpy( pt_d->puc_start, pt_s->puc_start, ul);
			pt_d->ul_length = ul;

			//if we won't be writing anymore to the new destination item, 
			//	release it
			if (!f_keepNewItemOpenForMore)	{
				OSUnlockBlock( pt_d->bid_contents);
				pt_d->puc_start = NULL;
			}

			//if we locked in the source item, release it now
			if (f_ItemJustLocked)	{
				OSUnlockBlock( pt_s->bid_contents);
				pt_s->puc_start = NULL;
				f_ItemJustLocked = FALSE;
			}

			//if there's a follow-on source item...
			if (pt_s = pt_s->pt_next)	{
				//Tack on another destination virtual item. The item will be 
				//	"locked in" by the us_InsertNextVirtualItem() call.
				ul = pt_s->ul_length;
				f_keepNewItemOpenForMore = f_hasActualityToCopy && 
															!pt_s->pt_next;
				if (us_err = us_InsertNextVirtualItem( pt_d, 
												mi_REGULAR_VIRTUAL, 
												f_CONSERVE_MEMORY && 
												!f_keepNewItemOpenForMore ? 
												ul : NULL))	{
					ClearItemList( &pt_dest->pt_Virtuality);
					return us_err;
				}
				pt_d = pt_d->pt_next;
			}else
				break;
		} //for (; ei_FOREVER;)

		pt_dest->pt_endVirtual = pt_d;
	} //if (pt_s)

	//if unvirtualized actual content remains to be copied...
	if (f_hasActualityToCopy)	{
		//construct a temporary context to manage the remaining 
		//	virtualization needed
		RtfTrackingInfo  t_context = *pt_SRC;
		t_context.pt_Virtuality = pt_dest->pt_Virtuality;
		t_context.pt_endVirtual = pt_dest->pt_endVirtual;

		//virtualize the remaining actual content
		if (us_err = us_VirtualizeRestOfRtf( &t_context, NULL))	{
			ClearItemList( &t_context.pt_Virtuality);
			pt_dest->pt_Virtuality = pt_dest->pt_endVirtual = NULL;
			return us_err;
		}

		//update the destination context structure with the final results
		pt_dest->pt_Virtuality = t_context.pt_Virtuality;
		pt_dest->pt_endVirtual = t_context.pt_endVirtual;
	} //if (f_hasActualityToCopy)

	return eus_SUCCESS;
} //ef_CopyRtfContent(


/** eus_ReplaceRtfWithCdStream( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
12/12/98 PR: created		*/
//DOC!!
STATUS eus_ReplaceRtfWithCdStream( const BYTE *const  PUC, 
									const DWORD  ul_LEN, 
									RtfTrackingInfo *const  pt_context)	{
	ItemInfo * pt_item;
	int  i_offset;

	if (!( PUC && ul_LEN > (DWORD) (i_offset = (*(WORD *) PUC == 
											TYPE_COMPOSITE ? sizeof( WORD) : 
											NULL)) && pt_context))
		return !eus_SUCCESS;

	//set the actual-frontier member of the tracking structure to say that 
	//	there is no more frontier
	pt_context->ul_ActualFrontier = mul_FRONTIER_NO_MORE;

	//if virtuality has been started, clear all the virtual items except the 
	//	first and set the first item's length member to the shortest 
	//	possible length
	if (pt_item = pt_context->pt_Virtuality)	{
		if (pt_item->pt_next)
			ClearItemList( &pt_item->pt_next);
		pt_item->ul_length = sizeof( WORD);
	}

	//virtualize the input stream of CD records to the context associated 
	//	with the specified rich-text field
	return us_VirtualAppend( PUC + i_offset, ul_LEN - i_offset, pt_context, 
																	ul_LEN);
} //eus_ReplaceRtfWithCdStream(


/** FreeItemList( ***
Purpose is to free allocated memory associated with the given list of 
rich-text item structures.

--- parameter ----------
ppt_ListHead: Input & Output. Pointer to the head node of the list 
	to be freed. The head node will return nullified.

--- revision history ---
10/29/98 PR: created		*/
static __inline void FreeItemList( ItemInfo * *const  ppt_ListHead)	{
	ItemInfo * pt_item, * pt_next;

	_ASSERTE( ppt_ListHead && *ppt_ListHead && 
										(*ppt_ListHead)->bid_contents.pool);

	//loop through the nodes in the passed-in list, freeing associated memory 
	pt_item = *ppt_ListHead;
	do	{
		//if the item's a virtual item that was never written to a note, 
		//	free the block of memory associated with the item
		if (pt_item->i_type == mi_VIRTUAL && pt_item->bid_contents.pool)
			OSMemFree( pt_item->bid_contents.pool);

		pt_next = pt_item->pt_next;
		free( pt_item);
	} while (pt_item = pt_next);

	//nullify the head node to signify that the given list has been freed
	*ppt_ListHead = NULL;
} //FreeItemList(


/** us_getCdRecLength( ***
Purpose is to return the length of the passed-in rich-text CD record.

--- parameter & return ----
puc_CD_RECORD: pointer to the CD record whose length needs to be determined
RETURN: the length in bytes of the passed-in CD record

--- revision history ------
2/23/00 PR: cosmetic code clean-up
10/29/98 PR: created		*/
static __inline WORD us_getCdRecLength( const BYTE *const  puc_CD_RECORD)	{
	DWORD  ul_recLength;

	_ASSERTE( puc_CD_RECORD);

	//Get the length from the length member of the signature structure. The 
	//	first byte of a CD Record defines the "signature" type of the record. 
	//	This byte is or'ed with a second-byte mask specifying whether the 
	//	length of the record can be provided in the space of a BYTE, WORD or 
	//	DWORD.
	switch (HIBYTE( *(WORD *) puc_CD_RECORD))	{
		case muc_SIGTYPE_LSIG:
			ul_recLength = ((LSIG *) puc_CD_RECORD)->Length;
			break;
		case muc_SIGTYPE_WSIG:
			ul_recLength = ((WSIG *) puc_CD_RECORD)->Length;
			break;
		default: //muc_SIGTYPE_BSIG:
			ul_recLength = ((BSIG *) puc_CD_RECORD)->Length;
	}

	return (WORD) ul_recLength;
} //us_getCdRecLength(


/** ef_GetPgraphStyleCount( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
1/17/99 PR: created			*/
//DOC!!
BOOL ef_GetPgraphStyleCount( CdRecordCursor  t_cursor, 
								const RtfTrackingInfo *const  pt_CONTEXT, 
								WORD *const  pus_styles)	{
	WORD  us = 0;
	BOOL  f_ItemGotLocked;
	ItemInfo * pt_itemLocked = NULL;

	if (!( t_cursor.puc_location && t_cursor.pt_item && 
													t_cursor.us_recLength && 
													pt_CONTEXT && pus_styles))
		return FALSE;

	*pus_styles = NULL;

	//loop record-by-record through the rich-text content
	while (t_cursor.pt_item)	{
		if (LOBYTE( SIG_CD_PABDEFINITION) == *t_cursor.puc_location)
			us++;

		//advance the cursor to the next CD record
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, &f_ItemGotLocked);

		//if we just moved to a follow-on rich-text item and the previous 
		//	item was locked by this procedure, unlock the previous item
		if (pt_itemLocked && pt_itemLocked != t_cursor.pt_item)	{
			OSUnlockBlock( pt_itemLocked->bid_contents);
			pt_itemLocked = (ItemInfo *) pt_itemLocked->puc_start = NULL;
		}

		//if a follow-on rich-text item was locked by this procedure, flag 
		//	the item for later unlocking
		if (f_ItemGotLocked)
			pt_itemLocked = t_cursor.pt_item;
	} //while (t_cursor.pt_item)

	*pus_styles = us;

	return TRUE;
} //ef_GetPgraphStyleCount(


/** ef_FindPgraphStyleUnusedSpan( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
1/17/99 PR: created			*/
//DOC!!
BOOL ef_FindPgraphStyleUnusedSpan( const WORD  us_LEN_SPAN, 
									CdRecordCursor  t_cursor, 
									const RtfTrackingInfo *const  pt_CONTEXT, 
									WORD *const  pus_style, 
									BOOL *const  pf_gap)	{
	WORD  us = 0, us_highestStyle = 0;
	BOOL  f_ItemGotLocked;
	ItemInfo * pt_itemLocked = NULL;

	if (!( t_cursor.puc_location && t_cursor.pt_item && 
													t_cursor.us_recLength && 
													pt_CONTEXT && pus_style))
		return FALSE;

	*pus_style = NULL;
	if (pf_gap)
		*pf_gap = FALSE;

	if (!us_LEN_SPAN)
		return TRUE;

	//loop through the rich-text field, record by record
	while (t_cursor.pt_item)	{
		//If this CD record is a paragraph-definition or -reference and the 
		//	associated ID is greater than the highest ID we've encountered so 
		//	far, reset the highest-ID variable. We're checking 
		//	paragraph-reference records as well as -definition records 
		//	because sometimes Notes leaves reference records in that point to 
		//	non-existent definition records.
		if (LOBYTE( SIG_CD_PABDEFINITION) == *t_cursor.puc_location && 
											(us = ((CDPABDEFINITION *) 
											t_cursor.puc_location)->PABID) > 
											us_highestStyle)
			us_highestStyle = us;
		else if (LOBYTE( SIG_CD_PABREFERENCE) == *t_cursor.puc_location && 
											(us = ((CDPABREFERENCE *) 
											t_cursor.puc_location)->PABID) > 
											us_highestStyle)
			us_highestStyle = us;

		//advance the cursor to the next CD record
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, &f_ItemGotLocked);

		//if we just moved to a follow-on rich-text item and the previous 
		//	item was locked by this procedure, unlock the previous item
		if (pt_itemLocked && pt_itemLocked != t_cursor.pt_item)	{
			OSUnlockBlock( pt_itemLocked->bid_contents);
			pt_itemLocked = (ItemInfo *) pt_itemLocked->puc_start = NULL;
		}

		//if a follow-on rich-text item was locked by this procedure, flag 
		//	the item for later unlocking
		if (f_ItemGotLocked)
			pt_itemLocked = t_cursor.pt_item;
	} //while (t_cursor.pt_item)

	//if the desired span length cannot be accommodated between the 
	//	highest-possible ID and the highest ID found, we need to look for a 
	//	gap within the ID set where the span can be accommodated (NOT YET
	//	DONE)
	if (us_highestStyle > mus_MAX_STYLE_ID - us_LEN_SPAN)	{
return FALSE;

		*pf_gap = TRUE;
	}else
		*pus_style = us_highestStyle + 1;

	return TRUE;
} //ef_FindPgraphStyleUnusedSpan(


/** ef_RenumberPgraphStylesToSpan( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- suggested enhancement ---
1/16/99 PR
The starting memory-lock state should be restored upon function exit, but no 
good mechanism is yet available with which to accomplish this. See this 
module's suggested-enhancement note for a description of the problem and 
proposed solution.

--- revision history -------
1/19/99 PR: created			*/
//DOC!!
BOOL ef_RenumberPgraphStylesToSpan( 
								const WORD  us_MIN_ID, 
								const WORD  us_MAX_ID, 
								CdRecordCursor  t_cursor, 
								const RtfTrackingInfo *const  pt_CONTEXT)	{
	CdRecordCursor  t_StartOff, t_cursr;
	WORD  us, us_IdToChange;
	CDPABDEFINITION * pt_pab;
	CDPABREFERENCE * pt_pgref;
	CDPABFORMULAREF * pt_phref;
	CDPABHIDE * pt_phide;

	if (!( us_MIN_ID && us_MIN_ID <= us_MAX_ID && t_cursor.puc_location && 
										t_cursor.pt_item && 
										t_cursor.us_recLength && pt_CONTEXT))
		return FALSE;

	//loop through the rich-text content, record by record
	us = us_MIN_ID - 1;
	do	{
		//if the record is a paragraph-style definition and the associated ID 
		//	falls outside the remaining span...
		if (LOBYTE( SIG_CD_PABDEFINITION) == *t_cursor.puc_location)	{
			us_IdToChange = (pt_pab = (CDPABDEFINITION *) 
												t_cursor.puc_location)->PABID;
			if (us_IdToChange < us_MIN_ID || us_IdToChange > us_MAX_ID)	{
				//reset the start-off cursor to the ensuing CD record
				t_StartOff = t_cursor;
				AdvanceCdCursor( &t_StartOff, pt_CONTEXT, NULL, NULL);

				//reset the current ID to the next-available ID in the span
				if (++us > us_MAX_ID)
					return FALSE;
				pt_pab->PABID = us;

				//starting from the start-off cursor, loop through the 
				//	rich-text content, record by record
				t_cursr = t_StartOff;
				while (t_cursr.pt_item)	{
					//If the record is a paragraph-style reference and its ID 
					//	matches the ID we need to change, reset the reference 
					//	equal to the ID we're changing to. (There's no need 
					//	for virtualization here since we're not changing the 
					//	length of the rich-text items at all.)
					if (LOBYTE( SIG_CD_PABREFERENCE) == 
											*t_cursr.puc_location && 
											(pt_pgref = (CDPABREFERENCE *) 
											t_cursr.puc_location)->PABID == 
											us_IdToChange)
						pt_pgref->PABID = us;
					else if (LOBYTE( SIG_CD_PABFORMREF) == 
													*t_cursr.puc_location)	{
						if ((pt_phref = (CDPABFORMULAREF *) 
										t_cursr.puc_location)->DestPABID == 
										us_IdToChange)
							pt_phref->DestPABID = us;
						if (pt_phref->SourcePABID == us_IdToChange)
							pt_phref->SourcePABID = us;
					}else if (LOBYTE( SIG_CD_PABHIDE) == 
											*t_cursr.puc_location && 
											(pt_phide = (CDPABHIDE *) 
											t_cursr.puc_location)->PABID == 
											us_IdToChange)
						pt_phide->PABID = us;

					//advance the operative cursor to next record in the 
					//	rich-text content
					AdvanceCdCursor( &t_cursr, pt_CONTEXT, NULL, NULL);
				} //while (t_cursr.pt_item)
			} //if (us_IdToChange < us_MIN_ID ||
		} //if (LOBYTE( SIG_CD_PABDEFINITION) ==

		//advance to the next record in the rich-text content
		AdvanceCdCursor( &t_cursor, pt_CONTEXT, NULL, NULL);
	} while (t_cursor.pt_item);

	return TRUE;
} //ef_RenumberPgraphStylesToSpan(


/** ef_FreeRtfContext( ***
Purpose is to polish off the structures and allocations made to support 
an instance of rich-text handling here in this module.

--- parameter & return ---
pt_context: Input & Output. The structure managing environment information 
	about an instance of rich-text handling. The ItemInfo lists contained 
	by the structure are freed and cleared by this function.
RETURN: TRUE if procedure was successful; FALSE if the pointer to the 
	rich-text item-tracking context is null and therefore clearly invalid

--- revision history -----
2/23/00 PR: minor documentation adjustment
12/7/98 PR: created		*/
BOOL ef_FreeRtfContext( RtfTrackingInfo *const  pt_context)	{
	if (!pt_context)
		return FALSE;

	//free & clear all information concerning the rich-text items managed 
	//	by the given instance of rich-text handling
	ClearItemList( &pt_context->pt_Actuality);
	ClearItemList( &pt_context->pt_Virtuality);
	ClearItemList( &pt_context->pt_SemiVirtuality);

	//Nullify the container structure itself. Less likely to be reused that 
	//	way.
	memset( pt_context, sizeof( RtfTrackingInfo), NULL);

	return TRUE;
} //ef_FreeRtfContext(


/** e_InitUtilityRtfContext( ***


--- parameter & return ----


--- revision history ------
12/8/98 PR: created			*/
//DOC!!
RtfTrackingInfo * ept_InitUtilityRtfContext( 
										RtfTrackingInfo *const  pt_context)	{
	if (!pt_context)
		return NULL;

	memset( pt_context, NULL, sizeof( RtfTrackingInfo));

	return pt_context;
} //e_InitUtilityRtfContext(


/** eus_AppendRtParagraph( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: minor logic shortening, token renaming
12/8/98 PR: created			*/
//DOC!!
STATUS eus_AppendRtParagraph( const WORD  us_ID_PARAGRAPH_DEF, 
								const COMPOUNDSTYLE *const  pt_PARAGRAPH_DEF, 
								RtfTrackingInfo *const  pt_context, 
								WORD *const  pus_usedParagraphDefId)	{
	const WORD  us_START_CUSTOM_IDS = 0xA000;
	const WSIG  t_PARAGRAPH_STYLE_HEADER = {SIG_CD_PABDEFINITION, 
													sizeof( CDPABDEFINITION)};
	
	const COMPOUNDSTYLE * pt_ParagraphDef = NULL;
	BYTE * puc;
	WORD  us_ParagraphDefId;
	CDPABREFERENCE  t_ParagraphDefinitionReference = 
							{{SIG_CD_PABREFERENCE, sizeof( CDPABREFERENCE)}};
	STATUS  us_err;

	if (!( pt_context && (us_ID_PARAGRAPH_DEF || pt_PARAGRAPH_DEF)))
		return !eus_SUCCESS;

//PGP development short-cut
_ASSERTE( pt_PARAGRAPH_DEF && us_ID_PARAGRAPH_DEF);

	//if applicable, default to null the variable in which the caller wants 
	//	to store the ID of the paragraph-definition record we use
	if (pus_usedParagraphDefId)
		*pus_usedParagraphDefId = NULL;

	//if a paragraph style ID was provided by the caller...
	if (us_ParagraphDefId = us_ID_PARAGRAPH_DEF)	{
		//see if the specified paragraph style record is already 
		//	defined within the rich-text field

		//If the record is not yet defined and if a paragraph-attribute 
		//	structure was provided, set the internal style-structure variable 
		//	to point to the structure provided by the caller. If the caller 
		//	didn't provide a structure, we just won't reference a paragraph 
		//	definition at all.
		if (pt_PARAGRAPH_DEF)
			pt_ParagraphDef = pt_PARAGRAPH_DEF;
	//else just a paragraph-attribute structure was provided, so we need 
	//	to create our own style ID...
	}else	{
		//loop until an acceptable ID is generated...
			//generate a random ID number within the non-custom range
return !eus_SUCCESS;
			//if the style ID is not already taken within virtuality, 
			//	break out of the loop

			//if we've tried this loop ten times already, something's 
			//	gotta be wrong, so return general failure

		//set the internal style-structure variable to point to the structure 
		//	provided by the caller
	} //if (us_ParagraphDefId = us_ID_PARAGRAPH_DEF)

	//if there's any non-virtualized actuality involved here, virtualize it
	if (us_err = us_VirtualizeRestOfRtf( pt_context, NULL))
		return us_err;

	//get a pointer to where we can append the paragraph record we need
	if (us_err = us_GetVirtualCdRecAppendPointer( pt_context, 
												sizeof( CDPARAGRAPH), &puc))
		return us_err;

	//write the paragraph record
	memcpy( puc, &mt_PARAGRAPH, sizeof( CDPARAGRAPH));

	//update the length of the item written to
	pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDPARAGRAPH);

	//if we need to create and append a paragraph style structure...
	if (pt_ParagraphDef)	{
		//get a pointer at which the record can be written
		if (us_err = us_GetVirtualCdRecAppendPointer( pt_context, sizeof( 
													CDPABDEFINITION), &puc))
			return us_err;

		//write the header bytes to the full style record, then copy in the 
		//	provided style structure, then write the trailing record bytes
		memcpy( puc, &t_PARAGRAPH_STYLE_HEADER, sizeof( WSIG));
		memcpy( puc += sizeof( WSIG), &us_ParagraphDefId, sizeof( WORD));
		memcpy( puc += sizeof( WORD), pt_ParagraphDef, 
													sizeof( COMPOUNDSTYLE));
		memset( puc + sizeof( COMPOUNDSTYLE), NULL, sizeof( DWORD) + 
															sizeof( WORD));

		//update the length of the item written to
		pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDPABDEFINITION);
	} //if (pt_ParagraphDef)

	//if needed, update the generic paragraph-reference record with our 
	//	favorite style ID, and write the paragraph-definition reference record
	if (us_ParagraphDefId)	{
		//get a pointer at which the record can be written
		if (us_err = us_GetVirtualCdRecAppendPointer( pt_context, sizeof( 
													CDPABREFERENCE), &puc))
			return us_err;

		t_ParagraphDefinitionReference.PABID = us_ParagraphDefId;
		memcpy( puc, &t_ParagraphDefinitionReference, 
													sizeof( CDPABREFERENCE));

		//update the length of the item written to
		pt_context->pt_endVirtual->ul_length += 
									pt_context->pt_endVirtual->ul_length % 
									2 + sizeof( CDPABREFERENCE);
	} //if (us_ParagraphDefId)

	//if applicable, update the variable in which the caller wants to store 
	//	the ID of the paragraph-definition record we used
	if (pus_usedParagraphDefId && us_ParagraphDefId)
		*pus_usedParagraphDefId = us_ParagraphDefId;
	
	return eus_SUCCESS;
} //eus_AppendRtParagraph(


/** eus_AppendAttachmentHotspot( ***


--- parameters & return ----
pc_UniqueNm: Input & Output.

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
1/29/99 PR: created			*/
//DOC!!
STATUS eus_AppendAttachmentHotspot( NOTEHANDLE  h_NOTE, 
									const char  pc_FILENM[], 
									char  pc_UniqueNm[], 
									const BYTE *const  puc_GRAPHIC_CD_RECS, 
									const WORD  us_LEN_GRAPHIC_CD_RECS, 
									RtfTrackingInfo *const pt_context)	{
	CDHOTSPOTBEGIN  t_HotspotBegin = {{SIG_CD_HOTSPOTBEGIN}, 
										HOTSPOTREC_TYPE_FILE, 
										HOTSPOTREC_RUNFLAG_BEGIN | 
										HOTSPOTREC_RUNFLAG_NOBORDER};
	UINT  ui_lenUniqueNm, ui;
	BYTE * puc;
	STATUS  us_error;

	if (!( h_NOTE && pc_UniqueNm && (puc_GRAPHIC_CD_RECS ? 
								us_LEN_GRAPHIC_CD_RECS : TRUE) && pt_context))
		return !eus_SUCCESS;

	//if no unique name has been provided...
		//if a filename has been provided...
			//if the filename is unique within the given note and rich-text 
			//	context...
				//copy the filename into the unique name

	//if a unique name still has not been set, determine a filename unique 
	//	within the given note and rich-text context

//PGP development shortcut
ui_lenUniqueNm = strlen( pc_UniqueNm);
if (pc_FILENM && !ui_lenUniqueNm)	{
  strcpy( pc_UniqueNm, pc_FILENM);
  ui_lenUniqueNm = strlen( pc_UniqueNm);
}

	ui = ui_lenUniqueNm + (pc_FILENM ? strlen( pc_FILENM) : (UINT) NULL) + 2;
	if (ui > (WORD) ui)
		return !eus_SUCCESS;
	t_HotspotBegin.DataLength = (WORD) ui;
	t_HotspotBegin.Header.Length = (WORD) (ui += sizeof( CDHOTSPOTBEGIN));

	//if there's any non-virtualized actuality involved here, virtualize it
	if (us_error = us_VirtualizeRestOfRtf( pt_context, NULL))
		return us_error;

	//get a pointer to where we can append the attachment-hotspot records 
	//	we need -- PGP SHORTCUT: left out graphics stuff!!
	if (us_error = us_GetVirtualCdRecAppendPointer( pt_context, ui += 
												sizeof( CDHOTSPOTEND), &puc))
		return us_error;

	//write in the attachment hotspot
	memcpy( puc, &t_HotspotBegin, sizeof( CDHOTSPOTBEGIN));
	strcpy( puc += sizeof( CDHOTSPOTBEGIN), pc_UniqueNm);
	puc += ui_lenUniqueNm + 1;
	if (pc_FILENM)	{
		strcpy( puc, pc_FILENM);
		puc += strlen( pc_FILENM);
	}else
		puc[0] = NULL;

	memcpy( ++puc + (DWORD) puc % 2, &mt_HOTSPOT_END, sizeof( CDHOTSPOTEND));

	//lastly, update the length of the item written to
	pt_context->pt_endVirtual->ul_length += 
								pt_context->pt_endVirtual->ul_length % 2 + ui;

	return eus_SUCCESS;
} //eus_AppendAttachmentHotspot(


/** f_ConstructStartCursor( ***


--- parameters & return ----

RETURN: TRUE if a valid cursor could be set; FALSE otherwise

--- revision history -------
1/16/99 PR: created			*/
//DOC!!
static BOOL f_ConstructStartCursor( const RtfTrackingInfo *const  pt_CONTEXT, 
									ItemInfo *const  pt_item, 
									CdRecordCursor *const  pt_cursor, 
									BOOL *const  pf_ItemGotLocked)	{
	const BOOL  f_SemiVirtual = pt_CONTEXT ? (!pt_CONTEXT->pt_Virtuality && 
											pt_CONTEXT->pt_SemiVirtuality ? 
											TRUE : FALSE) : FALSE;

	ItemInfo * pt_itm;
	DWORD  ul;
	BOOL  f_ItemGotLocked = FALSE;
	CdRecordCursor  t;

	_ASSERTE( (pt_CONTEXT || pt_item) && pt_cursor);

	memset( pt_cursor, (BYTE) NULL, sizeof( CdRecordCursor));
	if (pf_ItemGotLocked)
		*pf_ItemGotLocked = FALSE;
	
	//If full tracking information is available, use it. Else use the 
	//	individual item.
	pt_itm = pt_CONTEXT ? (pt_CONTEXT->pt_Virtuality ? 
								pt_CONTEXT->pt_Virtuality : (f_SemiVirtual ? 
								pt_CONTEXT->pt_SemiVirtuality : 
								pt_CONTEXT->pt_Actuality)) : pt_item;

	if (pt_itm->ul_length < (ul = !f_SemiVirtual ? sizeof( WORD) : NULL))
		return FALSE;

	while (pt_itm->ul_length == ul && (pt_itm = pt_itm->pt_next));
	if (!pt_itm)
		return FALSE;

	if (!pt_itm->puc_start)	{
		pt_itm->puc_start = OSLockBlock( BYTE, pt_itm->bid_contents);
		f_ItemGotLocked = TRUE;
	}

	t.puc_location = pt_itm->puc_start + (!f_SemiVirtual ? sizeof( WORD) : 
																		NULL);
	if (!(t.us_recLength = us_getCdRecLength( t.puc_location)))
		goto errJump;
	t.pt_item = pt_itm;

	*pt_cursor = t;
	if (pf_ItemGotLocked)
		*pf_ItemGotLocked = f_ItemGotLocked;

	return TRUE;

errJump:
	if (f_ItemGotLocked)	{
		OSUnlockBlock( pt_itm->bid_contents);
		pt_itm->puc_start = NULL;
	}

	return FALSE;
} //f_ConstructStartCursor(


/** eus_AppendItemsToRtf( ***
Appends the virtual items tracked by one rich-text context to another 
rich-text context.

--- parameters & return ---
pt_APPENDAGE: pointer to the rich-text context whose virtuality is to be 
	appended to the target context
pt_mainContext: Input & Output. Pointer to the rich-text context to be 
	appended to.
RETURN: eus_SUCCESS if successful; the Notes API error code otherwise

--- revision history ------
2/23/00 PR
+ minor signature adjustment for constness
+ completed standard documentation

12/8/98 PR: created			*/
STATUS eus_AppendItemsToRtf( const RtfTrackingInfo *const  pt_APPENDAGE, 
								RtfTrackingInfo *const  pt_mainContext)	{
	ItemInfo * pt_item;
	STATUS  us_error;

	if (!( pt_APPENDAGE && pt_APPENDAGE->pt_Virtuality && pt_mainContext))
		return !eus_SUCCESS;

	if (us_error = us_VirtualizeRestOfRtf( pt_mainContext, NULL))
		return us_error;

	if (pt_mainContext->pt_endVirtual)	{
		pt_item = pt_mainContext->pt_endVirtual;
		pt_item->pt_next = pt_APPENDAGE->pt_Virtuality;
	}else
		pt_item = pt_mainContext->pt_Virtuality = pt_APPENDAGE->pt_Virtuality;

	while (pt_item->pt_next)
		pt_item = pt_item->pt_next;

	pt_mainContext->pt_endVirtual = pt_item;

	return eus_SUCCESS;
} //eus_AppendItemsToRtf(


/** ef_UnappendRtfItems( ***
Undo a prior item-appendage (likely via eus_AppendItemsToRtf()) made to a 
particular rich-text context.

--- parameters & return ----
pt_APPENDED: pointer to the rich-text context describing the virtual-item 
	stream already incorporated into the other specified rich-text context
pt_mainContext: Input & Output. Pointer to the rich-text context that 
	received the specified appendage. The context will be adjusted such that 
	the appendage is disincorporated.
RETURN: TRUE if successful; FALSE otherwise

--- revision history -------
2/23/00 PR: created			*/
BOOL ef_UnappendRtfItems( const RtfTrackingInfo *const  pt_APPENDED, 
							RtfTrackingInfo *const  pt_mainContext)	{
	BLOCKID  bid;
	ItemInfo * pt, * pt_itm;

	if (!( pt_APPENDED && pt_APPENDED->pt_Virtuality && pt_mainContext && 
							(pt = pt_itm = pt_mainContext->pt_Virtuality)))
		return FALSE;

	//if the point at which the appendage occurred cannot be located, 
	//	short-circuit with failure
	bid = pt_APPENDED->pt_Virtuality->bid_contents;
	do
		if (memcmp( &pt_itm->bid_contents, &bid, sizeof( BLOCKID)) == ei_SAME)
			break;
	while (pt_itm = pt_itm->pt_next);
	if (!pt_itm)
		return FALSE;

	//excise the appendage from the virtual-item stream by updating 
	//	associated pointers
	do
		if (pt->pt_next = pt_itm)
			break;
	while (pt = pt->pt_next);
	if (pt)
		pt->pt_next = pt_mainContext->pt_endVirtual = NULL;
	else
		pt_mainContext->pt_endVirtual = pt_mainContext->pt_Virtuality = NULL;

	return TRUE;
} //ef_UnappendRtfItems(


/** eus_InsertSubformTop( ***


--- parameters & return ----

RETURN: eus_SUCCESS if no error occured; the Notes API error code otherwise

--- revision history -------
2/23/00 PR: Extension to handle subforms consisting of multiple rich-text 
	items. Caused procedure's signature to change.
1/16/99 PR: created			*/
//DOC!!
STATUS eus_InsertSubformTop( RtfTrackingInfo *const  pt_contextForm, 
								RtfTrackingInfo *const  pt_contextSubform, 
								const char  pc_SUBFORMNM[])	{
	const CDTEXT  t_BLANK = { {SIG_CD_TEXT, sizeof( CDTEXT)}, 
															DEFAULT_FONT_ID};
	//Notes includes the null terminator when writing the subform name with 
	//	the begin-hotspot record, so we do too. BTW, without the null 
	//	terminator, Notes doesn't open the form properly.
	const WORD  us_LEN_SUBFORMNM = (WORD) (pc_SUBFORMNM ? strlen( 
												pc_SUBFORMNM) + 1 : NULL), 
				us_LEN_HOTSPOT_BEGIN = sizeof( CDHOTSPOTBEGIN) + 
															us_LEN_SUBFORMNM;
	//I don't know what the 0x10000000 flag is, but Notes includes it with 
	//	all subforms, so we do too.
	const CDHOTSPOTBEGIN  t_HOTSPOT_BEGIN = { {SIG_CD_V4HOTSPOTBEGIN, 
												us_LEN_HOTSPOT_BEGIN}, 
												HOTSPOTREC_TYPE_SUBFORM, 
												HOTSPOTREC_RUNFLAG_BEGIN | 
												HOTSPOTREC_RUNFLAG_NOBORDER | 
												0x10000000, us_LEN_SUBFORMNM};
	const CDHOTSPOTEND  t_V4HOTSPOT_END = {{SIG_CD_V4HOTSPOTEND, 
													sizeof( CDHOTSPOTEND)}};

	CdRecordCursor  t_cursorBeginForm, t_cursorBeginSubform;
	ItemInfo * pt_FormItemLocked = NULL, * pt_SubformItemLocked = NULL, 
				* pt_item;
	BOOL  f_ItemGotLocked, f_failure, f_gap, f_SubformBegun = FALSE;
	WORD  us, us_styleStart;
	RtfTrackingInfo  t_Subform;
	BYTE * puc;
//	DWORD  ul;
	STATUS  us_error = eus_SUCCESS;

	if (!( pt_contextForm && pt_contextSubform && us_LEN_SUBFORMNM))
		return !eus_SUCCESS;

	//construct cursors pointing to the start of the form and the subform
	if (!f_ConstructStartCursor( pt_contextForm, NULL, &t_cursorBeginForm, 
															&f_ItemGotLocked))
		return !eus_SUCCESS;
	if (f_ItemGotLocked)
		pt_FormItemLocked = t_cursorBeginForm.pt_item;
	if (!f_ConstructStartCursor( pt_contextSubform, NULL, 
									&t_cursorBeginSubform, &f_ItemGotLocked))
		return !eus_SUCCESS;
	if (f_ItemGotLocked)
		pt_SubformItemLocked = t_cursorBeginSubform.pt_item;

	//determine the number of paragraph style IDs in the subform
	if (f_failure = !ef_GetPgraphStyleCount( t_cursorBeginSubform, 
													pt_contextSubform, &us))
		goto errJump;

	//in the form, determine a span of unused paragraph style-ID numbers that 
	//	will accommodate the number of style IDs in the subform
	if (f_failure = !ef_FindPgraphStyleUnusedSpan( (WORD) (us + 1), 
										t_cursorBeginForm, pt_contextForm, 
										&us_styleStart, &f_gap))
		goto errJump;

	//renumber the style IDs in the subform using the span of IDs not used in 
	//	the form, leaving the first unused ID for the hidden paragraph placed 
	//	before the subform hotspot
	if (f_failure = !ef_RenumberPgraphStylesToSpan( (WORD) (us_styleStart + 
									1), (WORD) (f_gap ? 
									us_styleStart + us : mus_MAX_STYLE_ID), 
									t_cursorBeginSubform, pt_contextSubform))
		goto errJump;

	//Initialize a rich-text context into which the subform will be copied 
	//	before attaching it to the form. Notes places a hidden paragraph 
	//	before a subform at the top of the form, so we do too.
	if (us_error = eus_AppendRtParagraph( us_styleStart, 
												&mt_HIDDEN_PARAGRAPH_DEF, 
												ept_InitUtilityRtfContext( 
												&t_Subform), NULL))
		goto errJump;
	f_SubformBegun = TRUE;

	//Virtualize a blank CDTEXT and a beginning subform-hotspot record. 
	//	(Notes puts in a blank CDTEXT for reason unknown, so we emulate.)
	if (us_error = us_GetVirtualCdRecAppendPointer( &t_Subform, sizeof( 
										CDTEXT) + us_LEN_HOTSPOT_BEGIN, &puc))
		goto errJump;
	memcpy( puc, &t_BLANK, sizeof( CDTEXT));
	memcpy( puc += sizeof( CDTEXT), &t_HOTSPOT_BEGIN, 
													sizeof( CDHOTSPOTBEGIN));
	memcpy( puc + sizeof( CDHOTSPOTBEGIN), pc_SUBFORMNM, us_LEN_SUBFORMNM);

	//update the length of the item written to
	t_Subform.pt_endVirtual->ul_length += 
									t_Subform.pt_endVirtual->ul_length % 2 + 
									sizeof( CDTEXT) + us_LEN_HOTSPOT_BEGIN;

	//virtualize in the body of the subform
	if (us_error = us_VirtualizeRestOfRtf( pt_contextSubform, &t_Subform))
		goto errJump;
//	ul = ul_LogicalRtfLength( pt_contextSubform);
//	if (us_error = us_VirtualAppend( t_cursorBeginSubform.puc_location, ul, 
//															&t_Subform, ul))
//		goto errJump;

	//Virtualize a hotspot-end record and a blank CDTEXT. Notes puts a blank 
	//	CDTEXT after subforms, so we will too.
	if (us_error = us_GetVirtualCdRecAppendPointer( &t_Subform, 
													sizeof( CDHOTSPOTEND) + 
													sizeof( CDTEXT), &puc))
		goto errJump;
	memcpy( puc, &t_V4HOTSPOT_END, sizeof( CDHOTSPOTEND));
	memcpy( puc + sizeof( CDHOTSPOTEND), &t_BLANK, sizeof( CDTEXT));

	//update the length of the item written to
	t_Subform.pt_endVirtual->ul_length += 
									t_Subform.pt_endVirtual->ul_length % 2 + 
									sizeof( CDHOTSPOTEND) + sizeof( CDTEXT);

	//prepend the now-compiled subform
	pt_item = t_Subform.pt_Virtuality;
	while (pt_item->pt_next)
		pt_item = pt_item->pt_next;
	pt_item->pt_next = pt_contextForm->pt_Virtuality;
	pt_contextForm->pt_Virtuality = t_Subform.pt_Virtuality;
	if (!pt_contextForm->pt_endVirtual)
		pt_contextForm->pt_endVirtual = pt_item;

errJump:
	if (us_error && f_SubformBegun)
		ef_FreeRtfContext( &t_Subform);
	if (pt_FormItemLocked)	{
		OSUnlockBlock( pt_FormItemLocked->bid_contents);
		pt_FormItemLocked->puc_start = NULL;
	}
	if (pt_SubformItemLocked)	{
		OSUnlockBlock( pt_SubformItemLocked->bid_contents);
		pt_SubformItemLocked->puc_start = NULL;
	}

	return us_error;
} //eus_InsertSubformTop(


/** ClearItemList( ***
Purpose is to clear out completely a list of rich-text item structures, 
including the freeing of associated memory.

--- parameter ----------
ppt_ListHead: Input & Output. Pointer to the node at which to start the 
	clearing out of the list. The node will be nullified upon return.

--- revision history ---
10/29/98 PR: created		*/
static void ClearItemList( ItemInfo * *const  ppt_ListHead)	{
	_ASSERTE( ppt_ListHead);

	//if there's something to deal with, clear out the list
	if (*ppt_ListHead)	{
		UnlockItems( *ppt_ListHead, NULL);
		FreeItemList( ppt_ListHead);
	}
} //ClearItemList(


/** UnlockItems( ***
Purpose is to unlock the memory associated with a given succession of 
rich-text items.

--- parameters ---------
pt_ListHead: Input & Output. Pointer to the head node of the list of 
	rich-text items to unlock. Note that this "head node" need not be the 
	actual head node of the source linked-list of items. For each item 
	unlocked, the item's member pointer puc_start will be nullified.
pt_STOP_NODE: Pointer to the node associtated with the rich-text item at 
	which the unlocking should stop. The item should appear later in the 
	list than the specified head node.

--- revision history ---
10/29/98 PR: created		*/
static __inline void UnlockItems( ItemInfo *const  pt_ListHead, 
									const ItemInfo *const  pt_STOP_NODE)	{
	ItemInfo * pt_item = pt_ListHead;

	//Loop through the given list, unlocking the memory associated with each 
	//	rich-text-item node, as necessary. If no stop node was provided, the 
	//	loop will run through to the end of the list.
	while (pt_item && pt_item != pt_STOP_NODE)	{
		if (pt_item->puc_start)	{
			OSUnlockBlock( pt_item->bid_contents);
			pt_item->puc_start = NULL;
		}

		pt_item = pt_item->pt_next;
	}
} //UnlockItems(


