/*____________________________________________________________________________
	Copyright (C) 1999 Pretty Good Privacy, Inc.
	All rights reserved.
	

	$Id: StringHandling.c,v 1.4.12.8 2000/08/09 01:23:10 build Exp $
____________________________________________________________________________*/

/*::: MODULE OVERVIEW :::::::::::::

--- revision history --------
8/26/99 Version 1.1: Paul Ryan
+ integrated standard LibLinkList functionality
+ documentation adjustment

1/9/99 Version 1.0: Paul Ryan
::::::::::::::::::::::::::::::::::::*/

#include "StringHandling.h"


/** e_FreeStringArray( ***


--- parameters ----------


--- revision history ----
8/26/99 PR: documentation adjustment
1/5/99 PR: created		*/
//DOC!!
void e_FreeStringArray( char * * *const  pppc, 
						const unsigned long  ul_ELEMENTS)	{
	char * * ppc = *pppc;

	if (!pppc)
		return;

	e_FreeStringArrayElements( ppc, ul_ELEMENTS);

	//free the array pointer itself
	free( ppc);

	//nullify the array pointer to assist the caller in not using it anymore
	*pppc = NULL;
} //e_FreeStringArray(


/** e_FreeStringArrayElements( ***


--- parameters ---------


--- revision history ---
1/5/99 PR: created		*/
//DOC!!
void e_FreeStringArrayElements( char * *const  ppc, 
								const unsigned long  ul_ELEMENTS)	{
	unsigned long  ul;

	if (!ppc)
		return;

	//free each string within the array
	for (ul = 0; ul < ul_ELEMENTS; ul++)
		if (ppc[ ul])	{
			free( ppc[ ul]);
			ppc[ ul] = NULL;
		}
} //e_FreeStringArrayElements(


/** ef_AddStringNodeFifo( ***


--- parameters & return ----

RETURN: TRUE if no error occurred; FALSE otherwise

--- revision history -------
8/26/99 PR: integration of standard LibLinkList functionality
1/11/99 PR: created			*/
//DOC!!
BOOL ef_AddStringNodeFifo( char *const  pc_orig, 
							const BOOL  f_COPY, 
							StringNode * *const  ppt_nd)	{
	char * pc;

	if (!( pc_orig && ppt_nd))
		return FALSE;

	//according to the caller's request, either transfer control of the 
	//	string to the just-created node or allocate new space for the string 
	//	and copy it in
	if (!ef_AddListNodeFifo( pc = f_COPY ? malloc( strlen( pc_orig) + 1) : 
														pc_orig, ppt_nd))	{
		if (f_COPY)
			free( pc);
		return FALSE;
	}
	if (f_COPY)
		strcpy( (*ppt_nd)->pc, pc_orig);

	return TRUE;
} //ef_AddStringNodeFifo(


/** ef_AddStringNodeFifoUnique( ***
Purpose is to append a given string to the end of a given list, provided that 
that string is not already in the list.

--- parameters & return -----
pc: pointer to the string to append to the list if it's not already there
f_COPY: flag telling whether a copy of the string (pc) should be stored, 
	instead of the string itself
f_CASE_SENSITIVE: flag telling whether uniqueness should be determined on a 
	case-sensitive or -insensitive basis
ppt: Input & Output. Pointer to the _head_ node pointer in the list to be 
	appended to. If a node is added, the node pointer returned will point to 
	the added node, else the node pointer will be null to indicate that the 
	string was not unique to the list.
RETURN: TRUE if no error occurred; FALSE if memory is insufficient to add the 
	string node

--- suggested enhancement ---
8/26/99 PR: extend procedure so it can be used on a case-insensitive basis

--- revision history --------
8/26/99 PR: created			*/
BOOL ef_AddStringNodeFifoUnique( char *const  pc, 
									const BOOL  f_COPY, 
									const BOOL  f_CASE_SENSITIVE, 
									StringNode * *const  ppt)	{
	StringNode * pt_nd, * pt;

	if (!( pc && ppt))
		return FALSE;

	pt_nd = *ppt;
	*ppt = NULL;

	if (ef_ListContainsString( pc, pt_nd, f_CASE_SENSITIVE))
		return TRUE;

	//according to the caller's request, either transfer control of the 
	//	string to the just-created node or allocate new space for the string 
	//	and copy it in
	pt = pt_nd;
	if (!ef_AddStringNodeFifo( pc, f_COPY, &pt))
		return FALSE;
	if (!pt_nd)
		pt_nd = pt;

	*ppt = pt;

	return TRUE;
} //ef_AddStringNodeFifoUnique(


/** epc_strtokStrict( ***
Behaves as the standard strtok() function does, except that leading 
delimiters are not skipped but instead return a null string. Thus this 
procedure behaves more along the lines of a "split" function.

--- parameters & return ----
pc_refresh: Pointer to the source string to be searched for the next 
	delimiter. If provided, string will be preserved statically and ensuing 
	tokens obtained by specifying this parameter as NULL.
pc_DELIM: pointer to the string of one-character delimiters to use in parsing 
	the source string
RETURN: Pointer to the next token in the source string. NULL if no further 
	tokens are available in the source string or if no source string has been 
	statically stored by the procedure.

--- revision history -------
8/26/99 PR: created			*/
const char * epc_strtokStrict( char *const  pc_refresh, 
								const char  pc_DELIM[])	{
	static char * pc_src;

	const char * pc, * pc_ret = NULL;
	char * pc_end;

	if (!( pc_DELIM && (pc_refresh || pc_src)))
		return NULL;

	if (pc_refresh)
		pc_src = pc_refresh;

	pc = pc_DELIM;
	do	{
		if (!(pc_end = strchr( pc_src, *pc)))
			continue;

		*pc_end = (char) NULL;
		pc_ret = pc_src;
		pc_src = pc_end + 1;
		break;
	} while (*++pc);

	if (!*pc)	{
		pc_ret = pc_src;
		pc_src = NULL;
	}

	return pc_ret;
} //epc_strtokStrict(


/** ef_ListContainsString( ***
Returns whether a string list includes a given string.

--- parameters & return ----
PC: pointer to the string to be sought for within the list
pt: pointer to the head node of the list to be searched
f_CASE_SENSITIVE: flag telling whether uniqueness should be determined on a 
	case-sensitive or -insensitive basis
RETURN: TRUE if list contains string; FALSE otherwise

--- revision history -------
8/26/99 PR
+ extended procedure to accommodate case-insensitive comparisons
+ completed documentation

1/9/99 PR: created			*/
BOOL ef_ListContainsString( const char  PC[], 
							const StringNode * pt, 
							const BOOL  f_CASE_SENSITIVE)	{
	if (!( PC && pt))
		return FALSE;

	do
		if (pt->pc)
			if ( (f_CASE_SENSITIVE ? strcmp( PC, pt->pc) : 
											stricmp( PC, pt->pc)) == ei_SAME)
				return TRUE;
	while (pt = pt->pt_next);

	return FALSE;
} //ef_ListContainsString(


