/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	
	
	$Id: PGPSockets.cp,v 1.31.6.1 1999/06/04 23:33:17 heller Exp $
____________________________________________________________________________*/

#include <CodeFragments.h>
#include <NumberFormatting.h>

#include <string.h>
#include <ctype.h>

#include "CSocket.h"
#include "CInternetUtilities.h"

#define PGP_MACINTOSH 1

#include "PGPSockets.h"



static SInt32					sNumberOfInits = 0;
static map<PGPError, PGPError>	sErrorMap;
static char *					sNull[] = {0};
static char *					sDiscardAliases[] = {"sink", "null", 0};
static char *					sSystatAliases[] = {"users", 0};
static char *					sQotdAliases[] = {"quote", 0};
static char *					sChargenAliases[] = {"ttytst", "source", 0};
static char *					sSMTPAliases[] = {"mail", 0};
static char *					sTimeAliases[] = {"timserver", 0};
static char *					sNameServerAliases[] = {"name", 0};
static char *					sWhoisAliases[] = {"nicname", 0};
static char *					sRJEAliases[] = {"netrjs", 0};
static char *					sWWWAliases[] = {"http", 0};
static char *					sLinkAliases[] = {"ttylink", 0};
static char *					sKerberosAliases[] = {"krb5", 0};
static char *					sHostnamesAliases[] = {"hostname", 0};
static char *					sISO_TSAPAliases[] = {"tsap", 0};
static char *					sCSNET_NSAliases[] = {"cso-ns", 0};
static char *					sPOP2Aliases[] = {"postoffice", 0};
static char *					sAuthAliases[] = {"tap", "ident", 
												  "authentication", 0};
static char *					sNNTPAliases[] = {"readnews", "untp",
												  "usenet", 0};
static char *					sSNMP_TrapAliases[] = {"snmptrap", 0};
static char *					sNextstepAliases[] = {"NeXTStep", "NextStep",
													  0};
static char *					sZ3950Aliases[] = {"wais", 0};
static char *					sBiffAliases[] = {"comsat", 0};
static char *					sWhoAliases[] = {"whod", 0};
static char *					sShellAliases[] = {"cmd", 0};
static char *					sPrinterAliases[] = {"spooler", 0};
static char *					sRouteAliases[] = {"router", "routed", 0};
static char *					sTimedAliases[] = {"timeserver", 0};
static char *					sTempoAliases[] = {"newdate", 0};
static char *					sCourierAliases[] = {"rpc", 0};
static char *					sConferenceAliases[] = {"chat", 0};
static char *					sUUCPAliases[] = {"uucpd", 0};
static char *					sRemotefsAliases[] = {"rfs_server", "rfs",
													  0};
static char *					sKrbupdateAliases[] = {"kreg", 0};
static char *					sKpasswdAliases[] = {"kpwd", 0};
static PGPProtocolEntry			sProtocolArray[] = {
									{"tcp", sNull, 6},
									{"udp", sNull, 17}};
									
static PGPServiceEntry			sServiceArray[] = {
		{"tcpmux", 			sNull,					1,		"tcp"},
		{"echo",			sNull,					7,		"tcp"},
		{"echo",			sNull,					7,		"udp"},
		{"discard",			sDiscardAliases,		9,		"tcp"},
		{"discard", 		sDiscardAliases,		9,		"udp"},
		{"systat",			sSystatAliases,			11,		"tcp"},
		{"daytime",			sNull,					13,		"tcp"},
		{"daytime",			sNull,					13,		"udp"},
		{"netstat",			sNull,					15,		"tcp"},
		{"qotd",			sQotdAliases,			17,		"tcp"},
		{"msp",				sNull,					18,		"tcp"},
		{"msp",				sNull,					18,		"udp"},
		{"chargen",			sChargenAliases,		19,		"tcp"},
		{"chargen",			sChargenAliases,		19,		"udp"},
		{"ftp",				sNull,					21,		"tcp"},
		{"telnet",			sNull,					23,		"tcp"},
		{"smtp",			sSMTPAliases,			25,		"tcp"},
		{"time",			sTimeAliases,			37,		"tcp"},
		{"time",			sTimeAliases,			37,		"udp"},
		{"rlp",				sNull,					39,		"udp"},
		{"nameserver",		sNameServerAliases,		42,		"tcp"},
		{"whois",			sWhoisAliases,			43,		"tcp"},
		{"domain",			sNull,					53,		"tcp"},
		{"domain",			sNull,					53,		"udp"},
		{"bootps",			sNull,					67,		"tcp"},
		{"bootps",			sNull,					67,		"udp"},
		{"bootpc",			sNull,					68,		"tcp"},
		{"bootpc",			sNull,					68,		"udp"},
		{"tftp",			sNull,					69,		"udp"},
		{"gopher",			sNull,					70,		"tcp"},
		{"gopher",			sNull,					70,		"udp"},
		{"rje",				sRJEAliases,			77,		"tcp"},
		{"finger",			sNull,					79,		"tcp"},
		{"www",				sWWWAliases,			80,		"tcp"},
		{"www",				sNull,					80,		"udp"},
		{"link", 			sLinkAliases,			87,		"tcp"},
		{"kerberos",		sKerberosAliases,		88,		"tcp"},
		{"kerberos",		sNull,					88,		"udp"},
		{"supdup",			sNull,					95,		"tcp"},
		{"hostnames",		sHostnamesAliases,		101,	"tcp"},
		{"iso-tsap", 		sISO_TSAPAliases,		102,	"tcp"},
		{"csnet-ns",		sCSNET_NSAliases,		105,	"tcp"},
		{"csnet-ns",		sCSNET_NSAliases,		105,	"udp"},
		{"rtelnet",			sNull,					107,	"tcp"},
		{"rtelnet",			sNull,					107,	"udp"},
		{"pop2",			sPOP2Aliases,			109,	"tcp"},
		{"pop2",			sNull,					109,	"udp"},
		{"pop3",			sNull,					110,	"tcp"},
		{"pop3",			sNull,					110,	"udp"},
		{"sunrpc",			sNull,					111,	"tcp"},
		{"sunrpc",			sNull,					111,	"udp"},
		{"auth",			sAuthAliases,			113,	"tcp"},
		{"sftp",			sNull,					115,	"tcp"},
		{"uucp-path",		sNull,					117,	"tcp"},
		{"nntp",			sNNTPAliases,			119,	"tcp"},
		{"ntp",				sNull,					123,	"tcp"},
		{"ntp",				sNull,					123,	"udp"},
		{"netbios-ns",		sNull,					137,	"tcp"},
		{"netbios-ns",		sNull,					137,	"udp"},
		{"netbios-dgm",		sNull,					138,	"tcp"},
		{"netbios-dgm",		sNull,					138,	"udp"},
		{"netbios-ssn",		sNull,					139,	"tcp"},
		{"netbios-ssn",		sNull,					139,	"udp"},
		{"imap2",			sNull,					143,	"tcp"},
		{"imap2",			sNull,					143,	"udp"},
		{"snmp",			sNull,					161,	"udp"},
		{"snmp-trap",		sSNMP_TrapAliases,		162,	"udp"},
		{"cmip-man",		sNull,					163,	"tcp"},
		{"cmip-man",		sNull,					163,	"udp"},
		{"cmip-agent",		sNull,					164,	"tcp"},
		{"cmip-agent",		sNull,					164,	"udp"},
		{"xdmcp",			sNull,					177,	"tcp"},
		{"xdmcp",			sNull,					177,	"udp"},
		{"nextstep",		sNextstepAliases,		178,	"tcp"},
		{"nextstep",		sNextstepAliases,		178,	"udp"},
		{"bgp",				sNull,					179,	"tcp"},
		{"bgp",				sNull,					179,	"udp"},
		{"prospero",		sNull,					191,	"tcp"},
		{"prospero",		sNull,					191,	"udp"},
		{"irc",				sNull,					194,	"tcp"},
		{"irc",				sNull,					194,	"udp"},
		{"smux",			sNull,					199,	"tcp"},
		{"smux",			sNull,					199,	"udp"},
		{"at-rtmp",			sNull,					201,	"tcp"},
		{"at-rtmp",			sNull,					201,	"udp"},
		{"at-nbp",			sNull,					202,	"tcp"},
		{"at-nbp",			sNull,					202,	"udp"},
		{"at-echo",			sNull,					204,	"tcp"},
		{"at-echo",			sNull,					204,	"udp"},
		{"at-zis",			sNull,					206,	"tcp"},
		{"at-zis",			sNull,					206,	"udp"},
		{"z3950",			sZ3950Aliases,			210,	"tcp"},
		{"z3950",			sZ3950Aliases,			210,	"udp"},
		{"ipx",				sNull,					213,	"tcp"},
		{"ipx",				sNull,					213,	"udp"},
		{"imap3",			sNull,					220,	"tcp"},
		{"imap3",			sNull,					220,	"udp"},
		{"ulistserv",		sNull,					372,	"tcp"},
		{"ulistserv",		sNull,					372,	"udp"},
		{"https",			sNull,					443,	"tcp"},
		{"exec",			sNull,					512,	"tcp"},
		{"biff",			sBiffAliases,			512,	"udp"},
		{"login",			sNull,					513,	"tcp"},
		{"who",				sWhoAliases,			513,	"udp"},
		{"shell",			sShellAliases,			514,	"tcp"},
		{"syslog",			sNull,					514,	"udp"},
		{"printer",			sPrinterAliases,		515,	"tcp"},
		{"talk",			sNull,					517,	"udp"},
		{"ntalk",			sNull,					518,	"udp"},
		{"route",			sRouteAliases,			520,	"udp"},
		{"timed",			sTimedAliases,			525,	"udp"},
		{"tempo",			sTempoAliases,			526,	"tcp"},
		{"courier",			sCourierAliases,		530,	"tcp"},
		{"conference",		sConferenceAliases,		531,	"tcp"},
		{"netnews",			sNull,					532,	"tcp"},
		{"netwall",			sNull,					533,	"udp"},
		{"uucp",			sUUCPAliases,			540,	"tcp"},
		{"remotefs",		sRemotefsAliases,		556,	"tcp"},
		{"klogin",			sNull,					543,	"tcp"},
		{"kshell",			sNull,					544,	"tcp"},
		{"kerberos-adm",	sNull,					749,	"tcp"},
		{"webster",			sNull,					765,	"tcp"},
		{"webster",			sNull,					765,	"udp"},
		{"ingreslock",		sNull,					1524,	"tcp"},
		{"ingreslock",		sNull,					1524,	"udp"},
		{"prospero-np",		sNull,					1525,	"tcp"},
		{"prospero-np",		sNull,					1525,	"udp"},
		{"rfe",				sNull,					5002,	"tcp"},
		{"rfe",				sNull,					5002,	"udp"},
		{"krbupdate",		sKrbupdateAliases,		760,	"tcp"},
		{"kpasswd",			sKpasswdAliases,		761,	"tcp"},
		{"eklogin",			sNull,					2105,	"tcp"},
		{"supfilesrv",		sNull,					871,	"tcp"},
		{"supfiledbg",		sNull,					1127,	"tcp"}};
												
class StSocketsThreadBusy {
public:
			StSocketsThreadBusy(SThreadContext * inContext, CSocket * inSocket);
	virtual ~StSocketsThreadBusy();

protected:
	SThreadContext *	mContext;
};

void			InitErrorMap();
PGPError		MapPGPSocketsError(PGPError inErr);
PGPInt32		PGPSocketsTLSReceive(void *inData, void *outBuffer,
					PGPInt32 inBufferSize);
PGPInt32		PGPSocketsTLSSend(void *inData, const void *inBuffer,
										PGPInt32 inBufferLength);



StSocketsThreadBusy::StSocketsThreadBusy(
	SThreadContext *	inContext,
	CSocket *			inSocket)
	: mContext(inContext)
{
	if (mContext == nil) {
		ThrowPGPError_(kPGPError_UnknownError);
	}
	if ((mContext->isBusy) || ((inSocket != nil) && inSocket->IsInCallback())) {
		ThrowPGPError_(kPGPError_SocketsInProgress);
 	} else {
 	 	mContext->isBusy = true;
 	}
}



StSocketsThreadBusy::~StSocketsThreadBusy()
{
	if (mContext == nil) {
		ThrowPGPError_(kPGPError_UnknownError);
	}
	mContext->isBusy = false;
}


	PGPError
PGPSocketsCreateThreadStorage(
	PGPSocketsThreadStorageRef *	outPreviousStorage)
{
	PGPError	err = kPGPError_NoErr;
	
	try {
		if (outPreviousStorage == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		*outPreviousStorage = kInvalidPGPSocketsThreadStorageRef;
					
		CSocket::CreateThreadStorage(
					(SThreadContext **) outPreviousStorage);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}

	return err;
}



	PGPError
PGPSocketsDisposeThreadStorage(
	PGPSocketsThreadStorageRef	inPreviousStorage)
{
	PGPError	err = kPGPError_NoErr;
	
	try {
		CSocket::DisposeThreadStorage(
					(SThreadContext *) inPreviousStorage);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}

	return err;
}


	PGPError
PGPSocketsInit()
{
	PGPError	err = kPGPError_NoErr;
	
	try {
		if (sNumberOfInits == 0) {
			InitErrorMap();
		}
		
		sNumberOfInits++;
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}

	return err;
}



	void
PGPSocketsCleanup()
{
	try {
		sNumberOfInits--;
		if (sNumberOfInits == 0) {
			CSocket::CleanupSockets();
			sErrorMap.erase(sErrorMap.begin(), sErrorMap.end());
		} else if (sNumberOfInits < 0) {
			sNumberOfInits = 0;
		}
	}

	catch (...) {
	}
}



	PGPSocketRef
PGPOpenSocket(
	PGPInt32	inAddressFamily,
	PGPInt32	inSocketType,
	PGPInt32	inSocketProtocol)
{
	PGPSocketRef		theSocket = kInvalidPGPSocketRef;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		StSocketsThreadBusy	theBusyThread(theContext, nil);
		
		CSocket::Initialize();

		// Switch on the family. Currently we only support the AF_INET family
		switch (inAddressFamily) {
			case kPGPAddressFamilyInternet:
			{
				// Switch on the type
				switch (inSocketType) {
					case kPGPSocketTypeStream:
					case kPGPSocketTypeDatagram:
					{
						// Switch on the protocol
						switch (inSocketProtocol) {
							case kPGPTCPProtocol:
							case kPGPUDPProtocol:
							{
								theSocket =
									(PGPSocketRef) CSocket::CreateSocket(
													inSocketType);
							}
							break;
							
							
							default:
							{
								ThrowPGPError_(
									kPGPError_SocketsProtocolNotSupported);
							}
							break;
						}
					}
					break;
					
					
					default:
					{
						ThrowPGPError_(kPGPError_SocketsProtocolNotSupported);
					}
					break;
				}
			}
			break;
			
			
			default:
			{
				ThrowPGPError_(kPGPError_SocketsAddressFamilyNotSupported);
			}
			break;
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return theSocket;
}




	PGPInt32
PGPCloseSocket(
	PGPSocketRef	inSocketRef)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		// Everything is good, so close the object
		((CSocket *) inSocketRef)->Close();
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPBindSocket(
	PGPSocketRef				inSocketRef,
	const PGPSocketAddress *	inAddress,
	PGPInt32					inAddressLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		if (inAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((inAddressLength != sizeof(PGPSocketAddressInternet))
		|| (inAddress->sa_family != kPGPAddressFamilyInternet)) {
			ThrowPGPError_(kPGPError_SocketsAddressFamilyNotSupported);
		}
		
		// Everything is good, so bind the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->
				Bind((const PGPSocketAddressInternet *) inAddress);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return result;
}



	PGPInt32
PGPConnect(
	PGPSocketRef				inSocketRef,
	const PGPSocketAddress *	inServerAddress,
	PGPInt32					inAddressLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		if (inServerAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((inAddressLength != sizeof(PGPSocketAddressInternet))
		|| (inServerAddress->sa_family != kPGPAddressFamilyInternet)) {
			ThrowPGPError_(kPGPError_SocketsAddressFamilyNotSupported);
		}
		
		// Everything is good, so bind the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->
				Connect((const PGPSocketAddressInternet *) inServerAddress);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return result;
}



	PGPInt32
PGPSend(
	PGPSocketRef	inSocketRef,
	const void *	inBuffer,
	PGPInt32		inBufferLength,
	PGPInt32		inFlags)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if ((inBufferLength < 0) ||
		(inFlags != kPGPSendFlagNone)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so send through the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		if (((CSocket *) inSocketRef)->HasTLSSession()) {
			err = PGPtlsSend(	((CSocket *) inSocketRef)->GetTLSSession(),
								inBuffer,
								inBufferLength);
			if (err == kPGPError_NoErr) {
				result = inBufferLength;
			} else if (err == kPGPError_TLSUnexpectedClose) {
				err = theContext->lastError;
			}
		} else {
			result = ((CSocket *) inSocketRef)->Send(	inBuffer,
														inBufferLength,
														inFlags);
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return result;
}



	PGPInt32
PGPWrite(
	PGPSocketRef	inSocketRef,
	const void *	inBuffer,
	PGPInt32		inBufferLength)
{
	return PGPSend(inSocketRef, inBuffer, inBufferLength, kPGPSendFlagNone);
}



	PGPInt32
PGPSendTo(
	PGPSocketRef		inSocketRef,
	const void *		inBuffer,
	PGPInt32			inBufferLength,
	PGPInt32			inFlags,
	PGPSocketAddress *	inAddress,
	PGPInt32			inAddressLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if ((inBufferLength < 0) || (inFlags != kPGPSendFlagNone)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (inAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((inAddressLength != sizeof(PGPSocketAddressInternet))
		|| (inAddress->sa_family != kPGPAddressFamilyInternet)) {
			ThrowPGPError_(kPGPError_SocketsAddressFamilyNotSupported);
		}
		
		// Everything is good, so send through the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		result = ((CSocket *) inSocketRef)->SendTo(inBuffer, inBufferLength,
					(const PGPSocketAddressInternet *) inAddress);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPReceive(
	PGPSocketRef	inSocketRef,
	void *			outBuffer,
	PGPInt32		inBufferSize,
	PGPInt32		inFlags)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (outBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if ((inBufferSize < 0) || (inFlags != kPGPReceiveFlagNone)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so send through the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		if (((CSocket *) inSocketRef)->HasTLSSession()) {
			PGPSize	bufferSize = inBufferSize;
			
			err = PGPtlsReceive(((CSocket *) inSocketRef)->GetTLSSession(),
								outBuffer,
								&bufferSize);
			if (err == kPGPError_NoErr) {
				result = bufferSize;
			} else if (err == kPGPError_TLSUnexpectedClose) {
				err = theContext->lastError;
				if (err == kPGPError_NoErr) {
					result = 0;
				}
			}
		} else {
			result = ((CSocket *) inSocketRef)->
							Receive(outBuffer, inBufferSize, inFlags);
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPRead(
	PGPSocketRef	inSocketRef,
	void *			outBuffer,
	PGPInt32		inBufferSize)
{
	return PGPReceive(	inSocketRef,
						outBuffer,
						inBufferSize,
						kPGPReceiveFlagNone);
}



	PGPInt32
PGPReceiveFrom(
	PGPSocketRef		inSocketRef,
	void *				outBuffer,
	PGPInt32			inBufferSize,
	PGPInt32			inFlags,
	PGPSocketAddress *	outAddress,
	PGPInt32 *			ioAddressLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (outBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if ((inBufferSize < 0) || (inFlags != kPGPReceiveFlagNone)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((outAddress != 0)
		&& (*ioAddressLength < sizeof(PGPSocketAddressInternet))) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so send through the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		result = ((CSocket *) inSocketRef)->
						ReceiveFrom(outBuffer,
									inBufferSize,
									(PGPSocketAddressInternet *) outAddress,
									(SInt32 *) ioAddressLength);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPListen(
	PGPSocketRef	inSocketRef,
	PGPInt32		inMaxBacklog)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inMaxBacklog < 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so send through the object
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->Listen(inMaxBacklog);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}


	PGPSocketRef
PGPAccept(
	PGPSocketRef		inSocketRef,
	PGPSocketAddress *	outAddress,
	PGPInt32 *			ioAddressLength)
{
	PGPSocketRef		result = kInvalidPGPSocketRef;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		if (outAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (*ioAddressLength < sizeof(PGPSocketAddressInternet)) {
			ThrowPGPError_(kPGPError_SocketsAddressFamilyNotSupported);
		}
		
		// Everything is good, so accept a connection
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		result = (PGPSocketRef) ((CSocket *) inSocketRef)->
						Accept(	(PGPSocketAddressInternet *) outAddress,
								(SInt32 *) ioAddressLength);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPSelect(
	PGPInt32					inNumSetCount,
	PGPSocketSet *				ioReadSet,
	PGPSocketSet *				ioWriteSet,
	PGPSocketSet *				ioErrorSet,
	const PGPSocketsTimeValue *	inTimeout)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	(void) inNumSetCount;
	
	try {
		StSocketsThreadBusy	theBusyThread(theContext, nil);
		
		SInt16	i;
	
		// Make sure arguments are valid
		if ((ioReadSet == 0) && (ioWriteSet == 0)
		&& (ioErrorSet == 0)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (ioReadSet != 0) {
			if (ioReadSet->fd_count > PGPSOCKETSET_SETSIZE) {
				ThrowPGPError_(kPGPError_BadParams);
			}			
			for (i = 0; i < ioReadSet->fd_count; i++) {
				if (! CSocket::VerifyPGPSocketRef((CSocket *) ioReadSet->
				fd_array[i])) {
					ThrowPGPError_(kPGPError_SocketsNotASocket);
				}
			}
		}
		if (ioWriteSet != 0) {
			if (ioWriteSet->fd_count > PGPSOCKETSET_SETSIZE) {
				ThrowPGPError_(kPGPError_BadParams);
			}
			for (i = 0; i < ioWriteSet->fd_count; i++) {
				if (! CSocket::VerifyPGPSocketRef((CSocket *) ioWriteSet->
				fd_array[i])) {
					ThrowPGPError_(kPGPError_SocketsNotASocket);
				}
			}
		}
		if (ioErrorSet != 0) {
			if (ioErrorSet->fd_count > PGPSOCKETSET_SETSIZE) {
				ThrowPGPError_(kPGPError_BadParams);
			}
			for (i = 0; i < ioErrorSet->fd_count; i++) {
				if (! CSocket::VerifyPGPSocketRef((CSocket *) ioErrorSet->
				fd_array[i])) {
					ThrowPGPError_(kPGPError_SocketsNotASocket);
				}
			}
		}
		if (inTimeout != 0) {
			if ((inTimeout->tv_sec < 0) || (inTimeout->tv_usec < 0)) {
				ThrowPGPError_(kPGPError_BadParams);
			}
		}
		
		result = CSocket::Select(	ioReadSet,
									ioWriteSet,
									ioErrorSet,
									inTimeout);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
__PGPSocketsIsSet(
	PGPSocketRef	inSocketRef,
	PGPSocketSet *	inSocketSet)
{
	PGPInt32 	result = 0;
	PGPUInt16 i;
	
	for (i = 0; i < inSocketSet->fd_count; i++) {
		if (inSocketSet->fd_array[i] == inSocketRef) {
			result = 1;
			break;
		}
	}
	
	return result;
}



	PGPHostEntry *
PGPGetHostByName(
	const char * inName)
{
	PGPHostEntry *		result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		StSocketsThreadBusy	theBusyThread(theContext, nil);
		
		CSocket::Initialize();

		// Make sure that the parameters are correct
		if (inName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}

		result = CSocket::sInternetUtilities->GetHostByName(inName);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPHostEntry *
PGPGetHostByAddress(
	const char *	inAddress,
	PGPInt32		inLength,
	PGPInt32		inType)
{
	PGPHostEntry *		result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		StSocketsThreadBusy	theBusyThread(theContext, nil);
		
		CSocket::Initialize();

		// Make sure that the parameters are correct
		if (inAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (inLength != sizeof(PGPInternetAddress)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (inType != kPGPProtocolFamilyInternet) {
			ThrowPGPError_(kPGPError_BadParams);
		}

		result = CSocket::sInternetUtilities->GetHostByAddress(
							*(PGPInternetAddress *) inAddress);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPGetHostName(
	char *		outName,
	PGPInt32	inNameLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		StSocketsThreadBusy	theBusyThread(theContext, nil);
		
		CSocket::Initialize();

		// Make sure that the parameters are correct
		if (outName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (inNameLength < 1) {
			ThrowPGPError_(kPGPError_BadParams);
		}

		CSocket::sInternetUtilities->GetHostName(outName, inNameLength);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPProtocolEntry *
PGPGetProtocolByName(
	const char *	inName)
{
	PGPProtocolEntry *	result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		if (theContext == nil) {
			ThrowPGPError_(kPGPError_UnknownError);
		}
		
		// Make sure that the parameters are correct
		if (inName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		for (UInt16 x = 0;
		  x < (sizeof(sProtocolArray)/sizeof(PGPProtocolEntry)); x++) {
		  	if (strcmp(inName, sProtocolArray[x].p_name) == 0) {
		  		theContext->protocolEntry = sProtocolArray[x];
		  		result = &theContext->protocolEntry;
		  		break;
		  	}
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPProtocolEntry *
PGPGetProtocolByNumber(
	PGPInt32	inNumber)
{
	PGPProtocolEntry *	result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		if (theContext == nil) {
			ThrowPGPError_(kPGPError_UnknownError);
		}
		
		for (UInt16 x = 0;
		  x < (sizeof(sProtocolArray)/sizeof(PGPProtocolEntry)); x++) {
		  	if (inNumber == sProtocolArray[x].p_proto) {
		  		theContext->protocolEntry = sProtocolArray[x];
		  		result = &theContext->protocolEntry;
		  		break;
		  	}
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPServiceEntry *
PGPGetServiceByName(
	const char *	inName,
	const char *	inProtocol)
{
	PGPServiceEntry *	result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		if (theContext == nil) {
			ThrowPGPError_(kPGPError_UnknownError);
		}
		
		// Make sure that the parameters are correct
		if (inName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (PGPGetProtocolByName(inProtocol) == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Walk the list looking for the service
		for (UInt16 x = 0;
		x < (sizeof(sServiceArray)/sizeof(PGPServiceEntry)); x++) {
			if (strcmp(inProtocol, sServiceArray[x].s_proto) == 0) {
			  	if (strcmp(inName, sServiceArray[x].s_name) == 0) {
			  		theContext->serviceEntry = sServiceArray[x];
			  		result = &theContext->serviceEntry;
			  		break;
			  	} else {
			  		// Walk aliases
			  		for (char ** alias = sServiceArray[x].s_aliases;
			  		*alias != 0; alias++) {
			  		  	if (strcmp(inName, *alias) == 0) {
					  		theContext->serviceEntry = sServiceArray[x];
					  		result = &theContext->serviceEntry;
					  		break;
					  	}
					}
				}
			}
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPServiceEntry *
PGPGetServiceByPort(
	PGPInt32		inPort,
	const char *	inProtocol)
{
	PGPServiceEntry *	result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		if (theContext == nil) {
			ThrowPGPError_(kPGPError_UnknownError);
		}
		
		// Make sure that the parameters are correct
		if (PGPGetProtocolByName(inProtocol) == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Walk the list looking for the service
		for (UInt16 x = 0;
		x < (sizeof(sServiceArray)/sizeof(PGPServiceEntry)); x++) {
			if ((strcmp(inProtocol, sServiceArray[x].s_proto) == 0)
			&& (sServiceArray[x].s_port == inPort)) {
		  		theContext->serviceEntry = sServiceArray[x];
		  		result = &theContext->serviceEntry;
		  		break;
			}
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}


	PGPInt32
PGPGetSocketName(
	PGPSocketRef		inSocketRef,
	PGPSocketAddress *	outName,
	PGPInt32 *			ioNameLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (outName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (*ioNameLength != sizeof(PGPSocketAddressInternet)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so get the address
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->GetSocketName(
						(PGPSocketAddressInternet *) outName);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPGetPeerName(
	PGPSocketRef		inSocketRef,
	PGPSocketAddress *	outName,
	PGPInt32 *			ioNameLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (outName == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (*ioNameLength != sizeof(PGPSocketAddressInternet)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so get the address
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->GetPeerName(
						(PGPSocketAddressInternet *) outName);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPUInt32
PGPDottedToInternetAddress(
	const char *	inAddress)
{
	PGPUInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (inAddress == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		size_t	inAddressLength = strlen(inAddress);
		
		if ((inAddressLength == 0) || (inAddressLength > 16)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		for (const char * c = inAddress; *c != '\0'; c++) {
			if ((*c != '.') && (! isdigit(*c))) {
				ThrowPGPError_(kPGPError_BadParams);
			}
		} 
		
		// Translate
		PGPInternetAddress	theAddress;
		UInt32				temp;
		char				theString[20];
		char *				first;
		char *				second;
		char *				third;
		char *				fourth;
		
		strcpy(theString, inAddress);
		first = strtok(theString, ".");
		if (first == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((second = strtok(0, ".")) == 0) {
			theAddress.s_addr = strtoul(first, 0, 10);
		} else if ((third = strtok(0, ".")) == 0) {
			theAddress.s_addr = 0;
			temp = strtoul(first, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b1 = temp;
			temp = strtoul(second, 0, 10);
			if (temp > 0x00FFFFFF) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.s_addr |= temp;
		} else if ((fourth = strtok(0, ".")) == 0) {
			temp = strtoul(first, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b1 = temp;
			temp = strtoul(second, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b2 = temp;
			temp = strtoul(third, 0, 10);
			if (temp > 0xFFFF) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_w.s_w2 = temp;
		} else if (strtok(0, ".") != 0) {
			ThrowPGPError_ (kPGPError_BadParams);
		} else {
			temp = strtoul(first, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b1 = temp;
			temp = strtoul(second, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b2 = temp;
			temp = strtoul(third, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b3 = temp;
			temp = strtoul(fourth, 0, 10);
			if (temp > 255) {
				ThrowPGPError_ (kPGPError_BadParams);
			}
			theAddress.S_un.S_un_b.s_b4 = temp;
		}
		result = theAddress.s_addr;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	char *
PGPInternetAddressToDottedString(
	PGPInternetAddress	inAddress)
{
	char *				result = 0;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		if (theContext == nil) {
			ThrowPGPError_(kPGPError_UnknownError);
		}
		
		Str32	temp;
		char *	curr = theContext->dottedStringBuffer;
		
		::NumToString(inAddress.S_un.S_un_b.s_b1, temp);
		::BlockMoveData(&temp[1], curr, temp[0]);
		curr += temp[0];
		*curr++ = '.';
		::NumToString(inAddress.S_un.S_un_b.s_b2, temp);
		::BlockMoveData(&temp[1], curr, temp[0]);
		curr += temp[0];
		*curr++ = '.';
		::NumToString(inAddress.S_un.S_un_b.s_b3, temp);
		::BlockMoveData(&temp[1], curr, temp[0]);
		curr += temp[0];
		*curr++ = '.';
		::NumToString(inAddress.S_un.S_un_b.s_b4, temp);
		::BlockMoveData(&temp[1], curr, temp[0]);
		curr += temp[0];
		*curr = '\0';
		result = theContext->dottedStringBuffer;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPIOControlSocket(
	PGPSocketRef	inSocketRef,
	PGPInt32		inCommand,
	PGPUInt32 *		ioParam)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inCommand != kPGPSocketCommandGetUnreadData) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (ioParam == nil) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so get the address
		StSocketsThreadBusy	theBusyThread(theContext, (CSocket *) inSocketRef);
		
		((CSocket *) inSocketRef)->IOControlSocket(inCommand,
				(UInt32 *) ioParam);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPGetSocketOptions(
	PGPSocketRef	inSocketRef,
	PGPInt32		inLevel,
	PGPInt32		inOptionName,
	char *			outOptionValue,
	PGPInt32 *		ioOptionLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inLevel != kPGPSocketOptionLevelSocket) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if ((inOptionName != kPGPSocketOptionAcceptingConnections)
		&& (inOptionName != kPGPSocketOptionType)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (outOptionValue == nil) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		if (*ioOptionLength < sizeof(PGPInt16)) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		// Everything is good, so get the address
		*ioOptionLength = sizeof(PGPInt32);
		((CSocket *) inSocketRef)->GetSocketOptions(inOptionName,
				(SInt32 *) outOptionValue);
		result = 0;
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPSetSocketOptions(
	PGPSocketRef	inSocketRef,
	PGPInt32		inLevel,
	PGPInt32		inOptionName,
	const char *	inOptionValue,
	PGPInt32		inOptionLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	(void) inSocketRef;
	(void) inLevel;
	(void) inOptionName;
	(void) inOptionValue;
	(void) inOptionLength;
	
	try {
		ThrowPGPError_ (kPGPError_SocketsOperationNotSupported);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}


	PGPError
PGPSocketsEstablishTLSSession(
	PGPSocketRef		inSocketRef,
	PGPtlsSessionRef	inTLSSession)
{
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inSocketRef)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		StSocketsThreadBusy	theBusyThread(theContext,
			(CSocket *) inSocketRef);
		
		if (((CSocket *) inSocketRef)->GetSocketType() !=
			kPGPSocketTypeStream) {
				ThrowPGPError_(kPGPError_SocketsOperationNotSupported);
		}
		
		err = PGPtlsSetReceiveCallback(	inTLSSession,
										PGPSocketsTLSReceive,
										inSocketRef);
		if (err == kPGPError_NoErr) {
			err = PGPtlsSetSendCallback(	inTLSSession,
											PGPSocketsTLSSend,
											inSocketRef);
			if (err == kPGPError_NoErr) {
				PGPSocketAddressInternet	address;
				
				((CSocket *) inSocketRef)->GetPeerName(&address);
				err = PGPtlsSetRemoteUniqueID(	inTLSSession,
												address.sin_addr.s_addr);
				if (err == kPGPError_NoErr) {
					err = PGPtlsHandshake(inTLSSession);
					if (err == kPGPError_NoErr) {
						((CSocket *) inSocketRef)->
							SetTLSSession(inTLSSession);
					} else if (err == kPGPError_TLSUnexpectedClose
					&& (theContext->lastError != kPGPError_NoErr)) {
						err = theContext->lastError;
					}
				}
			}
		}
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	

	return err;
}



	PGPInt32
PGPSocketsTLSReceive(
	void *		inData,
	void *		outBuffer,
	PGPInt32	inBufferSize)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inData)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (outBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if (inBufferSize < 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		result = ((CSocket *) inData)->Receive(	outBuffer,
												inBufferSize,
												kPGPReceiveFlagNone);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}

	return result;
}



	PGPInt32
PGPSocketsTLSSend(
	void *			inData,
	const void *	inBuffer,
	PGPInt32		inBufferLength)
{
	PGPInt32			result = kPGPSockets_Error;
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		// Make sure that the parameters are correct
		if (! CSocket::VerifyPGPSocketRef((CSocket *) inData)) {
			ThrowPGPError_(kPGPError_SocketsNotASocket);
		}
		
		if (inBuffer == 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}
		
		if (inBufferLength < 0) {
			ThrowPGPError_(kPGPError_BadParams);
		}

		result = ((CSocket *) inData)->Send(	inBuffer,
												inBufferLength,
												kPGPSendFlagNone);
	}
	
	catch (PGPError exception) {
		err = MapPGPSocketsError(exception);
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return result;
}



	PGPError
PGPGetLastSocketsError(void)
{
	SThreadContext *	theContext = CSocket::GetThreadContext();

	if (theContext != nil) {
		return theContext->lastError;
	} else {
		return kPGPError_UnknownError;
	}
}



	PGPError
PGPSetSocketsIdleEventHandler(
	PGPEventHandlerProcPtr	inCallback,
	PGPUserValue			inUserData)
{
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		CSocket::SetIdleEventHandler(inCallback, inUserData);
	}
	
	catch (PGPError exception) {
		err = exception;
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return err;
}



	PGPError
PGPGetSocketsIdleEventHandler(
	PGPEventHandlerProcPtr *	outCallback,
	PGPUserValue *				outUserData)
{
	PGPError			err = kPGPError_NoErr;
	SThreadContext *	theContext = CSocket::GetThreadContext();
	
	try {
		CSocket::GetIdleEventHandler(outCallback, outUserData);
	}
	
	catch (PGPError exception) {
		err = exception;
	}
	
	catch (...) {
		err = kPGPError_UnknownError;
	}
	
	if (theContext != nil) {
		theContext->lastError = err;
	}
	
	return err;
}



	PGPError
MapPGPSocketsError(
	PGPError	inErr)
{
	PGPError	result = kPGPError_UnknownError;
	
	if ((inErr > kPGPPFLErrorBase) && (inErr < kPGPError_Last)) {
		result = inErr;
	} else {
		map<PGPError, PGPError>::iterator	theIterator = sErrorMap.find(
															inErr);
		
		if (theIterator != sErrorMap.end()) {
			result = (*theIterator).second;
		}
	}
	
	return result;
}

	void
InitErrorMap()
{
	pgpLeaksSuspend();
	
	sErrorMap[insufficientResources] = kPGPError_OutOfMemory;
	sErrorMap[kOTAddressBusyErr] = kPGPError_SocketsAddressInUse;
	sErrorMap[kOTNoAddressErr] = kPGPError_OutOfMemory;
	sErrorMap[kOTBadAddressErr] = kPGPError_BadParams;
	sErrorMap[kOTBadOptionErr] = kPGPError_BadParams;
	sErrorMap[kOTBadDataErr] = kPGPError_SocketsBufferOverflow;
	sErrorMap[kOTBufferOverflowErr] = kPGPError_SocketsBufferOverflow;
	sErrorMap[kOTBadFlagErr] = kPGPError_BadParams;
	sErrorMap[kOTNotSupportedErr] = kPGPError_SocketsOperationNotSupported;
	sErrorMap[kOTBadQLenErr] = kPGPError_BadParams;
	sErrorMap[kOTQFullErr] = kPGPError_SocketsListenQueueFull;
	sErrorMap[kENOMEMErr] = kPGPError_OutOfMemory;
	sErrorMap[kOTOutOfMemoryErr] = kPGPError_OutOfMemory;
	sErrorMap[kOTDuplicateFoundErr] = kPGPError_SocketsAddressInUse;
	sErrorMap[kEINVALErr] = kPGPError_BadParams;
	sErrorMap[kEADDRINUSEErr] = kPGPError_SocketsAddressInUse;
	sErrorMap[kEADDRNOTAVAILErr] = kPGPError_SocketsAddressInUse;
	sErrorMap[kENETDOWNErr] = kPGPError_SocketsNetworkDown;
	sErrorMap[kENETUNREACHErr] = kPGPError_SocketsNetworkDown;
	sErrorMap[kECONNREFUSEDErr] = kPGPError_SocketsNotConnected;
	sErrorMap[kEHOSTDOWNErr] = kPGPError_SocketsNetworkDown;
	sErrorMap[kEHOSTUNREACHErr] = kPGPError_SocketsNetworkDown;
	sErrorMap[kENOSRErr] = kPGPError_OutOfMemory;
	sErrorMap[kOTUserRequestedErr] = kPGPError_SocketsNotConnected;
	sErrorMap[kOTPortLostConnection] = kPGPError_SocketsNotConnected;
	sErrorMap[kOTCanceledErr] = kPGPError_UserAbort;

	sErrorMap[kOTBadNameErr] = kPGPError_SocketsHostNotFound;
	sErrorMap[noNameServer] = kPGPError_SocketsHostNotFound;
	sErrorMap[authNameErr] = kPGPError_SocketsHostNotFound;
	sErrorMap[noAnsErr] = kPGPError_SocketsDomainServerError;
	sErrorMap[dnrErr] = kPGPError_SocketsDomainServerError;
	sErrorMap[outOfMemory] = kPGPError_OutOfMemory;
	
	pgpLeaksResume();
}
