/* $Id: sshd_main.c,v 1.47.2.4 2001/02/11 04:58:54 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */


/*-
 * Copyright (c) 1983, 1988, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* This file contains routines to do the initial setup: read */
/* configuration files.  Open necessary sockets.  Listen for */
/* incoming connections. */

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "options.h"

#include "sshd.h"
#include "ssh_auth.h"
#include "ssh_cipher.h"
#include "ssh_global.h"
#include "ssh_logging.h"
#include "ssh_rsakeys.h"
#include "ssh_sys.h"
#include "ssh_util.h"

static int exitflag = 0;
static sig_atomic_t numconns = 0;

void main_sigchld_handler(int);

void sched_sk_regen(int);

int main(int argc, char *argv[]) {

    g_argc = argc;
    g_argv = argv;

    if (setup(0) != 0)
	exit(1);

    /* Setup all done, switch to daemon now. */
    if (!nodaemon) {
	if (ssh_sys_daemonize() != 0) {
	    warn("Unable to daemonize.");
	    cleanup(0);
	    exit(1);
	}
    }

    /* Turn off logging to stderr. */
    if (!nodaemon)
	debug_nostderr();

    dolisten();

    cleanup(0);

    exit(0);
}

/*
 * setup: Read configuration info.
 *	Args: how	0 = clean setup.
 *			!0 = re-read config.
 */
int setup(int how) {
  struct in_addr laddr;
  struct rlimit rl;
  struct itimerval it;

    g_context.cipher = &g_cipher;

    if (loginit(NULL, 1) != 0) {
	warnx("Unable to initialize logging.\n");
	return(1);
    }

    if (how != 0) {
	SSH_ERROR("re-read config not implemented.\n");
	return(0);
    }

    /* Set signal handlers. */
    signal(SIGHUP, (void *)setup);
    signal(SIGINT, doquit);
    signal(SIGCHLD, main_sigchld_handler);
#ifndef UNSAFE_DEBUG
    signal(SIGQUIT, (void *)cleanup); signal(SIGILL, (void *)cleanup);
    signal(SIGTRAP, (void *)cleanup); signal(SIGEMT, (void *)cleanup);
    signal(SIGFPE, (void *)cleanup); signal(SIGBUS, (void *)cleanup);
    signal(SIGSEGV, (void *)cleanup); signal(SIGSYS, (void *)cleanup);
    signal(SIGPIPE, (void *)cleanup); signal(SIGALRM, (void *)cleanup);
    signal(SIGTERM, (void *)cleanup); signal(SIGTSTP, SIG_IGN);
    signal(SIGTTIN, SIG_IGN); signal(SIGTTOU, SIG_IGN);
    signal(SIGXCPU, SIG_IGN); signal(SIGXFSZ, (void *)cleanup);
    signal(SIGPROF, (void *)cleanup);
#endif
    signal(SIGUSR1, debug_inc);
    signal(SIGUSR2, debug_dec);

    if (ssh_sys_configuration(&g_context) != 0)
	return(1);

    if (ssh_sys_writepid(g_context.opts.pidfile) != 0)
	SSH_DLOG(0, ("Warning: unable to write pid file.\n"));

    if (init_serverkey() != 0)
	return(1);

    /* Set the alarm to regenerate the server key: */
    signal(SIGALRM, sched_sk_regen);
    it.it_value.tv_sec = SSHD_REGENINTERVAL;   /* Default: 20 minutes */
    it.it_value.tv_usec = 0;
    it.it_interval.tv_sec = SSHD_REGENINTERVAL;
    it.it_interval.tv_usec = 0;
    setitimer(ITIMER_REAL, &it, NULL);

#ifndef UNSAFE_DEBUG
    rl.rlim_cur = (rlim_t)0;
    rl.rlim_max = (rlim_t)0;

    setrlimit(RLIMIT_CORE, &rl);
#else
    rl = rl;		/* Shut up gcc */
#endif

    ssh_sys_randinit();

    set_supported_ciphers(&(g_context.supported_ciphers));

    set_supported_auths(&(g_context.supported_auths));

    /* Open all addresses we're going to listen to. */
    if (inet_aton(g_context.opts.address, &laddr) != 1) {
        SSH_ERROR("inet_aton failed on address %s: %s\n", 
	      g_context.opts.address, strerror(errno));
	return(1);
    }
    if (OpenListenSocket(laddr, &g_context) != 0) {
	SSH_ERROR("Failed to listen on default address: %s\n", 
	      strerror(errno));
	return(1);
    }
    return(0);
}

/*
 * OpenListenSocket: take an address, open a socket on it, get it ready
 *		to be listen'd to.
 */
int OpenListenSocket(struct in_addr laddr, sshd_context_t *sc) {
  int s;
  int retval;
  struct sockaddr_in sa;
  int dummy;
  struct linger linger;

    memset(&sa, 0, sizeof(sa));

    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	SSH_ERROR("Unable to create socket: %s\n", strerror(errno));
	return(1);
    }

    dummy = 1;

    setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&dummy, sizeof(dummy));

    linger.l_onoff = 1;
    linger.l_linger = 10;
    setsockopt(s, SOL_SOCKET, SO_LINGER, (void *)&linger, sizeof(linger));

    sa.sin_port = htons(atoi(sc->opts.port));
    sa.sin_family = AF_INET;
    sa.sin_addr = laddr;
    sa.sin_len = sizeof(sa);

    if ((retval = bind(s, (struct sockaddr *)&sa, sizeof(sa))) != 0) {
	SSH_ERROR("Unable to bind socket: %s\n", strerror(errno));
	SSH_ERROR("sockaddr:%s\n", inet_ntoa(sa.sin_addr));
	close(s);
	return(1);
    }

    if ((retval = listen(s, 7)) != 0) {
	SSH_ERROR("Unable to listen on socket: %s\n", strerror(errno));
	close(s);
	return(1);
    }

    sc->listen_socket = s;
    
    return(0);
}

/*
 * doquit: set exitflag so the daemon exits gracefully.
 */
void doquit(int sig) {
    exitflag = 1;
    return;
}

/*
 * cleanup: Cleanup before accepting a connection.
 *
 *	Args: sig	0 = normal shutdown
 *			!0 = shutdown due to signal.
 */
int cleanup(int sig) {
    SSH_DLOG(1, ("Received signal %d.  Exiting.\n", sig));
    if (g_context.s > 0)
	close(g_context.s);
    if (g_context.listen_socket > 0)
	close(g_context.listen_socket);
    if (g_context.hostkey)
	ssh_rsa_free(g_context.hostkey);
    if (g_context.serverkey)
	ssh_rsa_free(g_context.serverkey);
    memset(g_context.session_id, 0, 16);
    memset(g_context.session_key, 0, SSH_SESSION_KEY_SIZE);
    packet_cleanup(&g_context.in_pc);
    packet_cleanup(&g_context.out_pc);
    logclose();
    if (sig == SIGHUP || sig == SIGINT)
	exit(0);
    else
	exit(1);
}

/*
 * Just reap any childen.
 */
void main_sigchld_handler(int foo) {
 int child_status;
 int child_pid;
    child_pid = wait(&child_status);
    if (WIFEXITED(child_status)) {
	SSH_DLOG(3, ("main_sigchld_handler: child %d exited with %d\n",
			child_pid, WEXITSTATUS(child_status)));
    } else {
	SSH_DLOG(2, ("main_sigchld_handler: child %d  status %d\n",
			child_pid, child_status));
    }
    return;
}

/*
 * dolisten: Wait for a client to connect.
 */
int dolisten() {
  int num;
  int s;
  int cpid;
  struct timeval tv;
  fd_set readfds; 
  struct itimerval itval;
  struct hostent *hp;
  char *hostname, hostnamebuf[200];
  int slen;

    while (!exitflag) {
	FD_ZERO(&readfds);
	FD_SET(g_context.listen_socket, &readfds);
	num = select(g_context.listen_socket + 1, &readfds, NULL, NULL, NULL);
	if (num < 0) {
	    /* non-fatal. */
	    if (errno == EINTR) {
		continue;
	    }
	    /* fatal. */
	    SSH_ERROR("select error: %s\n", strerror(errno));
	    return(1);
	}
	if (FD_ISSET(g_context.listen_socket, &readfds)) {
	    slen = sizeof(struct sockaddr_in);
	    s = accept(g_context.listen_socket,
				(struct sockaddr *)&g_context.saddr, &slen);
	    if (s < 0) {
		SSH_ERROR("Unable to accept connection: %s\n", strerror(errno));
		continue;
	    }
	    SSH_DLOG(1, ("Connection from IP address %s\n",
					inet_ntoa(g_context.saddr.sin_addr)));
	    if(++numconns >= SSHD_REGENCONNS) {
		SSH_DLOG(1, ("Regenerating serverkey (%d connections)\n",
			     numconns));
		regen_serverkey();
		numconns = 0;
	    }
	    if ((cpid = fork()) == 0) {
		/* Child. */

		/* Don't detach if someone's watching. */
		/* Side effect: killing the original server will */
		/*		kill all it's children too. */
		if (!nodaemon)
		    daemon(0, 0);

		if (loginit(NULL, 1) != 0) {
		    warnx("Unable to initialize logging.\n");
		    return(1);
		}

		if (is_debug_level(6))
		    sleep(10);

		SSH_DLOG(3, ("Connection from IP address %s\n",
					inet_ntoa(g_context.saddr.sin_addr)));

		close(g_context.listen_socket);
		g_context.listen_socket = -1;

		/* Disable server-key regen alarm */
		signal(SIGALRM, SIG_IGN);
		timerclear(&itval.it_interval);
		timerclear(&itval.it_value);
		setitimer(ITIMER_REAL, &itval, NULL);

		gettimeofday(&tv, NULL);
		ssh_sys_randadd();
		ssh_sys_randadd();

		signal(SIGHUP, (void *)cleanup);

		/* Next ~40 lines from src/libexec/rlogind/rlogind.c:196 */
		/* NetBSD: rlogind.c,v 1.18 1998/08/29 17:31:56 tsarna Exp */
		/* XXX it would be nice if we could do this resolv in */
		/* XXX the background. */
#if 0 /* XXX WTF? */		
		saddr.sin_port = ntohs((in_port_t)saddr.sin_port);
#endif
		hp = gethostbyaddr((char *)&g_context.saddr.sin_addr,
				sizeof(struct in_addr), 
				g_context.saddr.sin_family);
		if (hp) {
		    hostname = hp->h_name;
		    strncpy(hostnamebuf, hp->h_name, sizeof(hostnamebuf) - 1);
		    hostnamebuf[sizeof(hostnamebuf) - 1] = 0;
		    hp = gethostbyname(hostnamebuf);
		    endhostent();
		    if (hp == NULL) {
			SSH_DLOG(2, ("Couldn't look up address for %s\n",
				    hostnamebuf));
			hostname = inet_ntoa(g_context.saddr.sin_addr);
		    } else for (; ; hp->h_addr_list++) {
			if (hp->h_addr_list[0] == NULL) {
			    SSH_DLOG(2, 
				("Host addr %s not listed for host %s\n",
				    inet_ntoa(g_context.saddr.sin_addr),
				    hp->h_name));
			    hostname = inet_ntoa(g_context.saddr.sin_addr);
			    break;
			}
			if (!memcmp(hp->h_addr_list[0],
					(caddr_t)&g_context.saddr.sin_addr,
					sizeof(g_context.saddr.sin_addr))) {
			    hostname = hp->h_name;
			    break;
			}
		    }

		    hostname = strncpy(hostnamebuf, hostname,
				   sizeof(hostnamebuf) - 1);
		} else {
		    hostname = strncpy(hostnamebuf, 
				   inet_ntoa(g_context.saddr.sin_addr),
				   sizeof(hostnamebuf) - 1);
		}

		hostnamebuf[sizeof(hostnamebuf) - 1] = '\0';

		SSH_DLOG(1, ("Received connection from %s.\n", hostnamebuf));

		g_context.s = s;
		g_context.cipher->type = -1;
		packet_init(&g_context.out_pc);
		packet_init(&g_context.in_pc);
		g_context.compressing = 0;

		doServer(&g_context);		/* Talk to the client. */

		/* This shouldn't be reached. */
		SSH_ERROR("Error!  doServer returned!\n");
		exit(1);
	    } else if (cpid < 0) {
		SSH_ERROR("Unable to fork: %s\n", strerror(errno));
		sleep(5);		/* wait a while, try later. */
		break;
	    } else {
		/* XX: This isn't quite right with the daemon call above. */
		SSH_DLOG(2, ("child: %d\n", cpid));
		close(s);
		s = -1;
	    }
	}

    } /* while (!exitflag) */
    return(0);
}

void sched_sk_regen(int sig)  
{
        numconns = SSHD_REGENCONNS;             /* XXX shouldn't be static */
}

