/* $Id: ssh-keygen.c,v 1.15.2.1 2000/08/25 09:32:02 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <string.h>
#include <termios.h>
#include <sys/types.h>
#include <pwd.h>

#include "options.h"

#include "sshd.h"
#include "ssh_parse.h"
#include "ssh_rsakeys.h"
#include "ssh_sys.h"

int whichprime;
struct sshd_context g_context;

/*

Generate a key and write to a file:

	ssh-keygen [-b bits] [-N new-pass] [-C comment]
		generate new key.
	ssh-keygen -u -P pass
		remove passphrase
	ssh-keygen [-P pass] -N new-pass
		change passphrase
	ssh-keygen [-P pass] -C comment
		change comment

	common opts:	[-t pub|priv|both] [-F FRESSSH|F-Secure]
			[-f file[:publickeyfile]]
			[-t pub|priv|both ]
	[-p] [-c] are ignored.

 * Defaults to both private and public files.
 * Default filenames are $HOME/.ssh/identity.{priv,pub}
 * Default comment is `id -nu`@`hostname`
 * Default file format is FreSSH
 * Default bits is 1024.
 * Minimum bits this version will generate is 1024.
 * Automatically rewrites in the same format as read.
 * Can convert/write to different format with -F.  (Default to FreSSH)
 * Can regenerate a public key from a private. (-t pub)
 * Can write only a private key. (-t priv)
 * -F and -t are case insensitive and only check first two characters.

 * Examples:
 *	Generate a 2048 bit FreSSH format key with no passphrase:
 *		ssh-keygen -b 2048 -N ""
 *	Generate a F-Secure format key:
 *		ssh-keygen -F F-Secure
 *	  or	ssh-keygen -F F-
 *	  or	ssh-keygen -F fs
 *	Convert a key in file foo to FreSSH format:
 *		ssh-keygen -f foo -F fr
 *	Generate a FreSSH key, write the private key to foo and
 *	the public key to bar:
 *		ssh-keygen -f foo:bar
 *	Generate a FreSSH key in foo and foo.pub:
 *		ssh-keygen -f foo
 *	Change the passphrase from "aaa" to "bbb", set the comment
 *	to "joe@bob.com" and convert to a F-Secure format:
 *		ssh-keygen -P "aaa" -N "bbb" -C "joe@bob.com -F FS
 */

struct target_file {
    char *name;
    int len;
    char *buf;
};

struct ops {
#define KEYFILE_BOTH	0
#define KEYFILE_PRIV	1
#define KEYFILE_PUB	2
    int filetype;		/* Type of key file to write. */
    int format;			/* Format it was written as. */
    int new_format;		/* Format to write it in. */

    struct target_file fpriv;
    struct target_file fpub;

    int remove_pp;
    char *passphrase;
    char *new_pp;

    char *comment;
    char *new_comment;

    int newkey;
    int bits;
    ssh_RSA *key;
} ops;

void Usage(char *);
void parse_filename_opt(struct ops *, char *);
void keygen_cleanup(struct ops *);
void default_filename(struct target_file *, struct target_file *);
int ssh_generate_key(struct ops *, ssh_RSA **);
int ssh_getpassphrase(char **, int);
int ssh_setdefaultcomment(char **);
void genkey_callback();
int write_file(struct target_file *);
int read_file(struct target_file *);

int write_file(struct target_file *finfo) {
  int d;

    SSH_DLOG(4, ("writing keyfile: %s %d\n", finfo->name, finfo->len));
    if ((d = open(finfo->name, O_RDWR|O_CREAT|O_TRUNC, 0600)) < 0) {
	SSH_ERROR("Unable to open keyfile: %s\n", strerror(errno));
	return(-1);
    }
    if (write(d, finfo->buf, finfo->len) != finfo->len) {
	SSH_ERROR("Unable to write keyfile: %s\n", strerror(errno));
	close(d);
	return(-1);
    }
    close(d);
    return(0);
}

int read_file(struct target_file *finfo) {
  int d;
  struct stat statinfo;
    if ((d = open(finfo->name, O_RDWR, 0600)) < 0)
	return(-1);
    if (fstat(d, &statinfo) < 0) {
	close(d);
	return(-1);
    }
    if ((finfo->buf = malloc(statinfo.st_size)) == NULL) {
	close(d);
	return(-1);
    }
    if (((finfo->len = read(d, finfo->buf, statinfo.st_size)) < 0) ||
	(finfo->len < statinfo.st_size)) {
	free(finfo->buf);
	finfo->buf = NULL;
	close(d);
	return(-1);
    }
    
    close(d);
    return(0);
}

int main (int argc, char **argv) {
  char ch;
  int ret;

    ret = 0;

    loginit("ssh-keygen", 2);	/* Initialize logging, only to stderr. */
    debug_inc(0);
    debug_inc(0);
    debug_inc(0);
    debug_inc(0);
    debug_inc(0);

    /* Clear everything. */
    memset(&ops, 0, sizeof(struct ops));

    while((ch = getopt(argc, argv, "t:F:pcuf:b:P:N:C:")) > 0) {
	switch (ch) {
	  case 't':
		/* Type for keyfile: public or private. */
		if (strlen(optarg) < 2)
		    Usage("Unknown keyfile type");
		optarg[0] = toupper(optarg[0]);
		optarg[1] = toupper(optarg[1]);
		if (strncmp(optarg, "PR", 2) == 0)
		    ops.filetype = KEYFILE_PRIV;
		else if (strncmp(optarg, "PU", 2) == 0)
		    ops.filetype = KEYFILE_PUB;
		else if (strncmp(optarg, "BO", 2) == 0)
		    ops.filetype = KEYFILE_BOTH;	/* a.k.a. 0 */
		else
		    Usage("Unknown keyfile type");
		break;
	  case 'F':
		/* Format of keyfile: FreSSH or F-Secure. */
		if (strlen(optarg) < 2)
		    Usage("Unknown keyfile format");
		optarg[0] = toupper(optarg[0]);
		optarg[1] = toupper(optarg[1]);
		if (strncmp(optarg, "FR", 2) == 0)
		    ops.new_format = FRESSH_PRIV_KEYFILE;
		else if (strncmp(optarg, "F-", 2) == 0 ||
			strncmp(optarg, "FS", 2) == 0)
		    ops.new_format = FSECURE_PRIV_KEYFILE;
		else
		    Usage("Unknown keyfile format");
		break;
	  case 'u':
		ops.remove_pp = 1;
		break;
	  case 'c':
	  case 'p': break;
	  case 'f':
		parse_filename_opt(&ops, optarg);
		break;
	  case 'b':
		ops.bits = atoi(optarg);
		break;
	  case 'P':
		ops.passphrase = optarg;
		break;
	  case 'N':
		ops.new_pp = optarg;
		break;
	  case 'C':
		ops.new_comment = optarg;
		break;
	  default:
		Usage("Unknown option");
		/* NOTREACHED */
	}
    }

    if (ops.remove_pp && ops.new_pp)
	Usage("New passphrase/remove passphrase options are incompatible.");

    /* Build the public keyfile name from the private. */
    if (!ops.fpub.name && ops.filetype == KEYFILE_BOTH) {
	if (ops.fpriv.name) {
	  int temp;
	    temp = strlen(ops.fpriv.name) + 5;
	    ops.fpub.name = malloc(temp);
	    snprintf(ops.fpub.name, temp, "%s.pub",
			ops.fpriv.name);
	}
    }
    /* Fill in defaults for unset names. */
    default_filename(&ops.fpriv, &ops.fpub);

    SSH_DLOG(4, ("Loading:%s\n", ops.fpriv.name));
    if (read_file(&ops.fpriv) < 0) { 
	SSH_DLOG(4, ("read_file(priv) failed: %s\n", strerror(errno)));
	SSH_DLOG(4, ("setting newkey flag.\n"));
	ops.newkey = 1;
    }
    if (ops.fpub.buf)
	free(ops.fpub.buf);	/* Don't need the public key */
    ops.fpub.buf = NULL;

    if (ops.filetype == KEYFILE_PUB && ops.newkey)
	Usage("Can't generate public key without a private key.");
    if (!ops.newkey && ops.bits)
	Usage("Can't generate new key: file exists.");
    if (ops.newkey && ops.passphrase)
	Usage("Use -N to set the passphrase for a new key.");

    if (ops.newkey) {
	SSH_DLOG(4, ("Generating new key\n"));
	if (ssh_generate_key(&ops, &(ops.key)) != 0) {
	    SSH_ERROR("Unable to generate key.\n");
	    ret = 1; goto out;
	}
	if (!ops.new_pp)
	    while (ssh_getpassphrase(&(ops.new_pp), 1) != 0) /* nothing */;
	if (!ops.new_comment)
	    ssh_setdefaultcomment(&(ops.new_comment));
	SSH_DLOG(4, ("Comment set to: %s\n", ops.new_comment));
	ops.format = FRESSH_PRIV_KEYFILE;
    } else {
	/* Try decoding w/o a passphrase first. */
	if (decode_keyfile(ops.fpriv.buf, ops.fpriv.len,
			NULL, 0, &ops.key, &ops.comment, &ops.format) != 0) {
	    if (!ops.passphrase) {
		while (ssh_getpassphrase(&(ops.passphrase), 0) != 0)
		    /* nothing */;
	    }

	    /* Now try decoding with a passphrase. */
	    if (decode_keyfile(ops.fpriv.buf, ops.fpriv.len,
			ops.passphrase, strlen(ops.passphrase),
			&ops.key, &ops.comment, &ops.format) != 0) {
		SSH_ERROR("Unable to decode key file: Bad passphrase?\n");
		ret = 1; goto out;
	    }
	}
	free(ops.fpriv.buf);
	ops.fpriv.buf = NULL;
    }

    /* We now are guaranteed a key in memory. */

    /* Set the comment to write to the file. */
    if (!ops.new_comment)
	ops.new_comment = ops.comment;

    if (is_debug_level(4)) {
	if (!ops.comment || (strcmp(ops.new_comment, ops.comment) != 0))
	    SSH_DLOG(4, ("new comment: %s\n", ops.new_comment));
    }

    /* Set the format to write with. */
    if (!ops.new_format)
	ops.new_format = ops.format;

    if (ops.remove_pp)
	ops.new_pp = NULL;
    else {
	if (!ops.new_pp)
	    ops.new_pp = ops.passphrase;
    }


    if (ops.filetype == KEYFILE_PRIV || ops.filetype == KEYFILE_BOTH) {
	ops.fpriv.len = encode_keyfile(ops.new_format,
			&ops.fpriv.buf, ops.new_pp, ops.new_pp ?
			strlen(ops.new_pp) : 0, ops.key, ops.new_comment);
    }
    if (ops.filetype == KEYFILE_PUB || ops.filetype == KEYFILE_BOTH) {
	ops.fpub.len = encode_keyfile(FRESSH_PUB_KEYFILE,
			&ops.fpub.buf, ops.new_pp, ops.new_pp ?
			strlen(ops.new_pp) : 0, ops.key, ops.new_comment);
    }
	

    free(ops.comment);
    ssh_rsa_free(ops.key);
    ops.comment = NULL;
    ops.key = NULL;

    if (ops.fpriv.len < 0 || ops.fpub.len < 0) {
	SSH_ERROR("Unable to encode keyfile.\n");
	ret = 1; goto out;
    }

    if (ops.filetype == KEYFILE_PRIV || ops.filetype == KEYFILE_BOTH)
	ret = write_file(&ops.fpriv);

    if (ops.filetype == KEYFILE_PUB || ops.filetype == KEYFILE_BOTH)
	ret = write_file(&ops.fpub);

out:
    keygen_cleanup(&ops);
    exit(ret);
}

void Usage(char *err) {
    if (err)
	SSH_ERROR(err);

    fprintf(stderr, "%s\n	%s\n",
	"ssh-keygen [-b bits] [-N new-pass] [-C comment]",
	"generate new key.");
    fprintf(stderr, "%s\n	%s\n",
	"ssh-keygen -u -P pass",
	"remove passphrase");
    fprintf(stderr, "%s\n	%s\n",
	"ssh-keygen [-P pass] -N new-pass",
	"change passphrase");
    fprintf(stderr, "%s\n	%s\n",
	"ssh-keygen [-P pass] -C comment",
	"change comment");
    fprintf(stderr, "	%s\n", "[-p] [-c] are ignored.");
    fprintf(stderr, "common options:\n");
    fprintf(stderr, "	%s\n", "[-t pub|priv|both]");
    fprintf(stderr, "	%s\n", "[-F FreSSH|FSecure|F-Secure]");
    fprintf(stderr, "	%s\n", "[-f privatefile[:publicfile]]");
    fprintf(stderr, "	%s\n", "[-f [privatefile]:publicfile]");

    exit(1);
}

void parse_filename_opt(struct ops *ops, char *optarg) {
  char *colon_pos;

    ops->fpub.name = NULL;

    colon_pos = strchr(optarg, ':');
    if (colon_pos == optarg)
	ops->fpriv.name = NULL;
    else
	ops->fpriv.name = optarg;
    if (colon_pos != NULL) {
	*colon_pos = '\0';
	optarg = colon_pos + 1;
	if (strlen(optarg) > 0) {
	    ops->fpub.name = malloc(strlen(optarg) + 1);
	    strncpy(ops->fpub.name, optarg, strlen(optarg) + 1);
	}
    }
}

void keygen_cleanup(struct ops *ops) {
    if (ops->fpriv.buf)
	free(ops->fpriv.buf);
    if (ops->fpub.buf)
	free(ops->fpub.buf);

    if (ops->key)
	ssh_rsa_free(ops->key);
    ops->key = NULL;
    if (ops->comment)
	free(ops->comment);
    ops->comment = NULL;
}

/*
 * Generate a ssh RSA key with progress display.
 * Returns 0 on success.
 */
int ssh_generate_key(struct ops *ops, ssh_RSA **key) {
  int rbits;
    if (!key)
	return(1);
    if (ops->bits < 128)
	rbits = 1024;
    else
	rbits = ops->bits;
   
    whichprime = 0;

    *key = ssh_rsa_generate_key(rbits, 3, genkey_callback, NULL);

    return(0);
}

void genkey_callback(int type, int num, void *arg) {
    if (whichprime == 0) {
	fprintf(stderr, "Generating p:");
	whichprime = 1;
    }
    if (whichprime == 2) {
	fprintf(stderr, "Generating q:");
	whichprime = 3;
    }
    switch (type) {
      case 0:
	fprintf(stderr, ">"); break;
      case 1:
	fprintf(stderr, "*"); break;
      case 2:
	fprintf(stderr, "Failed, regenerating.\n"); break;
      case 3:
	fprintf(stderr, ":done\n");
	whichprime++;
	break;
      default:
	printf("Xunknown %d(%d)X", type, num);
    }
    return;
}

/*
 * Ask for the passphrase with optional verification.
 * 	On successful return the location pointed to by pp
 *	is set to the _internal buffer_ used to read the passphrase.
 *
 *	If this function is to be called more than once the
 *	passphrase should be copied elsewhere.
 */
int ssh_getpassphrase(char **pp, int verify) {
  char buf[40];
  char buf2[40];
  int len;
  struct termios term, oterm;

    printf("Enter passphrase:");
    fflush(stdout);

    tcgetattr(fileno(stdin), &term);
    oterm = term;
    term.c_lflag &= ~(ECHO|ECHOE|ECHOK|ECHONL);
    tcsetattr(fileno(stdin), TCSANOW, &term);
    len = read(fileno(stdin), buf, sizeof(buf) - 1);
    tcsetattr(fileno(stdin), TCSANOW, &oterm);
    printf("\n");

    /* NL or interrupt. */
    if (len <= 1)
	buf[0] = '\0';
    else
	buf[len - 1] = '\0';

    if (verify) {
	printf("Enter the same passphrase again:");
	fflush(stdout);

	tcgetattr(fileno(stdin), &term);
	oterm = term;
	term.c_lflag &= ~(ECHO|ECHOE|ECHOK|ECHONL);
	tcsetattr(fileno(stdin), TCSANOW, &term);
	len = read(fileno(stdin), buf2, sizeof(buf2) - 1);
	tcsetattr(fileno(stdin), TCSANOW, &oterm);
	printf("\n");

	/* NL or interrupt. */
	if (len <= 1)
	    buf2[0] = '\0';
	else
	    buf2[len - 1] = '\0';

	if (strcmp(buf, buf2) != 0) {
	    printf("Passphrases do not match.  Try again.\n");
	    return(1);
	}
    }
    *pp = buf;
    return(0);
}

int ssh_setdefaultcomment(char **new_comment) {
  char hostname[100];
  struct passwd *pwp;
  int len;
    if (gethostname(hostname, 100) < 0)
	return(1);
    SSH_DLOG(4, ("hostname: %s\n", hostname));
    
    pwp = getpwuid(getuid());
    if (pwp == NULL)
	return(1);
    SSH_DLOG(4, ("username: %s\n", pwp->pw_name));

    len = strlen(hostname) + strlen(pwp->pw_name) + 2;
    if ((*new_comment = malloc(len)) == NULL) {
	endpwent();
	return(1);
    }
    snprintf(*new_comment, len, "%s@%s", pwp->pw_name, hostname);
    endpwent();
    return(0);
}

void default_filename(struct target_file *fpriv, struct target_file *fpub) {
  char *homedir;
    homedir = getenv("HOME");
    if (homedir == NULL)
	return;
    if (!fpriv->name) {
	fpriv->name = malloc(200);
	snprintf(fpriv->name, 200, "%s/.ssh/identity.priv", homedir);
    }
    if (!fpub->name) {
	fpub->name = malloc(200);
	snprintf(fpub->name, 200, "%s/.ssh/identity.pub", homedir);
    }
    return;
}
