/* $Id: ssh_sys_bsd44+.c,v 1.34.2.6 2000/10/17 02:01:58 erh Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */


#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#include <sys/mman.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifdef __FreeBSD__
#include <libutil.h>
#else
#include <util.h>
#endif

#include "options.h"

#include "sshd.h"
#include "ssh_buffer.h"
#include "ssh_crypto.h"
#include "ssh_global.h"
#include "ssh_paths.h"
#include "ssh_sys.h"
#include "ssh_util.h"
#include "ssh_parse.h"
#include "ssh_keyfile.h"

#ifndef BSD
#error This file only works on BSD systems!
#endif

#if (BSD < 199506)
#error This file only works on 4.4BSD-Lite2 or newer systems!
#endif

static char *childenv[16];		/* XXX */
static char shell[MAXINTERP + 1];

/*
 * ssh_sys_allocpty: doesn't actually allocate the pty on 4.4BSD!
 *
 *                Instead, it sets everything up, and ssh_sys_execcmd uses a
 *                function from libutil which includes the pty allocation.
 *                This means the other end may think we successfully
 *                allocated a pty, but then failed to execute the requested
 *                command, if, for example, we're out of ptys.  NBD.
 */
int ssh_sys_allocpty(struct sshd_context *context, char *term, int y_row, int x_col,
			int x_pix, int y_pix, u_int8_t *modes, size_t msize) {

    static char envterm[255 + 1];
    char **newenv;

    if ((context->term = malloc(strlen(term) + 1)) == NULL) {
	SSH_ERROR("ssh_sys_allocpty: unable to alloc mem for term: %s\n",
							strerror(errno));
	return(1);
    }
    strncpy(context->term, term, strlen(term));

    snprintf(envterm, sizeof(envterm), "TERM=%s", term);

    for(newenv = childenv; *newenv != NULL ; newenv++);

    *newenv = envterm;
    newenv++;
    *newenv = NULL;

    context->modes = malloc(msize);
    if (context->modes == NULL) {
	SSH_ERROR("ssh_sys_allocpty: unable to alloc mem for tty modes: %s\n",
		strerror(errno));
	free(context->term);
	context->term = NULL;
	return(1);
    }
    memcpy(context->modes, modes, msize);
    context->msize = msize;

    /* Set window size. */
    context->win.ws_row = y_row;
    context->win.ws_col = x_col;
    context->win.ws_xpixel = x_pix;
    context->win.ws_ypixel = y_pix;
	
    /* Set flag: ssh_sys_execcmd will check this and allocate a pty then. */
    context->usepty = 1;

    return(0);
}

/*
 * ssh_sys_setbufsize: Set socket buffer size to increase throughput.
 */
void ssh_sys_setbufsize(struct sshd_context *context, int fd0, int fd1) {
  int size;
  int a, b, c, d;
    size = context->max_sock_bufsize;
    while (1) {
	size -= 2048;
        if (size <= 0) {
	    break;
	}
	a = setsockopt(fd0, SOL_SOCKET, SO_SNDBUF, &size, sizeof(int));
	b = setsockopt(fd0, SOL_SOCKET, SO_RCVBUF, &size, sizeof(int));
	c = setsockopt(fd1, SOL_SOCKET, SO_SNDBUF, &size, sizeof(int));
	d = setsockopt(fd1, SOL_SOCKET, SO_RCVBUF, &size, sizeof(int));
	if (a == 0 && b == 0 && c == 0 && d == 0)
	    break;
    }
    if (size < SSH_MIN_READSIZE)
	context->sock_bufsize = SSH_MIN_READSIZE;
    else
	context->sock_bufsize = size;
}

/*
 * ssh_sys_execcmd: execute a command.
 */
int ssh_sys_execcmd(struct sshd_context *context, char *cmd) {
  int stdfd[2];
  int errfd[2];
  int mfd;
  char slave_name[MAXPATHLEN + 1];
  struct termios term;
  char *shell_name;
  

    SSH_DLOG(3, ("ssh_sys_execcmd: CMD=%s\n", cmd));
    if (context->usepty) {
	if (forkpty(&mfd, slave_name, NULL, &context->win) == 0) {
	/* child: */
	    if (tcgetattr(0, &term) < 0) {
		SSH_ERROR("ssh_sys_execcmd: couldn't get modes for pty.\n");
		exit(1);
	    }
	    if (ssh_sys_set_tty_modes(&term, context->modes,
				  context->msize) != 0) {
		SSH_ERROR("ssh_sys_execcmd: set_tty_modes failed: %s\n",
							strerror(errno));
		exit(1);
	    }
	    if (tcsetattr(0, TCSANOW, &term) < 0) {
		SSH_ERROR("ssh_sys_execcmd: couldn't set modes for pty.\n");
		exit(1);
	    }

	    if (close_files_for_exec(context) < 0) {
		SSH_ERROR("ssh_sys_execcmd: unable to close files.\n");
		exit(1);
	    }

#if 0
	    clear_sensitive_data();
#endif
	    reset_signals();

		/* XXX set environment: LOGNAME, TZ, SSH_CLIENT,
					SSH_TTY */
	    shell_name = strrchr(shell, '/');
	    if (cmd != NULL && strlen(cmd) != 0) {
		execle(shell, (shell_name++ == NULL) ? shell : shell_name, "-c", cmd, NULL, childenv);
	    } else {
		/* exec login. */
		execle(PATH_LOGIN, "login", "-f", "-p", /*"-h", 
				 inet_ntoa(context->saddr.sin_addr),*/ "--",
				 context->username, NULL, childenv);
	    }
	    SSH_DLOG(1, ("exec failed: %s\n", strerror(errno)));
	    exit(1);
	} else {
	  int size, len;
	  int ret;
	    /* parent: */
	    context->child_stdin = mfd;
	    context->child_stdout = mfd;
	    context->child_stderr = mfd;
	    len = sizeof(int);
	    ret = getsockopt(mfd, SOL_SOCKET, SO_RCVBUF, &size, &len);
	    if (ret < 0)
		context->sock_bufsize = SSH_MIN_READSIZE;
	    else
		context->sock_bufsize = size;
	}
    } else {
	/* create socket pairs. */
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, stdfd) < 0 ||
	    socketpair(AF_UNIX, SOCK_STREAM, 0, errfd) < 0) {
	    SSH_DLOG(1, ("doExecCmd: socketpair failed: %s\n", 
						strerror(errno)));
	    return(1);
	}

	ssh_sys_setbufsize(context, stdfd[0], stdfd[1]);

	if (fork() == 0) {
	    /* child: */
	    {
	      int i;
        	for(i = 0; i <= NSIG; i++)
		  signal(i, SIG_DFL);
	    }
	    if (dup2(stdfd[0], fileno(stdin)) < 0) {
		SSH_ERROR("doExecCmd: dup2 stdfd -> stdin failed: %s\n",
							strerror(errno));
		exit(1);
	    }
	    if (dup2(stdfd[0], fileno(stdout)) < 0) {
		SSH_ERROR("doExecCmd: dup2 stdfd -> stdout failed: %s\n",
							strerror(errno));
		exit(1);
	    }

	    close(stdfd[0]);
	    close(stdfd[1]);
	    if (dup2(errfd[0], fileno(stderr)) < 0) {
		SSH_ERROR("doExecCmd: dup2 errfd -> stderr failed: %s\n",
							strerror(errno));
		exit(1);
	    }
	    close(errfd[0]);
	    close(errfd[1]);

	    if (close_files_for_exec(context) < 0) {
		SSH_ERROR("ssh_sys_execcmd: unable to close files.\n");
		exit(1);
	    }

#if 0
	    clear_sensitive_data();
#endif
	    reset_signals();

	    shell_name = strrchr(shell, '/');
	    if (cmd != NULL && strlen(cmd) != 0) {
		execle(shell, (shell_name++ == NULL) ? shell : shell_name, "-c", cmd, NULL, childenv);
	    } else {
		execle(PATH_LOGIN, "login", "-f", "-p", /*"-h", 
			inet_ntoa(context->saddr.sin_addr),*/ "--",
			context->username, NULL, childenv);
	    }
	    SSH_DLOG(1, ("exec failed: %s\n", strerror(errno)));
	    exit(1);
	} else {
	   /* parent: */

	    /* Close child's side descriptors. */
	    close(errfd[0]);
	    close(stdfd[0]);

	    /* Save the descriptors. */
	    context->child_stdin = stdfd[1];
	    context->child_stdout = stdfd[1];
	    context->child_stderr = errfd[1];
	}
    }
    return(0);
}

/*
 * PRNG wrapper functions.  Replace with calls to your favorite PRNG if
 * desired.
 *
 * A note on randinit/randclean:  If you have hardware randomness, you
 * may want to modify randinit to use it.  On NetBSD, we use the system
 * RNG/PRNG, specifying the device node which generates to a PRNG if no
 * "hard" random data is available so that we don't block at startup.  We
 * do mix in some other gunk just in case, but you *really* want the kernel
 * random source turned on -- no great effort is made elsewise.
 *
 * In such an implementation, randclean just zeroes out the internal PRNG
 * state for neatness/safety.  But an alternate implementation might try to
 * preserve PRNG state across runs; then randclean would save the state, and
 * randinit would load it, hopefully mixing in some new seed randomness.
 *
 */

/*
 * ssh_sys_randinit:	initialize random number generator.
 */
void ssh_sys_randinit()
{

  struct timeval tv;
  int urandom;
  u_int8_t ubuf[128];


    gettimeofday(&tv, (struct timezone *)NULL);

    
    ssh_rand_feed((void *)&(tv.tv_sec), sizeof(tv.tv_sec));
    ssh_rand_feed((void *)&(tv.tv_usec), sizeof(tv.tv_usec));

    memset(&tv, 0, sizeof(tv));

    system("/usr/bin/wc -c /usr/share/dict/words > /dev/null 2>&1");
    system("/bin/ps auxl > /dev/null 2>&1");
    system("/bin/ls -lR /home > /dev/null 2>&1");


    urandom = open("/dev/urandom", O_RDONLY);

    if (urandom >= 0) {
        if (read(urandom, ubuf, sizeof(ubuf)) != sizeof(ubuf)) {
	    close(urandom);
	    goto wingit;
    	} else {
	    close(urandom);
	    ssh_rand_feed(ubuf, sizeof(ubuf));
	}
    } else /* System PRNG unavailable -- seed with some more data and pray */
    {
	wingit:
	{
	  int numfs, whichfs = 0;
	  struct statfs *mntbuf;
	    numfs = getmntinfo(&mntbuf, MNT_NOWAIT);
	    while(whichfs < numfs) {
		ssh_rand_feed((void *)mntbuf, sizeof(struct statfs));
		memset(mntbuf, 0, sizeof(struct statfs));
		mntbuf++;
		whichfs++;
	    }
	}
    }

    gettimeofday(&tv, NULL);

    ssh_rand_feed((void *)&(tv.tv_usec), sizeof(tv.tv_usec));

    memset(ubuf, 0, sizeof(ubuf));
    memset(&tv, 0, sizeof(tv));

}

/*
 * ssh_sys_randclean:       clean up (possibly save) PRNG state.
 */
void ssh_sys_randclean()
{
    ssh_rand_clean();
}

/*
 * ssh_sys_randadd:	Add some external randomness to the mix.  Must be
 *			fast, called during normal operation.
 */
void ssh_sys_randadd() {
    
  struct timeval tv;
  int urandom;
  u_int8_t ubuf[128];

    urandom = open("/dev/urandom", O_RDONLY);
    
    if (urandom < 0) {
	gettimeofday(&tv, NULL);                
	ssh_rand_feed((void *)&(tv.tv_usec), sizeof(tv.tv_usec));
    } else {
	if (read(urandom, ubuf, sizeof(ubuf)) != sizeof(ubuf)) {
	    close(urandom);
	    gettimeofday(&tv, NULL);                     
    	    ssh_rand_feed((void *)&(tv.tv_usec), sizeof(tv.tv_usec));
        } else {
            close(urandom);   
            ssh_rand_feed(ubuf, sizeof(ubuf));   
        }
    }
    memset(&tv, 0, sizeof(struct timeval));
    memset(ubuf, 0, sizeof(ubuf));
}
    
void ssh_sys_exit(int retval) {
    /* cleanup whatever. */
    SSH_DLOG(5, ("ssh_sys_exit: exiting with %d\n", retval));
    exit(retval);
}

/* XXX option decoding and usage probably doesn't need to be in this file. */
void ssh_sys_usage() {
    fprintf(stderr, "Usage: sshd %s %s %s %s %s\n", "[-Ddk]",
			"[-a address]", "[-h host key]", "[-p port]",
			"[-P pidfile]");
}

int ssh_sys_configuration(struct sshd_context *context) {
  int ch, d;
  struct servent *srv;
  off_t hksiz;
  void *keydata;

    while ((ch = getopt(g_argc, g_argv, "Ddka:h:p:P:")) != -1) {
        switch(ch) {
	  case 'D':
		nodaemon = 1;
		break;
          case 'd':
                debug_inc(0);
                break;
	  case 'k':
		context->opts.keepalive = 1;
		break;
          case 'h':
                /* Set the host key file */
                context->opts.hostkey = optarg;
                break;
	  case 'a':
		/* IP address to listen on */
		context->opts.address = optarg;
		break;
	  case 'p':
		/* Port to listen on */
		context->opts.port = optarg;
		break;
	  case 'P':
		context->opts.pidfile = optarg;
		break;
	  case '?':
	  default:
		ssh_sys_usage();
		return(1);
        }
    }

    if (!context->max_sock_bufsize)
	context->max_sock_bufsize = SSH_MAX_READSIZE;

    if(!context->opts.hostkey)
	context->opts.hostkey = SSHD_HOSTKEY;

    if(!context->opts.pidfile)
	context->opts.pidfile = SSHD_PIDFILE;

    if(!context->opts.address)
	context->opts.address = "0.0.0.0";

    if (!context->max_packet_size)
	context->max_packet_size = SSH_MAX_PACKETSIZE;

    /* One might call this a foolish consistency.  We keep 'port', like
       'address', as a string.  But getservbyname gives us a ready-to-use
       short in network byte order.  So we convert it back.  Oh well. */

    if(!context->opts.port) {
	srv = getservbyname("ssh", "tcp");
	if(srv != NULL) {
	    context->opts.port = malloc(6 * sizeof(char));
	    snprintf(context->opts.port, 6, "%d", ntohs(srv->s_port));
	} else {
	    context->opts.port = SSHD_PORT;
	}
    }

    if ((d = open(context->opts.hostkey, O_RDONLY, 0600)) < 0) {
	SSH_ERROR("couldn't open host key file %s: %s\n", 
	    context->opts.hostkey, strerror(errno));
	ssh_sys_usage();
	return(1);
    }

    hksiz = lseek(d, 0, SEEK_END);

    keydata = mmap(NULL, (size_t)hksiz, PROT_READ, MAP_PRIVATE, d, 
		  (off_t)0);
    close(d); 

    if (keydata == NULL) {
	SSH_ERROR("couldn't access host key %s: %s\n",
	    context->opts.hostkey, strerror(errno));
	ssh_sys_usage();
	return(1);
    }

    if (decode_keyfile(keydata, hksiz, NULL, 0, &context->hostkey,
        NULL, NULL) != 0) {
	SSH_ERROR("host key %s is corrupt (did not decode).\n");
	munmap(keydata, hksiz);
	ssh_sys_usage();
	return(1);
    }

    munmap(keydata, hksiz);
    return(0);
}

/*
 * ssh_sys_setuser:
 *	Set up the initial environment.
 *	Switch uid/gid.
 *	chdir to home directory.
 */

int ssh_sys_setuser(struct sshd_context *context, char *uname) {

  char **newenv = childenv;
  struct ssh_password *pwent;
  static char ename[5 + MAXLOGNAME + 1];
  static char ehome[5 + MAXPATHLEN + 1];
  static char eshell[6 + MAXINTERP + 1];
  static char epath[32];

    if (!context->pwent.pw_name)
	if (ssh_sys_lookupuser(context, uname) != 0)
	    return(-1);

    pwent = &context->pwent;

    (void)setgid(pwent->pw_gid);
    (void)initgroups(pwent->pw_name, pwent->pw_gid);
    (void)setuid(pwent->pw_uid);

    snprintf(ename, sizeof(ename), "USER=%s", pwent->pw_name);
    snprintf(ehome, sizeof(ehome), "HOME=%s", pwent->pw_dir);
    snprintf(eshell, sizeof(eshell), "SHELL=%s", pwent->pw_shell);
    snprintf(epath, sizeof(epath), "PATH=/bin:/usr/bin:/usr/pkg/bin");

    *newenv = ename;
    newenv++;
    *newenv = ehome;
    newenv++;
    *newenv = eshell;
    newenv++;
    *newenv = epath;
    newenv++;
    *newenv = NULL;

    chdir(pwent->pw_dir);

    snprintf(shell, sizeof(shell), "%s", pwent->pw_shell);

    return 0;
}

int ssh_sys_daemonize() {

    return daemon(0, 0);

}

