/* $Id: sshd_iactive.c,v 1.30.2.1 2000/08/25 09:32:21 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */


#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>

#include "options.h"

#include "sshd.h"
#include "ssh_util.h"
#include "ssh_messages.h"

static int child_exited;
static int child_exitstat;
static int child_exitpid;

/*
 * doInteractive: Interactive session state.
 *
 *	Args: s		socket to talk on.
 *
 *	Returns: 0	usually.
 *		 1	an error occurred.
 *			(this does not include a client disconnecting.)
 *
 * Note: when this function returns the connection
 *	will be completed and the socket closed.
 */
int doInteractive(sshd_context_t *context) {
  int error_flag;
  char *error_why;
  int client_gone;
  fd_set readfds;
  int maxfd, numfd;
  char *readbuf;
  int readsize;
  int num;
  struct ssh_packet *pc;

    pc = &(context->in_pc);
    client_gone = 0;
    error_flag = 0;
    error_why = "No error.";

    signal(SIGCHLD, sigchld_handler);

    /* Allocate a buffer to read into. */
    if ((readbuf = malloc(context->sock_bufsize)) == NULL) {
	context->sock_bufsize = SSH_MIN_READSIZE;
	readbuf = malloc(SSH_MIN_READSIZE);
    }
    if (readbuf == NULL)
	return(1);
    readsize = context->sock_bufsize;

    /* Adjust max size if compressing. (zlib isn't perfect) */
    if (context->compressing)
	readsize = (readsize - 12) / 1.001;

    /* Set the type of service. */
    set_tos(context->s, context->usepty);

    while(1) {
	FD_ZERO(&readfds);
	FD_SET(context->s, &readfds);
	FD_SET(context->child_stdout, &readfds);
	if (context->child_stdout != context->child_stderr)
	    FD_SET(context->child_stderr, &readfds);
	maxfd = context->s|context->child_stdout|context->child_stderr;
	numfd = select(maxfd + 1, &readfds, NULL, NULL, NULL);
	if (numfd < 0) {
	    if (errno == EINTR)
		continue;
	    SSH_ERROR("doInteractive: select failed: %s\n", strerror(errno));
	    error_why = "Select failed";
	    error_flag = 1;
	}

	/* If the sending packet hasn't finished yet, continue it. */
	if (!packet_done(&(context->out_pc))) {
	    if (xmit_packet(context, 0, NULL, 0) < 0) {
		if (errno != EPIPE) {
		    error_why = "doInteractive: xmit_data(continue) failed";
		    error_flag = 1;
		    client_gone = 1;
		} else {
		    SSH_DLOG(3, ("doInteractive: client dropped connection\n"));
		    client_gone = 1;
		}
		break;
	    }
	}

	/* If the previous packet is done, start the next one. */
	if (packet_done(&(context->out_pc))) {
	    if (FD_ISSET(context->child_stdout, &readfds)) {
		if ((num = read(context->child_stdout, readbuf, readsize)) < 0)
		    if (errno != EINTR) {
			SSH_ERROR("doInteractive: stdout read failed: %s\n",
							strerror(errno));
			error_why = "stdout read error.";
			error_flag = 1;
			break;
		    }
		if (num == 0) {
		    SSH_DLOG(4, ("EOF on stdout.\n"));
		    break;
		}
		if (xmit_data(context, SSH_SMSG_STDOUT_DATA,
							readbuf, num, 0) < 0) {
		    if (errno != EPIPE) {
			error_why = "doInteractive: xmit_data(stdout) failed";
			error_flag = 1;
			client_gone = 1;	/* Maybe not, but we can't */
						/* talk to him anyway.     */
		    } else {
			SSH_DLOG(3, 
			    ("doInteractive: client dropped connection\n"));
			client_gone = 1;
		    }
		    break;
		}
	    } else if (context->child_stdout != context->child_stderr) {
		if (FD_ISSET(context->child_stderr, &readfds)) {
		    if ((num =
			   read(context->child_stderr, readbuf, readsize)) < 0)
			if (errno != EINTR) {
			    SSH_ERROR("doInteractive: stderr read failed: %s\n",
							strerror(errno));
			    error_why = "stderr read error.";
			    error_flag = 1;
			    break;
			}
		    if (num == 0) {
			SSH_DLOG(4, ("EOF on stderr\n"));
			break;
		    }
		    if (xmit_data(context, SSH_SMSG_STDERR_DATA,
							readbuf, num, 0) < 0) {
			if (errno != EPIPE) {
			    error_why =
				"doInteractive: xmit_data(stderr) failed";
			    error_flag = 1;
			    client_gone = 1;
			} else {
			    SSH_DLOG(5,
				("doInteractive: client dropped connection\n"));
			    client_gone = 1;
			}
			break;
		    }
		}
	    }
	}

	if (!FD_ISSET(context->s, &readfds))
	    continue;

	if ((num = read_packet(context, pc, 0)) < 0) {
	    if (errno == EPIPE) {
		SSH_DLOG(5, ("doInteractive: client dropped connection.\n"));
		client_gone = 1;
	    } else {
		SSH_DLOG(1, ("doInteractive: read_packet failed: %s\n",
							strerror(errno)));
		error_why = "Unable to read packet";
		error_flag = 1;
	    }
	    break;		/* error. */
	}
	if (num == 0)
	    continue;		/* not enought data yet. */

	while ((num = process_packet(context, pc)) > 0) {
	    switch(packet_type(pc)) {
	    case SSH_MSG_NONE:
		  break;
	    case SSH_MSG_PORT_OPEN:
	    case SSH_MSG_CHANNEL_OPEN_CONFIRMATION:
	    case SSH_MSG_CHANNEL_OPEN_FAILURE:
	    case SSH_MSG_CHANNEL_DATA:
	    case SSH_MSG_CHANNEL_CLOSE:
	    case SSH_MSG_CHANNEL_CLOSE_CONFIRMATION:
		/* unimplemented. */
		SEND_DEBUG(context, "Unimplemented.");
		SEND_FAILURE(context);
		break;
	    case SSH_CMSG_STDIN_DATA:
		{
	          u_int8_t *indata;
	          size_t dlen;
		    packet_get_binstr(pc, &indata, &dlen);
		    if (write(context->child_stdin, indata, dlen) != dlen)
			SEND_FAILURE(context);
		    free(indata);
		    break;
	        }
	    case SSH_CMSG_EOF:
		/* All done with input. */
		/* Closing stdin should cause the program to exit. */
		SSH_DLOG(4, ("Client sent EOF, closing child_stdin.\n"));

		/* If stdin == stdout we can't just close b/c */
		/* we will still try to read until an EOF.    */
		if (context->child_stdin == context->child_stdout)
		    shutdown(context->child_stdin, SHUT_WR);
		else
		    close(context->child_stdin);
		context->child_stdin = -1;
		break;
	    case SSH_CMSG_WINDOW_SIZE:
		SEND_FAILURE(context);
		break;
	    case SSH_MSG_DISCONNECT:
		client_gone = 1;
		break;
	    default:
		SSH_DLOG(2, ("Client sent unknown packet type: %d\n",
					packet_type(pc)));
		error_why = "Unkown packet type";
		error_flag = 1;
		break;

	    }	/* end switch */

	    if (error_flag)
		break;
	    if (client_gone)
		break;
	} /* end while(process_packet) */
	if (num < 0) {
	    SSH_DLOG(2, ("doInteractive: process_packet failed: %s\n",
							strerror(errno)));
	    error_why = "Unable to process packet.";
	    error_flag = 1;
	    break;
	}
	if (client_gone || error_flag)
	    break;

    } /* while(1): select loop. */

    /* Check for any errors from above. */
    if (error_flag) {
	SSH_DLOG(5, ("doInteractive: error_flag set.\n"));
	if (!client_gone) {
	  char tmpbuf[200];
	    snprintf(tmpbuf, 200, "ssh server: %s", error_why);
	    SEND_DISCONNECT(context, tmpbuf);
	    client_gone = 1;
	}
    }

    /* Close connection to child; should cause it to exit, */
    /* if it hasn't already. */
    if (context->child_stdin > 0)
	close(context->child_stdin);
    if (context->child_stdin != context->child_stdout)
	close(context->child_stdout);
    if (context->child_stdout != context->child_stderr)
	close(context->child_stderr);
    context->child_stdin = -1;
    context->child_stdout = -1;
    context->child_stderr = -1;

    /* -- child should have exited by now, one way or another. */
    /* -- if the client is still around we can notify him of the exit */
    /* -- We can only get here w/o the exit status if an error happens. */
    /* -- in which case the client will have been send a disconnect. */

    /* Wait for the child to exit, but not forever. */
    for (num = 0 ; num < 10 && !child_exited; num++) {
	sleep(1);
    }
    if (num == 10)
	SSH_DLOG(1, ("doInteractive: Child refused to exit.\n"));


    if (!client_gone) {
	/* Client is still around.  Figure out what to notify him of. */
	/* We should only end up here if the child has exited. */
	if (WIFEXITED(child_exitstat)) {
	  u_int32_t retval;

	    /* Send the exit status. */
	    retval = WEXITSTATUS(child_exitstat);
	    SSH_DLOG(3, ("doInteractive: sending exitstatus: %d\n", retval));
	    xmit_int32(context, SSH_SMSG_EXITSTATUS, retval, PKT_WAITALL);

	    /* Wait for EXIT_CONFIRMATION.  There might be other packets */
	    /* so just keep reading until read_packet returns an error.   */
	    while (packet_type(pc) != SSH_CMSG_EXIT_CONFIRMATION) {
		do {
		    if ((num = read_packet(context, pc, PKT_WAITALL)) > 0)
			num = process_packet(context, pc);
		} while (num == 0);
	        if (num > 0) {
		    if (packet_type(pc) != SSH_CMSG_EXIT_CONFIRMATION) {
			SSH_DLOG(1, ("Weird exit confirmation: %d\n",
					packet_type(pc)));
		    } else {
		        SSH_DLOG(3, ("Received EXIT_CONFIRMATION\n"));
			break;
		    }
		} else {
		    if (num < 0) {
			SSH_DLOG(1,
			  ("doInteractive failed to get exit confirm: %s\n",
							strerror(errno)));
			break;
		    } else {
			SSH_DLOG(3, ("doInteractive: partial exit confirm:%d\n",
									num));
		    }
		}
	    }
	} else if (WIFSIGNALED(child_exitstat)) {
	  char errbuf[50];
	    SSH_DLOG(3, ("doInteractive: child exited due to signal %d\n",
						WTERMSIG(child_exitstat)));
	    snprintf(errbuf, 50, "Command exited on signal %d.",
						WTERMSIG(child_exitstat));
	    SEND_DISCONNECT(context, errbuf);
	} else {
	    SSH_DLOG(3, ("doInteractive: Sending disconnect to client.\n"));
	    SSH_DLOG(3, ("doInteractive: unknown child exit method.\n"));
	    SEND_DISCONNECT(context, "Exit status error\n");
	}
    }

    /* Close socket to client. */
    close(context->s);

    return(0);
}

void sigchld_handler(int sig) {
  int pid;
    while ((pid = wait(&child_exitstat)) < 0) {
	if (errno == EINTR)
	    continue;
	if (errno == ECHILD) {
	    SSH_ERROR("sigchld_handler: SIGCHLD w/o any child processes.\n");
	    child_exited = 1;
	    return;
	}
	SSH_ERROR("sigchld_handler: wait returned error: %s\n", 
						strerror(errno));
	break;
    }
    if (pid >= 0) {
	if (child_exited)
	    SSH_ERROR("sigchld_handler: multiple children!\n");
	else
	    child_exitpid = pid;
    }
    child_exited = 1;
    return;
}
