/* $Id: ssh_buffer.h,v 1.32 2001/02/11 03:35:10 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000 Andrew Brown and Eric Haszlakiewicz.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#ifndef _SSH_BUFFER_H
#define _SSH_BUFFER_H

#include "ssh_crypto.h"

/*
 * Various functions to deal with buffers.
 */

/*
 * data:	actual data buffer
 * dlen:	length of real data in memory
 * dsize:	allocate size of the buffer
 * get_loc:	location in the buffer that the *get* functions work from.
 *		*put* function always append to the end of the buffer.
 */
struct ssh_buf {
    u_int8_t	*data;
    u_int8_t	*get_loc;
    int		dlen;
    int		dsize;
};

/* All functions set errno to indicate the error. */

/* If buf==NULL, allocate buffer, init and return.  Else just init and return*/
struct ssh_buf *buf_alloc(struct ssh_buf *buf, int size);
int buf_grow(struct ssh_buf *buf, int size);
int buf_makeavail(struct ssh_buf *buf, int space);
void buf_clear(struct ssh_buf *buf);
void buf_cleanup(struct ssh_buf *buf);
#define buf_reset(buf) \
	do { (buf)->get_loc = (buf)->data; (buf)->dlen = 0; } while(0)

int buf_append(struct ssh_buf *dstbuf, const struct ssh_buf *srcbuf);

/* Actual data in the buffer: */
#define buf_data(b)	((b)->get_loc)
#define buf_alldata(b)	((b)->data)
/* Size of the data in the buffer: */
#define buf_len(b)	((b)->dlen - ((b)->get_loc - (b)->data))
#define buf_alllen(b)	((b)->dlen)
/* Total size of the buffer. */
#define buf_size(b)	((b)->dsize)
/* Size available for more data. */
#define buf_avail(b)	(buf_size(b) - buf_alllen(b))
/* Rewind pointer: */
#define buf_rewind(b)	do { (b)->get_loc = (b)->data; } while(0)
#define buf_endofdata(b) (buf_alldata((b)) + buf_alllen((b)))

/* Load a particular pointer and size into a buffer.  Be careful! */
#define buf_setbuf(b,p,s) \
	do { (b)->data = (b)->get_loc =(p); (b)->dlen = (b)->dsize =(s); } while(0)

/* Adjust the size of the data in the buffer. */
/*	i.e. if you just wrote some data into the buffer using buf_data */
int buf_adjlen(struct ssh_buf *buf, int delta);

/* Remove bytes from the beginning of the buffer.  Copy rest of data over. */
int buf_trim(struct ssh_buf *buf, int nbytes);

int buf_get_skip(struct ssh_buf *buf, int nbytes);
int buf_get_int32(struct ssh_buf *buf, u_int32_t *ret);
int buf_get_int16(struct ssh_buf *buf, u_int16_t *ret);
int buf_get_int8(struct ssh_buf *buf, u_int8_t *ret);
#define buf_get_byte(b,r) buf_get_int8(b,r)

int buf_get_nbytes(struct ssh_buf *buf, int nbytes, u_int8_t **val);
int buf_get_unk_bytes(struct ssh_buf *buf, u_int8_t **val, int *len);
int buf_get_binstr(struct ssh_buf *buf, u_int8_t **vals, u_int32_t *len);
int buf_get_asciiz(struct ssh_buf *buf, char **astring, int *len);
int buf_get_line(struct ssh_buf *buf, char **astring, int *len);
int buf_get_bignum(struct ssh_buf *buf, ssh_BIGNUM **num);
int buf_get_mpint(struct ssh_buf *, struct ssh_mpint *);
int buf_get_rsa_publickey(struct ssh_buf *, ssh_RSA *, u_int32_t *);

int buf_use_binstr(struct ssh_buf *buf, u_int8_t **vals, u_int32_t *len);

int buffer_expand(struct ssh_buf *buf, size_t size, const char *funcname);

int buf_put_byte(struct ssh_buf *buf, u_int8_t val);
int buf_put_nbytes(struct ssh_buf *buf, int nbytes, const u_int8_t *vals);
int buf_put_int32(struct ssh_buf *buf, u_int32_t val);
int buf_put_int16(struct ssh_buf *buf, u_int16_t val);
#define buf_put_int8(buf,val)	buf_put_byte(buf,val)
int buf_put_binstr(struct ssh_buf *buf, const u_int8_t *vals, u_int32_t len);
int buf_put_asciiz(struct ssh_buf *buf, const char *astring);
int buf_put_bignum(struct ssh_buf *buf, const ssh_BIGNUM *num);
int buf_put_mpint(struct ssh_buf *, struct ssh_mpint *);
int buf_put_rsa_publickey(struct ssh_buf *buf, const ssh_RSA *key);

/* Read from filedescriptor d into buf. if size==-1 read until EOF or EOB */
int buf_readfile(struct ssh_buf *buf, int d, int size);
int buf_fillbuf(ssize_t (*)(int, void *, size_t),struct ssh_buf *buf, int d, int size);

void hex_dump(const unsigned char *buf, int len);
void buf_printnx(const struct ssh_buf *buf, int nbytes, int printall);
#define buf_print(b)       buf_printnx((b), buf_alllen(b), 1)
#define buf_printn(b,n)    buf_printnx((b), (n), 1)
#define buf_print_rest(b)  buf_printnx((b), buf_len(b), 0)

#endif /* _SSH_BUFFER_H */
