/* $Id: ssh_logging.c,v 1.24 2001/02/11 03:35:12 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000 Andrew Brown and Eric Haszlakiewicz.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <syslog.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>

#include "options.h"

#include "ssh_logging.h"

int debuglevel;
const char *g_func;
struct timeval tv_start;

static const char *logident;
static int debugpid;
static int debug_to_stderr;
static int loginitdone = 0;

/*
 * progname:	Prefix to use, if not sshd.
 * usestderr:	if 1, also log to stderr
 *		if 2, only log to stderr
 */
int 
loginit(const char *progname, int usestderr)
{

	if (loginitdone == 1)
		closelog();

	if (usestderr)
		debug_to_stderr = usestderr;
	else
		debug_to_stderr = 0;

	if (!logident || progname)
		logident = (progname == NULL) ? "sshd" : progname;

	if (debug_to_stderr < 2)
		openlog(logident, LOG_PID, LOG_DAEMON);

	debugpid = getpid();
	loginitdone = 1;
	return (0);
}

int 
logclose()
{
	closelog();
	loginitdone = 0;
	return (0);
}

int 
logit(const char *fmt,...)
{
	va_list args;
	char *output_fmt;
	int flen, olen;

	/* Make sure we've got a reasonable function name */
	if (!g_func)
		g_func = "unknown";
	else if (strchr(g_func, '%') != NULL)
		g_func = "UNKNOWN";

	/* prepend the name of the function */
	flen = strlen(fmt);
	/* trim off the \n (if given) and add \r\n */
	if (fmt[flen - 1] == '\n')
		flen--;
	olen = strlen(g_func) + 2 + flen + 2 + 1;
	output_fmt = malloc(olen);
	snprintf(output_fmt, olen, "%s: %.*s\r\n", g_func, flen, fmt);

	va_start(args, fmt);
	if (debug_to_stderr < 2)
		vsyslog(LOG_DEBUG, output_fmt, args);
	if (debug_to_stderr) {
		fprintf(stderr, "%s[%d]: ", logident, debugpid);
		vfprintf(stderr, output_fmt, args);
	}
	va_end(args);
	free(output_fmt);
	return (0);
}

void 
debug_nostderr()
{
	debug_to_stderr = 0;
}
/*
 * debug_inc: increase the debug level.
 */
void 
debug_inc(int sig)
{
	debuglevel++;
	if (debuglevel > 10)
		debuglevel = 10;
	SSH_DLOG(0, ("Debug level increased to %d.\n", debuglevel));
	return;
}
/*
 * debug_dec: decrease the debug level.
 */
void 
debug_dec(int sig)
{
	debuglevel--;
	if (debuglevel < 0)
		debuglevel = 0;
	SSH_DLOG(0, ("Debug level decreased to %d.\n", debuglevel));
	return;
}

int 
is_debug_level(int l)
{
	return (l <= debuglevel);
}
