/* $Id: ssh_v1_child.c,v 1.27 2001/02/11 03:35:18 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000 Eric Haszlakiewicz.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <errno.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "sshd.h"

#include "ssh_buffer.h"
#include "ssh_transport.h"
#include "ssh_util.h"
#include "ssh_v1_messages.h"
#include "ssh_v1_child.h"

static void sigchld_handler(int);
static void sigpipe_handler(int);
static void sigalrm_handler(int);
static int sigalrm_triggered;

static int child_exitstat;
static int child_exitpid;

static ssh_context_t *l_context;

/*
 * v1_close_children: Close any children we spawned.
 *                    So far there's only one (the shell/command).
 *
 * After this function is called the fdinfo array should be thrown away
 * since all of it's file descriptors will be gone.
 *
 */
int 
v1_close_children(ssh_context_t * context)
{
	struct sigaction sa;
	int child_exited;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sigalrm_handler;

	if (sigaction(SIGALRM, &sa, NULL) < 0) {
		SSH_ERROR("sigaction failed: %s\n", strerror(errno));
		return (RET_FATAL);
	}
	/* reset sigchld handler */
	signal(SIGCHLD, SIG_DFL);

	/*
	 * Close connection to child; should cause it to exit if it hasn't
	 * yet.
	 */
	cleanup_channels(context);

	/* don't bother removing fd's from the fdinfo array, it won't be used */

	/* Wait for the child to exit, but not forever. */
	child_exited = 0;
	sigalrm_triggered = 0;
	alarm(10);
	while (!child_exited) {
		child_exitpid = wait(&child_exitstat);
		if (errno == EINTR && sigalrm_triggered)
			break;
		if (errno == ECHILD) {
			/* apparently i have no children */
			return (RET_OK);
		}
		if (child_exitpid != -1) {
			child_exited = 1;
			break;
		}
		if (errno != EINTR) {
			SSH_ERROR("wait failed: %s\n", strerror(errno));
			return (RET_FATAL);
		}
	}
	alarm(0);

	if (!child_exited) {
		SSH_DLOG(1, ("Child refused to exit, killing %d.\n",
			     context->child_pid));
		if (context->child_pid > 0)
			kill(context->child_pid, SIGTERM);
	}
	if (child_exitpid != context->child_pid) {
		SSH_DLOG(1, ("Unexpected child pid (%d instead of %d)\n",
			     child_exitpid, context->child_pid));
		if (context->child_pid > 0)
			kill(context->child_pid, SIGTERM);
	}
	return (RET_OK);
}

/*
 * v1_send_child_exitstat
 *
 * Called only from SIGNAL context after the child has
 * gone away.
 */
static void 
sigalrm_handler(int sig)
{
	sigalrm_triggered = 1;
}

int 
v1_send_child_exitstat(ssh_context_t * context)
{
	int ret;
	struct ssh_event ev;
	struct ssh_ev_send *evdata;

	ret = RET_OK;

	/* Client is still around.  Figure out what to notify him of. */
	/* We should only end up here if the child has exited. */
	if (WIFEXITED(child_exitstat)) {
		u_int32_t retval;
		struct ssh_buf buf;

		retval = WEXITSTATUS(child_exitstat);

		/* Turn the raw data into a ssh_buf. */
		memset(&buf, 0, sizeof(buf));
		if (buf_alloc(&buf, sizeof(u_int32_t)) == NULL) {
			SSH_ERROR("Unable to buf_alloc: %s\n", strerror(errno));
			return (RET_FATAL);
		}
		if (buf_put_int8(&buf, SSH_V1_SMSG_EXITSTATUS) != 0 ||
		    buf_put_int32(&buf, retval) != 0) {
			SSH_ERROR("Unable to bufferize data: %s\n",
			    strerror(errno));
			buf_cleanup(&buf);
			return (RET_FATAL);
		}
		/*
		 * Send it.
	 	 * This event will cause the sendThread to close the
		 * child FDs and send the packet we give it.
		 */
		SSH_DLOG(3, ("sending exitstatus event: %d\n", retval));
		ev.event_type = SSH_EVENT_EXITSTATUS;
		evdata = (struct ssh_ev_send *) ev.event_data;
		/* This won't be too long. */
		evdata->dlen = buf_alllen(&buf);
		memcpy(evdata->data, buf_alldata(&buf), buf_alllen(&buf));

		ret = ssh_sys_sendevent(context, &ev);
		memset(&ev, 0, sizeof(ev));
		buf_cleanup(&buf);

		context->v1_ctx.go_to_exit = 1;
	} else if (WIFSIGNALED(child_exitstat)) {
		char errbuf[50];
		SSH_DLOG(3, ("child exited due to signal %d\n",
			     WTERMSIG(child_exitstat)));
		snprintf(errbuf, 50, "Command exited on signal %d.",
			 WTERMSIG(child_exitstat));
		SEND_DISCONNECT(context, errbuf);
	} else {
		SSH_DLOG(3, ("unknown child exit method.\n"));
		SEND_DISCONNECT(context, "Exit status error\n");
	}

	return (ret);
}

/*
 * sigchld_handler:  This handler should be used in v1 iactive state.
 */
static void 
sigchld_handler(int sig)
{
	FUNC_DECL(sigchld_handler);
	int pid;

	while ((pid = wait(&child_exitstat)) < 0) {
		if (errno == EINTR)
			continue;
		if (errno == ECHILD) {
			SSH_ERROR("SIGCHLD w/o any child processes.\n");
			return;
		}
		SSH_ERROR("wait returned error: %s\n", strerror(errno));
		break;
	}
	if (pid >= 0) {
		child_exitpid = pid;

		if (pid == l_context->send_pid) {
			struct itimerval itval;
			/*
			 * Whoops, our send thread went away.
			 * This happens if it gets a signal, or we sent it
			 * and EXITTHREAD message.
			 * Set a flag so non-signal handler code knows this.
			 * Also set an alarm so we get kicked out of any
			 * system call if the SIGCHLD happened to occur
			 * just before we entered the kernel.
			 * XXX XAX huh?? don't we merely want to exit here?
			 */
			l_context->send_pid = -1;
			/* Null (mostly) handler. */
			signal(SIGALRM, sigalrm_handler);
			memset(&itval, 0, sizeof(itval));
			itval.it_interval.tv_usec = 100;
			itval.it_value.tv_usec = 100;
			setitimer(ITIMER_REAL, &itval, NULL);
		} else if (pid == l_context->child_pid) {
			/*
			 * If the exited child is the primary command,
			 * cleanup, exit
			 */
			/*
			 * XXX In some cases we don't want to exit: i.e.
			 * XXX if we've got other channels open.  Once
			 * XXX we have a way to keep track of those we'll
			 * XXX need to check.
			 */
			SSH_DLOG(4, ("Primary child(%d) exited\n", pid));
			if (v1_send_child_exitstat(l_context) != RET_OK)
				ssh_exit(l_context, 0, EXIT_NOW);
			ssh_exit(l_context, 0, EXIT_FLUSH);
		} else {
			/* Handle other FDs, channels here. */
			SSH_DLOG(4, ("Other child(%d) exited\n", pid));
		}
	}
}

void 
setup_sigchld_handler(ssh_context_t * context)
{
	struct sigaction sa;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sigchld_handler;
	sa.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGCHLD, &sa, NULL);
	l_context = context;
}

void 
setup_sigpipe_handler(ssh_context_t * context)
{
	struct sigaction sa;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sigpipe_handler;
	sa.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGPIPE, &sa, NULL);
	l_context = context;
}

static void 
sigpipe_handler(int sig)
{
	if (is_debug_level(4))
		SSH_DLOG(4, ("Got SIGPIPE"));
}
