/* $Id: sshd_main.c,v 1.89 2001/02/11 03:35:20 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */


/*-
 * Copyright (c) 1983, 1988, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Copyright (c) 2000, 2001 Andrew Brown, Eric Haszlakiewicz,
 *	and Thor Lancelot Simon.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* This file contains routines to do the initial setup: read */
/* configuration files.  Open necessary sockets.  Listen for */
/* incoming connections. */

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "options.h"

#include "sshd.h"
#include "ssh_auth.h"
#include "ssh_channel.h"
#include "ssh_cipher.h"
#include "ssh_global.h"
#include "ssh_parse.h"
#include "ssh_logging.h"
#include "ssh_rsakeys.h"
#include "ssh_sys.h"
#include "ssh_util.h"
#include "ssh_threads.h"
#include "ssh_v1_child.h"
#include "ssh_v1_proto.h"

#include "pathnames.h"

static int exitflag = 0;
static sig_atomic_t numconns = 0;

void main_sigchld_handler(int);

static int sshd_setup(int, char *[], int);
static void setup_sshd_listen_signals(void);
static void setup_sshd_connected_signals(void);
static void sshd_dolisten(void);
static void sshd_usage(void);
static int sshd_configuration(int, char *[], ssh_context_t *);
static int sshd_parse_conffile(ssh_context_t *, char *);

void cleanup_preconnect(int);
void cleanup_postconnect(int);

void sched_sk_regen(int);

/*
 * main routine for ssh server process
 */
int main(int argc, char *argv[])
{
	/* Ensure know initial values for everything */
	memset(&g_context, 0, sizeof(g_context));

	if (sshd_setup(argc, argv, 0) != 0)
		exit(1);

	/* Setup all done, switch to daemon now. */
	if (!nodaemon) {
		if (ssh_sys_daemonize() != 0) {
			warn("Unable to daemonize.");
			cleanup_preconnect(0);
			exit(1);
		}
	}

	/* Turn off logging to stderr. */
	if (!nodaemon)
		debug_nostderr();

	if (ssh_sys_writepid(g_context.opts.pidfile) != 0)
		SSH_DLOG(0, ("Warning: unable to write pid file.\n"));

	sshd_dolisten();

	cleanup_preconnect(0);

	exit(0);
}

/*
 * sshd_setup: Read configuration info for ssh server mode.
 *	Args: how:
 *           0 = clean setup.
 *           1 = re-read config.
 */
static int sshd_setup(int argc, char *argv[], int how)
{
	struct itimerval it;

	g_context.cipher = &g_cipher;
	g_cipher.type = -1;

	g_context.running_as_server = SSH_ROLE_SERVER;

	g_context.send_pid = -1;

	if (loginit(NULL, 1) != 0) {
		warnx("Unable to initialize logging.\n");
		return(1);
	}

	if (how != 0) {
		SSH_ERROR("re-read config not implemented.\n");
		return(0);
	}

	/* Set signal handlers. */

	setup_sshd_listen_signals();

	if (sshd_configuration(argc, argv, &g_context) != 0)
		return(1);

	if (init_serverkey() != 0)
		return(1);

        /* Set the alarm to regenerate the server key: */
	signal(SIGALRM, sched_sk_regen);
	it.it_value.tv_sec = SSHD_REGENINTERVAL;   /* Default: 20 minutes */
	it.it_value.tv_usec = 0;
	it.it_interval.tv_sec = SSHD_REGENINTERVAL;
	it.it_interval.tv_usec = 0;
	setitimer(ITIMER_REAL, &it, NULL);

#ifndef UNSAFE_DEBUG
	{
		struct rlimit rl;
		rl.rlim_cur = (rlim_t)0;
		rl.rlim_max = (rlim_t)0;

		setrlimit(RLIMIT_CORE, &rl);
	}
#endif

	ssh_sys_randinit();

	set_supported_ciphers(&(g_context.supported_ciphers));

	set_supported_auths(&(g_context.supported_auths));

	channel_init(&g_context);

	if (OpenListenSocket(&g_context) != 0)
		return(1);

	return(0);
}

static void
sshd_usage() {
	fprintf(stderr, "Usage: sshd %s %s %s %s %s %s\n", "[-Ddk]",
			"[-a address]", "[-f config]", "[-h host key]",
			"[-p port]", "[-P pidfile]");
}

static int
sshd_configuration(int argc, char *argv[], ssh_context_t *context) {
  int ch;
  off_t hksiz;
  void *keydata;
  char *conffilename = (char *)NULL;

	while ((ch = getopt(argc, argv, "Ddka:h:p:P:f:")) != -1) {
		switch(ch) {
		case 'D':
			nodaemon++;
			break;
		case 'd':
			debug_inc(0);
			break;
		case 'k':
			context->opts.keepalive = 1;
			break;
		case 'h':
			/* Set the host key file */
			context->opts.hostkey = optarg;
			break;
		case 'a':
			/* address to listen on */
			context->opts.address = optarg;
			break;
		case 'p':
			/* Port to listen on */
			context->opts.port = optarg;
			break;
		case 'P':
			context->opts.pidfile = optarg;
			break;
		case 'f':
			conffilename = optarg;
			break;
		case '?':
		default:
			sshd_usage();
			return(1);
		}
	}

	if(conffilename != NULL)
		sshd_parse_conffile(context, conffilename);

	/* XXX Default to V1 for now */
	context->protocol_version = SSH_V1;
	context->opts.v1_compat_mode = 0;

	if(!context->opts.pidfile)
		context->opts.pidfile = _PATH_SSHD_PIDFILE;

	if(!context->opts.hostkey) {
		context->opts.hostkey = _PATH_SSHD_SERVER_PRIVKEY;
		keydata = ssh_sys_readin(context->opts.hostkey, &hksiz);
		if (keydata == NULL && errno == ENOENT) {
			context->opts.hostkey =
			    _PATH_SSHD_SERVER_PRIVKEY_FRESSH;
			keydata = ssh_sys_readin(context->opts.hostkey, &hksiz);
		}
	} else
		 keydata = ssh_sys_readin(context->opts.hostkey, &hksiz);

	if (keydata == NULL) {
		SSH_ERROR("couldn't access host key %s: %s\n",
			context->opts.hostkey, strerror(errno));
		sshd_usage();
		return (1);
	}

	if (decode_keyfile(context, keydata, hksiz, NULL, 0,
		&context->v1_ctx.hostkey, NULL, NULL) != 0) {
		SSH_ERROR("host key %s is corrupt (did not decode).\n");
		sshd_usage();
		return (1);
	}

	return (0);
}

int sshd_parse_conffile(ssh_context_t *context, char *filename) 
{

  int known;
  FILE *conffile;
  char option[32 + 1], optval[255 + 1];
  static char hostkey[255 +1], address[16 + 1], port[5 + 1], pidfile[255 +1];

	conffile = fopen(filename, "r");
	if(conffile == NULL) {
		SSH_DLOG(1, ("could not open config file %s: %s\n", 
			 strerror(errno)));
		return -1;
	}

	while(!feof(conffile)) {

		known = 0;
		fscanf(conffile, "%32s%255s\n", option, optval);

		if(!strcasecmp(option, "keepalive")) {
			if(is_yes(optval))
				context->opts.keepalive = 1;
			known++;
		}

		if(!strcasecmp(option, "hostkey")) {
			if(context->opts.hostkey == NULL) {
				strncpy(hostkey, optval, sizeof(hostkey) - 1);
				hostkey[sizeof(hostkey) - 1] = NULL;
				context->opts.hostkey = hostkey;
			}
			known++;
		}

		if(!strcasecmp(option, "listenaddress")) {
			if(context->opts.address == NULL) {
				strncpy(address, optval, sizeof(address) - 1);
				address[sizeof(address) - 1] = NULL;
				context->opts.address = address;
			}
			known++;
		}

		if(!strcasecmp(option, "port")) {
			if(context->opts.port == NULL) {
				strncpy(port, optval, sizeof(port) -1);
				port[sizeof(port) - 1] = NULL;
				context->opts.port = port;
			}
			known++;
		}

		if(!strcasecmp(option, "pidfile")) {
			if(context->opts.pidfile == NULL) {
				strncpy(pidfile, optval, sizeof(pidfile) -1);
				pidfile[sizeof(pidfile) - 1] = NULL;
				context->opts.pidfile = pidfile;
			}
			known++;
		}

		if(!known)
			SSH_DLOG(1, ("Unknown option %s (value %s) in config "
				 "file %s\n", option, optval, filename));
	}
	
	return 0;
}

/*
 * OpenListenSocket: Create the listen sockets that sshd will use.  If
 * no address is specified in the context, we bind to the wildcard address
 * for all supported protocol families.
 */
int OpenListenSocket(ssh_context_t *context) {
  struct addrinfo hints, *res, *res0;
  struct sshd_listen_socket *ls;
  int dummy, error;
  struct linger linger;
  char addrbuf[NI_MAXHOST];

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_PASSIVE;

	if (context->opts.port == NULL) {
		error = getaddrinfo(context->opts.address, SSHD_SERVICE,
		    &hints, &res0);
		if (error == EAI_SERVICE)
			error = getaddrinfo(context->opts.address, SSHD_PORT,
			    &hints, &res0);
	} else
		error = getaddrinfo(context->opts.address, context->opts.port,
		    &hints, &res0);

	if (error) {
		SSH_ERROR("Unable to get listen address info: %s\n",
		    gai_strerror(errno));
		return (1);
	}

	for (res = res0; res != NULL; res = res->ai_next) {
		ls = malloc(sizeof(*ls));
		if (ls == NULL) {
			SSH_ERROR("Unable to allocate listen data: %s\n",
			    strerror(errno));
			freeaddrinfo(res0);
			return (1);
		}

		ls->sock = socket(res->ai_family, res->ai_socktype,
		    res->ai_protocol);
		if (ls->sock < 0) {
			/*
			 * Protocol probably isn't supported by the
			 * kernel -- not the end of the world.
			 */
			free(ls);
			ls = NULL;
			continue;
		}

		dummy = 1;
		(void) setsockopt(ls->sock, SOL_SOCKET, SO_REUSEADDR,
		    &dummy, sizeof(dummy));

		linger.l_onoff = 1;
		linger.l_linger = 10;
		(void) setsockopt(ls->sock, SOL_SOCKET, SO_LINGER,
		    &linger, sizeof(linger));

		if (bind(ls->sock, res->ai_addr, res->ai_addrlen) < 0) {
			SSH_ERROR("Unable to bind to %s: %s\n",
			    ssh_sockaddr_ntop(res->ai_addr, addrbuf,
			    sizeof(addrbuf)), strerror(errno));
			(void) close(ls->sock);
			free(ls);
			ls = NULL;
			continue;
		}

		if (listen(ls->sock, 7) < 0) {
			SSH_ERROR("Unable to listen on %s: %s\n",
			    ssh_sockaddr_ntop(res->ai_addr, addrbuf,
			    sizeof(addrbuf)), strerror(errno));
			(void) close(ls->sock);
			free(ls);
			ls = NULL;
			continue;
		}

		ls->next = context->listen_sockets;
		context->listen_sockets = ls;
		if (ls->sock > context->listen_socket_max)
			context->listen_socket_max = ls->sock;
	}

	freeaddrinfo(res0);

	if (context->listen_sockets == NULL) {
		SSH_ERROR("No listen sockets available!\n");
		return (1);
	} else
		return (0);
}

/*
 * doquit: set exitflag so the daemon exits gracefully.
 */
void doquit(int sig) {
	exitflag = 1;
	return;
}

/*
 * Just reap any childen.
 */
void main_sigchld_handler(int foo) {
 int child_status;
 int child_pid;
	child_pid = wait(&child_status);
	if (WIFEXITED(child_status)) {
		SSH_DLOG(3, ("child %d exited with %d\n",
				child_pid, WEXITSTATUS(child_status)));
	} else {
		SSH_DLOG(2, ("child %d  status %d\n",
				child_pid, child_status));
	}
	return;
}

static void
sshd_process_connection(ssh_context_t *context)
{
	FUNC_DECL(sshd_process_context);

	int pid;
	int retval;

	context->v1_ctx.state = SSH_STATE_NONE;

	/* Check for compatible versions. */
	if ((retval = get_version(context, NULL)) != 0)
		ssh_sys_exit(0);

	/*
	 * At this point, we should know what version we'll be using
	 * and what transport layer.
	 */

	if (ssh_sys_create_eventq(context) < 0) {
		SSH_ERROR("Unable to create event queue!\n");
		ssh_sys_exit(1);
	}

	/*
	 * Get the receive thread pid before forking so both sides
	 * have it.
	 */
	context->recv_pid = getpid();

	/*
	 * Setup a SIGCHLD handler so the receive thread can tell when
	 * the send thread exits.
	 */
	setup_sigchld_handler(context);

	switch ((pid = fork())) {
	case -1:
		SSH_ERROR("Unable to fork\n");
		ssh_sys_exit(1);
		break;

	case 0:
		if (loginit(NULL, 1) != 0) {
			SSH_ERROR("Unable to re-initialize logging.\n");
			kill(context->recv_pid, SIGINT);
			break;
		}
		ssh_send_thread(context);
		break;

	default:
		context->send_pid = pid;
		ssh_recv_thread(context, dispatch_v1_server_msg);
		break;
	}

	/*
	 * We shouldn't get here.  Kill ourself and any other
	 * processes in our group.
	 */
	kill(0, SIGINT);
	ssh_sys_exit(1);

	exit(2);		/* not reached */
}

/*
 * sshd_dolisten: Wait for a client to connect.
 */
static void sshd_dolisten()
{
	struct sockaddr *sa = (struct sockaddr *) &g_context.saddr;
	struct sshd_listen_socket *ls;
	int num;
	pid_t cpid;
	struct timeval tv;
	fd_set readfds; 
	char hostname[NI_MAXHOST];
	int slen, error, sobuf;

	while (exitflag == 0) {
		FD_ZERO(&readfds);
		for (ls = g_context.listen_sockets; ls != NULL; ls = ls->next)
			FD_SET(ls->sock, &readfds);
		num = select(g_context.listen_socket_max + 1, &readfds,
			     NULL, NULL, NULL);
		if (num < 0) {
			/* non-fatal. */
			if (errno == EINTR)
				continue;

			/* fatal. */
			SSH_ERROR("select error: %s\n", strerror(errno));
			return;
		}

		for (ls = g_context.listen_sockets; ls != NULL; ls = ls->next) {
			if (FD_ISSET(ls->sock, &readfds) == 0)
				continue;

			slen = sizeof(g_context.saddr);
			g_context.transport_ctx.commsock =
			    accept(ls->sock, sa, &slen);
			if (g_context.transport_ctx.commsock < 0) {
				SSH_ERROR("Unable to accept connection: %s\n",
				    strerror(errno));
				continue;
			}
			gettimeofday(&tv_start, NULL);
			SSH_DLOG(1, ("Connection from address %s\n",
			    ssh_sockaddr_ntop(sa, hostname, sizeof(hostname))));
			if(++numconns >= SSHD_REGENCONNS) {
				SSH_DLOG(1, ("Regenerating serverkey (%d connections)\n",
					numconns));
				regen_serverkey();
				numconns = 0;
			}
			if (nodaemon < 2) {
				cpid = fork();
				if (cpid < 0) {
					SSH_ERROR("Unable to fork: %s\n",
							  strerror(errno));
					/* Wait a while, try again. */
					sleep(5);
				} else if (cpid != 0) {
					/* Parent. */
					SSH_DLOG(2, ("child: %d\n", cpid));
					close(g_context.transport_ctx.commsock);
					g_context.transport_ctx.commsock = -1;
					continue;
				}
			}

			sobuf = 65536;
			setsockopt(g_context.transport_ctx.commsock,
				   SOL_SOCKET, SO_RCVBUF, &sobuf,
				   sizeof(sobuf));
			setsockopt(g_context.transport_ctx.commsock,
				   SOL_SOCKET, SO_SNDBUF, &sobuf,
				   sizeof(sobuf));

			while ((ls = g_context.listen_sockets) != NULL) {
				g_context.listen_sockets = ls->next;
				(void) close(ls->sock);
				free(ls);
			}
			g_context.listen_socket_max = 0;

			/*
			 * Don't detach if someone's watching.
			 * Side effect: killing the original
			 * server will kill at its children,
			 * too.
			 */
			if (!nodaemon)
				daemon(0, 0);

			if (loginit(NULL, 1) != 0) {
				warnx("Unable to re-initialize logging.\n");
				exit(1);
			}

			/*
			 * Switch signals to the state we
			 * want for a connected and forked
			 * process.
			 */
			setup_sshd_connected_signals();

			if (is_debug_level(6))
				sleep(6);

			(void) ssh_sockaddr_ntop(sa, hostname,
			    sizeof(hostname));

			SSH_DLOG(3, ("Connection from address %s\n", hostname));

			gettimeofday(&tv, NULL);
			ssh_sys_randadd();
			ssh_sys_randadd();

			/*
			 * Find the canonical name for the host
			 * that has connected to us.
			 */
#ifdef SA_LEN
			if (getnameinfo(sa, SA_LEN(sa), hostname,
#else
			if (getnameinfo(sa, sa->sa_len, hostname,
#endif
			    sizeof(hostname), NULL, 0, NI_NAMEREQD) == 0) {
				struct addrinfo hints, *res0;

				memset(&hints, 0, sizeof(hints));
				hints.ai_family = PF_UNSPEC;
				hints.ai_socktype = SOCK_STREAM;
				hints.ai_flags = AI_CANONNAME;

				error = getaddrinfo(hostname,
				    g_context.opts.port, &hints, &res0);
				if (error) {
					SSH_DLOG(2, ("Unable to get "
					    "address info for %s: %s\n",
					    hostname, gai_strerror(error)));
				} else {
					strncpy(hostname, res0->ai_canonname,
					    sizeof(hostname) - 1);
					hostname[sizeof(hostname) - 1] = '\0';
					freeaddrinfo(res0);
				}
			}

			SSH_DLOG(1, ("Received connection from %s.\n",
			    hostname));

			sshd_process_connection(&g_context);

			/* This shouldn't be reached. */
			SSH_ERROR("Error!  doSSH returned!\n");
			exit(1);
		}
	} /* while (!exitflag) */
	return;
}

static void setup_sshd_listen_signals()
{
	int i;
	struct sigaction sa;
#ifndef UNSAFE_DEBUG
	int cleanup_sigs[] = { SIGQUIT, SIGILL, SIGTRAP, //SIGEMT,
	                       SIGFPE, SIGBUS, SIGSEGV, SIGSYS,
	                       SIGPIPE, SIGALRM, SIGTERM, SIGXFSZ,
	                       SIGPROF, -1 };

	int ign_sigs[] = { SIGTSTP, SIGTTIN, SIGTTOU, SIGXCPU, -1 };
#endif

	signal(SIGHUP, (void *)sshd_setup);
	signal(SIGINT, doquit);

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = main_sigchld_handler;
	sa.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGCHLD, &sa, NULL);

	signal(SIGUSR1, debug_inc);
	signal(SIGUSR2, debug_dec);

#ifndef UNSAFE_DEBUG
	for (i = 0; cleanup_sigs[i] != -1 ; i++)
	{
		signal(cleanup_sigs[i], (void *)cleanup_preconnect);
	}

	for (i = 0; ign_sigs[i] != -1 ; i++)
	{
		signal(ign_sigs[i], SIG_IGN);
	}
#endif
}

/*
 * setup_sshd_connected_signals:
 *
 *   Setup signal handlers for the length of time
 *  from when we fork a process to handle a connection
 *  until that process creates it's threads.
 *  doSSH should set its own signal handlers at that point.
 */
static void setup_sshd_connected_signals()
{
	int i;
	struct itimerval itval;
#ifndef UNSAFE_DEBUG
	int cleanup_sigs[] = { SIGQUIT, SIGILL, SIGTRAP, //SIGEMT,
	                       SIGFPE, SIGBUS, SIGSEGV, SIGSYS,
	                       SIGPIPE, SIGALRM, SIGTERM, SIGXFSZ,
	                       SIGPROF, SIGHUP, SIGINT, -1 };

	int ign_sigs[] = { SIGALRM, SIGTSTP, SIGTTIN, SIGTTOU, SIGXCPU, -1 };
#endif

	/* Disable server-key regen alarm */
	timerclear(&itval.it_interval);
	timerclear(&itval.it_value);
	setitimer(ITIMER_REAL, &itval, NULL);

#ifndef UNSAFE_DEBUG
	for (i = 0; cleanup_sigs[i] != -1 ; i++)
	{
		signal(cleanup_sigs[i], (void *)cleanup_postconnect);
	}

	for (i = 0; ign_sigs[i] != -1 ; i++)
	{
		signal(ign_sigs[i], SIG_IGN);
	}
#endif
}

/*
 * cleanup_preconnect: Cleanup before accepting a connection.
 *
 *	Args: sig	0 = normal shutdown
 *			!0 = shutdown due to signal.
 */
void cleanup_preconnect(int sig)
{
	struct sshd_listen_socket *ls;

	SSH_DLOG(1, ("Received signal %d.  Exiting %d.\n", sig, getpid()));

	if (g_context.transport_ctx.commsock > 0) {
		(void) close(g_context.transport_ctx.commsock);
		g_context.transport_ctx.commsock = -1;
	}

	while ((ls = g_context.listen_sockets) != NULL) {
		g_context.listen_sockets = ls->next;
		(void) close(ls->sock);
		/* Don't free() from a signal handler! */
	}

	if (sig == SIGHUP || sig == SIGINT)
		exit(0);
	else
		exit(1);
}

void cleanup_postconnect(int sig)
{

/* XAX this should go into a handler installed by doSSH */
	if (g_context.v1_ctx.hostkey)
		ssh_rsa_free(&g_context.v1_ctx.hostkey);
	if (g_context.v1_ctx.serverkey)
		ssh_rsa_free(&g_context.v1_ctx.serverkey);
	if (g_context.send_pid > 0 && g_context.send_pid != getpid())
		kill(g_context.send_pid, SIGINT);
	memset(g_context.v1_ctx.session_id, 0, 16);
	memset(g_context.v1_ctx.session_key, 0, SSH_V1_SESSION_KEY_SIZE);

	logclose();

	exit(1);
}

void sched_sk_regen(int sig)
{
	numconns = SSHD_REGENCONNS;		/* XXX shouldn't be static */
}
