/* $Id: ssh_cipher.c,v 1.16 2001/02/11 03:35:26 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2001 Jason R. Thorpe.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* routines to implement the various crypt routines. */

#include <string.h>

#include "options.h"
#include "sshd.h"
#include "ssh_global.h"
#include "ssh_cipher.h"

/*
 * set_supported_ciphers: set the bitmask for supported ciphers.
 */
void 
set_supported_ciphers(int *arg)
{

	*arg = 0;
#ifdef WITH_CIPHER_NONE
	*arg = (1 << SSH_CIPHER_NONE);
#endif
#ifdef WITH_CIPHER_IDEA
	*arg |= (1 << SSH_CIPHER_IDEA);
#endif
#ifdef WITH_CIPHER_DES
	*arg |= (1 << SSH_CIPHER_DES);
#endif
#ifdef WITH_CIPHER_3DES
	*arg |= (1 << SSH_CIPHER_3DES);
	*arg |= (1 << SSH_CIPHER_DES3);
#endif
#ifdef WITH_CIPHER_RC4
	*arg |= (1 << SSH_CIPHER_RC4);
#endif
#ifdef WITH_CIPHER_BLOWFISH
	*arg |= (1 << SSH_CIPHER_BLOWFISH);
	*arg |= (1 << SSH_CIPHER_FISHBLOW);
#endif
}

int 
cipher_supported(ssh_context_t *context, int cipher)
{

	if (cipher < 0)
		return (context->supported_ciphers);
	else
		return ((1 << cipher) & context->supported_ciphers);
}

const char *
cipher_name(int cipher_type)
{

#ifdef WITH_CIPHER_NONE
	if (cipher_type == SSH_CIPHER_NONE)
		return ("none");
#endif
#ifdef WITH_CIPHER_IDEA
	if (cipher_type == SSH_CIPHER_IDEA)
		return ("idea");
#endif
#ifdef WITH_CIPHER_DES
	if (cipher_type == SSH_CIPHER_DES)
		return ("des");
#endif
#ifdef WITH_CIPHER_3DES
	if (cipher_type == SSH_CIPHER_3DES)
		return ("3des");
	if (cipher_type == SSH_CIPHER_DES3)
		return ("des3");
#endif
#ifdef WITH_CIPHER_RC4
	if (cipher_type == SSH_CIPHER_RC4)
		return ("rc4");
#endif
#ifdef WITH_CIPHER_BLOWFISH
	if (cipher_type == SSH_CIPHER_BLOWFISH)
		return ("blowfish");
	if (cipher_type == SSH_CIPHER_FISHBLOW)
		return ("fishblow");
#endif

	return ("unknown");
}

int 
cipher_number(const char *name)
{

#ifdef WITH_CIPHER_NONE
	if (strcasecmp(name, "NONE") == 0)
		return SSH_CIPHER_NONE;
#endif
#ifdef WITH_CIPHER_IDEA
	if (strcasecmp(name, "IDEA") == 0)
		return SSH_CIPHER_IDEA;
#endif
#ifdef WITH_CIPHER_DES
	if (strcasecmp(name, "DES") == 0)
		return SSH_CIPHER_DES;
#endif
#ifdef WITH_CIPHER_3DES
	if (strcasecmp(name, "3DES") == 0)
		return SSH_CIPHER_3DES;
	if (strcasecmp(name, "DES3") == 0)
		return SSH_CIPHER_DES3;
#endif
#ifdef WITH_CIPHER_RC4
	if (strcasecmp(name, "RC4") == 0)
		return SSH_CIPHER_RC4;
#endif
#ifdef WITH_CIPHER_BLOWFISH
	if (strcasecmp(name, "BLOWFISH") == 0)
		return SSH_CIPHER_BLOWFISH;
	if (strcasecmp(name, "FISHBLOW") == 0)
		return SSH_CIPHER_FISHBLOW;
#endif

	return -1;
}

/*
 * set_cipher_type: set up the context for the given cipher type.
 */
int 
set_cipher_type(struct ssh_cipher *p_cipher, int type)
{

	switch (type) {
#ifdef WITH_CIPHER_NONE
		case SSH_CIPHER_NONE:
		ssh_none_attach(p_cipher);
		break;
#endif
#ifdef WITH_CIPHER_IDEA
	case SSH_CIPHER_IDEA:
		ssh_idea_attach(p_cipher);
		break;
#endif
#ifdef WITH_CIPHER_DES
	case SSH_CIPHER_DES:
		ssh_des_attach(p_cipher);
		break;
#endif
#ifdef WITH_CIPHER_3DES
	case SSH_CIPHER_3DES:
		ssh_3des_attach(p_cipher);
		break;
	case SSH_CIPHER_DES3:
		ssh_des3_attach(p_cipher);
		break;
#endif
#ifdef WITH_CIPHER_RC4
	case SSH_CIPHER_RC4:
		ssh_rc4_attach(p_cipher);
		break;
#endif
#ifdef WITH_CIPHER_BLOWFISH
	case SSH_CIPHER_BLOWFISH:
		ssh_blowfish_attach(p_cipher);
		break;
	case SSH_CIPHER_FISHBLOW:
		ssh_fishblow_attach(p_cipher);
		break;
#endif
	default:
		SSH_DLOG(1, ("Unknown cipher type: %d\n", type));
		return (1);
	}
	return (0);
}

/*
 * cipher_initialize_passphrase: Initialize a cipher based on a passphrase.
 * This works by doing an MD5 digest on the passphrase, and then using
 * that as the key.
 */
void *
cipher_initialize_passphrase(struct ssh_cipher * cipher,
			     const char *passphrase, int role)
{
	MD5_CTX ctx;
	unsigned char digest[16];

	MD5_Init(&ctx);
	MD5_Update(&ctx, (const unsigned char *) passphrase,
		   strlen(passphrase));
	MD5_Final(digest, &ctx);

	cipher_initialize(cipher, digest, sizeof(digest), role);

	memset(digest, 0, sizeof(digest));
	memset(&ctx, 0, sizeof(ctx));

	return (cipher_key_data(cipher));
}
