/* $Id: ssh_crypto_openssl.c,v 1.25 2001/02/11 03:35:27 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000, 2001 Andrew Brown, Thor Lancelot Simon,
 *      and Jason R. Thorpe.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <openssl/rand.h>

#include <openssl/opensslv.h>

#if OPENSSL_VERSION_NUMBER >= 0x00903000L
#define TO_CBLOCK(x)	((des_cblock *)(x))
#else
#define	TO_CBLOCK(x)	(x)
#endif

#include "options.h"
#include "ssh_crypto.h"
#include "ssh_crypto_openssl.h"
#include "ssh_cipher.h"
#include "ssh_logging.h"
#include "ssh_rsakeys.h"

inline void
ssh_rsa_free(ssh_RSA **keyp)
{
	ssh_RSA *key = *keyp;

	*keyp = NULL;
	RSA_free(&(key->rsa));
}

inline ssh_RSA *
ssh_rsa_new(void)
{
	return ((ssh_RSA *) RSA_new());
}

inline ssh_RSA *
ssh_rsa_generate_key(int numbits, int exp, void *cb, char *cbarg)
{

	return ((ssh_RSA *) RSA_generate_key(numbits, exp, cb, cbarg));
}

/*
 * Make sure a key has all the entries
 * filled in correctly.  If not, calculate them.
 *
 * Calculates dmp1, dmq1, iqmp.
 */
int
ssh_key_fixup(ssh_RSA *key)
{
	BIGNUM *temp;
	BN_CTX *ctx;

	if (key->rsa.q == NULL || key->rsa.p == NULL)
		return (1);

	temp = BN_new();
	ctx = BN_CTX_new();

	key->rsa.dmp1 = BN_new();
	BN_sub(temp, key->rsa.p, BN_value_one());
	BN_mod(key->rsa.dmp1, key->rsa.d, temp, ctx);

	key->rsa.dmq1 = BN_new();
	BN_sub(temp, key->rsa.q, BN_value_one());
	BN_mod(key->rsa.dmq1, key->rsa.d, temp, ctx);

	if (key->rsa.iqmp == NULL) {
		key->rsa.iqmp = BN_new();
		BN_mod_inverse(key->rsa.iqmp, key->rsa.q, key->rsa.p, ctx);
	}

	return (0);
}

inline int
ssh_rsa_private_decrypt(int len, char *from, char *buf, ssh_RSA *key)
{

	return (RSA_private_decrypt(len, from, buf, &(key->rsa),
				    RSA_PKCS1_PADDING));
}

inline int
ssh_rsa_public_encrypt(int len, char *from, char *buf, ssh_RSA *key)
{
	return (RSA_public_encrypt(len, from, buf, &(key->rsa),
				   RSA_PKCS1_PADDING));
}
/* ---------------------- */

inline void
bignum_free(ssh_BIGNUM * num)
{

	return (BN_clear_free(&(num->num)));
}

inline int
bignum_compare(const ssh_BIGNUM * num1, const ssh_BIGNUM * num2)
{

	return (BN_cmp(&(num1->num), &(num2->num)));
}

inline int
bignum_num_bits(const ssh_BIGNUM * num)
{

	return (BN_num_bits(&(num->num)));
}

inline int
bignum_num_bytes(const ssh_BIGNUM * num)
{

	return (BN_num_bytes(&(num->num)));
}

/*
 * bignum_bn2bin:
 *	returns number of bytes used.
 */
inline int
bignum_bn2bin(const ssh_BIGNUM * num, char *buf)
{

	return (BN_bn2bin(&(num->num), buf));
}

inline int
bignum_bin2bn(ssh_BIGNUM ** num, const u_int8_t * data, u_int16_t bytes)
{
	BIGNUM *new_bn;

	new_bn = BN_bin2bn(data, bytes, &((*num)->num));
	if (new_bn == NULL)
		return (-1);
	(BIGNUM *) * num = new_bn;
	return (0);
}

inline char *
bignum_bn2hex(const ssh_BIGNUM * num)
{

	return (BN_bn2hex(&(num->num)));
}

inline int
bignum_hex2bn(ssh_BIGNUM ** num, const char *buf)
{

	return (BN_hex2bn((BIGNUM **) num, buf));
}

inline char *
bignum_bn2dec(const ssh_BIGNUM * num)
{

	return (BN_bn2dec(&(num->num)));
}

inline int
bignum_dec2bn(ssh_BIGNUM ** num, const char *buf)
{

	return (BN_dec2bn((BIGNUM **) num, buf));
}

/*
 * bignum_to_mpint: convert from a BIGNUM to a ssh_mpint.
 *		    mpi should point to an empty mpint.
 *
 * Returns -1 on error.
 */
int
bignum_to_mpint(const ssh_BIGNUM * bn, struct ssh_mpint * mpi)
{

	if ((mpi->data = malloc(BN_num_bytes(&(bn->num)))) == NULL) {
		SSH_ERROR("bignum_to_mpint: malloc failed: %s\n",
			  strerror(errno));
		mpi->bits = 0;
		return (-1);
	}
	mpi->bits = BN_num_bits(&(bn->num));
	return (BN_bn2bin(&(bn->num), mpi->data));
}

/*
 * mpint_to_bignum: convert from a ssh_mpint to a BIGNUM
 *		bn will be allocated.
 *
 * Returns -1 on error.
 */
int
mpint_to_bignum(const struct ssh_mpint * mpi, ssh_BIGNUM ** bn)
{
	if ((*bn = (ssh_BIGNUM *) BN_bin2bn(mpi->data,
					    (mpi->bits + 7) / 8, NULL))
	    == NULL) {
		SSH_ERROR("mpint_to_bignum: BN_bin2bn failed.\n");
		return (-1);
	}
	return (0);
}

/*
 * -------------
 * Functions defined in ssh_crypto.h:
 * -------------
 */

/*
 * ssh_rand_feed:        feed some seed data to the random number generator.
 *                      Note: buffer is cleared before returning.
 */
void
ssh_rand_feed(u_int8_t * buf, size_t len)
{

	RAND_seed(buf, len);
	memset(buf, 0, len);
}

/*
 * ssh_rand_bytes:       fill a buffer with random data.
 */
void
ssh_rand_bytes(size_t len, u_int8_t * buf)
{

	RAND_bytes(buf, len);
}

/*
 * ssh_rand_clean:	clean up PRNG private data.
 */
void
ssh_rand_clean()
{

	RAND_cleanup();
}

/*
 * ssh_md5_init:
 *
 *	Initialize an MD5 context.
 */
void
ssh_md5_init(ssh_MD5 *ctx)
{

	MD5_Init(&ctx->md5);
}

/*
 * ssh_md5_update:
 *
 *	Update an MD5 context with some data.
 */
void
ssh_md5_update(ssh_MD5 *ctx, const void *data, size_t len)
{

	MD5_Update(&ctx->md5, data, len);
}

/*
 * ssh_md5_final:
 *
 *	Finalize an MD5 computation.
 */
void
ssh_md5_final(u_int8_t *res, ssh_MD5 *ctx)
{

	MD5_Final(res, &ctx->md5);
}

#ifdef WITH_CIPHER_IDEA
void
ssh_idea_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_IDEA;
	c->initialize = (void *) ssh_idea_initialize;
	c->encrypt = (void *) ssh_idea_encrypt;
	c->decrypt = (void *) ssh_idea_decrypt;
	c->destroy = (void *) ssh_idea_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = NULL;
	c->mac_validate = NULL;
}

ssh_idea_t *
ssh_idea_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_idea_initialize);

	ssh_idea_t *key_data;

	if (klen < 16) {
		SSH_ERROR("Key length %d is insufficient for IDEA.\n", klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_idea_t));
	if (key_data == NULL)
		return (NULL);

	idea_set_encrypt_key(session_key, &key_data->i_key);

	memset(key_data->i_ivec, 0, sizeof(key_data->i_ivec));

	return (key_data);
}

void
ssh_idea_destroy(ssh_idea_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

void
ssh_idea_encrypt(u_int8_t * clear, u_int8_t * enc, int length,
		 ssh_idea_t * key_data)
{
	int num = 0;

	idea_cfb64_encrypt(clear, enc, length, &key_data->i_key,
			   key_data->i_ivec, &num, 1);
}

void
ssh_idea_decrypt(u_int8_t * enc, u_int8_t * clear, int length,
		 ssh_idea_t * key_data)
{
	int num = 0;

	idea_cfb64_encrypt(enc, clear, length, &key_data->i_key,
			   key_data->i_ivec, &num, 0);
}

/*
 * XXX This is a total crock hack to work around a problem
 * XXX with NetBSD's libcrypto_idea.  This isn't ever called,
 * XXX but we need to generate a reference to OpenSSL's
 * XXX idea_cbc_encrypt() function.
 */
void
ssh_idea_hack(u_char * in, u_char * out, long length,
	      IDEA_KEY_SCHEDULE * ks, u_char * iv, int enc);
void
ssh_idea_hack(u_char * in, u_char * out, long length,
	      IDEA_KEY_SCHEDULE * ks, u_char * iv, int enc)
{

	idea_cbc_encrypt(in, out, length, ks, iv, enc);
}
#endif /* WITH_CIPHER_IDEA */

#ifdef WITH_CIPHER_DES
void
ssh_des_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_DES;
	c->initialize = (void *) ssh_des_initialize;
	c->encrypt = (void *) ssh_des_encrypt;
	c->decrypt = (void *) ssh_des_decrypt;
	c->destroy = (void *) ssh_des_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = NULL;
	c->mac_validate = NULL;
}

ssh_des_t *
ssh_des_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_des_initialize);

	int err;
	des_cblock key;
	ssh_des_t *key_data;

	if (klen < 8) {
		SSH_ERROR("Key length %d is insufficient for DES.\n", klen);
		return (NULL);
	}
	err = 0;

	key_data = malloc(sizeof(ssh_des_t));
	if (key_data == NULL)
		return NULL;

	memcpy(key, session_key, sizeof(des_cblock));
	des_set_odd_parity(TO_CBLOCK(key));
	if (!des_is_weak_key(TO_CBLOCK(key)))
		(void) des_set_key(TO_CBLOCK(key), key_data->des_ks);
	else
		err = 1;

	memset(key_data->des_ivec[0], 0, sizeof(key_data->des_ivec[0]));
	memset(key_data->des_ivec[1], 0, sizeof(key_data->des_ivec[1]));

	if (!err)
		return key_data;
	else {
		memset(key, 0, sizeof(key));
		memset(key_data, 0, sizeof(*key_data));
		free(key_data);
		return NULL;
	}
}

void
ssh_des_destroy(ssh_des_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

void
ssh_des_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		ssh_des_t *key_data)
{

	des_ncbc_encrypt(clear, enc, length, key_data->des_ks,
			 TO_CBLOCK(key_data->des_ivec[0]), DES_ENCRYPT);
}

void
ssh_des_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		ssh_des_t *key_data)
{

	des_ncbc_encrypt(enc, clear, length, key_data->des_ks,
			 TO_CBLOCK(key_data->des_ivec[1]), DES_DECRYPT);
}
#endif /* WITH_CIPHER_DES */

#ifdef WITH_CIPHER_3DES
void
ssh_3des_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_3DES;
	c->initialize = (void *) ssh_3des_initialize;
	c->encrypt = (void *) ssh_3des_encrypt;
	c->decrypt = (void *) ssh_3des_decrypt;
	c->destroy = (void *) ssh_3des_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = NULL;
	c->mac_validate = NULL;
}

void
ssh_des3_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_DES3;
	c->initialize = (void *) ssh_des3_initialize;
	c->encrypt = (void *) ssh_des3_encrypt;
	c->decrypt = (void *) ssh_des3_decrypt;
	c->destroy = (void *) ssh_des3_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = (void *) ssh_des3_mac_generate;
	c->mac_validate = (void *) ssh_des3_mac_validate;
}

ssh_3des_t *
ssh_3des_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_3des_initialize);

	int i, j;
	des_cblock key[3];
	ssh_3des_t *key_data;

	if (klen < 16) {
		SSH_ERROR("Key length %d is insufficient for 3DES.\n", klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_3des_t));
	if (key_data == NULL)
		return NULL;

	for (i = j = 0; i < 3; i++) {
		memcpy(key[i], session_key + j, sizeof(des_cblock));
		des_set_odd_parity(TO_CBLOCK(key[i]));
		if (des_is_weak_key(TO_CBLOCK(key[i])))
			break;
		(void) des_set_key(TO_CBLOCK(key[i]), key_data->des_ks[i]);
		/*
		 * when keying from a passphrase (after md5) we will run
		 * out of keying material after two keys, so be *very*
		 * general about how big we expect the keying material
		 * to be.
		 */
		j += sizeof(des_cblock);
		if (j + sizeof(des_cblock) > klen)
			j = 0;
	}

	memset(key_data->des_ivec[0], 0, sizeof(key_data->des_ivec[0]));
	memset(key_data->des_ivec[1], 0, sizeof(key_data->des_ivec[1]));
	memset(key_data->des_ivec[2], 0, sizeof(key_data->des_ivec[2]));
	memset(key_data->des_ivec[3], 0, sizeof(key_data->des_ivec[3]));
	memset(key_data->des_ivec[4], 0, sizeof(key_data->des_ivec[4]));
	memset(key_data->des_ivec[5], 0, sizeof(key_data->des_ivec[5]));

	if (i == 3)		/* We did all three keys okay */
		return key_data;
	else {
		for (i = 0; i < 3; i++)
			memset(key[i], 0, sizeof(key[i]));
		memset(key_data, 0, sizeof(*key_data));
		free(key_data);
		return NULL;
	}
}

void
ssh_3des_destroy(ssh_3des_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

ssh_des3_t *
ssh_des3_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_des3_initialize);

	int i;
	des_cblock key;
	ssh_des3_t *key_data;
	u_int8_t key1ofb[24] = {
		0x10, 0x23, 0x66, 0x20, 0x10, 0x1d, 0xb7, 0x37,
		0x8d, 0xfb, 0xdf, 0xa9, 0x36, 0xff, 0xd6, 0xb1,
		0x45, 0x25, 0x2c, 0x34, 0xad, 0x8b, 0x10, 0xf8,
	};
	u_int8_t key2ofb[24] = {
		0x96, 0xa2, 0x1d, 0x63, 0xdc, 0xd1, 0x90, 0xfa,
		0x58, 0xee, 0xad, 0xeb, 0x2c, 0xb8, 0x86, 0x24,
		0xe4, 0xa0, 0x3e, 0x40, 0x02, 0x29, 0x8c, 0x13,
	};
	u_int8_t mac1ofb[16] = {
		0x43, 0x34, 0xdc, 0x87, 0x39, 0x3d, 0x8b, 0x09,
		0x0d, 0x10, 0xd4, 0xe7, 0x3c, 0x34, 0xdf, 0xb1,
	};
	u_int8_t mac2ofb[16] = {
		0x3f, 0x49, 0xf3, 0x04, 0x01, 0xfb, 0x43, 0xcb,
		0xef, 0x5f, 0xef, 0xdb, 0x15, 0x03, 0x95, 0x31,
	};
	BF_KEY bfkey;
	u_int8_t key1ivec[8] = {0x70, 0xae, 0x29, 0xe8, 0xab, 0x53, 0xff, 0x0f};
	u_int8_t key2ivec[8] = {0xc3, 0x1b, 0x41, 0x06, 0x88, 0x33, 0xfb, 0x27};
	u_int8_t mac1ivec[8] = {0x7e, 0xc7, 0x93, 0x99, 0x40, 0x3d, 0x41, 0x5c};
	u_int8_t mac2ivec[8] = {0x18, 0x29, 0x2b, 0x5b, 0xb6, 0x2c, 0xa0, 0xe8};

	u_int8_t *temp;

	int ofbnum;
	int weak = 0;

	if (klen != SSH_V1_SESSION_KEY_SIZE) {
		SSH_ERROR("Key length %d is incorrect for DES3.\n", klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_des3_t));
	if (key_data == NULL)
		return NULL;

	BF_set_key(&bfkey, klen, session_key);

	ofbnum = 0;
	BF_ofb64_encrypt(mac1ofb, mac1ofb, 16, &bfkey, mac1ivec, &ofbnum);

	ofbnum = 0;
	BF_ofb64_encrypt(mac2ofb, mac2ofb, 16, &bfkey, mac2ivec, &ofbnum);

	ofbnum = 0;
	BF_ofb64_encrypt(key1ofb, key1ofb, 24, &bfkey, key1ivec, &ofbnum);

	ofbnum = 0;
	BF_ofb64_encrypt(key2ofb, key2ofb, 24, &bfkey, key2ivec, &ofbnum);

	memset(key_data->des_ivec[0], 0, sizeof(key_data->des_ivec[0]));
	memset(key_data->des_ivec[1], 0, sizeof(key_data->des_ivec[1]));

	switch (role) {
	case SSH_ROLE_SERVER:

		memcpy(key_data->mac_key[0], mac1ofb, 16);
		memset(mac1ofb, 0, 16);

		memcpy(key_data->mac_key[1], mac2ofb, 16);
		memset(mac2ofb, 0, 16);

		temp = key1ofb;
		for (i = 0; i < 3; i++) {
			memcpy(&key, temp, sizeof(des_cblock));
			des_set_odd_parity(TO_CBLOCK(&key));
			if (des_is_weak_key(TO_CBLOCK(&key)))
				weak++;
			(void) des_set_key(TO_CBLOCK(&key),
					   key_data->des_ks[i]);
			temp += sizeof(des_cblock);
		}
		memset(&key, 0, sizeof(des_cblock));
		memset(key1ofb, 0, 24);

		temp = key2ofb;
		for (i = 3; i < 6; i++) {
			memcpy(&key, temp, sizeof(des_cblock));
			des_set_odd_parity(TO_CBLOCK(&key));
			if (des_is_weak_key(TO_CBLOCK(&key)))
				weak++;
			(void) des_set_key(TO_CBLOCK(&key),
					   key_data->des_ks[i]);
			temp += sizeof(des_cblock);
		}
		memset(&key, 0, sizeof(des_cblock));
		memset(key2ofb, 0, 24);
		break;
	case SSH_ROLE_CLIENT:
		memcpy(key_data->mac_key[1], mac1ofb, 16);
		memset(mac1ofb, 0, 16);

		memcpy(key_data->mac_key[0], mac2ofb, 16);
		memset(mac2ofb, 0, 16);

		temp = key2ofb;
		for (i = 0; i < 3; i++) {
			memcpy(&key, temp, sizeof(des_cblock));
			des_set_odd_parity(TO_CBLOCK(&key));
			if (des_is_weak_key(TO_CBLOCK(&key)))
				weak++;
			(void) des_set_key(TO_CBLOCK(&key),
					   key_data->des_ks[i]);
			temp += sizeof(des_cblock);
		}
		memset(&key, 0, sizeof(des_cblock));
		memset(key2ofb, 0, 24);

		temp = key1ofb;
		for (i = 3; i < 6; i++) {
			memcpy(&key, temp, sizeof(des_cblock));
			des_set_odd_parity(TO_CBLOCK(&key));
			if (des_is_weak_key(TO_CBLOCK(&key)))
				weak++;
			(void) des_set_key(TO_CBLOCK(&key),
					   key_data->des_ks[i]);
			temp += sizeof(des_cblock);
		}
		memset(&key, 0, sizeof(des_cblock));
		memset(key1ofb, 0, 24);
		break;
	}


	if (weak != 0) {
		memset(key_data, 0, sizeof(*key_data));
		free(key_data);
		return NULL;
	} else
		return key_data;
}


void
ssh_des3_destroy(ssh_des3_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

void
ssh_3des_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		 ssh_3des_t *key_data)
{

	des_ncbc_encrypt(clear, enc, length, key_data->des_ks[0],
			 TO_CBLOCK(key_data->des_ivec[0]), DES_ENCRYPT);
	des_ncbc_encrypt(enc, enc, length, key_data->des_ks[1],
			 TO_CBLOCK(key_data->des_ivec[1]), DES_DECRYPT);
	des_ncbc_encrypt(enc, enc, length, key_data->des_ks[2],
			 TO_CBLOCK(key_data->des_ivec[2]), DES_ENCRYPT);
}

void
ssh_3des_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		 ssh_3des_t *key_data)
{

	des_ncbc_encrypt(enc, clear, length, key_data->des_ks[2],
			 TO_CBLOCK(key_data->des_ivec[3]), DES_DECRYPT);
	des_ncbc_encrypt(clear, clear, length, key_data->des_ks[1],
			 TO_CBLOCK(key_data->des_ivec[4]), DES_ENCRYPT);
	des_ncbc_encrypt(clear, clear, length, key_data->des_ks[0],
			 TO_CBLOCK(key_data->des_ivec[5]), DES_DECRYPT);

}

void
ssh_des3_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		 ssh_des3_t *key_data)
{

	des_ede3_cbc_encrypt(clear, enc, length, key_data->des_ks[0],
			     key_data->des_ks[1], key_data->des_ks[2],
			     TO_CBLOCK(key_data->des_ivec[0]), DES_ENCRYPT);
}

void
ssh_des3_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		 ssh_des3_t *key_data)
{

	des_ede3_cbc_encrypt(enc, clear, length, key_data->des_ks[3],
			     key_data->des_ks[4], key_data->des_ks[5],
			     TO_CBLOCK(key_data->des_ivec[1]), DES_DECRYPT);

}

u_int8_t *
ssh_des3_mac_generate(u_int8_t *clear, ssh_des3_t *key_data,
		      int len)
{

	static u_int8_t digest[SHA_DIGEST_LENGTH];

	HMAC(EVP_sha1(), ((ssh_des3_t *) key_data)->mac_key[0],
	     SSH_V1_SESSION_KEY_SIZE / 2,
	     clear, len, digest, NULL);

	return digest;
}

int
ssh_des3_mac_validate(u_int8_t *clear, u_int8_t *mac,
		      ssh_des3_t *key_data, int len, int maclen)
{
	u_int8_t digest[SHA_DIGEST_LENGTH];

	(void) HMAC(EVP_sha1(), ((ssh_des3_t *) key_data)->mac_key[1],
		    SSH_V1_SESSION_KEY_SIZE / 2, clear, len, digest, NULL);

	if (!memcmp(mac, digest, maclen))
		return SSH_MAC_OK;
	else
		return SSH_MAC_BAD;
}
#endif /* WITH_CIPHER_3DES */

#ifdef WITH_CIPHER_RC4
void
ssh_rc4_attach(struct ssh_cipher * c)
{

	c->type = SSH_CIPHER_RC4;
	c->initialize = (void *) ssh_rc4_initialize;
	c->encrypt = (void *) ssh_rc4_encrypt;
	c->decrypt = (void *) ssh_rc4_decrypt;
	c->destroy = (void *) ssh_rc4_destroy;
	/*
	 * We're a stream cipher, but we still have to pad out the
	 * packets.
	 */
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = NULL;
	c->mac_validate = NULL;
}

ssh_rc4_t *
ssh_rc4_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_rc4_initialize);

	ssh_rc4_t *key_data;

	if (klen & 1) {
		SSH_ERROR("Key length %d incorrect for RC4.\n", klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_rc4_t));
	if (key_data == NULL)
		return (NULL);

	memset(key_data, 0, sizeof(*key_data));

	switch (role) {
	case SSH_ROLE_CLIENT:
		/*
		 * First half for receiving, second for sending.
		 */
		RC4_set_key(&key_data->r_recv, klen / 2, session_key);
		RC4_set_key(&key_data->r_send, klen / 2,
			    session_key + (klen / 2));
		break;

	case SSH_ROLE_SERVER:
		/*
		 * First half for sending, second for receiving.
		 */
		RC4_set_key(&key_data->r_recv, klen / 2,
			    session_key + (klen / 2));
		RC4_set_key(&key_data->r_send, klen / 2, session_key);
		break;

	case SSH_ROLE_NEITHER:
		/*
		 * Just use one key.
		 */
		RC4_set_key(&key_data->r_send, klen, session_key);
		break;

	default:
		SSH_ERROR("Invalid role %d\n", role);
		abort();
	}

	key_data->r_role = role;

	return (key_data);
}

void
ssh_rc4_destroy(ssh_rc4_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

void
ssh_rc4_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		ssh_rc4_t *key_data)
{

	/*
	 * "encrypt" is always "send".
	 */
	RC4(&key_data->r_send, length, clear, enc);
}

void
ssh_rc4_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		ssh_rc4_t *key_data)
{

	/*
	 * "decrypt" is always "receive", unless we have no role.
	 */
	RC4((key_data->r_role == SSH_ROLE_NEITHER) ? &key_data->r_send :
	    &key_data->r_recv, length, enc, clear);
}
#endif /* WITH_CIPHER_RC4 */

#ifdef WITH_CIPHER_BLOWFISH
void
ssh_blowfish_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_BLOWFISH;
	c->initialize = (void *) ssh_blowfish_initialize;
	c->encrypt = (void *) ssh_blowfish_encrypt;
	c->decrypt = (void *) ssh_blowfish_decrypt;
	c->destroy = (void *) ssh_blowfish_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = NULL;
	c->mac_validate = NULL;
}

void
ssh_fishblow_attach(struct ssh_cipher *c)
{

	c->type = SSH_CIPHER_FISHBLOW;
	c->initialize = (void *) ssh_fishblow_initialize;
	c->encrypt = (void *) ssh_fishblow_encrypt;
	c->decrypt = (void *) ssh_fishblow_decrypt;
	c->destroy = (void *) ssh_fishblow_destroy;
	c->block_size = 8;
	c->key_data = NULL;
	c->mac_generate = (void *) ssh_fishblow_mac_generate;
	c->mac_validate = (void *) ssh_fishblow_mac_validate;
}

ssh_blowfish_t *
ssh_blowfish_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_blowfish_initialize);

	ssh_blowfish_t *key_data;

	if (klen < 8) {
		SSH_ERROR("Key length %d is insufficient for Blowfish.\n",
			  klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_blowfish_t));
	if (key_data == NULL)
		return NULL;

	BF_set_key(&key_data->bf_ks, klen, session_key);

	memset(key_data->bf_iv[0], 0, sizeof(key_data->bf_iv[0]));
	memset(key_data->bf_iv[1], 0, sizeof(key_data->bf_iv[1]));

	return key_data;
}

void
ssh_blowfish_destroy(ssh_blowfish_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

ssh_fishblow_t *
ssh_fishblow_initialize(u_int8_t *session_key, int klen, int role)
{
	FUNC_DECL(ssh_fishblow_initialize);

	ssh_fishblow_t *key_data;
	u_int8_t digest[SHA_DIGEST_LENGTH];

	if (klen != SSH_V1_SESSION_KEY_SIZE) {
		SSH_ERROR("Key length %d is incorrect for Fishblow.\n", klen);
		return (NULL);
	}
	key_data = malloc(sizeof(ssh_fishblow_t));
	if (key_data == NULL)
		return NULL;

	switch (role) {

	case SSH_ROLE_SERVER:
		HMAC(EVP_sha1(), "0123456789ABCDEF", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		memcpy(key_data->mac_key[0], digest, 16);

		HMAC(EVP_sha1(), "123456789ABCDEF0", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		memcpy(key_data->mac_key[1], digest, 16);

		HMAC(EVP_sha1(), "23456789ABCDEF01", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		BF_set_key(&key_data->bf_ks[0], 16, digest);

		HMAC(EVP_sha1(), "3456789ABCDEF012", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		BF_set_key(&key_data->bf_ks[1], 16, digest);

		memset(digest, 0, SHA_DIGEST_LENGTH);
		break;

	case SSH_ROLE_CLIENT:
		HMAC(EVP_sha1(), "123456789ABCDEF0", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		memcpy(key_data->mac_key[0], digest, 16);

		HMAC(EVP_sha1(), "0123456789ABCDEF", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		memcpy(key_data->mac_key[1], digest, 16);

		HMAC(EVP_sha1(), "3456789ABCDEF012", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		BF_set_key(&key_data->bf_ks[0], 16, digest);

		HMAC(EVP_sha1(), "23456789ABCDEF01", 16, session_key,
		     SSH_V1_SESSION_KEY_SIZE, digest, NULL);
		BF_set_key(&key_data->bf_ks[1], 16, digest);
		memset(digest, 0, SHA_DIGEST_LENGTH);
		break;

	default:
		SSH_ERROR("ssh_fishblow_initialize: I am asymmetric!\n");
		abort();
	}

	memset(key_data->bf_iv[0], 0, sizeof(key_data->bf_iv[0]));
	memset(key_data->bf_iv[1], 0, sizeof(key_data->bf_iv[1]));

	return key_data;
}

void
ssh_fishblow_destroy(ssh_fishblow_t *key_data)
{

	memset(key_data, 0, sizeof(*key_data));
	free(key_data);
}

void
ssh_blowfish_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		     ssh_blowfish_t *key_data)
{

	FLIP_BYTES(clear, length);
	BF_cbc_encrypt(clear, enc, length, &key_data->bf_ks,
		       key_data->bf_iv[0], BF_ENCRYPT);
	FLIP_BYTES(enc, length);

}

void
ssh_blowfish_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		     ssh_blowfish_t *key_data)
{
	FLIP_BYTES(enc, length);
	BF_cbc_encrypt(enc, clear, length, &key_data->bf_ks,
		       key_data->bf_iv[1], BF_DECRYPT);
	FLIP_BYTES(clear, length);
}


void
ssh_fishblow_encrypt(u_int8_t *clear, u_int8_t *enc, int length,
		     ssh_fishblow_t *key_data)
{

	BF_cbc_encrypt(clear, enc, length, &(key_data->bf_ks[0]),
		       key_data->bf_iv[0], BF_ENCRYPT);

}

void
ssh_fishblow_decrypt(u_int8_t *enc, u_int8_t *clear, int length,
		     ssh_fishblow_t *key_data)
{

	BF_cbc_encrypt(enc, clear, length, &(key_data->bf_ks[1]),
		       key_data->bf_iv[1], BF_DECRYPT);
}

u_int8_t *
ssh_fishblow_mac_generate(u_int8_t *clear, ssh_fishblow_t *key_data,
			  int len)
{
	static u_int8_t digest[SHA_DIGEST_LENGTH];

	(void) HMAC(EVP_sha1(), key_data->mac_key[0],
		    SSH_V1_SESSION_KEY_SIZE / 2, clear, len, digest, NULL);

	return digest;
}

int
ssh_fishblow_mac_validate(u_int8_t *clear, u_int8_t *mac,
			  ssh_fishblow_t *key_data, int len,
			  int maclen)
{
	u_int8_t digest[SHA_DIGEST_LENGTH];

	(void) HMAC(EVP_sha1(), key_data->mac_key[1],
		    SSH_V1_SESSION_KEY_SIZE / 2, clear, len, digest, NULL);

	if (!memcmp(mac, digest, maclen))
		return SSH_MAC_OK;
	else
		return SSH_MAC_BAD;
}
#endif /* WITH_CIPHER_BLOWFISH */
