/* $Id: ssh_intro.c,v 1.17 2001/02/11 03:35:31 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000, 2001 Andrew Brown and Eric Haszlakiewicz.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Functions used during non-encrypted introduction phase
 * of the ssh connection.
 * 
 * These are used to negotiate the protocol version and to
 * initialize the transport layer.
 *
 * This file also currently includes a couple functions to kick
 * off the key exchange conversation once we've figured out what
 * binary protocol we're using.
 */

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <strings.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>

#include <openssl/md5.h>

#include "options.h"

#include "sshd.h"

#include "ssh_auth.h"
#include "ssh_util.h"
#include "ssh_transport.h"
#include "ssh_v1_messages.h"
#include "ssh_v1_proto.h"
#include "ssh_v2_proto.h"

#define RECV_BUFSIZE  100

static int check_version_compat(ssh_context_t *, int, int, char *,
								struct ssh_version *);
#ifdef WITH_PROTO_V2
static int check_v2_version(ssh_context_t *, int, char *,
							struct ssh_version *);
#endif
#ifdef WITH_PROTO_V1
static int check_v1_version(ssh_context_t *, int, char *,
							struct ssh_version *);
#endif

static void login_timeout(int);		/* Login timeout signal handler */

static ssh_context_t *l_context;	/* For login timeout signal handler */

/*
 * check_version_compat: check compatibility between versions.
 *  set any necessary emulation parameters.
 */
static int check_version_compat(ssh_context_t *context,
                                int major, int minor, char *softver,
								struct ssh_version *v)
{
	const char *c = "Protocol mismatch.";
	int retval;

#ifdef WITH_PROTO_V1_COMPAT
	if (major == V1_PROTO_MAJOR && minor == V2_PROTO_MINOR_COMPAT)
	{
		major = V2_PROTO_MAJOR;
		minor = 0;
	}
#endif

	switch(context->transport_layer)
	{
	  case SSH_V1_TRANSPORT:
		retval = v1_init_transport(&context->transport_ctx,
								   context->running_as_server);
		break;
	  case SSH_V2SSH_TRANSPORT:
#ifdef WITH_TRANSPORT_V2SSH
		retval = v2ssh_init_transport(&context->transport_ctx,
									  context->running_as_server);
#else
		retval = RET_FAIL;
#endif
		break;
	  case SSH_V2TLS_TRANSPORT:
#ifdef WITH_TRANSPORT_V2TLS
		retval = v2tls_init_transport(&context->transport_ctx,
									  context->running_as_server);
#else
		retval = RET_FAIL;
#endif
		break;
	  default:
		retval = RET_FAIL;
		break;
	}

	if (retval != RET_OK)
		c = "Unable to initialize transport layer\n";
	else
	{
		switch(major)
		{
#ifdef WITH_PROTO_V1
		  case V1_PROTO_MAJOR:
			if (check_v1_version(context, minor, softver, v) != 0)
			{
				c = "Unable to parse v1 version string";
				retval = RET_FAIL;
			}
			break;
#endif
#ifdef WITH_PROTO_V2
		  case V2_PROTO_MAJOR:
			if (check_v2_version(context, minor, softver, v) != 0)
			{
				c = "Unable to parse v2 version string";
				retval = RET_FAIL;
			}
			break;
#endif
		  default:
			c = "Unknown version\n";
			retval = RET_FAIL;
			break;
		}
	}

	if (retval != RET_OK)
	{
		/* got an error, return it  */
		SSH_ERROR(c);
		write(context->transport_ctx.commsock, c, strlen(c));
		retval = RET_FAIL;
	}

	return retval;
}

/*
 * check_v1_version:
 *
 *  Check for quirks in various V1 implementations.
 */
static int check_v1_version(ssh_context_t *context, int minor, char *softver,
							struct ssh_version *v)
{
	int s_major, s_minor, s_point;
	int s_date;
	int setlowsize;

	setlowsize = 0;

	if (strncasecmp(softver, "FreSSH", 6) == 0) {
		SSH_DLOG(4, ("FreSSH version string\n"));
		v->v_flags = SSH_FRESSH;
	}
	/* Look for an F-secure softver string. */
	/* They can't handle really big packets. */
	else if (sscanf(softver, "%d.%d.%d", &s_major, &s_minor, &s_point) == 3) {
		SSH_DLOG(4, ("F-SECURE version string\n"));
		v->v_flags = SSH_FSECURE;
		if (s_major > 1)
			;		/* Assume it's been fixed. */
		else if (s_major == 1 && s_minor > 2)
			;		/* Assume it's been fixed. */
		else
			setlowsize = 1;
	}
	else if (sscanf(softver, "TTSSH-%d.%d", &s_major, &s_minor) == 2)
	{
		SSH_DLOG(4, ("TTSSH version string\n"));
		v->v_flags = SSH_TTSSH;
		if (s_major > 1)
			;		/* Assume it's been fixed. */
		else if (s_major == 1 && s_minor > 2)
			;		/* Assume it's been fixed. */
		else
			setlowsize = 1;
	}
	else if (sscanf(softver, "NetBSD_Secure_Shell-%d", &s_date) == 1)
	{
		SSH_DLOG(4, ("NetBSD OpenSSH version string\n"));
		v->v_flags = SSH_OPENSSH;
		setlowsize = 1;	/* XXX Check the date string when openssh is fixed. */
	}
	if (setlowsize) {
		/*
		 * Set the maximum packet size to something small enough
		 * for the peer to handle.
		 */
		if (context->transport_ctx.t_v1.max_packet_size > SSH_CRCCOMP_MAX_PACKETSIZE)
			context->transport_ctx.t_v1.max_packet_size = SSH_CRCCOMP_MAX_PACKETSIZE;
		SSH_DLOG(4, ("max_packet_size set to %d\n",
		             context->transport_ctx.t_v1.max_packet_size));
	}
	return(0);
}
/*
 * check_v2_version:
 *    Check a version 2 identification string.
 *
 *  We default to the standard SSH transport layer
 * unless the software version string is of the
 * form "<foo>|TLS"
 */
#ifdef WITH_PROTO_V2
static int check_v2_version(ssh_context_t *context, int minor, char *softver,
							struct ssh_version *v)
{
#ifdef WITH_TRANSPORT_V2TLS
	char *xport_ver;
#endif

	v = v; /* this variable is now used */
	context->transport_layer = SSH_V2SSH_TRANSPORT;

#ifdef WITH_TRANSPORT_V2TLS
	xport_ver = strchr(softver, '|');
	if (xport_ver != NULL && (strlen(xport_ver) >= (3+1)) )
	{
		xport_ver[1] = toupper(xport_ver[1]);
		xport_ver[2] = toupper(xport_ver[2]);
		xport_ver[3] = toupper(xport_ver[3]);
		if (strcmp(&xport_ver[1], "TLS") == 0)
		{
			context->transport_layer = SSH_V2TLS_TRANSPORT;
		}
	}
#endif
	return(0);
}
#endif

/*
 * login_timeout: cleanup and exit after the login grace timer expires.
 */
static void login_timeout(int sig)
{
	SSH_DLOG(1, ("Login timeout.\n"));
	ssh_exit(l_context, 0, EXIT_NOW);
}

/*
 * send_version:
 *   sends our ssh version to the remote server or client.
 */
static int
send_version(ssh_context_t *context, const char *Who)
{
	char *verstr;
	/* Send version string ("SSH-#.#-foo") */
	verstr = build_version(context);
	while (write(context->transport_ctx.commsock, verstr,
				 strlen(verstr)) < 0) {
		if (errno == ENOBUFS) {
			sleep(1);
			continue;
		}
		SSH_DLOG(3,
		         ("%s disconnected before we could send the version.\n", Who));
		free(verstr);
		return(1);
	}
	free(verstr);
	return 0;
}

/*
 * get_version:
 *   talk to client (or server) and negotiate a protocol version.
 *   Based on this we also figure out the transport layer
 *   and initialize the appropriate one.
 */
int get_version(ssh_context_t *context, struct ssh_version *rv)
{
  char abuf[RECV_BUFSIZE];
  int count;
  int major, minor;
  char softver[RECV_BUFSIZE];
  struct itimerval itval;
  const char *who, *Who;
  struct ssh_version lrv;

	if (context->running_as_server == YES) {
		who = "client";
		Who = "Client";
	}
	else {
		who = "server";
		Who = "Server";
	}

	if (!rv)
		rv = &lrv;
	memset(rv, 0, sizeof(*rv));

	/*	
	 * the server sends its version as soon as the client connects.
	 * the client should wait until it has parsed the server's version
	 * string before it sends its own.
	 */
	if (context->running_as_server)
		if (send_version(context, Who))
			return 1;
		
	/*
	 * Set the login grace timeout alarm.
	 * This is the maximum time we can spend from here until
	 * when we have an actual authenticated user.
	 */
	if (context->running_as_server == YES && !is_debug_level(4)) {
		signal(SIGALRM, login_timeout);
		timerclear(&itval.it_interval);
		itval.it_value.tv_sec = 300;
		itval.it_value.tv_usec = 0;
		setitimer(ITIMER_REAL, &itval, NULL);
	}

	/* listen for response. */
	while ((count = read(context->transport_ctx.commsock, abuf,
						 RECV_BUFSIZE - 1)) < 0)
	{
		if (errno == EINTR)
			continue;
		SSH_DLOG(3, ("%s disconnected before sending its version.\n", Who));
		return(1);
	}

	if (count == (RECV_BUFSIZE - 1)) {
		SSH_DLOG(2, ("Caution: %s filled get_version recv buffer.\n", who));
	}
	abuf[count] = '\0';
	
	softver[99] = 0;
	if (sscanf(abuf, "SSH-%d.%d-%99[^\n]\n", &major, &minor, softver) != 3) {
		SSH_DLOG(4, ("%s version response didn't pass sscanf\n", Who));
		SSH_DLOG(4, ("buf=%s\n", abuf));
		major = -1;	/* force incompatibility. */
	}
	else {
		rv->v_major = major;
		rv->v_minor = minor;
		snprintf(rv->v_vendor, sizeof(rv->v_vendor), "%s", softver);
	}

	if (check_version_compat(context, major, minor, softver, rv) != 0) {
		/* Incompatible versions.  Error message has already been sent. */
		SSH_DLOG(3, ("Incompatible %s version %d.%d-%s\n", who,
					 major, minor, softver));
		return(-1);
	}
	
	SSH_DLOG(3, ("%s connected with version %d.%d-%s\n", Who,
				 major, minor, softver));

	if (! context->running_as_server)
		if (send_version(context, Who))
			return 1;

	return 0;
} /* end get_version() */

int start_v1_conversation(ssh_context_t *context)
{
	if (context->running_as_server == YES)
	{
		SSH_DLOG(4, ("start_v1_conversation\n"));
		if ((context->transport_ctx.kexinit)(context, NULL) != 0)
			return(1);
	}
	else
	{
		/* let's see...so far:
		 * - the server sent me it's version
		 * - i read it and sent mine
		 * - the server probably sent me its keys
		 * and that will be handled somewhere else
		 *
		 * so i do nothing here
		 */
	}
	return(0);
}

int start_v2_conversation(ssh_context_t *context)
{
	if (context->running_as_server == YES)
	{
		/* Send KEXINIT, other ? */
	}
	else
	{
		/* Send KEXINIT, other ? */
	}
	return(0);
}

/*
 * build_version: build an appropriate version string.
 *		caller must free the returned string.
 *
 * XXX this should be called by both client and server.
 * XXX Client should call this after it 
 * XAX Need to build |TLS version string.
 */
char *build_version(ssh_context_t *context)
{
	char *c;
	int major, minor;
	int is_server;

#define __VERSTRSIZE 30

	is_server = context->running_as_server;

	switch(context->protocol_version)
	{
#ifdef WITH_PROTO_V1
		case SSH_V1:
			major = V1_PROTO_MAJOR;
			minor = V1_PROTO_MINOR;
			break;
#endif
#ifdef WITH_PROTO_V2
		case SSH_V2:
			major = V2_PROTO_MAJOR;
			minor = V2_PROTO_MINOR;
#ifdef WITH_PROTO_V1_COMPAT
			if (is_server && context->opts.v1_compat_mode)
			{
				major = V1_PROTO_MAJOR;
				minor = V2_PROTO_MINOR_COMPAT;
			}
#endif
			break;
#endif
		default:
			major = 0;
			minor = 0;
			break;
	}

	c = malloc(sizeof(char) * __VERSTRSIZE);
	snprintf(c, __VERSTRSIZE, "SSH-%d.%d-%s\n",
				major, minor, SSHD_REV);
#undef __VERSTRSIZE
	return(c);
}
