/* $Id: ssh_v1_transport.c,v 1.43.2.1 2001/02/11 08:41:08 tls Exp $ */

/*
 * Copyright 1999 RedBack Networks, Incorporated.
 * All rights reserved.
 *
 * This software is not in the public domain.  It is distributed 
 * under the terms of the license in the file LICENSE in the
 * same directory as this file.  If you have received a copy of this
 * software without the LICENSE file (which means that whoever gave
 * you this software violated its license) you may obtain a copy from
 * http://www.panix.com/~tls/LICENSE.txt
 */

/*
 * Copyright (c) 2000, 2001 Andrew Brown, Eric Haszlakiewicz,
 *      and Jason Thorpe.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#define alloc_func		z_alloc_func
#define free_func		z_free_func
#include <zlib.h>
#undef alloc_func
#undef free_func

#include <openssl/rsa.h>

#include "options.h"

#define SSH_V1_MESSAGE_STRINGS

#include "sshd.h"

#include "ssh_auth.h"
#include "ssh_buffer.h"
#include "ssh_cipher.h"
#include "ssh_event.h"
#include "ssh_transport.h"
#include "ssh_rsakeys.h"
#include "ssh_util.h"
#include "ssh_types.h"
#include "ssh_parse.h"
#include "ssh_v1_messages.h"

static int v1_send_serverkeys(ssh_context_t *, struct ssh_buf *);
static int v1_recv_serverkeys(ssh_context_t *, struct ssh_buf *);
static int v1_init_compression(ssh_context_t *, u_int32_t);
static int v1_init_decompress(ssh_context_t *, u_int32_t);

static int v1_read_packet(ssh_context_t *, struct ssh_buf *);
static int v1_decode_packet(ssh_context_t *, struct ssh_buf *,
                         struct ssh_buf *, int, int);

static int v1_xmit_data(ssh_context_t *, u_int8_t, const u_int8_t *, size_t);
static int v1_xmit_packet(ssh_context_t *, const struct ssh_buf *);
static int v1_decode_sessionkey(ssh_context_t *, struct ssh_buf *, size_t);
static int v1_encode_sessionkey(ssh_context_t *, struct ssh_buf *);
static int v1_dispatch_transport(ssh_context_t *,
                                 struct ssh_buf *, size_t, int);

static int v1_set_max_packet_size(struct ssh_transport *, int);
static int v1_get_sendsize(struct ssh_transport *);

/*
 * Initialize the transport structure for 
 * the transport used in version 1 of the ssh protocol.
 */
int v1_init_transport(struct ssh_transport *xport, int isserver)
{
	FUNC_DECL(v1_init_transport);

	int flg, s;
	struct ssh_v1_transport *v1xport;

	SSH_DLOG(4, ("erasing v1 transport area\n"));

	v1xport = &xport->t_v1;

	s = xport->commsock;
	memset(xport, 0, sizeof(*xport));
	xport->commsock = s;

	xport->kexinit = isserver ? v1_send_serverkeys : v1_recv_serverkeys;
	xport->init_compression = v1_init_compression;
	xport->init_decompress = v1_init_decompress;
	xport->dispatch_transport = v1_dispatch_transport;

	xport->read_packet = v1_read_packet;
	xport->xmit_data = v1_xmit_data;
	xport->xmit_packet = v1_xmit_packet;

	xport->set_max_packet_size = v1_set_max_packet_size;
	xport->get_sendsize = v1_get_sendsize;

	v1xport->send_buf = buf_alloc(NULL, 2048);
	v1xport->clr_buf = buf_alloc(NULL, 2048);
	v1xport->compress_buf = buf_alloc(NULL, 2048);

	v1xport->max_packet_size = SSH_MAX_PACKET_SIZE;

	flg = fcntl(xport->commsock, F_GETFL);
	fcntl(xport->commsock, F_SETFL, flg & ~O_NONBLOCK);
	return(0);
}

/*
 * send_serverkeys: Create and send PUBLIC_KEY message.
 *		Determine session_id.
 */
static int v1_send_serverkeys(ssh_context_t *context, struct ssh_buf *bufp)
{
	FUNC_DECL(send_serverkeys);

	struct ssh_buf buf;
	struct ssh_event ev;
	struct ssh_ev_send *sendptr;
	u_int8_t *abuf, *apos;
	u_int32_t sup_ciphers, sup_auths;
	MD5_CTX md5ctx;
	int plen;

	SSH_DLOG(4, ("start\n"));

	memset(&buf, 0, sizeof(struct ssh_buf));
	memset(&ev, 0, sizeof(struct ssh_event));

	plen =
	/* cookie */
	8 +
	/* Server key bits, exponent and modulus. */
	4 + 2 + bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.serverkey)) +
	2 + bignum_num_bytes(ssh_rsa_epart(context->v1_ctx.serverkey)) +
	/* Host key bits, exponent and modulus. */
	4 + 2 + bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.hostkey)) +
	2 + bignum_num_bytes(ssh_rsa_epart(context->v1_ctx.hostkey)) +
	/* Protocol flags, supported ciphers and auths */
	4 + 4 + 4;

	if (buf_alloc(&buf, plen) == NULL) {
		SSH_ERROR("Unable to init PUBLIC_KEY packet.\n");
		return(1);
	}

	/* Generate the random cookie. */
	ssh_rand_bytes(8, context->v1_ctx.cookie);

	/*
	 * Build the packet:
	 */

	/* Copy in the cookie. */
	buf_put_nbytes(&buf, 8, context->v1_ctx.cookie);

	/* Drop in the server key. */
	buf_put_rsa_publickey(&buf, context->v1_ctx.serverkey);

	/* Drop in the host key. */
	buf_put_rsa_publickey(&buf, context->v1_ctx.hostkey);

	/* XXX no protocol flags for now? */
	buf_put_int32(&buf, 0);

	/* Fill in supported ciphers. */
	sup_ciphers = cipher_supported(context, -1);
	buf_put_int32(&buf, sup_ciphers);

	/* Fill in supported authentication methods. */
	sup_auths = auths_supported(context);
	buf_put_int32(&buf, sup_auths);

	/* Generate the session id: */
	plen = bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.serverkey)) +
	        bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.hostkey)) + 8;
	if ((abuf = malloc(plen)) == NULL) {
		SSH_ERROR("malloc for session_id failed: %s\n",
		          strerror(errno));
		buf_cleanup(&buf);
		return(1);
	}
	bignum_bn2bin(ssh_rsa_npart(context->v1_ctx.hostkey), abuf);
	apos = abuf + bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.hostkey));
	bignum_bn2bin(ssh_rsa_npart(context->v1_ctx.serverkey), apos);
	apos += bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.serverkey));
	memcpy(apos, context->v1_ctx.cookie, 8);

	MD5_Init(&md5ctx);
	MD5_Update(&md5ctx, abuf, plen);
	MD5_Final(context->v1_ctx.session_id, &md5ctx);
	free(abuf);

	/* transmit packet. */

	ev.event_type = SSH_EVENT_SEND;
	sendptr = (struct ssh_ev_send *) &ev.event_data;

	sendptr->dlen = 1 + buf_alllen(&buf);
	sendptr->data[0] = SSH_V1_SMSG_PUBLIC_KEY;
	memcpy(&sendptr->data[1], buf_alldata(&buf), buf_alllen(&buf));
	if (ssh_sys_sendevent(context, &ev) < 0) {
		SSH_DLOG(1, ("Unable to push public key to send thread\n"));
		buf_cleanup(&buf);
		return(1);
	}
	buf_cleanup(&buf);
	return(0);
}

static int v1_recv_serverkeys(ssh_context_t *context, struct ssh_buf *bufp)
{
	FUNC_DECL(recv_serverkeys);

	u_int8_t *abuf, *apos, *trash;
	u_int32_t sup_ciphers, sup_auths, sbits, hbits;
	MD5_CTX md5ctx;
	int plen;
	struct ssh_buf *skbuf;
	struct ssh_event ev;
	struct ssh_ev_send *evsend;
	struct ssh_ev_init_crypt *evcry;

	SSH_DLOG(4, ("start\n"));
	abuf = apos = trash = NULL;

	/* Copy out the cookie. */
	buf_get_nbytes(bufp, 8, &trash);
	memcpy(context->v1_ctx.cookie, trash, 8);
	free(trash);

	/* Suck out the server key. */
	if (!context->v1_ctx.serverkey)
	    context->v1_ctx.serverkey = ssh_rsa_new();
	buf_get_rsa_publickey(bufp, context->v1_ctx.serverkey, &sbits);

	/* Suck out the host key. */
	if (!context->v1_ctx.hostkey)
	    context->v1_ctx.hostkey = ssh_rsa_new();
	buf_get_rsa_publickey(bufp, context->v1_ctx.hostkey, &hbits);

	if (context->client->Verbosity)
		fprintf(stderr, "%s: Received server public key (%d bits) "
				"and host key (%d bits).\r\n",
				context->client->localhost, sbits, hbits);

	/*
	 * note that for the host key, hbits is *possibly* not the same as
	 * bignum_num_bits(ssh_rsa_npart(context->v1_ctx.hostkey))
	 */

	verify_host_key(context, hbits, context->v1_ctx.hostkey);

	/* XXX no protocol flags for now? */
	buf_get_int32(bufp, &plen);

	/* Retrieve supported ciphers. */
	buf_get_int32(bufp, &sup_ciphers);
	/* Only try to use ciphers we both support. */
	context->supported_ciphers &= sup_ciphers;

	/* wibble */
	if (context->client->Cipher == SSH_CIPHER_BLOWFISH &&
		context->supported_ciphers & (1 << SSH_CIPHER_FISHBLOW)) {
		if (context->client->Verbosity)
			fprintf(stderr, "%s: Server has %s, using that instead of %s\r\n",
					context->client->localhost,
					cipher_name(SSH_CIPHER_FISHBLOW),
					cipher_name(context->client->Cipher));
		context->client->Cipher = SSH_CIPHER_FISHBLOW;
	}
	else if (context->client->Cipher == SSH_CIPHER_3DES &&
		context->supported_ciphers & (1 << SSH_CIPHER_DES3)) {
		if (context->client->Verbosity)
			fprintf(stderr, "%s: Server has %s, using that instead of %s\r\n",
					context->client->localhost,
					cipher_name(SSH_CIPHER_DES3),
					cipher_name(context->client->Cipher));
		context->client->Cipher = SSH_CIPHER_DES3;
	}
	else if (! context->supported_ciphers & (1 << context->client->Cipher)) {
		fprintf(stderr, "Selected cipher type %s not supported by server.\r\n",
				cipher_name(context->client->Cipher));
		exit(1);
	}

	/* Get supported authentication methods. */
	buf_get_int32(bufp, &sup_auths);

	/*
	 * We could clamp down the auths that we support, but that might
	 * not be intuitive to the user; they might be looking at verbose
	 * output to see why their login is failing...
	 */
#if 0
	context->supported_auths &= sup_auths;
#endif

	/* Generate the session id: */
	plen = bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.serverkey)) +
	    bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.hostkey)) + 8;
	if ((abuf = malloc(plen)) == NULL) {
		SSH_ERROR("malloc for session_id failed: %s\n",
				  strerror(errno));
		return(1);
	}
	bignum_bn2bin(ssh_rsa_npart(context->v1_ctx.hostkey), abuf);
	apos = abuf + bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.hostkey));
	bignum_bn2bin(ssh_rsa_npart(context->v1_ctx.serverkey), apos);
	apos += bignum_num_bytes(ssh_rsa_npart(context->v1_ctx.serverkey));
	memcpy(apos, context->v1_ctx.cookie, 8);

	MD5_Init(&md5ctx);
	MD5_Update(&md5ctx, abuf, plen);
	MD5_Final(context->v1_ctx.session_id, &md5ctx);
	free(abuf);

	/* generate and send the session key */
	ssh_rand_bytes(SSH_V1_SESSION_KEY_SIZE, context->v1_ctx.session_key);
	skbuf = buf_alloc(NULL, 256); /* 256 is enough, yesh? */
	v1_encode_sessionkey(context, skbuf);
	ev.event_type = SSH_EVENT_SEND;
	evsend = (struct ssh_ev_send *) ev.event_data;
	evsend->dlen = 1 + buf_alllen(skbuf);
	evsend->data[0] = SSH_V1_CMSG_SESSION_KEY;
	memcpy(&evsend->data[1], buf_alldata(skbuf), buf_alllen(skbuf));
	buf_cleanup(skbuf);
	free(skbuf);
    EVT_SEND(&ev, context);

	if (context->client->Verbosity)
		fprintf(stderr, "%s: Sent encrypted session key.\r\n",
				context->client->localhost);

	/*
	 * now that the session key has been sent, we will (naively)
	 * assume that everything from here will be encrypted.
	 */
    if (set_cipher_type(context->cipher, context->client->Cipher) != 0) {
        SSH_DLOG(1, ("Unable to set cipher type.\n"));
        return(1);
    }

	/*
	 * Tell the send thread to start encrypting
	 */
	ev.event_type = SSH_EVENT_INIT_CRYPT;
	evcry = (struct ssh_ev_init_crypt *)ev.event_data;
	evcry->cipher_type = context->cipher->type;
	memcpy(evcry->cipher_data, context->v1_ctx.session_key,
		   SSH_V1_SESSION_KEY_SIZE);
	EVT_SEND(&ev, context);

    if (cipher_initialize(context->cipher, context->v1_ctx.session_key,
			  SSH_V1_SESSION_KEY_SIZE,
			  context->running_as_server) == NULL) {
        SSH_DLOG(1, ("Unable to initialize cipher:%s\n", 
                     cipher_name(context->client->Cipher)));
        return(1);
    }
	context->v1_ctx.state = SSH_V1_STATE_GETUSER;
	
	return(0);
}

/*
 * v1_init_compression 
 *   Sets up compression.  Calls deflateInit.
 *     init_compression should always be called after init_decompress
 *   since init_decompress does any sanity checks.
 *
 */
static int v1_init_compression(ssh_context_t *context, u_int32_t clevel)
{
	FUNC_DECL(v1_init_compression);

	SSH_DLOG(5, ("Compression level: %d\n", clevel));
#ifdef WITH_COMPRESSION
	memset(&context->transport_ctx.t_v1.inz, 0, sizeof(z_stream));
	/* don't let the client eat all of our CPU compressing. */
	deflateInit(&(context->transport_ctx.t_v1.outz), clevel > 5 ? 5 : clevel);

	context->transport_ctx.t_v1.compressing = 1;
	return(RET_OK);
#else
	/* This shouldn't happen! */
	SEND_DISCONNECT(context, "Compression not supported.");
	return(RET_FATAL);
#endif
}

/*
 * v1_init_decompress:
 *   Sets up decompression.  Calls inflateInit.
 *   Called from recvThread only 
 */
static int v1_init_decompress(ssh_context_t *context, u_int32_t clevel)
{
	FUNC_DECL(v1_init_decompress);
	struct ssh_event ev;

	SSH_DLOG(5, ("Compression level: %d\n", clevel));
#ifdef WITH_COMPRESSION
	if (clevel < 1 || clevel > 9)
	{
		char tmpmsg[20];
		snprintf(tmpmsg, sizeof(tmpmsg), "Compression level: %d.", clevel);
		EVT_V1_DEBUG(ev, context, tmpmsg);
		EVT_SEND(&ev, context);

		EVT_V1_DISCONNECT(ev, context, "Invalid compression level");
		EVT_SEND(&ev, context);

		ssh_exit(context, 0, EXIT_FLUSH);
	}

	if (context->transport_ctx.t_v1.compressing) {
		EVT_V1_DISCONNECT(ev, context, "Already compressing!");
		EVT_SEND(&ev, context);

		ssh_exit(context, 0, EXIT_FLUSH);
	}

	memset(&context->transport_ctx.t_v1.outz, 0, sizeof(z_stream));

	inflateInit(&(context->transport_ctx.t_v1.inz));

	context->transport_ctx.t_v1.compressing = 1;
	return(RET_OK);
#else		/* WITH_COMPRESSION */

	EVT_V1_DEBUG(ev, context, "Compression not supported.");
	EVT_SEND(&ev, context);

	return(RET_FAIL);
#endif		/* WITH_COMPRESSION */
}

/*
 * v1_dispatch_transport:
 *  Called from recvThread only
 *    Dispatch transport layer messages.  For simplicity, the only
 *    message we actually handle here is CMSG_SESSION_KEY.  
 * 
 * Assume: buf is discarded after return.
 *
 * Returns:
 *	RET_NEXT_STATE   Transport layer all set to go.
 *  RET_FAIL         Nope.
 */
static int v1_dispatch_transport(ssh_context_t *context, struct ssh_buf *buf,
                                 size_t size, int msg_type)
{
	struct ssh_event ev;
	struct ssh_ev_init_crypt *evcry;
#ifdef SESSION_KEY_TIMING
	struct timeval scratch;
	char msg[104];
#endif /* SESSION_KEY_TIMING */

	if (msg_type != SSH_V1_CMSG_SESSION_KEY)
	{
		EVT_V1_DISCONNECT(ev, context, "Expected CMSG_SESSION_KEY\n");
		EVT_SEND(&ev, context);
		return(RET_FAIL);
	}

	if (v1_decode_sessionkey(context, buf, size) != 0)
		return(RET_FAIL);

#ifdef SESSION_KEY_TIMING
	gettimeofday(&scratch, NULL);
	if ((scratch.tv_usec -= tv_start.tv_usec) < 0) {
		scratch.tv_usec += 1000000;
		scratch.tv_sec --;
	}
	scratch.tv_sec -= tv_start.tv_sec;
	snprintf(msg, 104, "Accepted session key: %ld.%06ld secs",
		 scratch.tv_sec, scratch.tv_usec);
	SSH_DLOG(1, ("%s", msg));
#endif /* SESSION_KEY_TIMING */

	/*
	 * Tell the send thread to start encrypting
	 */
	ev.event_type = SSH_EVENT_INIT_CRYPT;
	evcry = (struct ssh_ev_init_crypt *)ev.event_data;
	evcry->cipher_type = context->cipher->type;
	memcpy(evcry->cipher_data, context->v1_ctx.session_key,
	       SSH_V1_SESSION_KEY_SIZE);
	EVT_SEND(&ev, context);

	/* Tell client we're ready to go. */
	EVT_V1_SUCCESS(ev);
	if (ssh_sys_sendevent(context, &ev) < 0)
	{
		SSH_ERROR("Unable to send encryption-on success event\n");
		return(RET_FAIL);
	}

#ifdef SESSION_KEY_TIMING
	EVT_V1_DEBUG(ev, context, msg);
#endif /* SESSION_KEY_TIMING */

	return(RET_NEXT_STATE);
}


/*
 * v1_decode_sessionkey: Decode SESSION_KEY message from client.
 *
 */
static int v1_decode_sessionkey(ssh_context_t *context, struct ssh_buf *buf,
                                size_t size)
{
	FUNC_DECL(decode_sessionkey);

	struct ssh_mpint enc_session_key;
	u_int8_t cipher_type;
	u_int32_t pflags;
	u_int8_t *cookie;

	SSH_DLOG(5, ("start\n"));

	cookie = NULL;
	memset(&enc_session_key, 0, sizeof(enc_session_key));

	/* Get data out of packet. */
	/* Note: be sure to free enc_session_key->data and cookie. */
	if (buf_get_byte(buf, &cipher_type) != 0 ||
	    buf_get_nbytes(buf, SSH_V1_COOKIE_SIZE, &cookie) != 0 ||
	    buf_get_mpint(buf, &enc_session_key) != 0 ||
	    buf_get_int32(buf, &pflags) != 0)
	{
		SSH_DLOG(1, ("packet error.\n"));
		if (cookie)
			free(cookie);
		if (enc_session_key.data)
			free(enc_session_key.data);
		return(1);
	}

	if (!cipher_supported(context, cipher_type)) {
		SSH_DLOG(1, ("Client tried to use unsupported cipher type.\n"));
		free(enc_session_key.data);
		free(cookie);
		return(1);
	}

	if (memcmp(cookie, context->v1_ctx.cookie, SSH_V1_COOKIE_SIZE) != 0) {
		SSH_DLOG(1, ("Non-matching cookie returned\n"));
		free(enc_session_key.data);
		free(cookie);
		return(1);
	}
	free(cookie);

	/* XXX check pflags?   Nothing to do with it for now. */

	if (decrypt_session_key(context, &enc_session_key) != 0) {
		SSH_DLOG(1, ("Failed to decrypt session key.\n"));
		memset(enc_session_key.data, 0, (enc_session_key.bits + 7) / 8);
		free(enc_session_key.data);
		return(1);
	}
	memset(enc_session_key.data, 0, (enc_session_key.bits + 7) / 8);
	free(enc_session_key.data);

	if (set_cipher_type(context->cipher, cipher_type) != 0)
	{
		SSH_DLOG(1, ("Unable to set cipher type.\n"));
		return(1);
	}

	if (cipher_initialize(context->cipher, context->v1_ctx.session_key,
			      SSH_V1_SESSION_KEY_SIZE,
			      context->running_as_server) == NULL)
	{
		SSH_DLOG(1, ("Unable to initialize cipher:%s\n", 
		             cipher_name(cipher_type)));
		return(1);
	}

	SSH_DLOG(5, ("finished.\n"));
	return(0);
}

/*
 * v1_encode_sessionkey: Encrypt and encode a (previously made)
 * SESSION_KEY message for the server, and stuff it in the buffer.
 *
 */
static int v1_encode_sessionkey(ssh_context_t *context, struct ssh_buf *buf)
{
	FUNC_DECL(encode_sessionkey);

	struct ssh_mpint enc_session_key;
	u_int8_t cipher_type = context->client->Cipher;
	u_int32_t pflags = 0; /* nothing there */

	SSH_DLOG(5, ("start\n"));

	enc_session_key.bits = 0;
	enc_session_key.data = NULL;
	encrypt_session_key(context, &enc_session_key);

	if (context->client->Verbosity)
		fprintf(stderr, "%s: Encryption type: %s\r\n",
				context->client->localhost,
				cipher_name(context->client->Cipher));

	buf_put_byte(buf, cipher_type);
	buf_put_nbytes(buf, SSH_V1_COOKIE_SIZE, context->v1_ctx.cookie);
	buf_put_mpint(buf, &enc_session_key);
	buf_put_int32(buf, pflags);

	memset(enc_session_key.data, 0, (enc_session_key.bits + 7) / 8);
	free(enc_session_key.data);

	SSH_DLOG(5, ("finished.\n"));
	return(0);
}

/*
 * Reads from the socket into buf.
 * Not thread safe, should only be used by recvThread.
 *
 * Returns:
 *		-1 on error
 *		>0 Enough data to form a complete packet.
 *			Returns amount of data in the read buffer.
 *		0 Not enough data yet.
 *
 * clr_buf will never have more than one packet worth of data returned in it.
 * clr_buf will always be cleared before anything is placed in it.
 */
static int v1_read_packet(ssh_context_t *context, struct ssh_buf *clr_buf)
{
#define INITIAL_FILLBUF_SIZE	(32 * 1024)
	int read_len, pad_len, dlen;
	int len_needed;  /* Total amount of data (w/o length field) for packet */
	int amt_to_read; /* Amount we still need to read */
	struct ssh_transport *xport;

	xport = &context->transport_ctx;

	if (context->v1_ctx.read_buf == NULL)
	{
		if ((context->v1_ctx.read_buf =
		      buf_alloc(NULL, INITIAL_FILLBUF_SIZE)) == NULL)
		{
			SSH_ERROR("Unable to alloc memory for read");
		}
	}

	dlen = 0;

	/* Check if we already have enough data to grab the length. */
	if (buf_len(context->v1_ctx.read_buf) >= SSH_MIN_PACKET_SIZE)
	{
		buf_get_int32(context->v1_ctx.read_buf, &dlen);
		pad_len = 8 - (dlen % 8);

		SSH_DLOG(4, ("initial dlen,padlen: %d %d\n", dlen, pad_len));

		if ((4 + pad_len + dlen) > SSH_MAX_PACKET_SIZE)
		{
			SSH_DLOG(1, ("Packet length too big: %d\n", dlen));
			errno = E2BIG;
			return(-1);
		}
	} else
		pad_len = 0;

	while(!dlen || (buf_len(context->v1_ctx.read_buf) < (dlen + pad_len)) )
	{
		if (dlen)
		{
			/* Is the length too big? */
			if ((4 + pad_len + dlen) > SSH_MAX_PACKET_SIZE)
			{
				SSH_DLOG(1, ("Packet length too big: %d\n", dlen));
				errno = E2BIG;
				return(-1);
			}
		}

		/* Figure out how much more room we're going to need */
		len_needed = dlen ? (pad_len + dlen) : SSH_MIN_PACKET_SIZE;
		amt_to_read = len_needed - buf_len(context->v1_ctx.read_buf);

		/* Make sure we're got enough room to read into. */
		if (amt_to_read > buf_avail(context->v1_ctx.read_buf))
		{
			SSH_DLOG(4, ("Not enough room in buffer have %d, need %d total\n",
						 buf_avail(context->v1_ctx.read_buf), amt_to_read));

			/* Allocate more space for the input buffer, if needed. */
			if (buf_makeavail(context->v1_ctx.read_buf, amt_to_read) != 0)
			{
				int tmp_err = errno;
				SSH_ERROR("Unable to make buffer space available %d: %s\n",
				          dlen, strerror(errno));
				errno = tmp_err;
				return(-1);
			}
		}

		read_len = buf_fillbuf(read, context->v1_ctx.read_buf, xport->commsock,
		                       amt_to_read);
		if (read_len < 0 && errno != EAGAIN)
			return(-1);
		if (read_len == 0) {
			SSH_DLOG(4, ("EOF\n"));
			errno = EPIPE;
			return(-1);
		}

		SSH_DLOG(4, ("Read data length: %d\n", read_len));

		if (context->send_pid == -1)
		{
			/* Whoops, send thread is gone.  Don't bother with anything else */
			SSH_DLOG(4, ("sendThread gone, returning -1\n"));
			return(-1);
		}

		/*
		 * Grab data-length-w/o-padding field from the packet.
		 * Note: this includes packet type, data, and CRC.
		 */
		if (!dlen)
		{
			/*
			 * Not enough to figure out the data length yet?
			 * (Note that the min packet size includes the length)
			 */
			if (buf_len(context->v1_ctx.read_buf) < SSH_MIN_PACKET_SIZE)
			{
				SSH_DLOG(4, ("Ultra-short packet.  Saving.\n"));
				continue;		/* try reading more data. */
			}
	
			buf_get_int32(context->v1_ctx.read_buf, &dlen);
			pad_len = 8 - (dlen % 8);
			SSH_DLOG(4, ("got dlen,pad_len: %d %d\n", dlen, pad_len));
		}

	} /* while(enough data) */

	if (context->send_pid == -1)
	{
		SSH_DLOG(4, ("sendThread gone, returning -1\n"));
		return(-1);
	}

	/* -- Now we know that there's enough data for an entire packet. -- */
	if (v1_decode_packet(context, context->v1_ctx.read_buf, clr_buf,
	                     pad_len, dlen) < 0)
	{
		int tmp_err = errno;
		SSH_ERROR("Unable to decode packet");
		errno = tmp_err;
		return(-1);
	}

	return(1);
}


/*
 * Decode the next packet from the p_enc buffer into the p_clr buffer.
 *
 * p_enc_buf should be the encrypted portion of the packet and
 * the CRC, but not the length.
 *
 * Decrypts, checksums and decompresses into p_clr.
 * DEBUG and IGNORE packets are discarded.
 *
 * Returns:	-1 on error
 *		0 if no packet.  (Not enough data)
 *		>0 if a packet is ready.  (length of packet)
 */
static int v1_decode_packet(ssh_context_t *context, struct ssh_buf *p_enc_buf,
                            struct ssh_buf *p_clr_buf, int pad_len, int dlen)
{
	u_int32_t crc;
	struct ssh_buf *p_tmp_clr;
	static struct ssh_buf tmp_buf;
	static int first_time = YES;

	struct ssh_cipher *cipher;
	struct ssh_v1_transport *xport;

	assert(p_enc_buf != NULL);
	assert(p_clr_buf != NULL);
	assert(&context->v1_ctx != NULL);

	cipher = context->cipher;
	xport = &context->transport_ctx.t_v1;

	buf_reset(p_clr_buf);

	if (is_debug_level(6)) {
		SSH_DLOG(5, ("Encrypted packet:\n"));
		buf_printn(p_enc_buf, pad_len + dlen);
	}

	/* XXX this code is kind of clumsy */
	if (first_time)
	{
		memset(&tmp_buf, 0, sizeof(tmp_buf));
		first_time = NO;
	}
	buf_reset(&tmp_buf);

	/* Allocate space for decryption. */
	/* If compressing decrypt into an extra buffer. */
	if (xport->compressing)
		p_tmp_clr = buf_alloc(&tmp_buf,  pad_len + dlen);
	else
		p_tmp_clr = buf_alloc(p_clr_buf, pad_len + dlen);

	if (p_tmp_clr == NULL)
	{
		SSH_ERROR("buf_alloc for decryption failed: %s\n",
		          strerror(errno));
		return(-1);
	}

	/* We feed each encrypted packet to a function which detects Ariel 
	   Futoransky's CRC compensation attack. */

	if (cipher->type != SSH_CIPHER_NONE && !cipher->mac_generate)
	{
		if (detect_attack(buf_data(p_enc_buf), dlen + pad_len, NULL) != 0)
		{
			SEND_DISCONNECT(context,
			                "CRC compensation attack detected, goodbye!");
			return(-1);
		}
	}

	/* Decrypt from the input buffer into the decrypted packet buffer. */
	if (cipher->decrypt)
		cipher_decrypt(cipher, buf_data(p_enc_buf),
			       buf_data(p_tmp_clr), dlen + pad_len);
	else
		memcpy(buf_data(p_tmp_clr), buf_data(p_enc_buf), dlen+pad_len);

	/* Fix the length of the decrypted buffer. */
	if (buf_adjlen(p_tmp_clr, dlen + pad_len) != 0)
	{
		SSH_ERROR("unable to adj decrypted buf len: %s\n", strerror(errno));
		buf_cleanup(p_tmp_clr);
		return(-1);
	}

	if (is_debug_level(6)) {
		fprintf(stderr, "Decrypted packet:\r\n");
		buf_print(p_tmp_clr);
	}

	/* Remove this packet from the input buffer. */
	buf_get_skip(p_enc_buf, dlen + pad_len);	/* Already got the len field */

	{
		u_int32_t packet_crc;

		buf_get_skip(p_tmp_clr, pad_len + dlen -4);
		buf_get_int32(p_tmp_clr, &packet_crc);
		buf_rewind(p_tmp_clr);

		if(cipher->mac_validate)
		{

			if (cipher_mac_validate(cipher,
						buf_data(p_tmp_clr),
						(u_int8_t *)&packet_crc,
						pad_len + dlen - 4, 4) !=
						SSH_MAC_OK)
			{
				SSH_DLOG(2, ("Bad Authenticator!\n"));
				return(-1);
			}
		} else {
			/* Calculate crc.  dlen is length of type, data and crc: */

			crc = ssh_crc32(buf_data(p_tmp_clr), pad_len + dlen - 4);
			if (crc != packet_crc)
			{
				SSH_DLOG(2, ("Bad CRC. %d\n", crc));
				return(-1);
			}
		}
	}

	buf_adjlen(p_tmp_clr, -4);		/* cut CRC */

	/* Skip the padding. */
	SSH_DLOG(5, ("padlen:%d\n", pad_len));
	buf_get_skip(p_tmp_clr, pad_len);

	/* If compressing, decompress into p_clr_buf */
	if (xport->compressing)
	{
		int ret;

		/* Allocate space for the decrypted data. */
		if (buf_alloc(p_clr_buf, SSH_MAX_PACKET_SIZE) == NULL) {
			SSH_ERROR("buf_alloc for inflate failed: %s\n",
								strerror(errno));
			return(-1);
		}
		xport->inz.next_in = buf_data(p_tmp_clr);/* After padding. */
		xport->inz.avail_in = dlen - 4;	/* Only type and data, but NOT! CRC */
		xport->inz.next_out = buf_data(p_clr_buf);
		xport->inz.avail_out = buf_size(p_clr_buf);

		do {
			ret = inflate(&(xport->inz), Z_PARTIAL_FLUSH);
			if (ret == Z_OK)
				continue;
			if (ret != Z_BUF_ERROR)
			{
				SSH_DLOG(1, ("inflate failed: %d\n", ret));
				return(-1);
			}
			if (xport->inz.avail_out == 0)
			{
				SSH_DLOG(1, ("inflate out of output space.\n"));
				return(-1);
			}
		} while (ret != Z_BUF_ERROR) ;

		if (xport->inz.avail_in != 0) {
			SSH_DLOG(1, ("incomplete inflation: %d\n", 
						xport->inz.avail_in));
			return(-1);
		}

		/* Fix the length of the buffer. */
		buf_adjlen(p_clr_buf, xport->inz.next_out - buf_data(p_clr_buf));

		/* Cleanup compressed cleartext buffer. */
		buf_cleanup(p_tmp_clr);
	}
	/* -- p_clr should now contain the packet type and data. */
	/* -- it may also contain padding before this, */
	/* -- but w/ get_loc set correctly */

	/* All done */
	return(buf_alllen(p_clr_buf));
}

/*
 * Packet layout:
 *	32-bit packet length, not including padding and length.
 *	1-8 bytes of padding.
 *	8-bit packet type.
 *	data.
 *	32-bit crc.
 */

static int v1_xmit_data(ssh_context_t *context,
                        u_int8_t msg_type, const u_int8_t *rawbuf, size_t len)
{
  struct ssh_buf buf;
  int ret;

	memset(&buf, 0, sizeof(struct ssh_buf));

	/* Turn the raw data into a ssh_buf. */
	if (buf_alloc(&buf, 1 + 4 + len) == NULL)
	{
		SSH_ERROR("Unable to buf_alloc(%d): %s\n", len, strerror(errno));
		return(-1);
	}
	if ((buf_put_int8(&buf, msg_type) != 0) ||
	    (len > 0 && buf_put_binstr(&buf, rawbuf, len) != 0))
	{
		SSH_ERROR("Unable to bufferize data: %s\n", strerror(errno));
		buf_cleanup(&buf);
		return(-1);
	}

	/* Send it. */
	ret = v1_xmit_packet(context, &buf);
	buf_cleanup(&buf);

	return(ret);
}


/*
 * Takes a buffer, turns it into a packet and starts xmitting it.
 * The packet in transit is stored in the context.  If packet_done(context->pc)
 * is 0 then this must not be called with a non-null buf.  If it is 0
 * then calling with a NULL buf will send more of it.
 *
 * buf is returned unchanged.
 *
 * Returns:
 *    0    Packet sent
 *    -1   Fatal error.
 *    1    EPIPE on write: other side is gone.
 */

static int v1_xmit_packet(ssh_context_t *context, const struct ssh_buf *buf)
{
	int ret;
	struct ssh_buf *newb, *cb;
	int padlen, size;
	u_int32_t crc, len;
	struct ssh_v1_transport *xport;

	xport = &context->transport_ctx.t_v1;
	cb = NULL;

	if (buf) {
#ifndef SSH_V1_MESSAGE_STRINGS
		SSH_DLOG(4, ("type=%d len=%d\n", buf_data(buf)[0],
		    buf_alllen(buf)));
#else
		SSH_DLOG(4, ("type=%s (%d) len=%d\n",
		    v1_msg_name[buf_data(buf)[0]], buf_data(buf)[0],
		    buf_alllen(buf)));
#endif /* SSH_V1_MESSAGE_STRINGS */
	}

	if (xport->compressing)
	{
		if (is_debug_level(6))
		{
			SSH_DLOG(6, ("xmit_packet/uncompressed: "));
			buf_print(buf);
		}

		/* Set up the output for the compression. */
		size = (buf_alllen(buf) + sizeof(u_int8_t)) * 1.001 + 12;
		if ((cb = buf_alloc(xport->compress_buf, size)) == NULL)
		{
			SSH_ERROR("buf_alloc for deflate failed: %s\n", strerror(errno));
			return(-1);
		}
		cb = xport->compress_buf;
		xport->outz.next_out = buf_data(cb);
		xport->outz.avail_out = size;

		/* Toss in the packet data. */
		xport->outz.next_in = buf_alldata(buf);
		xport->outz.avail_in = buf_alllen(buf);
		if ((ret = deflate(&(xport->outz), Z_PARTIAL_FLUSH)) != Z_OK)
		{
			SSH_ERROR("deflate #2 failed: %d\n", ret);
			buf_clear(cb);
			return(-1);
		}

		/* Set the length of compressed data. */
		buf_adjlen(cb, size - xport->outz.avail_out);
		buf = cb;
	}
	len = buf_alllen(buf) + 4;  /* data in buf + type + crc */

	if (len > context->transport_ctx.t_v1.max_packet_size)
	{
		SSH_ERROR("packet too big:%d (vs %d)\n", len,
				  context->transport_ctx.t_v1.max_packet_size);
		if (cb)
			buf_clear(cb);
		return(-1);
	}

	padlen = 8 - (len % 8);

	/* Initialize buffer for un-encrypted packet. */
	if ((newb = buf_alloc(xport->clr_buf, padlen + len)) == NULL) {
		SSH_ERROR("unable to buf_alloc clr_buf: %s\n", strerror(errno));
		if (cb)
			buf_clear(cb);
		return(-1);
	}
	xport->clr_buf = newb;

	/* Initialize buffer for encrypted data. */
	if ((newb = buf_alloc(xport->send_buf, 4 + padlen + len)) == NULL)
	{
		SSH_ERROR("unable to buf_alloc send_buf: %s\n", strerror(errno));
		if (cb)
			buf_clear(cb);
		buf_clear(xport->send_buf);
		return(-1);
	}
	xport->send_buf = newb;

	/* Put the length in the final packet (unencrypted). */
	buf_put_int32(xport->send_buf, len);

	/* Generate the padding. */
	SSH_DLOG(4, ("generating padding:%d\n", padlen));
	if (context->cipher->type <= SSH_CIPHER_NONE)
		memset(buf_data(xport->clr_buf), 0, padlen);
	else
		ssh_rand_bytes(padlen, buf_data(xport->clr_buf));
	buf_adjlen(xport->clr_buf, padlen);

	/* Append payload data. */
	buf_append(xport->clr_buf, buf);
	if (cb)
		buf_clear(cb);
	
	/* Calculate CRC and append to unencrypted buffer. */
	if (context->cipher->mac_generate) {
		memcpy(&crc,
			   cipher_mac_generate(context->cipher,
					       buf_alldata(xport->clr_buf),
					       buf_alllen(xport->clr_buf)),
		       sizeof(crc));
	}
	else {
		crc = ssh_crc32(buf_alldata(xport->clr_buf),
						buf_alllen(xport->clr_buf));
	}
	SSH_DLOG(4, ("crc=%x\n", crc));
	buf_put_int32(xport->clr_buf, crc);

	if (is_debug_level(6))
	{
		SSH_DLOG(6, ("xmit_packet/clear: "));
		buf_print(xport->send_buf);	/* length */
		buf_print(xport->clr_buf);	/* rest of data. */
	}

	/* Encrypt the payload into the final packet. */
	if (context->cipher->encrypt)
		cipher_encrypt(context->cipher,
			       buf_alldata(xport->clr_buf),
			       buf_endofdata(xport->send_buf),
			       buf_alllen(xport->clr_buf));
	else
		memcpy(buf_endofdata(xport->send_buf),
			   buf_alldata(xport->clr_buf),
			   buf_alllen(xport->clr_buf));
	buf_adjlen(xport->send_buf, buf_alllen(xport->clr_buf));

	/* Clear the unencrypted buffer. */
	buf_clear(xport->clr_buf);

	if (is_debug_level(6)) {
		SSH_DLOG(6, ("xmit_packet/enc: "));
		buf_print(xport->send_buf);
	}

doxmit:

	ret = write(context->transport_ctx.commsock, buf_data(xport->send_buf),
	            buf_len(xport->send_buf));

	SSH_DLOG(4, ("wrote %d bytes.\n", ret));

	if (ret < 0) {
		if (errno == EPIPE)
			return(1);
		SSH_ERROR("write failed: %s\n", strerror(errno));
		return(-1);
	} else {
		/* (current count)+(already used) == (total len) */
		if ((ret + (buf_data(xport->send_buf) - buf_alldata(xport->send_buf)))
				== buf_alllen(xport->send_buf))
		{
			SSH_DLOG(4, ("done.\n"));
			return(0);
		} else {
			SSH_DLOG(4, ("skipping %d\n", ret));
			buf_get_skip(xport->send_buf, ret);
			goto doxmit;
		}
	}
}

static int v1_set_max_packet_size(struct ssh_transport *t_ctx, int size)
{
	FUNC_DECL(v1_set_max_packet_size);

	t_ctx->t_v1.max_packet_size = size;
	return(0);
}

static int v1_get_sendsize(struct ssh_transport *t_ctx)
{
	FUNC_DECL(v1_get_sendsize);
	int sendsize;

	sendsize = t_ctx->t_v1.max_packet_size - 1 - 4;

	/* Adjust max size if compressing since zlib isn't perfect */
	if (t_ctx->t_v1.compressing)
		sendsize = (sendsize - 12) * 0.999;
	
	return(sendsize);
}

