/*
 * Copyright (c) 2005 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "SFTPTableView.h"

@implementation SFTPTableView

/* catch key events */
- ( void )keyDown: ( NSEvent * )theEvent
{
    if ( ! [[ self delegate ] handleEvent: theEvent fromTable: self ] ) {
        [ super keyDown: theEvent ];
    }
}

/* saved for potential addition of promised file drags */
#ifdef notdef
    NSDivideRect( imageframe, &imageframe, &textrect, imagesize.width + 1, NSMinXEdge );
    
    [[[[ self tableColumns ] objectAtIndex: 0 ] dataCell ]
            drawWithFrame: imageframe inView: nil ];
    [[[[[ self tableColumns ] objectAtIndex: 0 ] dataCell ] stringValue ]
            drawInRect: textrect withAttributes: nil ];
#endif notdef

- ( void )dragImage: ( NSImage * )anImage at: ( NSPoint )imageLoc offset: ( NSSize )mouseOffset
        event: ( NSEvent * )theEvent pasteboard: ( NSPasteboard * )pboard source: ( id )sourceObject
        slideBack: ( BOOL )slideBack
{
    int                 row = [ self rowAtPoint: imageLoc ];
    NSRect		imageframe = [ self frameOfCellAtColumn: 0 row: row ];
    float		width = NSWidth( imageframe );
    float		height = NSHeight( [ self bounds ] );
    NSSize		imagesize = NSMakeSize( width, height );
    NSImage		*image = [[ NSImage alloc ] initWithSize: imagesize ];

    /* only draw the portion from the filename column */
    [ image lockFocus ];
    //[[
    [ anImage dissolveToPoint: NSZeroPoint fraction: 1.0 ];
    [ image unlockFocus ];

    [ super dragImage: image at: imageLoc offset: mouseOffset
            event: theEvent pasteboard: pboard source: sourceObject
            slideBack: slideBack ];
    [ image release ];
}

#ifdef notdef
- ( void )mouseDown: ( NSEvent * )event
{
    NSEvent		*e;
    NSPoint 		dragPosition;
    NSRect 		imageLocation;
    int 		column, restoreDragRow;

    if ( ! [self respondsToSelector:
                @selector(dragPromisedFilesOfTypes:fromRect:source:slideBack:event:) ] ) {
	NSLog( @"Promised file drag-and-drop is not supported." );
	[ super mouseDown: event ];
        return;
    }

    dragPosition = [ self convertPoint: [ event locationInWindow ]
                            fromView: nil ];
    restoreDragRow = [ self rowAtPoint: dragPosition ];
    column = [ self columnAtPoint: dragPosition ];

     [ self selectRow: restoreDragRow byExtendingSelection: NO ];

    [ NSEvent startPeriodicEventsAfterDelay: 0.3 withPeriod: 0 ];
    e = [[ self window ] nextEventMatchingMask:
                ( NSLeftMouseUpMask | NSLeftMouseDraggedMask ) ];
	
    switch ( [ e type] ) {
    case NSLeftMouseDragged:
        [ NSEvent stopPeriodicEvents ];
	
	if (( restoreDragRow != -1 ) && ( restoreDragRow < [ self numberOfRows ] )) {
            NSString		*extension = @"'docs'";
	    //dragPosition.x -= 28;
	    //dragPosition.y -= 10;
	    imageLocation.origin = dragPosition;
	    imageLocation.size = NSMakeSize( 16, 16 );

	    //NSString *extension = [[[self delegate] getOriginalPath: restoreDragRow ] pathExtension];
	
	    //if ( ![extension length] || ([extension length] > 4) )
		//extension = @"'docs'";

	    [ self dragPromisedFilesOfTypes: [ NSArray arrayWithObject: extension ]
		    fromRect: imageLocation source: self slideBack: YES event: event ];
	}
	
	break;
	
     default:
	[ NSEvent stopPeriodicEvents ];
        [ super mouseDown: e ];         /* so clicks register if we're not dragging */
    }
}

- ( NSArray * )namesOfPromisedFilesDroppedAtDestination: ( NSURL * )url
{
     NSLog( @"Dropped to: %@", [ url path ] );

     return nil;
}
#endif notdef

- ( NSDragOperation )draggingSourceOperationMaskForLocal: ( BOOL )isLocal
{
    return( NSDragOperationCopy );
}

- ( void )doSpringLoadedAction: ( NSTimer * )timer
{
    [[ self delegate ] performSelector:
                    @selector( performSpringLoadedActionInTable: )
                                    withObject: self
                                    afterDelay: 0.0 ];
                                    
    [ springLoadedTimer invalidate ];
    springLoadedTimer = nil;
}

#ifdef SPRINGLOADED
- ( NSDragOperation )draggingEntered: ( id <NSDraggingInfo> )sender
{
    [ springLoadedTimer invalidate ];
    springLoadedTimer = nil;
    
    if ( [[ self delegate ] respondsToSelector:
                    @selector( performSpringLoadedActionInTable: ) ] ) {
        springLoadedTimer = [ NSTimer scheduledTimerWithTimeInterval: 2.0
                                        target: self
                                        selector: @selector( doSpringLoadedAction: )
                                        userInfo: nil
                                        repeats: NO ];
    }
    
    return( [ super draggingEntered: sender ] );
}
#endif /* SPRINGLOADED */

/* watch for keydown events while dragging for spring-loaded folders */
- ( NSDragOperation )draggingUpdated: ( id <NSDraggingInfo> )sender
{
    NSEvent         *e = [ NSApp currentEvent ];
    NSEventType     type = [ e type ];
    NSDragOperation op = [ super draggingUpdated: sender ];
    
    if ( type == NSKeyDown ) {
        NSEvent         *newEvent = nil;
        NSPoint         pt;
        unichar         key = [[ e charactersIgnoringModifiers ] characterAtIndex: 0 ];
        
        if ( key == ' ' ) {
            if ( [[ self delegate ] respondsToSelector:
                            @selector( performSpringLoadedActionInTable: ) ] ) {
                [ springLoadedTimer invalidate ];
                springLoadedTimer = nil;
                
                [[ self delegate ] performSelector:
                    @selector( performSpringLoadedActionInTable: )
                                    withObject: self
                                    afterDelay: 0.0 ];
            }
        }
        
        pt = NSMakePoint( [ e locationInWindow ].x + 1, [ e locationInWindow ].y + 1 );
        
        /* send a fake event to avoid repeats */
        newEvent = [ NSEvent mouseEventWithType: NSLeftMouseDragged
                    location: pt modifierFlags: [ e modifierFlags ]
                    timestamp: ( [ e timestamp ] + 0.1 ) windowNumber: [ e windowNumber ]
                    context: [ e context ] eventNumber: 1
                    clickCount: 0 pressure: 0.0 ];
                    
        [ NSApp postEvent: newEvent atStart: YES ];
    }

    return( op );
}

- ( void )draggingExited: ( id <NSDraggingInfo> )sender
{
    if ( springLoadedTimer != nil ) {
        [ springLoadedTimer invalidate ];
        springLoadedTimer = nil;
    }
    
    if ( [[ self delegate ] respondsToSelector:
            @selector( springLoadedActionCancelledInTable: ) ] ) {
        [[ self delegate ] performSelector:
                    @selector( springLoadedActionCancelledInTable: )
                            withObject: self
                            afterDelay: 0.0 ];
    }
    
    [ super draggingExited: sender ];
}

#ifdef SPRINGLOADED
- ( void )mouseMoved: ( NSEvent * )event
{
    switch ( [ event type ] ) {
    case NSLeftMouseDragged:
        if ( springLoadedTimer != nil ) {
            [ springLoadedTimer invalidate ];
            springLoadedTimer = nil;
            springLoadedTimer = [ NSTimer scheduledTimerWithTimeInterval: 2.0
                                        target: self
                                        selector: @selector( doSpringLoadedAction: )
                                        userInfo: nil
                                        repeats: NO ];
        }
        
    default:
        break;
    }
    
    [ super mouseMoved: event ];
}
#endif /* SPRINGLOADED */

- ( BOOL )needsPanelToBecomeKey
{
    return( YES );
}

/* display contextual menus for both of the browsers */
- ( NSMenu * )menuForEvent: ( NSEvent * )event
{
    NSPoint	p = [ self convertPoint: [ event locationInWindow ]
                            fromView: nil ];
    int		column = [ self columnAtPoint: p ];
    int		row = [ self rowAtPoint: p ];
    
    rightClickedRow = 0;
    
    if ( column == 0 && row >= 0  &&
        [[ self delegate ] respondsToSelector:
                @selector( menuForTable:column:row: ) ] ) {
        rightClickedRow = row;
        [ self selectRow: row byExtendingSelection: NO ];
        
        return( [[ self delegate ] menuForTable: self
                                    column: column
                                    row: row ] );
    }
    return( nil );
}

- ( void )textDidEndEditing: ( NSNotification * )aNotification
{
    if ( [[[ aNotification userInfo ] objectForKey: @"NSTextMovement" ]
                                        intValue ] == NSReturnTextMovement ) {
        int		column = [ self editedColumn ];
        
        if ( ! [[ self delegate ] respondsToSelector:
                    @selector( handleChangedText:forTable:column: ) ] ) {
            NSBeep();
            [ super textDidEndEditing: aNotification ];
            return;
        }
        
        [[ self delegate ] handleChangedText: [[ aNotification object ] string ]
                forTable: self
                column: column ];
        [[ self window ] endEditingFor: self ];
        [[ self window ] makeFirstResponder: self ];
        [ self reloadData ];
    } else {
        [ super textDidEndEditing: aNotification ];
    }
}

- ( int )rightClickedRow
{
    return( rightClickedRow );
}

- ( NSPoint )originOfSelectedCell
{
    NSRect		cellframe;
    NSPoint		cellorigin;
    int			row = [ self selectedRow ];
    
    if ( row < 0 ) {
        return( NSZeroPoint );
    }
    
    cellframe = [ self frameOfCellAtColumn: 0 row: row ];
    cellorigin = NSMakePoint( cellframe.origin.x, cellframe.origin.y );
    
    cellorigin = [[[ self window ] contentView ] convertPoint: cellorigin fromView: self ];
    cellorigin = [[ self window ] convertBaseToScreen: cellorigin ];
    
    return( cellorigin );
}

- ( void )addTableColumnWithIdentifier: ( id )identifier
            columnTitle: ( NSString * )title width: ( float )width
{
    NSTableColumn	*column = nil;
    
    if ( identifier == nil || title == nil || [ title isEqualToString: @"" ] ) {
        return;
    }
    
    column = [[[ NSTableColumn alloc ] initWithIdentifier: identifier ]
                                        autorelease ];
                                        
    [[ column headerCell ] setStringValue: title ];
    [ column setMinWidth: ( width / 2 ) ];
    [ column setMaxWidth: ( width * 5 ) ];
    [ column setWidth: width ];
    if ( [ identifier isEqualToString: @"sizecolumn" ] ) {
        [[ column headerCell ] setAlignment: NSRightTextAlignment ];
        [[ column dataCell ] setAlignment: NSRightTextAlignment ];
    }
    
    [ self addTableColumn: column ];
}

    NSString *
ColumnTitleFromIdentifier( NSString *identifier )
{
    NSString		*title = @"";
    
    if ( [ identifier isEqualToString: @"datecolumn" ] ) {
        title = NSLocalizedString( @"Date", @"Date" );
    } else if ( [ identifier isEqualToString: @"groupcolumn" ] ) {
        title = NSLocalizedString( @"Group", @"Group" );
    } else if ( [ identifier isEqualToString: @"ownercolumn" ] ) {
        title = NSLocalizedString( @"Owner", @"Owner" );
    } else if ( [ identifier isEqualToString: @"permcolumn" ] ) {
        title = NSLocalizedString( @"Permissions", @"Permissions" );
    } else if ( [ identifier isEqualToString: @"sizecolumn" ] ) {
        title = NSLocalizedString( @"Size", @"Size" );
    }

    return( title );
}

    float
WidthForColumnWithIdentifier( NSString *identifier )
{
    float		width = 50.0; /* arbitrary non-zero width */
    
    if ( [ identifier isEqualToString: @"datecolumn" ] ) {
        width = DATE_COLUMN_WIDTH;
    } else if ( [ identifier isEqualToString: @"groupcolumn" ] ) {
        width = GROUP_COLUMN_WIDTH;
    } else if ( [ identifier isEqualToString: @"ownercolumn" ] ) {
        width = OWNER_COLUMN_WIDTH;
    } else if ( [ identifier isEqualToString: @"permcolumn" ] ) {
        width = MODE_COLUMN_WIDTH;
    } else if ( [ identifier isEqualToString: @"sizecolumn" ] ) {
        width = SIZE_COLUMN_WIDTH;
    }
    
    return( width );
}

@end
