/******************************************************************************
 *
 * Copyright (c) 1999-2001 AppGate AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 1.1, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate AB, Stora Badhusgatan 18-20, 41121 Goteborg, SWEDEN
 *
 *****************************************************************************/

/*
 * Below is some references to useful information about RSA:
 *
 * Bruce Schneier: Applied Cryptography 2nd ed., John Wiley & Sons, 1996
 * Arto Salomaa: Public-Key Cryptography 2nd ed., Springer-Verlag, 1996
 * R. Rivest, A. Shamir, and L. M. Adleman: Cryptographic Communications
 *    System and Method.  US Patent 4,405,829, 1983.
 */
package com.mindbright.security.publickey;

import java.math.BigInteger;

import com.mindbright.jca.security.SecureRandom;
import com.mindbright.jca.security.KeyPair;
import com.mindbright.jca.security.SignatureException;

public final class RSAAlgorithm {

    private final static BigInteger one = BigInteger.valueOf(1L);

    public static BigInteger doPublic(BigInteger input, BigInteger modulus,
				      BigInteger publicExponent)
    {
	return input.modPow(publicExponent, modulus);
    }

    public static BigInteger doPrivate(BigInteger input, BigInteger modulus,
				       BigInteger privateExponent) {
	return doPublic(input, modulus, privateExponent);
    }

    public static BigInteger doPrivateCrt(BigInteger input,
					  BigInteger privateExponent,
					  BigInteger primeP, BigInteger primeQ,
					  BigInteger crtCoefficient)
    {
	return doPrivateCrt(input,
			    primeP, primeQ,
			    getPrimeExponent(privateExponent, primeP),
			    getPrimeExponent(privateExponent, primeQ),
			    crtCoefficient);
    }

    public static BigInteger doPrivateCrt(BigInteger input,
					  BigInteger primeP, BigInteger primeQ,
					  BigInteger primeExponentP,
					  BigInteger primeExponentQ,
					  BigInteger crtCoefficient)
    {
	BigInteger p2;
	BigInteger q2;
	BigInteger k;
	BigInteger result;

	p2 = input.mod(primeP);
	p2 = p2.modPow(primeExponentP, primeP);

	q2 = input.mod(primeQ);
	q2 = q2.modPow(primeExponentQ, primeQ);

	if(p2.compareTo(q2) == 0)
	    return p2;

	k = q2.subtract(p2).mod(primeQ);
	k = k.multiply(crtCoefficient);
	k = k.mod(primeQ);

	result = k.multiply(primeP);
	result = result.add(p2);

	return result;
    }

    public static BigInteger getPrimeExponent(BigInteger privateExponent,
					      BigInteger prime) {
	BigInteger pe = prime.subtract(one);
	return privateExponent.mod(pe);
    }

    public static BigInteger addPKCS1Pad(BigInteger input, int type,
					 int padLen, SecureRandom rand)
	throws SignatureException
    {
	BigInteger result;
	BigInteger rndInt;
	int inByteLen  = (input.bitLength() + 7) / 8;

	if(inByteLen > padLen - 11) {
	    throw new SignatureException("PKCS1Pad: Input too long to pad");
	}

	byte[] padBytes = new byte[(padLen - inByteLen - 3) + 1];
	padBytes[0] = 0;

	for(int i = 1; i < (padLen - inByteLen - 3 + 1); i++) {
	    if(type == 0x01) {
		padBytes[i] = (byte)0xff;
	    } else {
		byte[] b = new byte[1];
		do {
		    rand.nextBytes(b);
		} while(b[0] == 0);
		padBytes[i] = b[0];
	    }
	}

	rndInt = new BigInteger(1, padBytes);
	rndInt = rndInt.shiftLeft((inByteLen + 1) * 8);
	result = new BigInteger("2");
	result = result.shiftLeft((padLen - 2) * 8);
	result = result.or(rndInt);
	result = result.or(input);

	return result;
    }

    public static BigInteger stripPKCS1Pad(BigInteger input, int type)
	throws SignatureException
    {
	byte[] strip = input.toByteArray();
	byte[] val;
	int    i;

	if(strip[0] != type) {
	    throw new SignatureException("Invalid PKCS1 padding, type != " +
					 type);
	}

	for(i = 1; i < strip.length; i++) {
	    if(strip[i] == 0) {
		break;
	    }
	    if(type == 0x01 && strip[i] != (byte)0xff) {
		throw new SignatureException("Invalid PKCS1 padding, " +
					     "corrupt data");
	    }
	}

	if(i == strip.length) {
	    throw new SignatureException("Invalid PKCS1 padding, corrupt data");
	}

	val = new byte[strip.length - i];
	System.arraycopy(strip, i, val, 0, val.length);
	return new BigInteger(1, val);
    }

    public static KeyPair generateKeyPair(int bits, SecureRandom secRand) {
	BigInteger p   = null;
	BigInteger q   = null;
	BigInteger t   = null;
	BigInteger p_1 = null;
	BigInteger q_1 = null;
	BigInteger phi = null;
	BigInteger G   = null;
	BigInteger F   = null;
	BigInteger e   = null;
	BigInteger d   = null;
	BigInteger u   = null;
	BigInteger n   = null;

	boolean finished = false;

	while(!finished) {
	    p = new BigInteger(bits / 2, 64, secRand);
	    q = new BigInteger(bits - (bits / 2), 64, secRand);

	    if(p.compareTo(q) == 0) {
		continue;
	    } else if(q.compareTo(p) < 0) {
		t = q;
		q = p;
		p = t;
	    }

	    t = p.gcd(q);
	    if(t.compareTo(one) != 0) {
		continue;
	    }

	    p_1 = p.subtract(one);
	    q_1 = q.subtract(one);
	    phi = p_1.multiply(q_1);
	    G   = p_1.gcd(q_1);
	    F   = phi.divide(G);

	    e   = one.shiftLeft(5);
	    e   = e.subtract(one);
	    do {
		e = e.add(one.add(one));
		t = e.gcd(phi);
	    } while(t.compareTo(one) != 0);

	    // !!! d = e.modInverse(F);
	    d = e.modInverse(phi);
	    n = p.multiply(q);
	    u = p.modInverse(q);

	    finished = true;
	}

	RSAPrivateCrtKey prvKey = new RSAPrivateCrtKey(n, e, d, p, q, u);
	RSAPublicKey     pubKey = new RSAPublicKey(n, e);

	return new KeyPair(pubKey, prvKey);
    }

}
