package com.sshtools.j2ssh.agent;


import java.net.*;
import java.io.*;
import com.sshtools.j2ssh.io.*;
import com.sshtools.j2ssh.subsystem.SubsystemMessage;
import java.util.*;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.transport.publickey.*;

import org.apache.log4j.Logger;

public class SshAgentConnection implements Runnable {

   private static Logger log = Logger.getLogger(SshAgentConnection.class);

   InputStream in;
   OutputStream out;
   KeyStore keystore;
   Thread thread;
   Vector forwardingNodes = new Vector();

   SshAgentConnection(KeyStore keystore, InputStream in, OutputStream out) {
     this.keystore = keystore;
     this.in = in;
     this.out = out;
     thread = new Thread(this);
     thread.start();
   }


   protected void sendAgentSuccess() throws IOException {
     SshAgentSuccess msg = new SshAgentSuccess();
     sendMessage(msg);
   }

   protected void sendAgentFailure(int errorcode) throws IOException {
     SshAgentFailure msg = new SshAgentFailure(errorcode);
     sendMessage(msg);
   }

   protected void sendVersionResponse() throws IOException {
     SshAgentVersionResponse msg = new SshAgentVersionResponse(2);
     sendMessage(msg);
   }

   protected void sendAgentKeyList() throws IOException {
     SshAgentKeyList msg = new SshAgentKeyList(keystore.getPublicKeys());
     sendMessage(msg);
   }

   protected void sendOperationComplete(byte[] data) throws IOException {
     SshAgentOperationComplete msg = new SshAgentOperationComplete(data);
     sendMessage(msg);
   }

   protected void sendRandomData(byte[] data) throws IOException {
     SshAgentRandomData msg = new SshAgentRandomData(data);
     sendMessage(msg);
   }

   protected void sendAgentAlive(byte[] padding) throws IOException {
     SshAgentAlive msg = new SshAgentAlive(padding);
     sendMessage(msg);
   }

   protected void sendMessage(SubsystemMessage msg)throws IOException {
     log.info("Sending message " + msg.getMessageName());
     byte[] msgdata = msg.toByteArray();
     out.write(ByteArrayWriter.encodeInt(msgdata.length));
     out.write(msgdata);
     out.flush();
   }

   protected void onForwardingNotice(SshAgentForwardingNotice msg) {
     forwardingNodes.add(new ForwardingNotice(msg.getRemoteHostname(),
         msg.getRemoteIPAddress(),
         msg.getRemotePort()));
   }

   protected void onRequestVersion(SshAgentRequestVersion msg) throws IOException {
     sendVersionResponse();
   }

   protected void onAddKey(SshAgentAddKey msg) throws IOException {
     if(keystore.addKey(msg.getPrivateKey(),
                        msg.getPublicKey(),
                        msg.getDescription(),
                        msg.getKeyConstraints())) {
       sendAgentSuccess();
     } else
       sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_FAILURE);
   }

   protected void onDeleteAllKeys(SshAgentDeleteAllKeys msg) throws IOException {
     keystore.deleteAllKeys();
     sendAgentSuccess();
   }

   protected void onListKeys(SshAgentListKeys msg) throws IOException {
     sendAgentKeyList();
   }

   protected void onPrivateKeyOp(SshAgentPrivateKeyOp msg) throws IOException {

    try {
      if (msg.getOperation().equals("sign")) {
        sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_KEY_NOT_SUITABLE);
      }
      else if (msg.getOperation().equals("hash-and-sign")) {
        byte[] sig = keystore.performHashAndSign(msg.getPublicKey(),
                                                 forwardingNodes,
                                                 msg.getOperationData());
        sendOperationComplete(sig);
      }
      else if (msg.getOperation().equals("decrypt")) {
        sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_KEY_NOT_SUITABLE);
      }
      else if (msg.getOperation().equals("ssh1-challenge-response")) {
        sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_KEY_NOT_SUITABLE);
      }
      else {
        sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_UNSUPPORTED_OP);
      }
    }
    catch (KeyTimeoutException ex) {
      sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_TIMEOUT);
    } catch(InvalidSshKeyException ex) {
      sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_KEY_NOT_FOUND);
    }

   }

   protected void onDeleteKey(SshAgentDeleteKey msg) throws IOException {

     if(keystore.deleteKey(msg.getPublicKey(),
                           msg.getDescription())) {
         sendAgentSuccess();
       } else
         sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_KEY_NOT_FOUND);

   }

   protected void onLock(SshAgentLock msg) throws IOException {
     if(keystore.lock(msg.getPassword())) {
       sendAgentSuccess();
     } else
       sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_DENIED);
   }

   protected void onUnlock(SshAgentUnlock msg) throws IOException {

     if(keystore.unlock(msg.getPassword())) {
       sendAgentSuccess();
     } else
       sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_DENIED);
   }

   protected void onPing(SshAgentPing msg) throws IOException {
     sendAgentAlive(msg.getPadding());
   }

   protected void onRandom(SshAgentRandom msg) throws IOException {

     if(msg.getLength() > 0){
       byte[] random = new byte[msg.getLength()];
       ConfigurationLoader.getRND().nextBytes(random);
       sendRandomData(random);
     } else
       sendAgentFailure(SshAgentFailure.SSH_AGENT_ERROR_FAILURE);
   }


    public void run() {

      try {

        log.info("Starting thread");

        byte[] lendata = new byte[4];
        byte[] msgdata;
        int len;

        while (true) {
          // Read the first 4 bytes to determine the length of the message
          len = 0;
          while(len < lendata.length)
            len += in.read(lendata, len, lendata.length - len);

          len = (int)ByteArrayReader.readInt(lendata, 0);
          msgdata = new byte[len];
          len = 0;
          while(len < msgdata.length)
            len += in.read(msgdata, len, msgdata.length - len);

          onMessageReceived(msgdata);


        }
      }
      catch (IOException ex) {
        ex.printStackTrace();
      }

      log.info("Exiting thread");

    }


    protected void onMessageReceived(byte[] msgdata) throws IOException {

      switch((int)(msgdata[0]&0xFF)) {

        case SshAgentForwardingNotice.SSH_AGENT_FORWARDING_NOTICE: {
          SshAgentForwardingNotice msg = new SshAgentForwardingNotice();
          msg.fromByteArray(msgdata);
          onForwardingNotice(msg);
          break;
        }

        case SshAgentRequestVersion.SSH_AGENT_REQUEST_VERSION: {
          SshAgentRequestVersion msg = new SshAgentRequestVersion();
          msg.fromByteArray(msgdata);
          onRequestVersion(msg);
          break;
        }

        case SshAgentAddKey.SSH_AGENT_ADD_KEY: {
          SshAgentAddKey msg = new SshAgentAddKey();
          msg.fromByteArray(msgdata);
          onAddKey(msg);
          break;
        }

        case SshAgentDeleteAllKeys.SSH_AGENT_DELETE_ALL_KEYS: {
          SshAgentDeleteAllKeys msg = new SshAgentDeleteAllKeys();
          msg.fromByteArray(msgdata);
          onDeleteAllKeys(msg);
          break;
        }

        case SshAgentListKeys.SSH_AGENT_LIST_KEYS: {
          SshAgentListKeys msg = new SshAgentListKeys();
          msg.fromByteArray(msgdata);
          onListKeys(msg);
          break;
        }

        case SshAgentPrivateKeyOp.SSH_AGENT_PRIVATE_KEY_OP: {
          SshAgentPrivateKeyOp msg = new SshAgentPrivateKeyOp();
          msg.fromByteArray(msgdata);
          onPrivateKeyOp(msg);
          break;
        }

        case SshAgentDeleteKey.SSH_AGENT_DELETE_KEY: {
          SshAgentDeleteKey msg = new SshAgentDeleteKey();
          msg.fromByteArray(msgdata);
          onDeleteKey(msg);
          break;
        }

        case SshAgentLock.SSH_AGENT_LOCK: {
          SshAgentLock msg = new SshAgentLock();
          msg.fromByteArray(msgdata);
          onLock(msg);
          break;
        }


        case SshAgentUnlock.SSH_AGENT_UNLOCK: {
          SshAgentUnlock msg = new SshAgentUnlock();
          msg.fromByteArray(msgdata);
          onUnlock(msg);
          break;
        }

        case SshAgentPing.SSH_AGENT_PING: {
          SshAgentPing msg = new SshAgentPing();
          msg.fromByteArray(msgdata);
          onPing(msg);
          break;
        }

        case SshAgentRandom.SSH_AGENT_RANDOM: {
          SshAgentRandom msg = new SshAgentRandom();
          msg.fromByteArray(msgdata);
          onRandom(msg);
          break;
        }

        default:
          throw new IOException("Unrecognized message type " + String.valueOf(msgdata[0]) + " received");

      }
    }
  }