/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.connection;

import com.sshtools.j2ssh.io.IOStreamConnector;

import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.SshMessageStore;
import com.sshtools.j2ssh.transport.TransportProtocolException;

import com.sshtools.j2ssh.util.InvalidStateException;

import org.apache.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.Map;
import java.util.Vector;
import java.util.Iterator;
import com.sshtools.j2ssh.transport.*;
import com.sshtools.j2ssh.transport.*;


public abstract class Channel {
    private static Logger log = Logger.getLogger(Channel.class);
    protected ChannelDataWindow localWindow = new ChannelDataWindow();
    protected ChannelDataWindow remoteWindow = new ChannelDataWindow();
    protected ConnectionProtocol connection;
    protected Map nativeSettings;
    protected SshMessageStore incoming = new SshMessageStore();
    protected long localChannelId;
    protected long localPacketSize;
    protected long remoteChannelId;
    protected long remotePacketSize;
    protected ChannelInputStream in;
    protected ChannelOutputStream out;
    protected ChannelState state = new ChannelState();
    private boolean isClosed = false;
    private boolean isLocalEOF = false;
    private boolean isRemoteEOF = false;
    private OutputStream bout = null;
    private InputStream bin = null;
    private IOStreamConnector ios = null;
    private Vector eventListeners = new Vector();
    private Channel boundChannel = null;

    private int seqIn = 0;
    private int seqOut = 0;

    public Channel() {
        this.localPacketSize = getMaximumPacketSize();
        this.localWindow.increaseWindowSpace(getMaximumWindowSpace());
    }

    public abstract byte[] getChannelOpenData();

    public abstract byte[] getChannelConfirmationData();

    public abstract String getChannelType();

    protected abstract int getMinimumWindowSpace();

    protected abstract int getMaximumWindowSpace();

    protected abstract int getMaximumPacketSize();

    public void bindOutputStream(OutputStream bout) throws IOException {

      // Synchronize on the incoming message store to ensure that no other
      // messages are added whilst we transfer to a bound state
      synchronized(incoming) {
        this.bout = bout;
        if (state.getValue() == ChannelState.CHANNEL_OPEN)
          sendOutstandingMessages();
      }
    }

    public void bindInputStream(InputStream bin) throws IOException {
        this.bin = bin;

        if(state.getValue()==ChannelState.CHANNEL_OPEN)
          ios = new IOStreamConnector(bin, out);
    }

    public void bindChannel(Channel boundChannel) throws IOException {
      this.boundChannel = boundChannel;

      // If the bound channel is open then bind the outputstreams
      if(boundChannel.getState().getValue()==ChannelState.CHANNEL_OPEN) {
        throw new IOException("You cannot bind to an open channel");
      }

      // Create an event listener so we can listen
      boundChannel.addEventListener(new ChannelEventListener() {
          public void onChannelOpen(Channel channel) {
            try {
              bindOutputStream(channel.getOutputStream());
              channel.bindOutputStream(getOutputStream());
            }
            catch (IOException ex) {
              log.info("Failed to bind the channel");
            }
          }

          public void onChannelEOF(Channel channel) {

          }

          public void onChannelClose(Channel channel) {
            try {
              close();
            }
            catch (IOException ex) {
              log.info("Failed to close the channel");
            }
          }

          public void onDataReceived(Channel channel, int numBytes) {

          }

          public void onDataSent(Channel channel, int numBytes) {

          }

        });


    }

    public void onChannelData(SshMsgChannelData msg) throws IOException {
        if (msg.getChannelDataLength() > localWindow.getWindowSpace()) {
            throw new ServiceOperationException(
                "More data recieved than is allowed by the channel data window");
        }

        long windowSpace = localWindow.consumeWindowSpace(msg.getChannelData().length);

        if (windowSpace < getMinimumWindowSpace()) {
            log.debug("Channel " + String.valueOf(localChannelId)
                + " requires more window space");
            windowSpace = getMaximumWindowSpace() - windowSpace;
            connection.sendChannelWindowAdjust(this, windowSpace);
            localWindow.increaseWindowSpace(windowSpace);
        }

        // Synchronize on the message store to ensure that another thread
        // does not try to read its data. This will make sure that the incoming
        // messages are not being flushed to an outputstream after a bind


        synchronized(incoming) {

          seqIn++;

          if (bout != null) {
            seqOut++;
            bout.write(msg.getChannelData());
          }
          else {

            incoming.addMessage(msg);
          }
        }

        Iterator it = eventListeners.iterator();
        ChannelEventListener eventListener;
        while (it.hasNext()) {
          eventListener = (ChannelEventListener) it.next();
          if (eventListener != null)
            eventListener.onDataSent(this, msg.getChannelData().length);
        }

    }

    public boolean isClosed() {
        synchronized (state) {
            return state.getValue() == ChannelState.CHANNEL_CLOSED;
        }
    }

    protected void sendChannelData(byte[] data) throws IOException {


      if(!connection.isConnected())
      throw new IOException("The connection has been closed");

    if(!isClosed()) {

          connection.sendChannelData(this, data);

          Iterator it = eventListeners.iterator();
          ChannelEventListener eventListener;
          while (it.hasNext()) {
            eventListener = (ChannelEventListener) it.next();
            if (eventListener != null)
              eventListener.onDataSent(this, data.length);
          }
        } else
          throw new IOException("The channel is closed");

    }

    protected void sendChannelExtData(int type, byte[] data)
        throws IOException {

        if(!connection.isConnected())
          throw new IOException("The connection has been closed");

        if(!isClosed()) {
          connection.sendChannelExtData(this, type, data);

          Iterator it = eventListeners.iterator();
          ChannelEventListener eventListener;
          while (it.hasNext()) {
            eventListener = (ChannelEventListener) it.next();
            if (eventListener != null)
              eventListener.onDataSent(this, data.length);
          }
        } else
          throw new IOException("The channel is closed");


    }

    public void onChannelExtData(SshMsgChannelExtendedData msg)
        throws IOException {
        if (msg.getChannelData().length > localWindow.getWindowSpace()) {
            throw new ServiceOperationException(
                "More data recieved than is allowed by the channel data window");
        }

        long windowSpace = localWindow.consumeWindowSpace(msg.getChannelData().length);

        if (windowSpace < getMinimumWindowSpace()) {
            log.debug("Channel " + String.valueOf(localChannelId)
                + " requires more window space");
            windowSpace = getMaximumWindowSpace() - windowSpace;
            connection.sendChannelWindowAdjust(this, windowSpace);
            localWindow.increaseWindowSpace(windowSpace);
        }

        incoming.addMessage(msg);

        Iterator it = eventListeners.iterator();
        ChannelEventListener eventListener;
        while (it.hasNext()) {
          eventListener = (ChannelEventListener) it.next();
          if (eventListener != null)
            eventListener.onDataReceived(this, msg.getChannelData().length);
        }

    }

    public ChannelInputStream getInputStream() {
        return in;
    }

    public long getLocalChannelId() {
        return localChannelId;
    }

    public long getLocalPacketSize() {
        return localPacketSize;
    }

    public synchronized ChannelDataWindow getLocalWindow() {
        return localWindow;
    }

    public ChannelOutputStream getOutputStream() {
        return out;
    }

    public long getRemoteChannelId() {
        return remoteChannelId;
    }

    public long getRemotePacketSize() {
        return remotePacketSize;
    }

    public synchronized ChannelDataWindow getRemoteWindow() {
        return remoteWindow;
    }

    public ChannelState getState() {
        return state;
    }

    public void close() throws IOException {
        synchronized (state) {

            state.setValue(ChannelState.CHANNEL_CLOSED);

            onChannelClose();

            // Close the IOStream connector if were bound
            if(ios!=null)
              ios.close();

            if(connection!=null)
              connection.closeChannel(this);

            Iterator it = eventListeners.iterator();
            ChannelEventListener eventListener;
            while(it.hasNext()){
                eventListener = (ChannelEventListener)it.next();
                if(eventListener!=null)
                   eventListener.onChannelClose(this);
            }

        }
    }

    protected void onOutputStreamEOF() throws IOException {
        synchronized (state) {
            if (state.getValue() == ChannelState.CHANNEL_OPEN && connection.isConnected()) {
                connection.sendChannelEOF(this);
            }

            //if(in.isClosed())
            //  close();
        }
    }

    protected void onInputStreamEOF() throws IOException {

      Iterator it = eventListeners.iterator();
      ChannelEventListener eventListener;
      while (it.hasNext()) {
        eventListener = (ChannelEventListener) it.next();
        if (eventListener != null)
          eventListener.onChannelClose(this);
      }

      synchronized(state) {
        in.close();

       // if (out.isClosed())
       //   close();
      }

}


    public void addEventListener(ChannelEventListener eventListener) {
      eventListeners.add(eventListener);
    }

    public void init(ConnectionProtocol connection, long localChannelId,
        long senderChannelId, long initialWindowSize, long maximumPacketSize)
        throws IOException {
        this.localChannelId = localChannelId;
        this.remoteChannelId = senderChannelId;
        this.remotePacketSize = maximumPacketSize;
        this.remoteWindow.increaseWindowSpace(initialWindowSize);
        this.connection = connection;
        this.nativeSettings = nativeSettings;
        this.in = new ChannelInputStream(incoming);
        this.out = new ChannelOutputStream(this);
        state.setValue(ChannelState.CHANNEL_OPEN);
    }



    protected void open() throws IOException {

      state.setValue(ChannelState.CHANNEL_OPEN);

      onChannelOpen();


      Iterator it = eventListeners.iterator();
      ChannelEventListener eventListener;
      while(it.hasNext()){
          eventListener = (ChannelEventListener)it.next();
          if(eventListener!=null)
             eventListener.onChannelOpen(this);
      }



     if(bout!=null)
         sendOutstandingMessages();

     if(bin!=null && ios==null)
         ios = new IOStreamConnector(bin, out);




    }

    private void sendOutstandingMessages() throws IOException {


        if (bout != null && incoming.hasMessages()) {
          while (true) {
            try {
              // Peek into the message store and look for the next message
              SshMsgChannelData msg =
                  (SshMsgChannelData)
                  incoming.peekMessage(SshMsgChannelData.SSH_MSG_CHANNEL_DATA);
              // Remove the message so we dont process again
              incoming.removeMessage(msg);

              seqOut++;

              // Write the message out to the bound OutputStream
              bout.write(msg.getChannelData());
            }
            catch (MessageStoreEOFException ex) {
              break;
            }
            catch (MessageNotAvailableException ex) {
              break;
            }
          }
        }

    }

    public void init(ConnectionProtocol connection, long localChannelId,
        long senderChannelId, long initialWindowSize, long maximumPacketSize,
        ChannelEventListener eventListener) throws IOException {
        if(eventListener!=null)
          addEventListener(eventListener);
        init(connection, localChannelId, senderChannelId, initialWindowSize,
            maximumPacketSize);
    }

    protected abstract void onChannelClose() throws IOException;



    protected abstract void onChannelOpen() throws InvalidChannelException;

    protected abstract void onChannelRequest(String requestType,
        boolean wantReply, byte[] requestData) throws IOException;
}
