/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.connection;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.transport.InvalidMessageException;
import com.sshtools.j2ssh.transport.SshMessage;

import java.io.IOException;


public class SshMsgChannelOpen extends SshMessage {
    protected final static int SSH_MSG_CHANNEL_OPEN = 90;
    private String channelType;
    private byte[] channelData;
    private long initialWindowSize;
    private long maximumPacketSize;
    private long senderChannelId;

    public SshMsgChannelOpen(String channelType, long senderChannelId,
        long initialWindowSize, long maximumPacketSize, byte[] channelData) {
        super(SSH_MSG_CHANNEL_OPEN);

        this.channelType = channelType;
        this.senderChannelId = senderChannelId;
        this.initialWindowSize = initialWindowSize;
        this.maximumPacketSize = maximumPacketSize;
        this.channelData = channelData;
    }

    public SshMsgChannelOpen() {
        super(SSH_MSG_CHANNEL_OPEN);
    }

    public byte[] getChannelData() {
        return channelData;
    }

    public String getChannelType() {
        return channelType;
    }

    public long getInitialWindowSize() {
        return initialWindowSize;
    }

    public long getMaximumPacketSize() {
        return maximumPacketSize;
    }

    public String getMessageName() {
        return "SSH_MSG_CHANNEL_OPEN";
    }

    public long getSenderChannelId() {
        return senderChannelId;
    }

    protected void constructByteArray(ByteArrayWriter baw)
        throws InvalidMessageException {
        try {
            baw.writeString(channelType);
            baw.writeInt(senderChannelId);
            baw.writeInt(initialWindowSize);
            baw.writeInt(maximumPacketSize);

            if (channelData != null) {
                baw.write(channelData);
            }
        } catch (IOException ioe) {
            throw new InvalidMessageException("Could not write message data");
        }
    }

    protected void constructMessage(ByteArrayReader bar)
        throws InvalidMessageException {
        try {
            channelType = bar.readString();
            senderChannelId = bar.readInt();
            initialWindowSize = bar.readInt();
            maximumPacketSize = bar.readInt();

            if (bar.available() > 0) {
                channelData = new byte[bar.available()];
                bar.read(channelData);
            }
        } catch (IOException ioe) {
            throw new InvalidMessageException("Invalid message data");
        }
    }
}
