/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.connection;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.transport.InvalidMessageException;
import com.sshtools.j2ssh.transport.SshMessage;

import java.io.IOException;


public class SshMsgChannelOpenConfirmation extends SshMessage {
    protected final static int SSH_MSG_CHANNEL_OPEN_CONFIRMATION = 91;
    private byte[] channelData;
    private long initialWindowSize;
    private long maximumPacketSize;
    private long recipientChannel;
    private long senderChannel;

    public SshMsgChannelOpenConfirmation(long recipientChannel,
        long senderChannel, long initialWindowSize, long maximumPacketSize,
        byte[] channelData) {
        super(SSH_MSG_CHANNEL_OPEN_CONFIRMATION);
        this.recipientChannel = recipientChannel;
        this.senderChannel = senderChannel;
        this.initialWindowSize = initialWindowSize;
        this.maximumPacketSize = maximumPacketSize;
        this.channelData = channelData;
    }

    public SshMsgChannelOpenConfirmation() {
        super(SSH_MSG_CHANNEL_OPEN_CONFIRMATION);
    }

    public byte[] getChannelData() {
        return channelData;
    }

    public long getInitialWindowSize() {
        return initialWindowSize;
    }

    public long getMaximumPacketSize() {
        return maximumPacketSize;
    }

    public String getMessageName() {
        return "SSH_MSG_CHANNEL_OPEN_CONFIRMATION";
    }

    public long getRecipientChannel() {
        return recipientChannel;
    }

    public long getSenderChannel() {
        return senderChannel;
    }

    protected void constructByteArray(ByteArrayWriter baw)
        throws InvalidMessageException {
        try {
            baw.writeInt(recipientChannel);
            baw.writeInt(senderChannel);
            baw.writeInt(initialWindowSize);
            baw.writeInt(maximumPacketSize);

            if (channelData != null) {
                baw.write(channelData);
            }
        } catch (IOException ioe) {
            throw new InvalidMessageException("Invalid message data");
        }
    }

    protected void constructMessage(ByteArrayReader bar)
        throws InvalidMessageException {
        try {
            recipientChannel = bar.readInt();
            senderChannel = bar.readInt();
            initialWindowSize = bar.readInt();
            maximumPacketSize = bar.readInt();

            if (bar.available() > 0) {
                channelData = new byte[bar.available()];
                bar.read(channelData);
            }
        } catch (IOException ioe) {
            throw new InvalidMessageException("Invalid message data");
        }
    }
}
