/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.forwarding;

import com.sshtools.j2ssh.SshException;
import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;

import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelState;
import com.sshtools.j2ssh.connection.InvalidChannelException;

import com.sshtools.j2ssh.io.ByteArrayWriter;
import com.sshtools.j2ssh.io.IOStreamConnector;
import com.sshtools.j2ssh.io.IOStreamConnectorState;

import com.sshtools.j2ssh.transport.ServiceOperationException;

import com.sshtools.j2ssh.util.MultipleStateMonitor;
import com.sshtools.j2ssh.util.State;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.*;


public class ForwardingChannel extends Channel {
  public final static String X11_FORWARDING_CHANNEL = "x11";
  public final static String LOCAL_FORWARDING_CHANNEL = "direct-tcpip";
  public final static String REMOTE_FORWARDING_CHANNEL = "forwarded-tcpip";
  private static Logger log = Logger.getLogger(ForwardingChannel.class);
  private ForwardingConfiguration config;
  private String forwardType;
  private String originatingHost;
  private int originatingPort;
  private Socket socket;

  public ForwardingChannel(String forwardType,
                           ForwardingConfiguration config,
                           String originatingHost, int originatingPort)
      throws ForwardingConfigurationException {
    if (!forwardType.equals(LOCAL_FORWARDING_CHANNEL)
        && !forwardType.equals(REMOTE_FORWARDING_CHANNEL)
        && !forwardType.equals(X11_FORWARDING_CHANNEL)) {
      throw new ForwardingConfigurationException(
          "The forwarding type is invalid");
    }
    this.config = config;
    this.forwardType = forwardType;
    this.originatingHost = originatingHost;
    this.originatingPort = originatingPort;
  }

  public byte[] getChannelOpenData() {
    try {
      ByteArrayWriter baw = new ByteArrayWriter();
      if (forwardType.equals(LOCAL_FORWARDING_CHANNEL)) {
        baw.writeString(config.getHostToConnect());
        baw.writeInt(config.getPortToConnect());
      }
      else {
        baw.writeString(config.getAddressToBind());
        baw.writeInt(config.getPortToBind());
      }
      baw.writeString(originatingHost);
      baw.writeInt(originatingPort);
      return baw.toByteArray();
    }
    catch (IOException ioe) {
      return null;
    }
  }

  public byte[] getChannelConfirmationData() {
    return null;
  }

  public String getChannelType() {
    return forwardType;
  }

  protected int getMinimumWindowSpace() {
    return 32768;
  }

  protected int getMaximumWindowSpace() {
    return 131072;
  }

  protected int getMaximumPacketSize() {
    return 32768;
  }

  public ForwardingConfiguration getForwardingConfiguration() {
    return config;
  }

  public InetSocketAddress getRemoteSocketAddress() {
    if (socket != null)
      return (InetSocketAddress) socket.getRemoteSocketAddress();
    else
      return new InetSocketAddress(originatingHost, originatingPort);
  }

  public InetAddress getLocalAddress() {
    try {
      if (socket != null)
        return socket.getInetAddress();
      else
        return InetAddress.getLocalHost();
    }
    catch (UnknownHostException ex) {
      return null;
    }
  }

  public void bindSocket(Socket socket)
      throws IOException {
    this.socket = socket;
    if (state.getValue() == ChannelState.CHANNEL_OPEN) {
      bindInputStream(socket.getInputStream());
      bindOutputStream(socket.getOutputStream());
    }

  }

    protected void onChannelClose() throws ServiceOperationException {
    }

    protected void onChannelOpen() throws InvalidChannelException {

      try {
        if (socket != null) {
          bindInputStream(socket.getInputStream());
          bindOutputStream(socket.getOutputStream());
        }
      }
      catch (IOException ex) {
        throw new InvalidChannelException(
            "Failed to bind the channel to the forwarded socket: " +
            ex.getMessage());
      }
    }

    protected void onChannelRequest(String request, boolean wantReply,
        byte[] requestData) throws IOException {
      connection.sendChannelRequestFailure(this);
    }


}
