/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.io;

import java.io.IOException;

import java.net.InetSocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.channels.ServerSocketChannel;

import java.util.Iterator;

import java.io.*;


public class SocketConnector implements Runnable {
    private SocketConnectorState state = new SocketConnectorState();
    private Thread thread;
    private SocketChannel socketChannel;
    private InputStream in;
    private OutputStream out;
    private ByteBuffer readbuf = ByteBuffer.allocateDirect(32768);
    private ByteBuffer writebuf = ByteBuffer.allocateDirect(32768);

    public SocketConnector(SocketChannel socketChannel, InputStream in, OutputStream out) {
        this.socketChannel = socketChannel;
        this.in = in;
        this.out = out;
        thread = new Thread(this);
        thread.run();
    }

    public void run() {
        // Create a selector and register two socket channels
        Selector selector = null;

        try {
          // Create the selector
          selector = Selector.open();

          // Configure the channel for non-blocking
          socketChannel.configureBlocking(false);

          // Register the channel with selector, listening for all events
          socketChannel.register(selector,  SelectionKey.OP_READ); // | SelectionKey.OP_WRITE);

          state.setValue(SocketConnectorState.CONNECTED);

          // Wait for events
          int events;
          while ((events = selector.select()) > 0) {

            System.out.println(events);
            // Wait for an event
            //selector.select();

            // Get list of selection keys with pending events
            Iterator it = selector.selectedKeys().iterator();

            // Process each key at a time
            while (it.hasNext()) {
              // Get the selection key
              SelectionKey selKey = (SelectionKey) it.next();


              // Remove it from the list to indicate that it is being processed
              it.remove();

              try {
                processSelectionKey(selKey);
              }
              catch (IOException e) {
                // Handle error with channel and unregister
                state.setValue(SocketConnectorState.UNEXPECTED_TERMINATION);
                selKey.cancel();
                break;
              }
            }
          }
      } catch (IOException e) {
          state.setValue(SocketConnectorState.UNEXPECTED_TERMINATION);
      }

    }

    // Creates a non-blocking socket channel for the specified host name and port.
    // connect() is called on the new channel before it is returned.
    public static SocketChannel createSocketChannel(String hostName, int port)
        throws IOException {
        // Create a non-blocking socket channel
        SocketChannel sChannel = SocketChannel.open();
        sChannel.configureBlocking(false);

        // Send a connection request to the server; this method is non-blocking
        sChannel.connect(new InetSocketAddress(hostName, port));

        return sChannel;
    }

    public static ServerSocketChannel createServerSocketChannel(int port) throws IOException {

        // Create a non-blocking server socket channel on port
        ServerSocketChannel ssChannel = ServerSocketChannel.open();
        ssChannel.configureBlocking(false);
        ssChannel.socket().bind(new InetSocketAddress(port));

        // Get port that received the connection request; this information
        // might be useful in determining how to handle the connection
        //int localPort = ssChannel.socket().getLocalPort();

        return ssChannel;

      }


      public static SocketChannel acceptNextConnection(ServerSocketChannel ssChannel) throws IOException {

        // Create the selector
        Selector selector = Selector.open();

        ssChannel.register(selector, SelectionKey.OP_ACCEPT);

        while (true) {
          // Wait for an event
          selector.select();

          // Get list of selection keys with pending events
          Iterator it = selector.selectedKeys().iterator();

          // Process each key
          while (it.hasNext()) {
            // Get the selection key
            SelectionKey selKey = (SelectionKey) it.next();

            // Remove it from the list to indicate that it is being processed
            it.remove();

            // Check if it's a connection request
            if (selKey.isAcceptable()) {
              // Get channel with connection request
              ServerSocketChannel server = (ServerSocketChannel) selKey.channel();
              return server.accept();
            }
          }
        }
      }



    public SocketConnectorState getState() {
        return state;
    }

    public void processSelectionKey(SelectionKey selKey)
        throws IOException {
        // Since the ready operations are cumulative,
        // need to check readiness for each operation
        if (selKey.isConnectable()) {
            // Get channel with connection request
            SocketChannel sChannel = (SocketChannel) selKey.channel();

            boolean success = sChannel.finishConnect();

            if (!success) {
                // An error occurred; handle it
                state.setValue(SocketConnectorState.UNEXPECTED_TERMINATION);

                // Unregister the channel with this selector
                selKey.cancel();
            }
        }

        if (selKey.isReadable()) {
            // Get channel with bytes to read
            SocketChannel socketChannel = (SocketChannel) selKey.channel();
            if(socketChannel.equals(this.socketChannel)) {
              try {
                // Clear the buffer and read bytes from socket
                readbuf.clear();

                int numBytesRead = socketChannel.read(readbuf);

                if (numBytesRead == -1) {
                  // No more bytes can be read from the channel
                  state.setValue(SocketConnectorState.CLOSED);
                  socketChannel.close();
                }
                else {
                  // To read the bytes, flip the buffer
                  readbuf.flip();
                  byte[] buf = new byte[readbuf.remaining()];
                  readbuf.get(buf);
                  out.write(buf);
                }
              }
              catch (IOException e) {
                // Connection may have been closed
                state.setValue(SocketConnectorState.UNEXPECTED_TERMINATION);
              }
            }
        }

        if (selKey.isWritable()) {
            // Get channel that's ready for more bytes
            SocketChannel socketChannel = (SocketChannel) selKey.channel();
            if(socketChannel.equals(this.socketChannel)) {
              try {

                int numBytesRead = in.available();

                if (numBytesRead > 0) {

                  byte[] buf = new byte[numBytesRead];
                  in.read(buf);
                  ByteBuffer b = ByteBuffer.wrap(buf);

                  // Write bytes
                  int numBytesWritten = socketChannel.write(b);

                }
              }
              catch (IOException e) {
                // Connection may have been closed
                state.setValue(SocketConnectorState.UNEXPECTED_TERMINATION);
              }
            }
        }
    }
}
