/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.subsystem;

import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.session.SessionDataProvider;

import com.sshtools.j2ssh.transport.MessageStoreEOFException;

import com.sshtools.j2ssh.util.StartStopState;

import org.apache.log4j.Logger;

import java.io.InputStream;
import java.io.OutputStream;


public abstract class SubsystemServer implements SessionDataProvider, Runnable {
    private static Logger log = Logger.getLogger(SubsystemServer.class);
    private SubsystemMessageStore incoming = new SubsystemMessageStore();
    private SubsystemMessageStore outgoing = new SubsystemMessageStore();
    private SubsystemInputStream in = new SubsystemInputStream(outgoing);
    private SubsystemOutputStream out = new SubsystemOutputStream(incoming);
    private SshThread thread;
    private StartStopState state = new StartStopState(StartStopState.STOPPED);

    public SubsystemServer() {
    }

    public InputStream getInputStream() {
        return in;
    }

    public OutputStream getOutputStream() {
        return out;
    }

    public void run() {
        state.setValue(StartStopState.STARTED);

        try {
            while (state.getValue() == StartStopState.STARTED) {
                SubsystemMessage msg = incoming.nextMessage();

                if (msg != null) {
                    onMessageReceived(msg);
                }
            }
        } catch (MessageStoreEOFException meof) {
        }

        thread = null;
    }

    public void start() {
        if (Thread.currentThread() instanceof SshThread) {
            thread = ((SshThread) Thread.currentThread()).cloneThread(this,
                    "SubsystemServer");
            thread.start();
        } else {
            log.error(
                "Subsystem Server must be called from within an SshThread context");
            stop();
        }
    }

    public void stop() {
        state.setValue(StartStopState.STOPPED);
        incoming.close();
        outgoing.close();
    }

    public StartStopState getState() {
        return state;
    }

    protected abstract void onMessageReceived(SubsystemMessage msg);

    protected void registerMessage(int messageId, Class implementor) {
        incoming.registerMessage(messageId, implementor);
    }

    protected void sendMessage(SubsystemMessage msg) {
        outgoing.addMessage(msg);
    }
}
