/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport;

import com.sshtools.j2ssh.SshException;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.PlatformConfiguration;

import com.sshtools.j2ssh.util.InvalidStateException;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


public abstract class Service {
    private static Logger log = Logger.getLogger(Service.class);
    public final static int REQUESTING_SERVICE = 1;
    public final static int ACCEPTING_SERVICE = 2;

    //protected Map nativeSettings;
    protected SshMessageStore messageStore = new SshMessageStore();
    protected TransportProtocol transport;
    protected Integer startMode = null;
    protected ServiceState state = new ServiceState();
    private String serviceName;

    public Service(String serviceName) {
        this.serviceName = serviceName;
    }

    public final String getServiceName() {
        return serviceName;
    }

    public final void start() throws IOException {
        if (startMode == null) {
            throw new ServiceOperationException(
                "Service must be initialized first!");
        }

        // If were accepted (i.e. client) we will call onServiceAccept()
        if (startMode.intValue() == REQUESTING_SERVICE) {
            log.info(serviceName + " has been accepted");
            onServiceAccept();
        } else {
            // We've recevied a request instead
            log.info(serviceName + " has been requested");
            onServiceRequest();
        }

        onStart();

        state.setValue(ServiceState.SERVICE_STARTED);
    }

    protected abstract void onStart() throws IOException;

    public ServiceState getState() {
        return state;
    }

    public void init(int startMode, TransportProtocol transport) //, Map nativeSettings)
        throws IOException {
        if ((startMode != REQUESTING_SERVICE)
                && (startMode != ACCEPTING_SERVICE)) {
            throw new ServiceOperationException("Invalid start mode!");
        }

        this.transport = transport;
        this.startMode = new Integer(startMode);

        //this.nativeSettings = nativeSettings;
        onServiceInit(startMode);

        transport.addMessageStore(messageStore);
    }

    public final void stop() {
        messageStore.close();

        state.setValue(ServiceState.SERVICE_STOPPED);
    }

    protected abstract void onServiceAccept() throws IOException;

    protected abstract void onServiceInit(int startMode)
        throws IOException;

    protected abstract void onServiceRequest() throws IOException;

    protected void sendServiceAccept() throws IOException {
        SshMsgServiceAccept msg = new SshMsgServiceAccept(serviceName);

        transport.sendMessage(msg, this);
    }
}
