/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.authentication;

import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;
import com.sshtools.j2ssh.transport.MessageNotAvailableException;
import com.sshtools.j2ssh.transport.MessageStoreEOFException;
import com.sshtools.j2ssh.transport.Service;
import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.SshMessage;

public class AuthenticationProtocolClient
    extends Service {
  private static Logger log = Logger.getLogger(AuthenticationProtocolClient.class);
  private int[] resultFilter = new int[2];
  private int[] singleIdFilter = new int[3];

  public AuthenticationProtocolClient() {
    super("ssh-userauth");
    resultFilter[0] = SshMsgUserAuthSuccess.SSH_MSG_USERAUTH_SUCCESS;
    resultFilter[1] = SshMsgUserAuthFailure.SSH_MSG_USERAUTH_FAILURE;

    singleIdFilter[0] = SshMsgUserAuthSuccess.SSH_MSG_USERAUTH_SUCCESS;
    singleIdFilter[1] = SshMsgUserAuthFailure.SSH_MSG_USERAUTH_FAILURE;
  }

  protected void onServiceAccept() throws java.io.IOException {
  }

  protected void onStart() {
  }

  protected void onServiceInit(int startMode) throws java.io.IOException {
    if (startMode == Service.ACCEPTING_SERVICE) {
      throw new ServiceOperationException(
          "The Authentication Protocol client cannot be accepted");
    }

    messageStore.registerMessage(SshMsgUserAuthFailure.SSH_MSG_USERAUTH_FAILURE,
                                 SshMsgUserAuthFailure.class);

    messageStore.registerMessage(SshMsgUserAuthSuccess.SSH_MSG_USERAUTH_SUCCESS,
                                 SshMsgUserAuthSuccess.class);

    messageStore.registerMessage(SshMsgUserAuthBanner.SSH_MSG_USERAUTH_BANNER,
                                 SshMsgUserAuthBanner.class);

    messageStore.registerMessage(SshMsgUserAuthPwdChangeReq.
                                 SSH_MSG_USERAUTH_PWD_CHANGEREQ,
                                 SshMsgUserAuthPwdChangeReq.class);
  }

  protected void onServiceRequest() throws java.io.IOException {
    throw new ServiceOperationException(
        "This class implements the client protocol only!");
  }

  public List getAvailableAuths(String username, String serviceName) throws
      IOException {
    log.info("Requesting authentication methods");

    SshMessage msg = new SshMsgUserAuthRequest(username, serviceName,
                                               "none", null);

    transport.sendMessage(msg, this);

    msg = messageStore.getMessage(resultFilter);

    if (msg instanceof SshMsgUserAuthFailure) {
      return ( (SshMsgUserAuthFailure) msg).getAvailableAuthentications();
    }
    else {
      throw new ServiceOperationException(
          "None request returned success! Insecure feature not supported");
    }
  }

  public int authenticate(SshAuthenticationClient auth, Service serviceToStart) throws
      IOException {
    try {

      if (!auth.canAuthenticate() && auth.canPrompt()) {
        SshAuthenticationPrompt prompt = auth.getAuthenticationPrompt();
        if (!prompt.showPrompt()) {
          return AuthenticationProtocolState.CANCELLED;
        }
      }

      auth.authenticate(this, serviceToStart.getServiceName());

      SshMessage msg = parseMessage(messageStore.getMessage(resultFilter));

      // We should not get this far
      throw new AuthenticationProtocolException(
          "Unexpected authentication message " + msg.getMessageName());
    }
    catch (TerminatedStateException tse) {
      if (tse.getState() == AuthenticationProtocolState.COMPLETE) {
        serviceToStart.init(Service.ACCEPTING_SERVICE, transport); //, nativeSettings);
        serviceToStart.start();
      }

      return tse.getState();
    }
  }

  public void sendMessage(SshMessage msg) throws IOException {
    transport.sendMessage(msg, this);
  }

  public byte[] getSessionIdentifier() {
    return transport.getSessionIdentifier();
  }

  public void registerMessage(Class cls, int messageId) {
    messageStore.registerMessage(messageId, cls);
  }

  public SshMessage readMessage(int messageId) throws TerminatedStateException,
      AuthenticationProtocolException {
    singleIdFilter[2] = messageId;

    return internalReadMessage(singleIdFilter);
  }

  private SshMessage internalReadMessage(int[] messageIdFilter) throws
      TerminatedStateException, AuthenticationProtocolException {
    try {
      SshMessage msg = messageStore.getMessage(messageIdFilter);

      return parseMessage(msg);
    }
    catch (MessageStoreEOFException meof) {
      throw new AuthenticationProtocolException("Failed to read messages");
    }
  }

  public SshMessage readMessage(int[] messageId) throws
      TerminatedStateException, AuthenticationProtocolException {
    int[] messageIdFilter = new int[messageId.length + resultFilter.length];
    System.arraycopy(resultFilter, 0, messageIdFilter, 0,
                     resultFilter.length);
    System.arraycopy(messageId, 0, messageIdFilter, resultFilter.length,
                     messageId.length);

    return internalReadMessage(messageIdFilter);
  }

  private SshMessage parseMessage(SshMessage msg) throws
      TerminatedStateException {
    if (msg instanceof SshMsgUserAuthFailure) {
      if ( ( (SshMsgUserAuthFailure) msg).getPartialSuccess()) {
        throw new TerminatedStateException(AuthenticationProtocolState.PARTIAL);
      }
      else {
        throw new TerminatedStateException(AuthenticationProtocolState.FAILED);
      }
    }
    else if (msg instanceof SshMsgUserAuthSuccess) {
      throw new TerminatedStateException(AuthenticationProtocolState.COMPLETE);
    }
    else {
      return msg;
    }
  }

  public String getBannerMessage(int timeout) {
    try {
      log.debug(
          "getBannerMessage is attempting to read the authentication banner");

      SshMessage msg = messageStore.peekMessage(SshMsgUserAuthBanner.
                                                SSH_MSG_USERAUTH_BANNER,
                                                timeout);

      return ( (SshMsgUserAuthBanner) msg).getBanner();
    }
    catch (MessageNotAvailableException e) {
      return "";
    }
    catch (MessageStoreEOFException eof) {
      log.error(
          "Failed to retreive banner becasue the message store is EOF");

      return "";
    }
  }
}
