package com.sshtools.j2ssh.authentication;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Iterator;

import org.apache.log4j.Logger;
import com.sshtools.j2ssh.configuration.Authorization;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.ServerConfiguration;
import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;
import com.sshtools.j2ssh.platform.NativeAuthenticationProvider;
import com.sshtools.j2ssh.transport.publickey.SshKeyPair;
import com.sshtools.j2ssh.transport.publickey.SshKeyPairFactory;
import com.sshtools.j2ssh.transport.publickey.SshPublicKey;
import com.sshtools.j2ssh.transport.publickey.SshPublicKeyFile;


public class AuthorizationFileVerification implements PublicKeyVerification {

  private static Logger log = Logger.getLogger(AuthorizationFileVerification.class);

  public boolean verifyKeySignature(String username, String algorithm, byte[] encoded, String service,
                                    byte[] sessionId, byte[] signature)
      throws IOException {

    NativeAuthenticationProvider provider = NativeAuthenticationProvider.getInstance();

    String userHome = provider.getHomeDirectory(username); //, nativeSettings);

      if (userHome == null) {
        log.warn("There is no home directory for " + username
                 + " is available");
      }

      // Replace '\' with '/' because when we use it in String.replaceAll
      // for some reason it removes them?
      userHome = userHome.replace('\\', '/');

      ServerConfiguration config = ConfigurationLoader
          .getServerConfiguration();
      String authorizationFile;
      String userConfigDir = config.getUserConfigDirectory();

      // First replace any '\' with '/' (Becasue replaceAll removes them!)
      userConfigDir = userConfigDir.replace('\\', '/');

      // Replace any home directory tokens
      if ( (userConfigDir.indexOf("%D") > -1) && (userHome == null)) {
        log.error(
            "<UserConfigDirectory> requires home directory, but none available for "
            + username);

        return false;
      }

      userConfigDir = userConfigDir.replaceAll("%D", userHome);

      // Replace any username tokens
      userConfigDir = userConfigDir.replaceAll("%U", username);

      // Replace the '/' with File.seperator and trim
      userConfigDir = userConfigDir.replace('/', File.separatorChar).trim();

      if (!userConfigDir.endsWith(File.separator)) {
        userConfigDir += File.separator;
      }

      authorizationFile = userConfigDir + config.getAuthorizationFile();

      // Load the authorization file
      File file = new File(authorizationFile);

      if (!file.exists()) {
        log.info("authorizationFile: " + authorizationFile
                 + " does not exist.");
        log.info(
            "Authentication failed because no authorization file is available");

        return false;
      }

      FileInputStream in = new FileInputStream(file);
      Authorization keys;

      try {
        keys = new Authorization(in);
      }
      catch (Exception e) {
        throw new AuthenticationProtocolException(
            "Failed to load authorized keys file " + authorizationFile);
      }

      Iterator it = keys.getAuthorizedKeys().iterator();

      SshKeyPair pair = SshKeyPairFactory.newInstance(algorithm);
      SshPublicKey authorizedKey = null;
      SshPublicKey key = pair.decodePublicKey(encoded);
      boolean valid = false;
      String keyfile;

      while (it.hasNext()) {
        keyfile = (String) it.next();

        // Look for the file in the user config dir first
        file = new File(userConfigDir + keyfile);

        // If it does not exist then look absolute
        if (!file.exists()) {
          file = new File(keyfile);
        }

        if (file.exists()) {
          // Try to open the public key in the default file format
          // otherwise attempt the supported key formats
          SshPublicKeyFile pkf = SshPublicKeyFile.parse(file);
          authorizedKey = pkf.toPublicKey();

          if (authorizedKey.getFingerprint().equals(key
              .getFingerprint())) {


            ByteArrayWriter data = new ByteArrayWriter();
            data.writeBinaryString(sessionId);
            data.write(SshMsgUserAuthRequest.SSH_MSG_USERAUTH_REQUEST);
            data.writeString(username);
            data.writeString(service);
            data.writeString("publickey");
            data.write(1);
            data.writeString(key.getAlgorithmName());
            data.writeBinaryString(key.getEncoded());

            if (key.verifySignature(signature, data.toByteArray())) {
              return true;
            }
          }
        }
        else {
          log.info("Failed attempt to load key file " + keyfile);
        }
      }

      return false;
  }

}