/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.forwarding;

import com.sshtools.j2ssh.SshException;
import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;

import com.sshtools.j2ssh.connection.ConnectionProtocol;

import com.sshtools.j2ssh.util.StartStopState;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;


import com.sshtools.j2ssh.io.*;
import java.nio.channels.*;

public abstract class ForwardingListener extends ForwardingConfiguration
    implements Runnable {
    private static Logger log = Logger.getLogger(ForwardingListener.class);
    private ConnectionProtocol connection;
    private ServerSocket server;
    private Thread thread;
    private boolean listening;

    public ForwardingListener(String name, ConnectionProtocol connection,
        String addressToBind, int portToBind, String hostToConnect,
        int portToConnect) {
        super(name, addressToBind, portToBind, hostToConnect, portToConnect);

        log.info("Creating forwarding listener named '" + name + "'");

        this.connection = connection;

        if (log.isDebugEnabled()) {
            log.debug("Address to bind: " + getAddressToBind());
            log.debug("Port to bind: " + String.valueOf(getPortToBind()));
            log.debug("Host to connect: " + hostToConnect);
            log.debug("Port to connect: " + portToConnect);
        }
    }

    public ForwardingListener(ConnectionProtocol connection,
        String addressToBind, int portToBind) {
        this(addressToBind + ":" + String.valueOf(portToBind), connection,
            addressToBind, portToBind, "[Specified by connecting computer]", -1);
    }

    public int getLocalPort() {
        return server == null ? -1 : server.getLocalPort();
    }

    public boolean isListening() {
        return listening;
    }

    public void run() {
        try {
            log.info("Starting forwarding listener thread for '" + name + "'");

//
//            ServerSocket server = new ServerSocket(getPortToBind(), 50, InetAddress.getByName(getAddressToBind()));
            //server = new ServerSocket(getPortToBind(), 50, InetAddress.getByName(getAddressToBind()));

            Socket socket;

            while (state.getValue() == StartStopState.STARTED) {
                listening = true;

                socket = server.accept();

                if ((state.getValue() == StartStopState.STOPPED)
                        || (socket == null)) {
                    break;
                }

                log.info("Connection accepted, creating forwarding channel");


                try {
                  ForwardingChannel channel = createChannel(hostToConnect,
                      portToConnect, socket);

                  channel.bindSocket(socket);

                  if(connection.openChannel(channel)) {
                    log.info("Forwarding channel for '" + name + "' is open");
                  } else
                    log.warn("Failed to open forwarding chanel " + name);
                }
                catch (Exception ex) {
                  log.warn("Failed to open forwarding chanel " + name, ex);
                }

            }
        } catch (IOException ioe) {
            /* only warn if the forwarding has not been stopped */
            if (state.getValue() == StartStopState.STARTED) {
                log.warn("Local forwarding listener to " + hostToConnect + ":"
                    + String.valueOf(portToConnect) + " has failed", ioe);
            }
        } finally {
            stop();
        }
    }

    public boolean isRunning() {
        return (thread != null) && thread.isAlive();
    }

    public void start() throws IOException{
        /* Set the state by calling the super method */
        super.start();

        /* Bind server socket */
        try {
            server = new ServerSocket(getPortToBind(), 50, InetAddress.getByName(getAddressToBind()));
        } catch (IOException ioe) {
            log.warn("Local forwarding listener to " + hostToConnect + ":"
              + String.valueOf(portToConnect) + " failed to open socket", ioe);
            super.stop();
            return;
        }

                 /* Create a thread and start it */
                 thread = new SshThread(this, "Forwarding listener", true);


        /* Create a thread and start it */
        thread = new SshThread(this, "Forwarding listener", true);
        thread.start();
    }

    public void stop() {
        /* Set the state by calling the super method */
        super.stop();

        try {
            /* Close the server socket */
            if (server != null) {
                server.close();
            }
        } catch (IOException ioe) {
            log.warn("Forwarding listener failed to stop", ioe);
        }

        thread = null;
        listening = false;
    }

    protected abstract ForwardingChannel createChannel(String hostToConnect,
        int portToConnect, Socket socket)
        throws ForwardingConfigurationException;
}
