/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport;

import com.sshtools.j2ssh.io.ByteArrayReader;

import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public final class SshMessageStore {
    private static Logger log = Logger.getLogger(SshMessageStore.class);

    // List to hold messages as they are received
    private List messages = new ArrayList();
    private Map register = new HashMap();
    private boolean isClosed = false;
    private int[] singleIdFilter = new int[1];

    public SshMessageStore() {
    }

    public boolean isClosed() {
        return isClosed;
    }

    public synchronized SshMessage getMessage(int[] messageIdFilter)
        throws MessageStoreEOFException {
        try {
            return getMessage(messageIdFilter, 0);
        } catch (MessageNotAvailableException e) {
            // This should never happen but throw just in case
            throw new MessageStoreEOFException();
        }
    }

    public synchronized SshMessage getMessage(int[] messageIdFilter, int timeout)
        throws MessageStoreEOFException, MessageNotAvailableException {
        if ((messages.size() <= 0) && isClosed) {
            throw new MessageStoreEOFException();
        }

        if (messageIdFilter == null) {
            return nextMessage();
        }

        SshMessage msg;
        boolean firstPass = true;

        if (timeout < 0) {
            timeout = 0;
        }

        while ((messages.size() > 0) || !isClosed) {
            // lookup the message
            msg = lookupMessage(messageIdFilter, true);

            if (msg != null) {
                return msg;
            }

            // Now wait
            try {
                if (!isClosed) {
                    wait(timeout);
                }
            } catch (InterruptedException ie) {
            }

            // If we wanted to timeout then throw
            if (firstPass && (timeout > 0)) {
                throw new MessageNotAvailableException();
            }

            firstPass = false;
        }

        throw new MessageStoreEOFException();
    }

    public synchronized SshMessage getMessage(int messageId)
        throws MessageStoreEOFException {
        try {
            return getMessage(messageId, 0);
        } catch (MessageNotAvailableException e) {
            // This should never happen by throw jsut in case
            throw new MessageStoreEOFException();
        }
    }

    public synchronized SshMessage getMessage(int messageId, int timeout)
        throws MessageStoreEOFException, MessageNotAvailableException {
        singleIdFilter[0] = messageId;

        return getMessage(singleIdFilter, timeout);

   }

   public boolean hasMessages() {
     return messages.size() > 0;
   }

   public int size() {
     return messages.size();
   }

    public boolean isRegisteredMessage(Integer messageId) {
        return register.containsKey(messageId);
    }

    public void addMessage(byte[] msgdata)
        throws MessageNotRegisteredException, InvalidMessageException {
        Integer messageId = new Integer(msgdata[5]);

        if (!isRegisteredMessage(messageId)) {
            throw new MessageNotRegisteredException(messageId);
        }

        Class cls = (Class) register.get(SshMessage.getMessageId(msgdata));

        try {
            SshMessage msg = (SshMessage) cls.newInstance();
            msg.fromByteArray(new ByteArrayReader(msgdata));
            addMessage(msg);
        } catch (IllegalAccessException iae) {
            throw new InvalidMessageException(
                "Illegal access for implementation class " + cls.getName());
        } catch (InstantiationException ie) {
            throw new InvalidMessageException("Instantiation failed for class "
                + cls.getName());
        }
    }

    public synchronized void addMessage(SshMessage msg)
        throws MessageNotRegisteredException {
        // Add the message
        messages.add(messages.size(), msg);

        // Notify the threads
        notifyAll();
    }

    public synchronized void close() {
        isClosed = true;

        // We need to notify all anyway as if there are messages still available
        // it should not affect the waiting threads as they are waiting for their
        // own messages to be received because non were avaialable in the first place
        //if (messages.size()<=0) {
        notifyAll();

        //}
    }

    public synchronized SshMessage nextMessage()
        throws MessageStoreEOFException {
        if ((messages.size() <= 0) && isClosed) {
            throw new MessageStoreEOFException();
        }

        // If there are no messages available then wait untill there are.
        while ((messages.size() <= 0)) {
            try {
                wait();
            } catch (InterruptedException e) {
            }
        }

        if (messages.size() > 0) {
            return (SshMessage) messages.remove(0);
        } else {
            throw new MessageStoreEOFException();
        }
    }

    public synchronized SshMessage peekMessage(int[] messageIdFilter)
        throws MessageStoreEOFException, MessageNotAvailableException {
        return peekMessage(messageIdFilter, 0);
    }

    public synchronized SshMessage peekMessage(int[] messageIdFilter,
        int timeout)
        throws MessageStoreEOFException, MessageNotAvailableException {
        SshMessage msg;

        // Do a straight lookup
        msg = lookupMessage(messageIdFilter, false);

        if (msg != null) {
            return msg;
        }


        // If were willing to wait the wait and look again
        if (timeout > 0) {

           log.debug("No message so waiting for " + String.valueOf(timeout)
            + " milliseconds");

           try {
                wait(timeout);
                msg = lookupMessage(messageIdFilter, false);

                if (msg != null) {
                    return msg;
                }
            } catch (InterruptedException e) {
                log.debug("peekMessage was interrupted");
            }
        }

        // Nothing even after a wait so throw the relevant exception
        if (isClosed) {
            throw new MessageStoreEOFException();
        } else {
            throw new MessageNotAvailableException();
        }
    }

    private SshMessage lookupMessage(int[] messageIdFilter, boolean remove) {
        SshMessage msg;

        for (int x = 0; x < messages.size(); x++) {
            msg = (SshMessage) messages.get(x);

            // Determine whether its one of the filtered messages
            for (int i = 0; i < messageIdFilter.length; i++) {
                if (msg.getMessageId() == messageIdFilter[i]) {
                    if (remove) {
                        messages.remove(msg);
                    }

                    return msg;
                }
            }
        }

        return null;
    }

    public synchronized SshMessage peekMessage(int messageId)
        throws MessageStoreEOFException, MessageNotAvailableException {
        return peekMessage(messageId, 0);
    }

    public synchronized void removeMessage(SshMessage msg) {
      messages.remove(msg);
    }

    public synchronized SshMessage peekMessage(int messageId, int timeout)
        throws MessageStoreEOFException, MessageNotAvailableException {
        singleIdFilter[0] = messageId;

        return peekMessage(singleIdFilter, timeout);

        /*SshMessage msg;

          if (messages.size()>0) {
              for (int x = 0;x<messages.size();x++) {
                  msg = (SshMessage) messages.get(x);

                  if (msg.getMessageId()==messageId) {
                      return msg;
                  }
              }
          }

          if (isClosed) {
              throw new MessageStoreEOFException();
          } else {
              throw new MessageNotAvailableException();
          }*/
    }

    public void registerMessage(int messageId, Class implementor) {
        Integer id = new Integer(messageId);
        register.put(id, implementor);
    }

    public Object[] getRegisteredMessageIds() {
        return register.keySet().toArray();
    }

    public SshMessage createMessage(byte[] msgdata)
        throws MessageNotRegisteredException, InvalidMessageException {
        Integer messageId = SshMessage.getMessageId(msgdata);

        if (!isRegisteredMessage(messageId)) {
            throw new MessageNotRegisteredException(messageId);
        }

        Class cls = (Class) register.get(SshMessage.getMessageId(msgdata));

        try {
            SshMessage msg = (SshMessage) cls.newInstance();
            msg.fromByteArray(new ByteArrayReader(msgdata));

            return msg;
        } catch (IllegalAccessException iae) {
            throw new InvalidMessageException(
                "Illegal access for implementation class " + cls.getName());
        } catch (InstantiationException ie) {
            throw new InvalidMessageException("Instantiation failed for class "
                + cls.getName());
        }
    }
}
