/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport.publickey;

import com.sshtools.j2ssh.configuration.ConfigurationException;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.ExtensionAlgorithm;
import com.sshtools.j2ssh.configuration.SshAPIConfiguration;

import com.sshtools.j2ssh.transport.AlgorithmNotSupportedException;

import com.sshtools.j2ssh.transport.publickey.dsa.SshDssKeyPair;
import com.sshtools.j2ssh.transport.publickey.rsa.SshRsaKeyPair;

import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.sshtools.j2ssh.io.ByteArrayReader;

import java.io.IOException;

public class SshKeyPairFactory {
    private static Map pks;
    private static String defaultAlgorithm;
    private static Logger log = Logger.getLogger(SshKeyPairFactory.class);

    static {
        pks = new HashMap();

        log.info("Loading public key algorithms");

        pks.put("ssh-dss", SshDssKeyPair.class);
        pks.put("ssh-rsa", SshRsaKeyPair.class);

        try {
            // Load external pks from configuration file
            SshAPIConfiguration config = ConfigurationLoader
                .getAPIConfiguration();

            if (config != null) {
                List list = config.getPublicKeyExtensions();

                if (list != null) {
                    Iterator it = list.iterator();

                    while (it.hasNext()) {
                        ExtensionAlgorithm algorithm = (ExtensionAlgorithm) it
                            .next();

                        String name = algorithm.getAlgorithmName();

                        if (pks.containsKey(name)) {
                            log.debug("Standard public key " + name
                                + " is being overidden by "
                                + algorithm.getImplementationClass());
                        } else {
                            log.debug(algorithm.getAlgorithmName()
                                + " public key is implemented by "
                                + algorithm.getImplementationClass());
                        }

                        try {
                            pks.put(algorithm.getAlgorithmName(),
                                ConfigurationLoader.getExtensionClass(
                                    algorithm.getImplementationClass()));
                        } catch (ClassNotFoundException cnfe) {
                            log.error("Could not locate "
                                + algorithm.getImplementationClass());
                        }
                    }
                }

                defaultAlgorithm = config.getDefaultPublicKey();
            }
        } catch (ConfigurationException ex) {
        }

        if ((defaultAlgorithm == null) || !pks.containsKey(defaultAlgorithm)) {
            log.debug("The default public key is not set! using first in list");

            Iterator it = pks.keySet().iterator();
            defaultAlgorithm = (String) it.next();
        }
    }

    protected SshKeyPairFactory() {
    }

    public static void initialize() {
    }

    public static String getDefaultPublicKey() {
        return defaultAlgorithm;
    }

    public static List getSupportedKeys() {
        // Get the list of pks
        return new ArrayList(pks.keySet());
    }

    public static SshKeyPair newInstance(String methodName)
        throws AlgorithmNotSupportedException {
        try {
            return (SshKeyPair) ((Class) pks.get(methodName)).newInstance();
        } catch (Exception e) {
            throw new AlgorithmNotSupportedException(methodName
                + " is not supported!");
        }
    }

    public static boolean supportsKey(String algorithm) {
        return pks.containsKey(algorithm);
    }

    public static SshPrivateKey decodePrivateKey(byte[] encoded) throws InvalidSshKeyException, AlgorithmNotSupportedException {
      try {

        ByteArrayReader bar = new ByteArrayReader(encoded);
        String algorithm = bar.readString();
        if (supportsKey(algorithm)) {
          SshKeyPair pair = newInstance(algorithm);
          return pair.decodePrivateKey(encoded);
        }
        else
          throw new AlgorithmNotSupportedException(algorithm +
                                                   " is not supported");
      } catch(IOException ioe) {
        throw new InvalidSshKeyException(ioe.getMessage());
      }
    }


        public static SshPublicKey decodePublicKey(byte[] encoded) throws InvalidSshKeyException,
            AlgorithmNotSupportedException {

          try {
            ByteArrayReader bar = new ByteArrayReader(encoded);
            String algorithm = bar.readString();
            if (supportsKey(algorithm)) {
              SshKeyPair pair = newInstance(algorithm);
              return pair.decodePublicKey(encoded);
            }
            else
              throw new AlgorithmNotSupportedException(algorithm +
                                                       " is not supported");
          } catch(IOException ioe) {
            throw new InvalidSshKeyException(ioe.getMessage());
          }
        }

}
