/* ssh2dos.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date:$
 * $Revision:$
 *
 * This module is the main part:
 *  - command line parsing
 *  - client loop
 *  - running remote command or interactive session
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <time.h>

#if defined (__DJGPP__)
 #include <unistd.h>
 #include <go32.h>
 #include <dpmi.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include <io.h>
 #include <dos.h> 
 #include "tcp.h"
#endif

#include "channel.h"
#include "transprt.h"
#include "ssh.h"
#include "version.h"
#include "vt100.h"
#include "xmalloc.h"
#include "config.h"
#include "macros.h"

/* external structures, variables */
void fatal(const char *fmt, ...);
short TCPConnect(char *, unsigned short);
short SSHConnect(char *, char *, char *);
extern struct Packet pktin;	/* incoming SSH2 packet */
extern struct Packet pktout;	/* outgoing SSH2 packet */
extern char *RemoteClosed;
extern char *ConnectionClosed;
extern char *protocolerror;
extern unsigned short statusline;

/* global variables */
Config GlobalConfig;		/* global configuration structure */
unsigned short Configuration = 0;	/* Configuration bits */

/* local static variables */
static char *terminals[]={"xterm","vt100","linux","xterm-color"}; /* terminal types */
static char *command = NULL;
static short tty = 0;
static char *RemoteHost = NULL;
static char *UserName = NULL;
static char *PassWord = NULL;
static char *KeyFile = NULL;
static char *term;
static unsigned short RemotePort = SSH_PORT;
static unsigned short Keepalives = 0;
static FILE *LogFile = NULL;

#if defined (__DJGPP__)
/*
 * we don't want DJGPP to extend wildcards, so include
 * this dummy function
 */
char **__crt0_glob_function (char *arg)
{
   return 0;
}

/*
 * Some keepalive support tricks. In DJGPP, we must ensure,
 * that interrupt handlers remain always in memory, not
 * swapped out to disk. That's why we need end_keepalive
 * dummy function. Also note, that we don't need to call
 * the old handler under DJGPP.
 */
unsigned long timer = 0;	/* increased by timer interrupt */

 _go32_dpmi_seginfo oldhandler, my_handler;	/* for old and new timer interrupt */

/* Timer handler */
void keepalive(void)
{
   timer++;
}
void end_keepalive(void) {}	/* dummy, not to swap out keepalive func. */

#elif defined (__TURBOC__)

volatile unsigned long timer = 0;	/* increased by timer interrupt */

void interrupt (*oldhandler)(void); /* for old timer interrupt */

void interrupt keepalive(void){  /* Timer handler */
   timer++;
   oldhandler();
}

#endif

/*
 * Initialize global variables
 */
static void Config_Init(void){
   term = terminals[0]; /* default is "xterm" */
   GlobalConfig.debugfile = NULL;
   GlobalConfig.brailab = NULL;
}

/*
 * Allocate a pseudo terminal
 */
static short SSH2_Request_Pty(char *termtype)
{
   if(Configuration & VERBOSE_MODE)
        puts("Requesting PTY");
   SSH2_Channel_PktInit(SSH_MSG_CHANNEL_REQUEST);
   SSH2_putstring("pty-req");
   SSH2_putbool(1);
   SSH2_putstring(termtype);
   SSH2_putuint32(80);
   SSH2_putuint32(25 - statusline);
   SSH2_putuint32(0);
   SSH2_putuint32(0);
   SSH2_putstring("\0");
   SSH2_pkt_send();

   if(SSH2_Channel_Read(NULL))
	return(1);

   switch(pktin.type){
        case SSH_MSG_CHANNEL_SUCCESS:
           break;

        case SSH_MSG_CHANNEL_FAILURE:
	   SSH2_Disconnect(SSH_DISCONNECT_BY_APPLICATION, "Cannot allocate PTY");
	   return(1);

        default:
	   SSH2_Disconnect(SSH_DISCONNECT_PROTOCOL_ERROR, protocolerror);
	   return(1);
   }
   return(0);
}

/*
 * Start interactive shell or run command
 */
static short SSH2_Start_Shell_Or_Command(void)
{
   if(command != NULL && command[0] != '\0') {
	if(Configuration & VERBOSE_MODE)
           puts("Running command");
	SSH2_Channel_PktInit(SSH_MSG_CHANNEL_REQUEST);
        SSH2_putstring("exec");
        SSH2_putbool(1);
        SSH2_putstring(command);
	xfree(command);
   } else {
	if(Configuration & VERBOSE_MODE)
           puts("Running shell");
	SSH2_Channel_PktInit(SSH_MSG_CHANNEL_REQUEST);
        SSH2_putstring("shell");
        SSH2_putbool(1);
   }
   SSH2_pkt_send();

   if(SSH2_Channel_Read(NULL))
	return(1);

   switch(pktin.type){
        case SSH_MSG_CHANNEL_SUCCESS:
           break;

        case SSH_MSG_CHANNEL_FAILURE:
	   SSH2_Disconnect(SSH_DISCONNECT_BY_APPLICATION, "Cannot run shell or command");
	   return(1);

        default:
           SSH2_Disconnect(SSH_DISCONNECT_PROTOCOL_ERROR, protocolerror);
	   return(1);
   }
   return(0);
}

/*
 * Client loop. This runs when the user successfully logged in,
 * until SSH connection is terminated
 */
static short dosession(void)
{
char *str;
int status;
unsigned long len;
unsigned short i;

   do{
        /* send keepalive SSH_MSG_IGNORE packet if configured */
        if( timer > Keepalives ){
	   SSH2_pkt_init(SSH_MSG_IGNORE);
	   SSH2_putuint32(0);
	   SSH2_pkt_send();
	   timer = 0;
        } /* if */

        sock_tick(&GlobalConfig.s, &status); /* TCP wait */
        while (ConChk())	/* examine STDIN */
	   DoKey();
   } while(!sock_dataready(&GlobalConfig.s));

   if(SSH2_Channel_Read(NULL)) /* uncrypt and get valuable data */
	return(EXIT_SSH);

   if(pktin.type == SSH_MSG_CHANNEL_DATA || /* we got data to display */
      pktin.type == SSH_MSG_CHANNEL_EXTENDED_DATA){
        if(pktin.type == SSH_MSG_CHANNEL_EXTENDED_DATA)
           pktin.ptr += 4;
	SSH2_getstring(&str, &len); /* get and display data */
	for (i = 0; i < len; i++){
           if(tty)
                ConOut(str[i]);
	   else
                putchar(str[i]);
	}
	if(LogFile)
	   fwrite(str, 1, len, LogFile);
   }
   return(0);

sock_err:
   switch (status){
        case 1 :
	   puts(ConnectionClosed);
           break;
        case -1:
           puts(RemoteClosed);
           break;
   }
   return(EXIT_SSH);
}

/*
 * Get command line arguments
 */
static void getargs(int argc, char *argv[])
{
unsigned short i, j, len;
char *s;
char *keymaperror="Specified keymap not found!";
char *usage="Usage: sshdos [options] username remotehost [command [args]]\n"
	    "Options:\n"
	    "-i <identity file>                 - key file for public key authentication\n"
	    "-t <xterm|vt100|linux|xterm-color> - terminal type (default: xterm)\n"
	    "-p <port number>                   - remote port\n"
	    "-k <keymap file>                   - path to keymap file\n"
	    "-s <password>                      - remote password\n"
	    "-l <log file>                      - log session to file\n"
	    "-a <minutes>                       - time between keepalive packets\n"
	    "-b <COM[1234]>                     - Brailab PC adapter on COM[1234] port\n"
	    "-P                                 - don't allocate a privileged port\n"
	    "-C                                 - enable compression\n"
	    "-S                                 - disable status line\n"
	    "-n                                 - add CR if server sends only LF\n"
	    "-d                                 - save SSH packets to debug.pkt\n"
	    "-v                                 - verbose output";

   for (i = 1; i < argc; ++i){
	s = argv[i];
	if(*s != '-')
           break;
	switch (*++s){
	   case '\0':
		fatal(usage);
		return;

	   case 'i':
		if(*++s)
		   KeyFile = s;
		else if(++i < argc)
		   KeyFile = argv[i];
		else
		   fatal(usage);
		continue;

	   case 's':
		if(*++s)
		   PassWord = strdup(s);
		else if(++i < argc)
		   PassWord = strdup(argv[i]);
		else
		   fatal(usage);
		PassWord[MAX_PASSWORD_LENGTH - 1] = '\0';
		continue;

	   case 'l':
		if(*++s){
		   if((LogFile = fopen(s,"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else if(++i < argc){
		   if((LogFile = fopen(argv[i],"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else
		   fatal(usage);
		continue;

	   case 't':
		if(*++s){
		  if(!strcmp(s,terminals[0]) ||
		     !strcmp(s,terminals[1]) ||
		     !strcmp(s,terminals[2]) ||
		     !strcmp(s,terminals[3]) )
		     term = s;
		  else
                        fatal(usage);
		}
		else if(++i < argc){
		  if(!strcmp(argv[i],terminals[0]) ||
		     !strcmp(argv[i],terminals[1]) ||
		     !strcmp(argv[i],terminals[2]) ||
		     !strcmp(argv[i],terminals[3]) )
		     term = argv[i];
		  else
                        fatal(usage);
		}
	    	else
                   fatal(usage);
		continue;

	   case 'p':
		if(*++s)
		   RemotePort = atoi(s);
		else if(++i < argc)
		   RemotePort = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'a':
		if(*++s)
		   Keepalives = atoi(s);
		else if(++i < argc)
		   Keepalives = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'b':
		if(*++s){
		   if(!strcmp(s, "COM1") ||
		      !strcmp(s, "COM2") ||
		      !strcmp(s, "COM3") ||
		      !strcmp(s, "COM4")){
			if((GlobalConfig.brailab = fopen(s,"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i], "COM1") ||
		      !strcmp(argv[i], "COM2") ||
		      !strcmp(argv[i], "COM3") ||
		      !strcmp(argv[i], "COM4")){
			if((GlobalConfig.brailab = fopen(argv[i],"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else
		   fatal(usage);
		continue;

	   case 'k':
		if(*++s){
		   if(keymap_init(s))
			fatal(keymaperror);
		}
		else if(++i < argc){
		   if(keymap_init(argv[i]))
			fatal(keymaperror);
		}
		else
		   fatal(usage);
		continue;

	   case 'P':
		Configuration |= NONPRIVILEGED_PORT;
		continue;

	   case 'S':
		statusline = 0;
		continue;

	   case 'C':
		Configuration |= COMPRESSION_REQUESTED;
		continue;

	   case 'n':
		Configuration |= NEWLINE;
		continue;

	   case 'd':
		if((GlobalConfig.debugfile = fopen("debug.pkt","w+")) == NULL)
		   fatal("Cannot create debug file");
		else
		   fputs("\n-------------------\n",GlobalConfig.debugfile);
		continue;

	   case 'v':
		Configuration |= VERBOSE_MODE;
		continue;

	   default:
		fatal(usage);
	} /* end switch */

   } /* end for */

   /* no_more_options */
   if(i + 2 > argc)
        fatal(usage);
   UserName = argv[i++];
   RemoteHost = argv[i++];
   if(i >= argc)			/* command args? */
	return;
   /* collect remaining arguments and make a command line of them */
   for(len = 0, j = i; j < argc; j++)
	len += strlen(argv[j]) + 1;	/* 1 for the separating space */
   command = (char *)xmalloc(len);
   for(*command = '\0', j = i; j < argc; j++){
	strcat(command, argv[j]);	/* inefficient, but no big deal */
	if(j < argc - 1)		/* last argument? */
            strcat(command, " ");
   }
}

/*
 * Main program starts here
 */
int main(int argc, char **argv)
{
   printf("SSH2DOS v%s\n", SSH_VERSION);

   Config_Init();	/* Initialize global variables */
   srand(time(NULL));	/* Initialize random number generator */

   getargs(argc, argv); /* Process command line */

   if(TCPConnect(RemoteHost, RemotePort))	/* Connect to server */
	fatal("Cannot start TCP connection");
   if(SSHConnect(UserName, PassWord, KeyFile))	/* begin SSH negotiation */
	fatal("");

   /* Open a channel */
   if(SSH2_Channel_Open())
	fatal("");

   /* Request a pseudo terminal */
   if(isatty(fileno(stdout)))
	tty = 1;
   if(SSH2_Request_Pty(term))
	fatal("");

   /* Start an interactive shell or run specified command */
   if(SSH2_Start_Shell_Or_Command())
	fatal("");

#if defined (__DJGPP__)
   tcp_cbreak(1);	/* No Break checking under DJGPP */
#endif

   KeyInit();
   VidInit(UserName, RemoteHost);
   VTInit();

   if(Keepalives){	/* install keepalive timer */
	Keepalives = Keepalives * 18 * 60; /* correct keepalives value */
#if defined (__DJGPP__)
        _go32_dpmi_lock_data(&timer, sizeof(timer));
        _go32_dpmi_lock_code(keepalive, (long)end_keepalive - (long) keepalive);
        _go32_dpmi_get_protected_mode_interrupt_vector(0x1C, &oldhandler);
        my_handler.pm_offset = (int) keepalive;
        my_handler.pm_selector = _go32_my_cs();
        _go32_dpmi_chain_protected_mode_interrupt_vector(0x1C, &my_handler);
#elif defined (__TURBOC__)
        oldhandler = getvect(0x1C);
        setvect(0x1C, keepalive);
#endif
   } /* if */

   while(EXIT_SSH != dosession());	/* Loop until session end */

#ifdef __TURBOC__
   if(Configuration & COMPRESSION_ENABLED)
	Disable_Compression();
#endif

   if(Keepalives)
#if defined (__DJGPP__)
	_go32_dpmi_set_protected_mode_interrupt_vector(0x1C, &oldhandler);
#elif defined (__TURBOC__)
	setvect(0x1C, oldhandler);
#endif

   sock_close(&GlobalConfig.s); /* Close TCP socket */

   /* Close open files */
   if(GlobalConfig.brailab)
	fclose(GlobalConfig.brailab);
   if(GlobalConfig.debugfile)
	fclose(GlobalConfig.debugfile);
   if(LogFile)
	fclose(LogFile);

   return(0);
}
