/* protocol.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/03/28 17:02:39 $
 * $Revision: 1.10 $
 *
 * This module deals with incoming and outgoing packets.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version, with the
 * additional permission that it may be linked against Erick Engelke's
 * WATTCP source code and Jerry Joplin's CVT100 source code.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <mem.h>

#if defined (__DJGPP__)
 #include <errno.h>
 #include <string.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include "tcp.h"
#endif

#include "ssh.h"
#include "protocol.h"
#include "des.h"
#include "blowfish.h"
#include "crc.h"
#include "zlib.h"
#include "xmalloc.h"

extern void fatal(const char *fmt, ...);

extern tcp_Socket s;
extern char verbose;
extern FILE *fileptr;
extern int status;
extern char done[];
int ssh_port=22;

struct Packet pktin = { 0, 0, 0, NULL, 0 };
struct Packet pktout = { 0, 0, 0, NULL, 0 };

DESCon keys[3];		/* 3DES contexts */
BlowfishContext ectx, dctx; /* blowfish context */

int cipher=0;				/* cipher on or off*/
int cipher_type=SSH_CIPHER_3DES;	/* type of cipher */

short ssh_compressing=0;	/* compression */
z_stream out_stream;		/* outgoing compressed stream */
z_stream in_stream;		/* incoming stream to uncompress */

char debug=0;

/* request compression in protocol negotiation phase. */
void RequestCompression(int level)
{
char nomem[]="Not enough memory for compression support";

	if(verbose) cputs("Requesting compression...");

	s_wrpkt_start(SSH_CMSG_REQUEST_COMPRESSION, 4);
	pktout.body[0] = pktout.body[1] = pktout.body[2] =0;
	pktout.body[3] = level; /* compression level */
	s_wrpkt();
	packet_read_block();
	switch (pktin.type){
	  case SSH_SMSG_SUCCESS:
                if((deflateInit(&out_stream, level))==Z_MEM_ERROR)
			fatal(nomem);
		if((inflateInit(&in_stream))==Z_MEM_ERROR)
			fatal(nomem);
		ssh_compressing = 1;
		if(verbose) cputs(done);
		break;

	  case SSH_SMSG_FAILURE:
		cputs("Server refused to compress\r\n");
		break;

	  default:
		fatal("Protocol confusion\n");
	} /* switch */
}

/* clean up compression structures at the end. This frees memory */
void UninitCompression(void)
{
	deflateEnd(&out_stream);
	inflateEnd(&in_stream);
}


/* compress a buffer */
int sshcompress (Bytef *dest, uLongf *destLen,
		 const Bytef *source, uLong sourceLen)
{
    int err;

    out_stream.next_in = (Bytef*)source;	/* source buffer */
    out_stream.avail_in = (uInt)sourceLen;	/* source length */
    out_stream.next_out = dest;			/* destination buffer */
    out_stream.avail_out = (uInt)*destLen;	/* destination buffer size */

    err = deflate(&out_stream, Z_PARTIAL_FLUSH);/* we do the whole at once,
						   because buffer is large */
    *destLen = *destLen - out_stream.avail_out;	/* return compressed size */

    return err;
}

/* uncompress a buffer */
int sshuncompress (Bytef *dest, uLongf *destLen,
			const Bytef *source, uLong sourceLen)
{
    int err;

    in_stream.next_in = (Bytef*)source;
    in_stream.avail_in = (uInt)sourceLen;
    in_stream.next_out = dest;
    in_stream.avail_out = (uInt)*destLen;

    err = inflate(&in_stream, Z_PARTIAL_FLUSH);

    *destLen = *destLen - in_stream.avail_out;

    return err;
}


/* save packets in hexadecimal format for debugging */
static void fwritehex(unsigned char *buf, unsigned int length)
{
unsigned int i;

	for(i = 1; i <= length; i++){
		fprintf(fileptr, "%02X ", *buf++);
		if(i%16==0) fputs("\n", fileptr);
	}

	fputs("\n", fileptr);
	buf-=length;

	for(i = 1; i <= length; i++){
		if(*buf >= ' ' && *buf < 126)
			fprintf(fileptr, "%c", *buf);
		else
			fputc('.', fileptr);
		if(i%16==0) fputs("\n", fileptr);
		buf++;
	}
}


/* Read and convert raw packet to readable structure.
 * Uncrypt and uncompress if necessary */
void ssh_gotdata(void)
{
unsigned long len;	/* packet length without length and padding fields */
unsigned long biglen;	/* full packet length with padding */
unsigned int pad;	/* number of padding bytes */
unsigned char PktInLength[4];	/* first four bytes of a packet (length) */
unsigned char *inbuf;           /* buffer for incoming packet */
unsigned char *decompblk;	/* buffer for decompression */
unsigned long decomplen;	/* Entry: buffer size. Exit: data langth */
unsigned int i;

       	sock_fastread(&s,PktInLength,4); /* get four bytes from packet */
       	for (i = len = 0; i < 4; i++)
       		len = (len << 8) + PktInLength[i]; /* get packet size */
	pad = 8 - (len%8);	/* calculate padding */
	biglen = len + pad;	/* calculate full packet length */
       	inbuf=(unsigned char *)xmalloc(biglen*(sizeof(char))+4);/* allocate memory for incoming packet */
	memcpy(inbuf, PktInLength, 4); /* copy length bytes too */
       	sock_read(&s, inbuf+4, biglen); /* Read rest */

	if(cipher){
	   switch(cipher_type){
		case SSH_CIPHER_3DES:
			des_3cbc_decrypt(inbuf+4, inbuf+4, biglen, keys);
			break;

		case SSH_CIPHER_BLOWFISH:
			blowfish_decrypt_cbc(inbuf+4, biglen, &dctx);
			break;
	   } /* switch */
	} /* cipher */

	pktin.length=biglen-4-pad-1;	/* minus CRC, padding and type*/
        pktin.data = (pktin.data == NULL ? (unsigned char *)xmalloc(biglen) :
			(unsigned char *)xrealloc(pktin.data, biglen));
	memcpy(pktin.data, inbuf+4, biglen);
       	xfree(inbuf); /* it's now in pktin structure, so free it */
	pktin.body = pktin.data+pad+1; /* skip padding and type fields */

	if (ssh_compressing){
	   decomplen=MAX_PACKET_SIZE;	/* we shouldn't receive larger */
	   decompblk=(unsigned char *)xmalloc(decomplen);
	   if((i = sshuncompress(decompblk, &decomplen, pktin.body - 1,
			 pktin.length + 1)) != Z_OK ) fatal("Decompression error: %d\n", i);
	   pktin.data = (unsigned char *)xrealloc(pktin.data, pad + decomplen);
	   pktin.body = pktin.data + pad + 1;
	   memcpy(pktin.body - 1, decompblk, decomplen); /* copy uncompressed */
	   xfree(decompblk);
	   pktin.length = decomplen-1;
	} /* ssh_compressing */

	if(debug){
	      fputs("\nRECEIVED packet:\n", fileptr);
	      fwritehex(pktin.data+pad, 1 + pktin.length);
	      fputc('\n', fileptr);
	} /* debug */

	pktin.type = pktin.data[pad];

}

/* calculate packet length */
void s_wrpkt_size(unsigned long len)
{
int pad;
unsigned long biglen;

    len += 5;			        /* add type and CRC */
    pad = 8 - (len%8);                  /* calculate padding */
    biglen = len + pad;                 /* add padding for total length */

    pktout.length = len - 5;
    pktout.data = (pktout.data == NULL ? (unsigned char *)xmalloc(biglen) :
		       (unsigned char *)xrealloc(pktout.data, biglen));
    pktout.body = pktout.data+4+pad+1;
}

/* create header for raw outgoing packet */
void s_wrpkt_start(int type, int len)
{
    s_wrpkt_size(len);
    pktout.type = type;
}

/* create outgoing packet */
void s_wrpkt(void)
{
int pad, len, biglen, i;
unsigned long crc;
unsigned char *compblk;
unsigned long complen;

    pktout.body[-1] = pktout.type;

    if (ssh_compressing) {
	complen=(pktout.length+1)+(pktout.length+1)/1000+13; /* from zlib */
	compblk=(unsigned char *)xmalloc(complen);
	if((i = sshcompress(compblk, &complen, (const Bytef*)pktout.body - 1,
			 pktout.length + 1)) != Z_OK) fatal("Compression error: %d\n", i);
	s_wrpkt_size(complen - 1);
	memcpy(pktout.body - 1, compblk, complen);
	xfree(compblk);
    } /* ssh_compressing */

    len = pktout.length + 5;	       /* type and CRC */
    pad = 8 - (len%8);
    biglen = len + pad;

    for (i=0; i<pad; i++)
	pktout.data[i+4] = rand() % 256;
    crc = sshcrc32(pktout.data+4, biglen-4);

    pktout.data[biglen+0] = (unsigned char) ((crc >> 24) & 0xFF);
    pktout.data[biglen+1] = (unsigned char) ((crc >> 16) & 0xFF);
    pktout.data[biglen+2] = (unsigned char) ((crc >> 8) & 0xFF);
    pktout.data[biglen+3] = (unsigned char) (crc & 0xFF);

    pktout.data[0] = (len >> 24) & 0xFF;
    pktout.data[1] = (len >> 16) & 0xFF;
    pktout.data[2] = (len >> 8) & 0xFF;
    pktout.data[3] = len & 0xFF;

    if(debug){
	fputs("\nSENT packet:\n", fileptr);
	fwritehex(pktout.body - 1, pktout.length + 1);
	fputc('\n', fileptr);
    }

    if(cipher){
	  switch(cipher_type){
		case SSH_CIPHER_3DES:
			des_3cbc_encrypt(pktout.data+4, pktout.data+4, biglen, keys);
			break;

		case SSH_CIPHER_BLOWFISH:
			blowfish_encrypt_cbc(pktout.data+4, biglen, &ectx);
			break;

	  } /* switch */
    } /* cipher */

    if(sock_write(&s, pktout.data, biglen+4) != biglen+4)
	        fatal("write: %s", strerror(errno));

}


/* Get a packet with blocking. Handle debug, ignore and disconnect packets */
int packet_read_block(void)
{

restart:
                sock_wait_input(&s,sock_delay,NULL,&status);
		ssh_gotdata();
                if(pktin.type==SSH_MSG_DEBUG){
			if(debug || verbose){
			   pktin.body[4+pktin.body[3]]=0;
			   printf("DEBUG: %s\n",pktin.body+4);
			}
                        goto restart;
		}
                if(pktin.type==SSH_MSG_IGNORE) goto restart;
                if(pktin.type==SSH_MSG_DISCONNECT){
			pktin.body[4+pktin.body[3]]=0;
                        fatal("DISCONNECT: %s\n",pktin.body+4);
		}

return(0);

sock_err:
  switch (status){
    case 1 : puts ("Connection closed");
             break;
    case -1: puts ("REMOTE HOST CLOSED CONNECTION");
             break;
  }
  fatal("Socket error");
  return(0);	/* for BorlandC */
}

/* expect a packet type */
void packet_read_expect(int type)
{
      packet_read_block();
      if(pktin.type!=type) fatal("Invalid answer from server\n");
}

/* Send n bytes as a packet */
void tton(const char *buff, size_t len)
{
	s_wrpkt_start(16,len + 4);
	pktout.body[0]=(len & 0xff000000) >> 24;
	pktout.body[1]=(len & 0xff0000) >> 16;
	pktout.body[2]=(len & 0xff00) >> 8;
	pktout.body[3]= len & 0xff;
	memcpy(&pktout.body[4], buff, len);
	s_wrpkt();
}

/* Send a character */
void ttoc(char c)
{
	tton(&c, 1);
}

/* Send two characters after an ESC */
void ttoe2c(char c1, char c2)
{
char buff[3];
buff[0] = 0x1b;
buff[1] = c1;
buff[2] = c2;
	tton(buff, 3);
}

/* Send two characters after ANSI */
void ttoea2c(char c1, char c2)
{
char buff[4];
buff[0] = 0x1b;
buff[1] = 0x5b;
buff[2] = c1;
buff[3] = c2;
	tton(buff, 4);
}

/* Send three characters after ANSI */
void ttoea3c(char c1, char c2, char c3)
{
char buff[5];
buff[0] = 0x1b;
buff[1] = 0x5b;
buff[2] = c1;
buff[3] = c2;
buff[4] = c3;
	tton(buff, 5);
}
