/* sshdos.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/03/28 17:02:39 $
 * $Revision: 1.13 $
 *
 * This module is the main part:
 *  - command line parsing
 *  - client loop
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <stdarg.h> 

#if defined (__DJGPP__)
 #include <unistd.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include <io.h>
 #include "tcp.h"
#endif

#include "ssh.h"
#include "protocol.h"
#include "des.h"
#include "blowfish.h"
#include "version.h"
#include "vt100.h"
#include "xmalloc.h"

/* connect to the remote host */
int connectssh(char *);

extern struct Packet pktin;
extern struct Packet pktout;
extern int cipher_type;         /* type of cipher */
extern int ssh_port;		/* SSH port of server */
extern char privileged;
extern char debug;
char verbose=0;
FILE *fileptr;

tcp_Socket s;		/* socket */
int status;		/* status of communication */
char *remotehost;
unsigned char username[20];
char password[MAX_PASSWORD_LENGTH];
short passwdcmd=0;	/* password from command line? */

char *terminals[]={"xterm","vt100","linux","xterm-color"}; /* terminal types */
char *term; /* user selected type */
char *command=NULL;
char tty=0;
char statusline=1;
char compression=0;
extern char newline;

/* fatal error handler */
void fatal(const char *fmt, ...)
{
va_list ap;
char buf[1024];

  va_start(ap, fmt);
  vsprintf(buf, fmt, ap);
  va_end(ap);
  fprintf(stderr, "%s\n", buf);
  if( (debug) && (fileptr!=NULL) ) fclose(fileptr);
  sock_close(&s);
  exit(255);
}


/* allocate a pseudo terminal */
static void request_pty(char *termtype)
{
int i;

        i = strlen(termtype);
	s_wrpkt_start(SSH_CMSG_REQUEST_PTY, i+5*4+1);
        pktout.body[0] = (i >> 24) & 0xFF;
        pktout.body[1] = (i >> 16) & 0xFF;
        pktout.body[2] = (i >> 8) & 0xFF;
        pktout.body[3] = i & 0xFF;
	memcpy(pktout.body+4, termtype, i);
        memset(pktout.body+4+i, 0, 17);
	if(!statusline)
		pktout.body[7+i]=25; /* we can have 25 lines now */
	else pktout.body[7+i]=24;
        pktout.body[11+i]=80;
	s_wrpkt();
}

/* start interactive shell or run command */
static void start_shell(void)
{
int len;

    if (command != NULL && command[0] != '\0') {
	s_wrpkt_start(SSH_CMSG_EXEC_CMD, (len = strlen(command)) + 4);
	pktout.body[0] = pktout.body[1] = 0;
	pktout.body[2] = (len >> 8) & 0xFF;
	pktout.body[3] = len & 0xFF;		/* up to 32k of string */
	memcpy(pktout.body+4, command, len);
	xfree(command);
	s_wrpkt();
    } else {
	s_wrpkt_start(SSH_CMSG_EXEC_SHELL, 0);
	s_wrpkt();
    }
}


/* Client loop. This runs until SSH connection is terminated */
static int dosession(void)
{
int i;

        while (ConChk())
		DoKey();

	sock_tick(&s, &status); /* TCP wait */
	if (!sock_dataready(&s))
		 return(0); /* Begin loop again if none */
	ssh_gotdata(); /* uncrypt and get valuable data */

	switch(pktin.type) {
		case SSH_SMSG_STDOUT_DATA:
			for (i = 0; i < pktin.length-4; i++)
	                        if (tty) ConOut(pktin.body[i+4]);
				else putchar(pktin.body[i+4]);
			break;

		case SSH_SMSG_STDERR_DATA:
			for (i = 0; i < pktin.length-4; i++)
			ConOut(pktin.body[i+4]);
			break;

		case SSH_SMSG_EXITSTATUS:
			s_wrpkt_start(SSH_CMSG_EXIT_CONFIRMATION,0);
			s_wrpkt();
	                return(EXIT_SSH);

		case SSH_MSG_DISCONNECT:
			 /* Connection reset by other party */
			pktin.body[4 + pktin.body[3]] = 0;
			cputs(pktin.body + 4);
			return(EXIT_SSH);

		case	SSH_SMSG_SUCCESS:
		case	SSH_MSG_IGNORE:
			break;

		default:
			cprintf("Unsupported packet received. Type: %d\n\r",pktin.type);
			return(0);

		}
return(0);

sock_err:
  switch (status)
  {
    case 1 : cputs ("Connection closed\n\r");
             break;
    case -1: cputs ("REMOTE HOST CLOSED CONNECTION\n\r");
             break;
  }
  return(EXIT_SSH);

}


/* Get command line arguments */

static void getargs(int argc, char *argv[])
{
int i, j, len;
char *s;
char *keymaperror="Specified keymap not found!";
char *usage="Usage: sshdos [options] username remotehost [command [args]]\n"
	    "Options:\n"
	    "-c <3des|blowfish>                 - cipher type\n"
	    "-t <xterm|vt100|linux|xterm-color> - terminal type\n"
	    "-p <port number>                   - remote port\n"
	    "-k <keymap file>                   - path to keymap file\n"
	    "-P                                 - use non privileged local port\n"
	    "-C                                 - enable compression\n"
	    "-s                                 - remote password\n"
	    "-S                                 - disable status line\n"
	    "-n                                 - add CR if server sends only LF\n"
	    "-d                                 - save SSH packets to debug.pkt\n"
	    "-v                                 - verbose output\n"
	    "Default is 3des cipher and xterm terminal emulation.";

    for (i = 1; i < argc; ++i)
    {
	s = argv[i];
	if (*s != '-') break;
	switch (*++s){
	   case '\0':
		fatal(usage);
		return;

	   case 'c':
		if (*++s){
			if(!strcmp(s,"3des")) cipher_type = SSH_CIPHER_3DES;
			else if(!strcmp(s,"blowfish")) cipher_type = SSH_CIPHER_BLOWFISH;
			else fatal(usage);
		}
		else if (++i < argc){
			if(!strcmp(argv[i],"3des")) cipher_type = SSH_CIPHER_3DES;
			else if(!strcmp(argv[i],"blowfish")) cipher_type = SSH_CIPHER_BLOWFISH;
			else fatal(usage);
		}
		else fatal(usage);
		continue;

	   case 's':
		passwdcmd=1;
		if (*++s) strncpy(password, s, MAX_PASSWORD_LENGTH);
		else if (++i < argc) strncpy(password, argv[i], MAX_PASSWORD_LENGTH);
		else fatal(usage);
		continue;

	   case 't':
		if (*++s){
		  if(!strcmp(s,terminals[0]) ||
		     !strcmp(s,terminals[1]) ||
		     !strcmp(s,terminals[2]) ||
		     !strcmp(s,terminals[3]) )
		     term = s;
		  else fatal(usage);
		}
		else if (++i < argc){
		  if(!strcmp(argv[i],terminals[0]) ||
		     !strcmp(argv[i],terminals[1]) ||
		     !strcmp(argv[i],terminals[2]) ||
		     !strcmp(argv[i],terminals[3]) )
		     term = argv[i];
		  else fatal(usage);
		}
	    	else fatal(usage);
		continue;

	   case 'p':
		if (*++s)
		   ssh_port = atoi(s);
		else if (++i < argc)
		   ssh_port = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'k':
		if (*++s){
		   if(keymap_init(s)) fatal(keymaperror);
		}
		else if (++i < argc){
		   if(keymap_init(argv[i])) fatal(keymaperror);
		}
		else
		   fatal(usage);
		continue;

	   case 'P':
		privileged=0;
		continue;

	   case 'S':
		statusline = 0;
		continue;

	   case 'C':
		compression = 1;
		continue;

	   case 'n':
		newline=1;
		continue;

	   case 'd':
		debug=1;
		continue;

	   case 'v':
		verbose=1;
		continue;

	   default:
		fatal(usage);
	} /* end switch */

    } /* end for */

/* no_more_options */
    if (i + 2 > argc) fatal(usage);
    strncpy(username,argv[i++],20);
    remotehost = argv[i++];
    if (i >= argc)			/* command args? */
	return;
    /* collect remaining arguments and make a command line of them */
    for (len = 0, j = i; j < argc; j++)
	len += strlen(argv[j]) + 1;	/* 1 for the separating space */
    command=(char *)xmalloc(len);
    for (command[0] = '\0', j = i; j < argc; j++) {
	strcat(command, argv[j]);	/* inefficient, but no big deal */
	if (j < argc - 1)		/* last argument? */
            strcat(command, " ");
    }
}



/* main program */
int main(int argc, char **argv)
{
    cprintf("SSHDOS v%s\n\r", SSH_VERSION);

/* Do some preinitializations */

    memset(username,0,20);
    memset(password, 0, MAX_PASSWORD_LENGTH);

    term=terminals[0]; /* default is "xterm" */

    getargs(argc, argv); /* get and parse command line */

    srand (time(NULL)); /* Initialize random number generator */

    if(debug){
	 if((fileptr=fopen("debug.pkt","w+"))==NULL)
		fatal("Cannot create debug file\n");
	 else
		fputs("\n-------------------\n",fileptr);
    }

    /* connect to server */
    if(connectssh(remotehost))
	fatal("Connection error\n");

    /* Request a pseudo terminal if stdout is the terminal*/

    if (isatty(fileno(stdout))) {
	tty=1;
        request_pty(term);
        packet_read_expect(SSH_SMSG_SUCCESS);
    }

    /* Start compression if configured */
    if(compression)
	RequestCompression(6);

    /* Start an interactive shell if no other command */
    start_shell();

#if defined (__DJGPP__)
    tcp_cbreak(1);	/* No Break checking under DJGPP */
#endif

    KeyInit();
    VidInit(remotehost);
    VTInit();

    while(EXIT_SSH!=dosession());	/* Loop until session end */

    sock_close(&s);	/* Close TCP socket */

    if(debug)
	 fclose(fileptr);

    if(compression)
	UninitCompression();

return(0);

}
