/* telnet.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/06/19 14:09:36 $
 * $Revision: 1.1 $
 *
 * This module is the main part:
 *  - command line parsing
 *  - client loop
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h> 
#include <xmalloc.h>

#if defined (__DJGPP__)
 #include <unistd.h>
 #include <errno.h>
 #include <dpmi.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include "tcp.h"
#endif

#include "version.h"
#include "vt100.h"
#include "config.h"
#include "telnet.h"

/* external functions */
SendFuncPtr SendPacket;
void ChrDelete(void);

/* global variables */
Config GlobalConfig;		/* global configuration structure */
unsigned short Configuration = 0;	/* Configuration bits */

/* local variables */
static char *terminals[]={"xterm","vt100","linux","xterm-color"};
					/* terminal types */
static unsigned char Inbuf[1024];	/* buffer for incoming data */
static unsigned char Outbuf[1024];	/* buffer for outgoing data */
static unsigned char echo = 0;		/* is the echoing on? */

/*
 * fatal error handler
 */
void fatal(const char *fmt, ...)
{
va_list ap;
char buf[512];

   va_start(ap, fmt);
   vsprintf(buf, fmt, ap);
   va_end(ap);
   fprintf(stderr, "%s\n", buf);
   if(GlobalConfig.logfile)
	fclose(GlobalConfig.logfile);
   if(GlobalConfig.brailab)
	fclose(GlobalConfig.brailab);
   sock_close(&GlobalConfig.s);
   exit(255);
}

/*
 * Send Telnet packet
 */
void SendTelnetPacket(unsigned char *buff, unsigned short len)
{
   if(sock_write(&GlobalConfig.s, buff, len) != len)
        fatal("Sock write: %s", strerror(errno));
}

/*
 * Send Telnet command
 */
static void SendTC(unsigned char c1, unsigned char c2)
{
unsigned char buff[3];

   buff[0] = IAC;
   buff[1] = c1;
   buff[2] = c2;
   SendPacket(buff, 3);
}

/*
 * Initialize global structures
 */
static void Config_Init(void)
{
   memset(Outbuf, 0, sizeof(Outbuf));
   GlobalConfig.remotehost = NULL;
   GlobalConfig.RemotePort = 23;
   GlobalConfig.term = terminals[0]; /* default is "xterm" */
   GlobalConfig.KeyMapFile = NULL;
   GlobalConfig.statusline = 1;
   SendPacket = SendTelnetPacket;
   GlobalConfig.logfile = NULL;
   GlobalConfig.brailab = NULL;
}

/*
 * Create TCP connection
 */
static short TCPConnect(char *remotehost)
{
unsigned short localport;
longword remoteip;
int status;

   /* Allocate local port */
   localport = (rand() % 512) + 512;
   if(Configuration & NONPRIVILEGED_PORT)
	localport = localport + 512;

   sock_init(); /* Initialize socket */

   /* Resolve hostname */
   if((remoteip = resolve(remotehost)) == 0) {
       cprintf("\n\rUnable to resolve `%s'\n\r", remotehost);
       return(1);
   }

   /* Open TCP */
   if(!tcp_open(&GlobalConfig.s, localport, remoteip,
                  GlobalConfig.RemotePort, NULL)){
       cputs("Unable to open connection\n\r");
       return(1);
   }

   /* Negotiate TCP connection */
   cputs("waiting for remote host to connect\r\n");
   fflush(stdout);
   sock_wait_established (&GlobalConfig.s, sock_delay, NULL, &status);

   return(0);

sock_err:
   switch(status){
	case 1:
	   cputs("Connection closed\r\n");
	   break;
	case -1:
	   cputs("Remote host closed connection\r\n");
	   break;
   }
   return(1);
}

/*
 * Client loop. This runs until Telnet connection is terminated
 */
static short dosession(void)
{
unsigned short n;
unsigned short len;
unsigned char Command, Option, y;
unsigned char *ptr;
unsigned char sb[40];
int status;

   while(ConChk()){	/* examine STDIN */
	if(echo)	/* If we have remote echo, don't bother much */
	   DoKey();
	else{		/* else we have to buffer user input */
	   n = strlen(Outbuf);
	   y = getch();
	   if(!y){		 /* we have a special character here? */
		if(getch() == 0x2d)	 /* ALT-X ? */
		   fatal("Terminating session");
		else
		   break;
	   }
	   ConOut(y);	/* echo to screen */
	   switch(y){
		case 8: /* backspace */
		   if(n > 0)
			Outbuf[n-1] = 0;
		   ChrDelete();
		   break;

		case('\r'): /* return */
		   ConOut('\n');
		   strcat(Outbuf, "\n");
		   SendPacket(Outbuf, strlen(Outbuf));
		   memset(Outbuf, 0, strlen(Outbuf));
		   break;

		default:
		   strncat(Outbuf, &y, 1);
		   break;
	   } /* switch */
	} /* else */
   } /* while */

	sock_tick(&GlobalConfig.s, &status); /* TCP wait */
	if(!sock_dataready(&GlobalConfig.s))
	   return(0); /* Begin loop again if none */
	len = sock_fastread(&GlobalConfig.s, Inbuf, 1024);

	for(ptr = Inbuf; len > 0; len--){
	   if(*ptr != IAC){ /* isn't it a Telnet command? */
		ConOut(*ptr);
		if(GlobalConfig.logfile)
		   fputc(*ptr, GlobalConfig.logfile);
                ptr++;
	   }
	   else{	/* It's a Telnet command */
		ptr++;
		Command = *ptr++;
		len--;
		if( Command == IAC ) /* double IAC means command terminated */
		   continue;
		Option = *ptr++;
		len--;

		switch(Option) {

		   case TELOPT_ECHO:
			switch(Command){
			   case WILL: /* server WILL echo. Do it */
				echo = 1;
				SendTC( DO, TELOPT_ECHO );
				break;

			   case WONT:
				echo = 0;
				SendTC( DONT, TELOPT_ECHO );
				break;

			   case DO: /* we won't echo no matter what */
			   case DONT:
				SendTC( WONT, TELOPT_ECHO );
				break;
			} /* switch */
                	break;

		   case TELOPT_SGA: /* suppress go ahead */
			if( Command == WONT )
			   SendTC( DONT, TELOPT_SGA );
			else if( Command == WILL){
			   SendTC( DO, TELOPT_SGA );
			   if(!echo)
				SendTC( DO, TELOPT_ECHO );
			}
			break;

		   case TELOPT_TTYPE:
			if( Command == DO ) /* we will send our terminal type */
			   SendTC( WILL, TELOPT_TTYPE );
			else if( Command == SB ){
			   /* find end of subnegotiation */
			   for(; *ptr != SE; ptr++, len--);
                           ptr++; len--;
			   SendTC(SB, TELOPT_TTYPE);
			   n = strlen(GlobalConfig.term);
			   sb[0] = '\0';
			   memcpy(sb + 1, GlobalConfig.term, n);
			   sb[n+1] = 0xFF;
			   sb[n+2] = SE;
			   SendPacket(sb, n + 3);
			} /* else if */
			break;

		   case TELOPT_NAWS: /* terminal window size */
			if( Command == DO ){
			   SendTC( WILL, TELOPT_NAWS );
                           SendTC( SB, TELOPT_NAWS);
                           sb[0] = 0;
                           sb[1] = 80;
                           sb[2] = 0;
                           sb[3] = 25 - GlobalConfig.statusline;
                           sb[4] = IAC;
                           sb[5] = SE;
                           SendPacket(sb, 6);
                        }
			break;

		   default:
			switch(Command){

			   case WILL:
			   case WONT:
				SendTC( DONT, Option );
				break;

			   case DO:
			   case DONT:
				SendTC( WONT, Option );
				break;
			} /* switch */
			break;
		} /* switch */
	   } /* else */
	}/* for */

return(0);

sock_err:
   switch(status){
	case 1:
	   cputs ("Connection closed\r\n");
	   break;
	case -1:
	   cputs ("Remote host closed connection\r\n");
	   break;
   }
   return(EXIT_TELNET);
}

/*
 * Get command line arguments
 */
static void getargs(int argc, char *argv[])
{
unsigned short i;
char *s;
char *keymaperror="Specified keymap not found!";
char *usage="Usage: telnet [options] remotehost\n"
	    "Options:\n"
	    "-t <xterm|vt100|linux|xterm-color> - terminal type\n"
	    "-p <port number>                   - remote port\n"
	    "-k <keymap file>                   - path to keymap file\n"
	    "-l <log file>                      - log session to file\n"
	    "-b <COM[1234]>                     - Brailab PC adapter on COM[1234] port\n"
	    "-P                                 - use non privileged local port\n"
	    "-S                                 - disable status line\n"
	    "-n                                 - add CR if server sends only LF\n"
	    "Default is xterm terminal emulation.";

   for(i = 1; i < argc; ++i){
	s = argv[i];
	if(*s != '-') break;
	switch (*++s){
	   case '\0':
		fatal(usage);
		return;

	   case 't':
		if(*++s){
		   if(!strcmp(s,terminals[0]) ||
		      !strcmp(s,terminals[1]) ||
		      !strcmp(s,terminals[2]) ||
		      !strcmp(s,terminals[3]) )
			GlobalConfig.term = s;
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i],terminals[0]) ||
		      !strcmp(argv[i],terminals[1]) ||
		      !strcmp(argv[i],terminals[2]) ||
		      !strcmp(argv[i],terminals[3]) )
			GlobalConfig.term = argv[i];
		   else
			fatal(usage);
		}
	    	else
		   fatal(usage);
		continue;

	   case 'p':
		if(*++s)
		   GlobalConfig.RemotePort = atoi(s);
		else if(++i < argc)
		   GlobalConfig.RemotePort = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'b':
		if(*++s){
		   if(!strcmp(s, "COM1") ||
		      !strcmp(s, "COM2") ||
		      !strcmp(s, "COM3") ||
		      !strcmp(s, "COM4")){
			if((GlobalConfig.brailab = fopen(s,"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i], "COM1") ||
		      !strcmp(argv[i], "COM2") ||
		      !strcmp(argv[i], "COM3") ||
		      !strcmp(argv[i], "COM4")){
			if((GlobalConfig.brailab = fopen(argv[i],"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else
		   fatal(usage);
		continue;

	   case 'k':
		if(*++s){
		   if(keymap_init(s))
			fatal(keymaperror);
		}
		else if(++i < argc){
		   if(keymap_init(argv[i]))
			fatal(keymaperror);
		}
		else
		   fatal(usage);
		continue;

	   case 'l':
		if(*++s){
		   if((GlobalConfig.logfile = fopen(s,"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else if(++i < argc){
		   if((GlobalConfig.logfile = fopen(argv[i],"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else
		   fatal(usage);
		continue;

	   case 'P':
		Configuration |= NONPRIVILEGED_PORT;
		continue;

	   case 'S':
		GlobalConfig.statusline = 0;
		continue;

	   case 'n':
		Configuration |= NEWLINE;
		continue;

	   default:
		fatal(usage);
	} /* end switch */

   } /* end for */

   /* no_more_options */
   if(i + 1 > argc)
	fatal(usage);
   GlobalConfig.remotehost = argv[i++];
}

/*
 * Main program starts here
 */
int main(int argc, char **argv)
{
   printf("TELNET v%s\n", SSH_VERSION);

   Config_Init();	/* Initialize global variables */
   getargs(argc, argv); /* Process command line */

   if(TCPConnect(GlobalConfig.remotehost))      /* connect to server */
	fatal("TCP connection error\n");

   sock_mode(&GlobalConfig.s, TCP_MODE_NAGLE );

#if defined (__DJGPP__)
   tcp_cbreak(1);	/* No Break checking under DJGPP */
#endif

   KeyInit();
   VidInit(GlobalConfig.remotehost);
   VTInit();

   while(EXIT_TELNET != dosession());	/* Loop until session end */

   sock_close(&GlobalConfig.s);	/* Close TCP socket */

   /* Close open files */
   if(GlobalConfig.brailab)
	fclose(GlobalConfig.brailab);
   if(GlobalConfig.logfile)
	fclose(GlobalConfig.logfile);

   return(0);
}
