/*
Copyright (c) 1998-2001, Robert O'Callahan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice, this list
of conditions and the following disclaimer in the documentation and/or other materials
provided with the distribution.

The name of Robert O'Callahan may not be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "ttxssh.h"
#include "util.h"

#include <bn.h>
#include <limits.h>

static char ssh_ttymodes[] = "\x01\x03\x02\x1c\x03\x08\x04\x15\x05\x04";

static void try_send_credentials(PTInstVar pvar);
static void prep_compression(PTInstVar pvar);

static int get_predecryption_amount(PTInstVar pvar) {
  static int small_block_decryption_sizes[] = { 5, 5, 6, 6, 8 };

  if (SSHv1(pvar)) {
    return 0;
  } else {
    int block_size = CRYPT_get_decryption_block_size(pvar);
    
    if (block_size < 5) {
      return small_block_decryption_sizes[block_size];
    } else {
      return block_size;
    }
  }
}

/* Get up to 'limit' bytes into the payload buffer.
   'limit' is counted from the start of the payload data.
   Returns the amount of data in the payload buffer, or
   -1 if there is an error.
   We can return more than limit in some cases. */
static int buffer_packet_data(PTInstVar pvar, int limit) {
  if (pvar->ssh_state.payloadlen >= 0) {
    return pvar->ssh_state.payloadlen;
  } else {
    int cur_decompressed_bytes = pvar->ssh_state.decompress_stream.next_out -
      pvar->ssh_state.postdecompress_inbuf;

    while (limit > cur_decompressed_bytes) {
      int result;

      pvar->ssh_state.payload = pvar->ssh_state.postdecompress_inbuf + 1;
      if (pvar->ssh_state.postdecompress_inbuflen == cur_decompressed_bytes) {
        buf_ensure_size(&pvar->ssh_state.postdecompress_inbuf,
          &pvar->ssh_state.postdecompress_inbuflen,
          min(limit, cur_decompressed_bytes*2));
      }
      
      pvar->ssh_state.decompress_stream.next_out
        = pvar->ssh_state.postdecompress_inbuf + cur_decompressed_bytes;
      pvar->ssh_state.decompress_stream.avail_out
        = min(limit, pvar->ssh_state.postdecompress_inbuflen)
            - cur_decompressed_bytes;

      result = inflate(&pvar->ssh_state.decompress_stream, Z_SYNC_FLUSH);
      cur_decompressed_bytes = pvar->ssh_state.decompress_stream.next_out -
        pvar->ssh_state.postdecompress_inbuf;

      switch (result) {
      case Z_OK:
        break;
      case Z_BUF_ERROR:
        pvar->ssh_state.payloadlen = cur_decompressed_bytes;
        return cur_decompressed_bytes;
      default:
        notify_fatal_error(pvar, "Invalid compressed data in received packet");
        return -1;
      }
    }

    return cur_decompressed_bytes;
  }
}

/* For use by the protocol processing code.
   Gets N bytes of uncompressed payload. Returns FALSE if data not available
   and a fatal error has been signaled.
   The data is available in the payload buffer. This buffer address
   can change during a call to grab_payload, so take care!
   The payload pointer is set to point to the first byte of the actual data
   (after the packet type byte).
*/
static BOOL grab_payload(PTInstVar pvar, int num_bytes) {
  /* Accept maximum of 4MB of payload data */
  int in_buffer = buffer_packet_data(pvar, 4*1024*1024);
  
  if (in_buffer < 0) {
    return FALSE;
  } else {
    pvar->ssh_state.payload_grabbed += num_bytes;
    if (pvar->ssh_state.payload_grabbed > in_buffer) {
      notify_fatal_error(pvar, "Received truncated packet");
      return FALSE;
    } else {
      return TRUE;
    }
  }
}

static BOOL grab_payload_limited(PTInstVar pvar, int num_bytes) {
  int in_buffer;

  pvar->ssh_state.payload_grabbed += num_bytes;
  in_buffer = buffer_packet_data(pvar, pvar->ssh_state.payload_grabbed);
  
  if (in_buffer < 0) {
    return FALSE;
  } else {
    if (pvar->ssh_state.payload_grabbed > in_buffer) {
      notify_fatal_error(pvar, "Received truncated packet");
      return FALSE;
    } else {
      return TRUE;
    }
  }
}

#define get_payload_uint32(pvar, offset) get_uint32_MSBfirst((pvar)->ssh_state.payload + (offset))
#define get_uint32(buf) get_uint32_MSBfirst((buf))
#define set_uint32(buf, v) set_uint32_MSBfirst((buf), (v))
#define get_mpint_len(pvar, offset) ((get_ushort16_MSBfirst((pvar)->ssh_state.payload + (offset)) + 7) >> 3)
#define get_ushort16(buf) get_ushort16_MSBfirst((buf))

#define do_crc(buf, len) (~(uint32)crc32(0xFFFFFFFF, (buf), (len)))

/* Decrypt the payload, checksum it, eat the padding, get the packet type
   and return it.
   'data' points to the start of the packet --- its length field.
   'len' is the length of the
   payload + padding (+ length of CRC for SSHv1). 'padding' is the length
   of the padding alone. */
static int prep_packet(PTInstVar pvar, char FAR * data, int len, int padding) {
  pvar->ssh_state.payload = data + 4;
  pvar->ssh_state.payloadlen = len;

  if (SSHv1(pvar)) {
    if (CRYPT_detect_attack(pvar, pvar->ssh_state.payload, len)) {
      notify_fatal_error(pvar, "'CORE insertion attack' detected.  Aborting connection.");
      return SSH_MSG_NONE;
    }

    CRYPT_decrypt(pvar, pvar->ssh_state.payload, len);
    /* PKT guarantees that the data is always 4-byte aligned */
    if (do_crc(pvar->ssh_state.payload, len - 4) != get_uint32_MSBfirst(pvar->ssh_state.payload + len - 4)) {
      notify_fatal_error(pvar, "Detected corrupted data; connection terminating.");
      return SSH_MSG_NONE;
    }

    pvar->ssh_state.payload += padding;
    pvar->ssh_state.payloadlen -= padding + 4;
  } else {
    int already_decrypted = get_predecryption_amount(pvar);

    CRYPT_decrypt(pvar, data + already_decrypted, len - already_decrypted);
    
    if (!CRYPT_verify_receiver_MAC(pvar, pvar->ssh_state.receiver_sequence_number,
      data, len + 4, data + len + 4)) {
      notify_fatal_error(pvar, "Detected corrupted data; connection terminating.");
      return SSH_MSG_NONE;
    }

    pvar->ssh_state.payload++;
    pvar->ssh_state.payloadlen -= padding + 1;
  }

  pvar->ssh_state.payload_grabbed = 0;

  if (pvar->ssh_state.decompressing) {
    if (pvar->ssh_state.decompress_stream.avail_in != 0) {
      notify_nonfatal_error(pvar, "Internal error: a packet was not fully decompressed.\n"
        "This is a bug, please report it.");
    }
    
    pvar->ssh_state.decompress_stream.next_in = pvar->ssh_state.payload;
    pvar->ssh_state.decompress_stream.avail_in = pvar->ssh_state.payloadlen;
    pvar->ssh_state.decompress_stream.next_out = pvar->ssh_state.postdecompress_inbuf;
    pvar->ssh_state.payloadlen = -1;
  } else {
    pvar->ssh_state.payload++;
  }
  
  if (!grab_payload_limited(pvar, 1)) {
    return SSH_MSG_NONE;
  }

  pvar->ssh_state.receiver_sequence_number++;

  return pvar->ssh_state.payload[-1];
}

/* Create a packet to be sent. The SSH protocol packet type is in 'type';
   'len' contains the length of the packet payload, in bytes (this
   does not include the space for any of the packet headers or padding,
   or for the packet type byte).
   Returns a pointer to the payload data area, a region of length 'len',
   to be filled by the caller. */
static unsigned char FAR * begin_send_packet(PTInstVar pvar, int type, int len) {
  unsigned char FAR * buf;

  pvar->ssh_state.outgoing_packet_len = len + 1;

  if (pvar->ssh_state.compressing) {
    buf_ensure_size(&pvar->ssh_state.precompress_outbuf, &pvar->ssh_state.precompress_outbuflen,
      1 + len);
    buf = pvar->ssh_state.precompress_outbuf;
  } else {
    /* For SSHv2,
       Encrypted_length is 4(packetlength) + 1(paddinglength) + 1(packettype)
                         + len(payload) + 4(minpadding), rounded up to nearest block_size
       We only need a reasonable upper bound for the buffer size */
    buf_ensure_size(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen,
      len + 30 + CRYPT_get_sender_MAC_size(pvar) + CRYPT_get_encryption_block_size(pvar));
    buf = pvar->ssh_state.outbuf + 12;
  }

  buf[0] = (unsigned char)type;
  return buf + 1;
}

#define finish_send_packet(pvar) finish_send_packet_special((pvar), 0)

static BOOL send_packet_blocking(PTInstVar pvar, char FAR * data, int len) {
  u_long do_block = 0;

  if ((pvar->PWSAAsyncSelect)(pvar->socket, pvar->NotificationWindow,
        0, 0) == SOCKET_ERROR
    || ioctlsocket(pvar->socket, FIONBIO, &do_block) == SOCKET_ERROR
    || (pvar->Psend)(pvar->socket, data, len, 0) != len
    || (pvar->PWSAAsyncSelect)(pvar->socket, pvar->NotificationWindow,
         pvar->notification_msg, pvar->notification_events) == SOCKET_ERROR) {
    notify_fatal_error(pvar, "A communications error occurred while sending an SSH packet.\n"
       "The connection will close.");
    return FALSE;
  } else {
    return TRUE;
  }
}

/* if skip_compress is true, then the data has already been compressed
   into outbuf + 12 */
static void finish_send_packet_special(PTInstVar pvar, int skip_compress) {
  int len = pvar->ssh_state.outgoing_packet_len;
  char FAR * data;
  int data_length;

  if (pvar->ssh_state.compressing) {
    if (!skip_compress) {
      buf_ensure_size(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen,
        len + (len >> 6) + 50 + CRYPT_get_sender_MAC_size(pvar));
      pvar->ssh_state.compress_stream.next_in = pvar->ssh_state.precompress_outbuf;
      pvar->ssh_state.compress_stream.avail_in = len;
      pvar->ssh_state.compress_stream.next_out = pvar->ssh_state.outbuf + 12;
      pvar->ssh_state.compress_stream.avail_out = pvar->ssh_state.outbuflen - 12;
    
      if (deflate(&pvar->ssh_state.compress_stream, Z_SYNC_FLUSH) != Z_OK) {
        notify_fatal_error(pvar, "An error occurred while compressing packet data.\n"
          "The connection will close.");
        return;
      }
    }

    len = pvar->ssh_state.outbuflen - 12 - pvar->ssh_state.compress_stream.avail_out;
  }

  if (SSHv1(pvar)) {
    int padding = 8 - ((len + 4) % 8);
    
    data = pvar->ssh_state.outbuf + 8 - padding;
    data_length = padding + len + 8;

    set_uint32(data, len + 4);
    if (CRYPT_get_receiver_cipher(pvar) != SSH_CIPHER_NONE) {
      CRYPT_set_random_data(pvar, data + 4, padding);
    } else {
      memset(data + 4, 0, padding);
    }
    set_uint32(data + data_length - 4, do_crc(data + 4, data_length - 8));
    CRYPT_encrypt(pvar, data + 4, data_length - 4);
  } else {
    int block_size = CRYPT_get_encryption_block_size(pvar);
    int encryption_size;
    int padding;

    if (block_size < 8) {
      block_size = 8;
    }
    encryption_size = ((len + 8)/block_size + 1)*block_size;
    data = pvar->ssh_state.outbuf + 7;
    data_length = encryption_size + CRYPT_get_sender_MAC_size(pvar);

    set_uint32(data, encryption_size - 4);
    padding = encryption_size - len - 5;
    data[4] = (unsigned char)padding;
    CRYPT_set_random_data(pvar, data + 5 + len, padding);
    CRYPT_build_sender_MAC(pvar, pvar->ssh_state.sender_sequence_number,
      data, encryption_size, data + encryption_size);
    CRYPT_encrypt(pvar, data, encryption_size);
  }

  send_packet_blocking(pvar, data, data_length);

  pvar->ssh_state.sender_sequence_number++;
}

static void destroy_packet_buf(PTInstVar pvar) {
  memset(pvar->ssh_state.outbuf, 0, pvar->ssh_state.outbuflen);
  if (pvar->ssh_state.compressing) {
    memset(pvar->ssh_state.precompress_outbuf, 0, pvar->ssh_state.precompress_outbuflen);
  }
}

/* The handlers are added to the queue for each message. When one of the
   handlers fires, if it returns FALSE, then all handlers in the set are
   removed from their queues. */
static void enque_handlers(PTInstVar pvar, int num_msgs, const int FAR * messages,
  const SSHPacketHandler FAR * handlers) {
  SSHPacketHandlerItem FAR * first_item;
  SSHPacketHandlerItem FAR * last_item = NULL;
  int i;

  for (i = 0; i < num_msgs; i++) {
    SSHPacketHandlerItem FAR * item = (SSHPacketHandlerItem FAR *)malloc(sizeof(SSHPacketHandlerItem));
    SSHPacketHandlerItem FAR * cur_item = pvar->ssh_state.packet_handlers[messages[i]];
    
    item->handler = handlers[i];
    
    if (cur_item == NULL) {
      pvar->ssh_state.packet_handlers[messages[i]] = item;
      item->next_for_message = item;
      item->last_for_message = item;
      item->active_for_message = messages[i];
    } else {
      item->next_for_message = cur_item;
      item->last_for_message = cur_item->last_for_message;
      cur_item->last_for_message->next_for_message = item;
      cur_item->last_for_message = item;
      item->active_for_message = -1;
    }

    if (last_item != NULL) {
      last_item->next_in_set = item;
    } else {
      first_item = item;
    }
    last_item = item;
  }

  if (last_item != NULL) {
    last_item->next_in_set = first_item;
  }
}

static SSHPacketHandler get_handler(PTInstVar pvar, int message) {
  SSHPacketHandlerItem FAR * cur_item = pvar->ssh_state.packet_handlers[message];

  if (cur_item == NULL) {
    return NULL;
  } else {
    return cur_item->handler;
  }
}

/* Called only by SSH_handle_packet */
static void deque_handlers(PTInstVar pvar, int message) {
  SSHPacketHandlerItem FAR * cur_item = pvar->ssh_state.packet_handlers[message];
  SSHPacketHandlerItem FAR * first_item_in_set = cur_item;

  do {
    SSHPacketHandlerItem FAR * next_in_set = cur_item->next_in_set;

    if (cur_item->active_for_message >= 0) {
      SSHPacketHandlerItem FAR * replacement = cur_item->next_for_message;

      if (replacement == cur_item) {
        replacement = NULL;
      } else {
        replacement->active_for_message = cur_item->active_for_message;
      }
      pvar->ssh_state.packet_handlers[cur_item->active_for_message] = replacement;
    }
    cur_item->next_for_message->last_for_message = cur_item->last_for_message;
    cur_item->last_for_message->next_for_message = cur_item->next_for_message;

    free(cur_item);
    cur_item = next_in_set;
  } while (cur_item != first_item_in_set);
}

static void enque_handler(PTInstVar pvar, int message, SSHPacketHandler handler) {
  enque_handlers(pvar, 1, &message, &handler);
}

static void chop_newlines(char FAR * buf) {
  int len = strlen(buf);

  while (len > 0 && (buf[len - 1] == '\n' || buf[len - 1] == '\r')) {
    buf[len - 1] = 0;
    len--;
  }
}

/********************/
/* Message handlers */
/********************/

static BOOL handle_forwarding_success(PTInstVar pvar) {
  return FALSE;
}

static BOOL handle_forwarding_failure(PTInstVar pvar) {
  return FALSE;
}

static void enque_forwarding_request_handlers(PTInstVar pvar) {
  static const int msgs[] = { SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE };
  static const SSHPacketHandler handlers[]
    = { handle_forwarding_success, handle_forwarding_failure };

  enque_handlers(pvar, 2, msgs, handlers);
}

static BOOL handle_auth_failure(PTInstVar pvar) {
  notify_verbose_message(pvar, "Authentication failed", LOG_LEVEL_VERBOSE);

  AUTH_set_generic_mode(pvar);
  AUTH_advance_to_next_cred(pvar);
  pvar->ssh_state.status_flags &= ~STATUS_DONT_SEND_CREDENTIALS;
  try_send_credentials(pvar);
  return FALSE;
}

static BOOL handle_rsa_auth_refused(PTInstVar pvar) {
  AUTH_destroy_cur_cred(pvar);
  return handle_auth_failure(pvar);
}

static BOOL handle_TIS_challenge(PTInstVar pvar) {
  if (grab_payload(pvar, 4)) {
    int len = get_payload_uint32(pvar, 0);
    
    if (grab_payload(pvar, len)) {
      notify_verbose_message(pvar, "Received TIS challenge", LOG_LEVEL_VERBOSE);

      AUTH_set_TIS_mode(pvar, pvar->ssh_state.payload + 4, len);
      AUTH_advance_to_next_cred(pvar);
      pvar->ssh_state.status_flags &= ~STATUS_DONT_SEND_CREDENTIALS;
      try_send_credentials(pvar);
    }
  }
  return FALSE;
}

static BOOL handle_auth_required(PTInstVar pvar) {
  notify_verbose_message(pvar, "Server requires authentication", LOG_LEVEL_VERBOSE);

  pvar->ssh_state.status_flags &= ~STATUS_DONT_SEND_CREDENTIALS;
  try_send_credentials(pvar);
  /* the first AUTH_advance_to_next_cred is issued early by ttxssh.c */

  return FALSE;
}

static BOOL handle_ignore(PTInstVar pvar) {
  if (grab_payload(pvar, 4) && grab_payload(pvar, get_payload_uint32(pvar, 0))) {
    /* ignore it! but it must be decompressed */
  }
  return TRUE;
}

static BOOL handle_debug(PTInstVar pvar) {
  BOOL always_display;
  char FAR * description;
  int description_len;
  char buf[2048];

  if (SSHv1(pvar)) {
    if (grab_payload(pvar, 4)
      && grab_payload(pvar, description_len = get_payload_uint32(pvar, 0))) {
      always_display = FALSE;
      description = pvar->ssh_state.payload + 4;
      description[description_len] = 0;
    } else {
      return TRUE;
    }
  } else {
    if (grab_payload(pvar, 5)
      && grab_payload(pvar, (description_len = get_payload_uint32(pvar, 1)) + 4)
      && grab_payload(pvar, get_payload_uint32(pvar, 5 + description_len))) {
      always_display = pvar->ssh_state.payload[0] != 0;
      description = pvar->ssh_state.payload + 5;
      description[description_len] = 0;
    } else {
      return TRUE;
    }
  }

  chop_newlines(description);
  _snprintf(buf, sizeof(buf), "DEBUG message from server: %s", description);
  buf[sizeof(buf) - 1] = 0;
  if (always_display) {
    notify_nonfatal_error(pvar, buf);
  } else {
    notify_verbose_message(pvar, buf, LOG_LEVEL_VERBOSE);
  }
  return TRUE;
}

static BOOL handle_disconnect(PTInstVar pvar) {
  int reason_code;
  char FAR * description;
  int description_len;
  char buf[2048];
  char FAR * explanation = "";

  if (SSHv1(pvar)) {
    if (grab_payload(pvar, 4)
      && grab_payload(pvar, description_len = get_payload_uint32(pvar, 0))) {
      reason_code = -1;
      description = pvar->ssh_state.payload + 4;
      description[description_len] = 0;
    } else {
      return TRUE;
    }
  } else {
    if (grab_payload(pvar, 8)
      && grab_payload(pvar, (description_len = get_payload_uint32(pvar, 4)) + 4)
      && grab_payload(pvar, get_payload_uint32(pvar, 8 + description_len))) {
      reason_code = get_payload_uint32(pvar, 0);
      description = pvar->ssh_state.payload + 8;
      description[description_len] = 0;
    } else {
      return TRUE;
    }
  }

  chop_newlines(description);
  if (description[0] == 0) {
    description = NULL;
  }

  if (get_handler(pvar, SSH_SMSG_FAILURE) == handle_forwarding_failure) {
    explanation = "\nIt may have disconnected because it was unable to forward a port you requested to be forwarded from the server.\n"
      "This often happens when someone is already forwarding that port from the server.";
  }
  
  if (description != NULL) {
    _snprintf(buf, sizeof(buf), "Server disconnected with message '%s'.%s",
      description, explanation);
  } else {
    _snprintf(buf, sizeof(buf), "Server disconnected (no reason given).%s",
      explanation);
  }
  buf[sizeof(buf) - 1] = 0;
  notify_fatal_error(pvar, buf);
  
  return TRUE;
}

static BOOL handle_unimplemented(PTInstVar pvar) {
  /* Should never receive this since we only send base 2.0 protocol messages */
  grab_payload(pvar, 4);
  return TRUE;
}

static BOOL handle_crypt_success(PTInstVar pvar) {
  notify_verbose_message(pvar, "Secure mode successfully achieved", LOG_LEVEL_VERBOSE);
  return FALSE;
}

static BOOL handle_noauth_success(PTInstVar pvar) {
  notify_verbose_message(pvar, "Server does not require authentication", LOG_LEVEL_VERBOSE);
  prep_compression(pvar);
  return FALSE;
}

static BOOL handle_auth_success(PTInstVar pvar) {
  notify_verbose_message(pvar, "Authentication accepted", LOG_LEVEL_VERBOSE);
  prep_compression(pvar);
  return FALSE;
}

static BOOL handle_server_public_key(PTInstVar pvar) {
  int server_key_public_exponent_len;
  int server_key_public_modulus_pos;
  int server_key_public_modulus_len;
  int host_key_bits_pos;
  int host_key_public_exponent_len;
  int host_key_public_modulus_pos;
  int host_key_public_modulus_len;
  int protocol_flags_pos;
  int supported_ciphers;
  char FAR * inmsg;
 
  if (!grab_payload(pvar, 14)) return FALSE;
  server_key_public_exponent_len = get_mpint_len(pvar, 12);

  if (!grab_payload(pvar, server_key_public_exponent_len + 2)) return FALSE;
  server_key_public_modulus_pos = 14 + server_key_public_exponent_len;
  server_key_public_modulus_len = get_mpint_len(pvar, server_key_public_modulus_pos);

  if (!grab_payload(pvar, server_key_public_modulus_len + 6)) return FALSE;
  host_key_bits_pos = server_key_public_modulus_pos + 2 + server_key_public_modulus_len;
  host_key_public_exponent_len = get_mpint_len(pvar, host_key_bits_pos + 4);

  if (!grab_payload(pvar, host_key_public_exponent_len + 2)) return FALSE;
  host_key_public_modulus_pos = host_key_bits_pos + 6 + host_key_public_exponent_len;
  host_key_public_modulus_len = get_mpint_len(pvar, host_key_public_modulus_pos);

  if (!grab_payload(pvar, host_key_public_modulus_len + 12)) return FALSE;
  protocol_flags_pos = host_key_public_modulus_pos + 2 + host_key_public_modulus_len;

  inmsg = pvar->ssh_state.payload;

  CRYPT_set_server_cookie(pvar, inmsg);
  if (!CRYPT_set_server_RSA_key(pvar, get_uint32(inmsg + 8), pvar->ssh_state.payload + 12,
    inmsg + server_key_public_modulus_pos)) return FALSE;
  if (!CRYPT_set_host_RSA_key(pvar, get_uint32(inmsg + host_key_bits_pos),
    inmsg + host_key_bits_pos + 4,
    inmsg + host_key_public_modulus_pos)) return FALSE;
  pvar->ssh_state.server_protocol_flags = get_uint32(inmsg + protocol_flags_pos);
  
  supported_ciphers = get_uint32(inmsg + protocol_flags_pos + 4);
  if (!CRYPT_set_supported_ciphers(pvar, supported_ciphers, supported_ciphers)) return FALSE;
  if (!AUTH_set_supported_auth_types(pvar, get_uint32(inmsg + protocol_flags_pos + 8))) return FALSE;

  /* this must be the LAST THING in this function, since it can cause
     host_is_OK to be called. */
  HOSTS_check_host_key(pvar, pvar->ssh_state.hostname,
    get_uint32(inmsg + host_key_bits_pos),
    inmsg + host_key_bits_pos + 4,
    inmsg + host_key_public_modulus_pos);

  return FALSE;
}

/*
The ID must have already been found to start with "SSH-". It must
be null-terminated.
*/
static BOOL parse_protocol_ID(PTInstVar pvar, char FAR * ID) {
  char FAR * str;

  for (str = ID + 4; *str >= '0' && *str <= '9'; str++) {
  }

  if (*str != '.') {
    return FALSE;
  }

  pvar->protocol_major = atoi(ID + 4);
  pvar->protocol_minor = atoi(str + 1);

  for (str = str + 1; *str >= '0' && *str <= '9'; str++) {
  }

  return *str == '-';
}

/*
On entry, the pvar->protocol_xxx fields hold the server's advertised
protocol number. We replace the fields with the protocol number we will
actually use, or return FALSE if there is no usable protocol version.
*/
static BOOL negotiate_protocol(PTInstVar pvar) {
  switch (pvar->protocol_major) {
  case 1:
    if (pvar->protocol_minor > 5) {
      pvar->protocol_minor = 5;
    }
    
    return TRUE;

  case 2:
  default:
    return FALSE;
  }
}

static void init_protocol(PTInstVar pvar) {
  CRYPT_initialize_random_numbers(pvar);
  HOSTS_prefetch_host_key(pvar, pvar->ssh_state.hostname);
  /* while we wait for a response from the server... */
  
  if (SSHv1(pvar)) {
    enque_handler(pvar, SSH_MSG_DISCONNECT, handle_disconnect);
    enque_handler(pvar, SSH_MSG_IGNORE, handle_ignore);
    enque_handler(pvar, SSH_MSG_DEBUG, handle_debug);
    enque_handler(pvar, SSH_SMSG_PUBLIC_KEY, handle_server_public_key);
  } else {
    enque_handler(pvar, SSH2_MSG_DISCONNECT, handle_disconnect);
    enque_handler(pvar, SSH2_MSG_IGNORE, handle_ignore);
    enque_handler(pvar, SSH2_MSG_DEBUG, handle_debug);
    enque_handler(pvar, SSH2_MSG_UNIMPLEMENTED, handle_unimplemented);
  }
}

BOOL SSH_handle_server_ID(PTInstVar pvar, char FAR * ID, int ID_len) {
  static const char prefix[] = "Received server prologue string: ";

  if (ID_len <= 0) {
    return FALSE;
  } else {
    char FAR * buf = (char FAR *)malloc(ID_len + NUM_ELEM(prefix));

    strcpy(buf, prefix);
    strncpy(buf + NUM_ELEM(prefix) - 1, ID, ID_len);
    buf[NUM_ELEM(prefix) + ID_len - 1] = 0;
    chop_newlines(buf);

    notify_verbose_message(pvar, buf, LOG_LEVEL_VERBOSE);

    free(buf);

    if (ID[ID_len - 1] != '\n') {
      pvar->ssh_state.status_flags |= STATUS_IN_PARTIAL_ID_STRING;
      return FALSE;
    } else if ((pvar->ssh_state.status_flags & STATUS_IN_PARTIAL_ID_STRING) != 0) {
      pvar->ssh_state.status_flags &= ~STATUS_IN_PARTIAL_ID_STRING;
      return FALSE;
    } else if (strncmp(ID, "SSH-", 4) != 0) {
      return FALSE;
    } else {
      ID[ID_len - 1] = 0;

      if (ID_len > 1 && ID[ID_len - 2] == '\r') {
        ID[ID_len - 2] = 0;
      }

      pvar->ssh_state.server_ID = _strdup(ID);

      if (!parse_protocol_ID(pvar, ID) || !negotiate_protocol(pvar)) {
        notify_fatal_error(pvar, "This program does not understand the server's version of the protocol.");
      } else {
        char TTSSH_ID[1024];
        int TTSSH_ID_len;

        _snprintf(TTSSH_ID, sizeof(TTSSH_ID), "SSH-%d.%d-TTSSH/1.5.0 Win32\n",
          pvar->protocol_major, pvar->protocol_minor);
        TTSSH_ID_len = strlen(TTSSH_ID);
        
        if ((pvar->Psend)(pvar->socket, TTSSH_ID, TTSSH_ID_len, 0) != TTSSH_ID_len) {
          notify_fatal_error(pvar, "An error occurred while sending the SSH ID string.\n"
            "The connection will close.");
        } else {
          init_protocol(pvar);
        }
      }
      
      return TRUE;
    }
  }
}

static BOOL handle_exit(PTInstVar pvar) {
  if (grab_payload(pvar, 4)) {
    begin_send_packet(pvar, SSH_CMSG_EXIT_CONFIRMATION, 0);
    finish_send_packet(pvar);
    notify_closed_connection(pvar);
  }
  return TRUE;
}

static BOOL handle_data(PTInstVar pvar) {
  if (grab_payload_limited(pvar, 4)) {
    pvar->ssh_state.payload_datalen = get_payload_uint32(pvar, 0);
    pvar->ssh_state.payload_datastart = 4;
  }
  return TRUE;
}

static BOOL handle_channel_open(PTInstVar pvar) {
  int host_len;
  int originator_len;

  if ((pvar->ssh_state.server_protocol_flags & SSH_PROTOFLAG_HOST_IN_FWD_OPEN) != 0) {
    if (grab_payload(pvar, 8)
      && grab_payload(pvar, 8 + (host_len = get_payload_uint32(pvar, 4)))
      && grab_payload(pvar, originator_len = get_payload_uint32(pvar, host_len + 12))) {
      int local_port = get_payload_uint32(pvar, 8 + host_len);

      pvar->ssh_state.payload[8 + host_len] = 0;
      FWD_open(pvar, get_payload_uint32(pvar, 0), pvar->ssh_state.payload + 8,
        local_port, pvar->ssh_state.payload + 16 + host_len, originator_len);
    }
  } else {
    if (grab_payload(pvar, 8)
      && grab_payload(pvar, 4 + (host_len = get_payload_uint32(pvar, 4)))) {
      int local_port = get_payload_uint32(pvar, 8 + host_len);

      pvar->ssh_state.payload[8 + host_len] = 0;
      FWD_open(pvar, get_payload_uint32(pvar, 0), pvar->ssh_state.payload + 8,
        local_port, NULL, 0);
    }
  }

  return TRUE;
}

static BOOL handle_X11_channel_open(PTInstVar pvar) {
  int originator_len;

  if ((pvar->ssh_state.server_protocol_flags & SSH_PROTOFLAG_HOST_IN_FWD_OPEN) != 0) {
    if (grab_payload(pvar, 8)
      && grab_payload(pvar, originator_len = get_payload_uint32(pvar, 4))) {
      FWD_X11_open(pvar, get_payload_uint32(pvar, 0), pvar->ssh_state.payload + 8,
        originator_len);
    }
  } else {
    if (grab_payload(pvar, 4)) {
      FWD_X11_open(pvar, get_payload_uint32(pvar, 0), NULL, 0);
    }
  }

  return TRUE;
}

static BOOL handle_channel_open_confirmation(PTInstVar pvar) {
  if (grab_payload(pvar, 8)) {
    FWD_confirmed_open(pvar, get_payload_uint32(pvar, 0), get_payload_uint32(pvar, 4));
  }
  return FALSE;
}

static BOOL handle_channel_open_failure(PTInstVar pvar) {
  if (grab_payload(pvar, 4)) {
    FWD_failed_open(pvar, get_payload_uint32(pvar, 0));
  }
  return FALSE;
}

static BOOL handle_channel_data(PTInstVar pvar) {
  int len;

  if (grab_payload(pvar, 8) && grab_payload(pvar, len = get_payload_uint32(pvar, 4))) {
    FWD_received_data(pvar, get_payload_uint32(pvar, 0),
      pvar->ssh_state.payload + 8, len);
  }
  return TRUE;
}

static BOOL handle_channel_input_eof(PTInstVar pvar) {
  if (grab_payload(pvar, 4)) {
    FWD_channel_input_eof(pvar, get_payload_uint32(pvar, 0));
  }
  return TRUE;
}

static BOOL handle_channel_output_eof(PTInstVar pvar) {
  if (grab_payload(pvar, 4)) {
    FWD_channel_output_eof(pvar, get_payload_uint32(pvar, 0));
  }
  return TRUE;
}

void SSH_handle_packet(PTInstVar pvar, char FAR * data, int len, int padding) {
  unsigned char message = prep_packet(pvar, data, len, padding);

  if (message != SSH_MSG_NONE) {
    SSHPacketHandler handler = get_handler(pvar, message);
    
    if (handler == NULL) {
      if (SSHv1(pvar)) {
        char buf[1024];
          
        _snprintf(buf, sizeof(buf), "Unexpected packet type received: %d", message);
        buf[sizeof(buf) - 1] = 0;
        notify_fatal_error(pvar, buf);
      } else {
        unsigned char FAR * outmsg = begin_send_packet(pvar, SSH2_MSG_UNIMPLEMENTED, 4);
        
        set_uint32(outmsg, pvar->ssh_state.receiver_sequence_number - 1);
        finish_send_packet(pvar);
        /* XXX need to decompress incoming packet, but how? */
      }
    } else {
      if (!handler(pvar)) {
        deque_handlers(pvar, message);
      }
    }
  }
}

static BOOL handle_pty_success(PTInstVar pvar) {
  FWD_enter_interactive_mode(pvar);
  enque_handler(pvar, SSH_SMSG_EXITSTATUS, handle_exit);
  enque_handler(pvar, SSH_SMSG_STDOUT_DATA, handle_data);
  enque_handler(pvar, SSH_SMSG_STDERR_DATA, handle_data);
  enque_handler(pvar, SSH_MSG_CHANNEL_DATA, handle_channel_data);
  enque_handler(pvar, SSH_MSG_CHANNEL_INPUT_EOF, handle_channel_input_eof);
  enque_handler(pvar, SSH_MSG_CHANNEL_OUTPUT_CLOSED, handle_channel_output_eof);
  enque_handler(pvar, SSH_MSG_PORT_OPEN, handle_channel_open);
  enque_handler(pvar, SSH_SMSG_X11_OPEN, handle_X11_channel_open);
  return FALSE;
}

static BOOL handle_pty_failure(PTInstVar pvar) {
  notify_nonfatal_error(pvar, "The server cannot allocate a pseudo-terminal. "
    "You may encounter some problems with the terminal.");
  return handle_pty_success(pvar);
}

static void prep_pty(PTInstVar pvar) {
  int len = strlen(pvar->ts->TermType);
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_REQUEST_PTY,
    4 + len + 16 + sizeof(ssh_ttymodes));
  static const int msgs[] = { SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE };
  static const SSHPacketHandler handlers[]
    = { handle_pty_success, handle_pty_failure };

  set_uint32(outmsg, len);
  memcpy(outmsg + 4, pvar->ts->TermType, len);
  set_uint32(outmsg + 4 + len, pvar->ssh_state.win_rows);
  set_uint32(outmsg + 4 + len + 4, pvar->ssh_state.win_cols);
  set_uint32(outmsg + 4 + len + 8, 0);
  set_uint32(outmsg + 4 + len + 12, 0);
  memcpy(outmsg + 4 + len + 16, ssh_ttymodes, sizeof(ssh_ttymodes));
  finish_send_packet(pvar);

  enque_handlers(pvar, 2, msgs, handlers);

  begin_send_packet(pvar, SSH_CMSG_EXEC_SHELL, 0);
  finish_send_packet(pvar);
}

static void prep_forwarding(PTInstVar pvar) {
  FWD_prep_forwarding(pvar);
  prep_pty(pvar);
}

static void enable_compression(PTInstVar pvar) {
  pvar->ssh_state.compress_stream.zalloc = NULL;
  pvar->ssh_state.compress_stream.zfree = NULL;
  pvar->ssh_state.compress_stream.opaque = NULL;
  if (deflateInit(&pvar->ssh_state.compress_stream, pvar->ssh_state.compression_level) != Z_OK) {
    notify_fatal_error(pvar, "An error occurred while setting up compression.\n"
      "The connection will close.");
    return;
  } else {
    pvar->ssh_state.compressing = TRUE;
  }

  pvar->ssh_state.decompress_stream.zalloc = NULL;
  pvar->ssh_state.decompress_stream.zfree = NULL;
  pvar->ssh_state.decompress_stream.opaque = NULL;
  if (inflateInit(&pvar->ssh_state.decompress_stream) != Z_OK) {
    deflateEnd(&pvar->ssh_state.compress_stream);
    notify_fatal_error(pvar, "An error occurred while setting up compression.\n"
      "The connection will close.");
    return;
  } else {
    pvar->ssh_state.decompressing = TRUE;
    buf_ensure_size(&pvar->ssh_state.postdecompress_inbuf,
      &pvar->ssh_state.postdecompress_inbuflen, 1000);
  }
}

static BOOL handle_enable_compression(PTInstVar pvar) {
  enable_compression(pvar);
  prep_forwarding(pvar);
  return FALSE;
}

static BOOL handle_disable_compression(PTInstVar pvar) {
  prep_forwarding(pvar);
  return FALSE;
}

static void prep_compression(PTInstVar pvar) {
  if (pvar->session_settings.CompressionLevel > 0) {
    static const int msgs[] = { SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE };
    static const SSHPacketHandler handlers[]
      = { handle_enable_compression, handle_disable_compression };

    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_REQUEST_COMPRESSION, 4);

    set_uint32(outmsg, pvar->session_settings.CompressionLevel);
    finish_send_packet(pvar);

    pvar->ssh_state.compression_level = pvar->session_settings.CompressionLevel;

    enque_handlers(pvar, 2, msgs, handlers);
  } else {
    prep_forwarding(pvar);
  }
}

static void enque_simple_auth_handlers(PTInstVar pvar) {
  static const int msgs[] = { SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE };
  static const SSHPacketHandler handlers[]
    = { handle_auth_success, handle_auth_failure };

  enque_handlers(pvar, 2, msgs, handlers);
}

static BOOL handle_rsa_challenge(PTInstVar pvar) {
  int challenge_bytes;
  
  if (!grab_payload(pvar, 2)) {
    return FALSE;
  }

  challenge_bytes = get_mpint_len(pvar, 0);

  if (grab_payload(pvar, challenge_bytes)) {
    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_RSA_RESPONSE, 16);

    if (CRYPT_generate_RSA_challenge_response(pvar, pvar->ssh_state.payload + 2,
      challenge_bytes, outmsg)) {
      AUTH_destroy_cur_cred(pvar);
      finish_send_packet(pvar);

      enque_simple_auth_handlers(pvar);
    } else {
      notify_fatal_error(pvar, "An error occurred while decrypting the RSA challenge.\n"
        "Perhaps the key file is corrupted.");
    }
  }

  return FALSE;
}

static void try_send_credentials(PTInstVar pvar) {
  if ((pvar->ssh_state.status_flags & STATUS_DONT_SEND_CREDENTIALS) == 0) {
    AUTHCred FAR * cred = AUTH_get_cur_cred(pvar);
    static const int RSA_msgs[] = { SSH_SMSG_AUTH_RSA_CHALLENGE, SSH_SMSG_FAILURE };
    static const SSHPacketHandler RSA_handlers[]
      = { handle_rsa_challenge, handle_rsa_auth_refused };
    static const int TIS_msgs[] = { SSH_SMSG_AUTH_TIS_CHALLENGE, SSH_SMSG_FAILURE };
    static const SSHPacketHandler TIS_handlers[]
      = { handle_TIS_challenge, handle_auth_failure };

    switch (cred->method) {
    case SSH_AUTH_NONE: return;
    case SSH_AUTH_PASSWORD: {
      int len = strlen(cred->password);
      unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_PASSWORD, 4 + len);

      notify_verbose_message(pvar, "Trying PASSWORD authentication...", LOG_LEVEL_VERBOSE);

      set_uint32(outmsg, len);
      memcpy(outmsg + 4, cred->password, len);
      AUTH_destroy_cur_cred(pvar);
      enque_simple_auth_handlers(pvar);
      break;
                            } 
    case SSH_AUTH_RHOSTS: {
      int len = strlen(cred->rhosts_client_user);
      unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_RHOSTS, 4 + len);

      notify_verbose_message(pvar, "Trying RHOSTS authentication...", LOG_LEVEL_VERBOSE);

      set_uint32(outmsg, len);
      memcpy(outmsg + 4, cred->rhosts_client_user, len);
      AUTH_destroy_cur_cred(pvar);
      enque_simple_auth_handlers(pvar);
      break;
    }
    case SSH_AUTH_RSA: {
      int len = BN_num_bytes(cred->key_pair->RSA_key->n);
      unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_RSA, 2 + len);

      notify_verbose_message(pvar, "Trying RSA authentication...", LOG_LEVEL_VERBOSE);

      set_ushort16_MSBfirst(outmsg, len*8);
      BN_bn2bin(cred->key_pair->RSA_key->n, outmsg + 2);
      /* don't destroy the current credentials yet */
      enque_handlers(pvar, 2, RSA_msgs, RSA_handlers);
      break;
    }
    case SSH_AUTH_RHOSTS_RSA: {
      int mod_len = BN_num_bytes(cred->key_pair->RSA_key->n);
      int name_len = strlen(cred->rhosts_client_user);
      int exp_len = BN_num_bytes(cred->key_pair->RSA_key->e);
      int index;
      unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_RHOSTS_RSA, 12 + mod_len
        + name_len + exp_len);

      notify_verbose_message(pvar, "Trying RHOSTS+RSA authentication...", LOG_LEVEL_VERBOSE);

      set_uint32(outmsg, name_len);
      memcpy(outmsg + 4, cred->rhosts_client_user, name_len);
      index = 4 + name_len;

      set_uint32(outmsg + index, 8*mod_len);
      set_ushort16_MSBfirst(outmsg + index + 4, 8*exp_len);
      BN_bn2bin(cred->key_pair->RSA_key->e, outmsg + index + 6);
      index += 6 + exp_len;

      set_ushort16_MSBfirst(outmsg + index, 8*mod_len);
      BN_bn2bin(cred->key_pair->RSA_key->n, outmsg + index + 2);
      /* don't destroy the current credentials yet */
      enque_handlers(pvar, 2, RSA_msgs, RSA_handlers);
      break;
    }
    case SSH_AUTH_TIS: {
      if (cred->password == NULL) {
        unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_TIS, 0);
        
        notify_verbose_message(pvar, "Trying TIS authentication...", LOG_LEVEL_VERBOSE);
        enque_handlers(pvar, 2, TIS_msgs, TIS_handlers);
      } else {
        int len = strlen(cred->password);
        unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_AUTH_TIS_RESPONSE, 4 + len);
        
        notify_verbose_message(pvar, "Sending TIS response", LOG_LEVEL_VERBOSE);
        
        set_uint32(outmsg, len);
        memcpy(outmsg + 4, cred->password, len);
        enque_simple_auth_handlers(pvar);
      }
      AUTH_destroy_cur_cred(pvar);
      break;
    }
    default:
      notify_fatal_error(pvar, "Internal error: unsupported authentication method");
      return;
    }

    finish_send_packet(pvar);
    destroy_packet_buf(pvar);

    pvar->ssh_state.status_flags |= STATUS_DONT_SEND_CREDENTIALS;
  }
}

static void try_send_user_name(PTInstVar pvar) {
  if ((pvar->ssh_state.status_flags & STATUS_DONT_SEND_USER_NAME) == 0) {
    char FAR * username = AUTH_get_user_name(pvar);

    if (username != NULL) {
      int len = strlen(username);
      unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_USER, 4 + len);
      char buf[1024] = "Sending user name: ";
      static const int msgs[] = { SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE };
      static const SSHPacketHandler handlers[]
        = { handle_noauth_success, handle_auth_required };

      set_uint32(outmsg, len);
      memcpy(outmsg + 4, username, len);
      finish_send_packet(pvar);

      pvar->ssh_state.status_flags |= STATUS_DONT_SEND_USER_NAME;

      strncpy(buf + strlen(buf), username, sizeof(buf) - strlen(buf) - 2);
      buf[sizeof(buf) - 1] = 0;
      notify_verbose_message(pvar, buf, LOG_LEVEL_VERBOSE);

      enque_handlers(pvar, 2, msgs, handlers);
    }
  }
}

static void send_session_key(PTInstVar pvar) {
  int encrypted_session_key_len = CRYPT_get_encrypted_session_key_len(pvar);
  unsigned char FAR * outmsg;

  if (!CRYPT_choose_ciphers(pvar)) return;
  outmsg = begin_send_packet(pvar, SSH_CMSG_SESSION_KEY, 15 + encrypted_session_key_len);
  outmsg[0] = (unsigned char)CRYPT_get_sender_cipher(pvar);
  memcpy(outmsg + 1, CRYPT_get_server_cookie(pvar), 8); /* antispoofing cookie */
  outmsg[9] = (unsigned char)(encrypted_session_key_len >> 5);
  outmsg[10] = (unsigned char)(encrypted_session_key_len << 3);
  if (!CRYPT_choose_session_key(pvar, outmsg + 11)) return;
  set_uint32(outmsg + 11 + encrypted_session_key_len,
    SSH_PROTOFLAG_SCREEN_NUMBER | SSH_PROTOFLAG_HOST_IN_FWD_OPEN);
  finish_send_packet(pvar);

  if (!CRYPT_start_encryption(pvar)) return;
  notify_established_secure_connection(pvar);

  enque_handler(pvar, SSH_SMSG_SUCCESS, handle_crypt_success);

  pvar->ssh_state.status_flags &= ~STATUS_DONT_SEND_USER_NAME;
  try_send_user_name(pvar);
}

/*************************
   END of message handlers
   ************************/

void SSH_init(PTInstVar pvar) {
  int i;

  buf_create(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen);
  buf_create(&pvar->ssh_state.precompress_outbuf, &pvar->ssh_state.precompress_outbuflen);
  buf_create(&pvar->ssh_state.postdecompress_inbuf, &pvar->ssh_state.postdecompress_inbuflen);
  pvar->ssh_state.payload = NULL;
  pvar->ssh_state.compressing = FALSE;
  pvar->ssh_state.decompressing = FALSE;
  pvar->ssh_state.status_flags = STATUS_DONT_SEND_USER_NAME | STATUS_DONT_SEND_CREDENTIALS;
  pvar->ssh_state.payload_datalen = 0;
  pvar->ssh_state.hostname = NULL;
  pvar->ssh_state.server_ID = NULL;
  pvar->ssh_state.receiver_sequence_number = 0;
  pvar->ssh_state.sender_sequence_number = 0;
  for (i = 0; i < NUM_ELEM(pvar->ssh_state.packet_handlers); i++) {
    pvar->ssh_state.packet_handlers[i] = NULL;
  }
}

void SSH_open(PTInstVar pvar) {
  pvar->ssh_state.hostname = _strdup(pvar->ts->HostName);
  pvar->ssh_state.win_cols = pvar->ts->TerminalWidth;
  pvar->ssh_state.win_rows = pvar->ts->TerminalHeight;
}

void SSH_notify_disconnecting(PTInstVar pvar, char FAR * reason) {
  int len = reason == NULL ? 0 : strlen(reason);
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_DISCONNECT, len + 4);

  set_uint32(outmsg, len);
  if (reason != NULL) {
    memcpy(outmsg + 4, reason, len);
  }
  finish_send_packet(pvar);
}

void SSH_notify_host_OK(PTInstVar pvar) {
  if ((pvar->ssh_state.status_flags & STATUS_HOST_OK) == 0) {
    pvar->ssh_state.status_flags |= STATUS_HOST_OK;
    send_session_key(pvar);
  }
}

void SSH_notify_win_size(PTInstVar pvar, int cols, int rows) {
  pvar->ssh_state.win_cols = cols;
  pvar->ssh_state.win_rows = rows;

  if (get_handler(pvar, SSH_SMSG_STDOUT_DATA) == handle_data) {
    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_WINDOW_SIZE, 16);
    
    set_uint32(outmsg, rows);
    set_uint32(outmsg + 4, cols);
    set_uint32(outmsg + 8, 0);
    set_uint32(outmsg + 12, 0);
    finish_send_packet(pvar);
  }
}

int SSH_get_min_packet_size(PTInstVar pvar) {
  if (SSHv1(pvar)) {
    return 12;
  } else {
    int block_size = CRYPT_get_decryption_block_size(pvar);

    return max(16, block_size);
  }
}

/* data is guaranteed to be at least SSH_get_min_packet_size bytes long
   at least 5 bytes must be decrypted */
void SSH_predecrpyt_packet(PTInstVar pvar, char FAR * data) {
  if (SSHv2(pvar)) {
    CRYPT_decrypt(pvar, data, get_predecryption_amount(pvar));
  }
}

int SSH_get_clear_MAC_size(PTInstVar pvar) {
  if (SSHv1(pvar)) {
    return 0;
  } else {
    return CRYPT_get_receiver_MAC_size(pvar);
  }
}

void SSH_notify_user_name(PTInstVar pvar) {
  try_send_user_name(pvar);
}

void SSH_notify_cred(PTInstVar pvar) {
  try_send_credentials(pvar);
}

void SSH_send(PTInstVar pvar, unsigned char const FAR * buf, int buflen) {
  if (get_handler(pvar, SSH_SMSG_STDOUT_DATA) != handle_data) {
    return;
  }

  while (buflen > 0) {
    int len = buflen > SSH_MAX_SEND_PACKET_SIZE ? SSH_MAX_SEND_PACKET_SIZE : buflen;
    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_STDIN_DATA, 4 + len);
    
    set_uint32(outmsg, len);
    
    if (pvar->ssh_state.compressing) {
      buf_ensure_size(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen, len + (len >> 6) + 50);
      pvar->ssh_state.compress_stream.next_in = pvar->ssh_state.precompress_outbuf;
      pvar->ssh_state.compress_stream.avail_in = 5;
      pvar->ssh_state.compress_stream.next_out = pvar->ssh_state.outbuf + 12;
      pvar->ssh_state.compress_stream.avail_out = pvar->ssh_state.outbuflen - 12;
      
      if (deflate(&pvar->ssh_state.compress_stream, Z_NO_FLUSH) != Z_OK) {
        notify_fatal_error(pvar, "Error compressing packet data");
        return;
      }

      pvar->ssh_state.compress_stream.next_in = (unsigned char FAR *)buf;
      pvar->ssh_state.compress_stream.avail_in = len;

      if (deflate(&pvar->ssh_state.compress_stream, Z_SYNC_FLUSH) != Z_OK) {
        notify_fatal_error(pvar, "Error compressing packet data");
        return;
      }
    } else {
      memcpy(outmsg + 4, buf, len);
    }
    
    finish_send_packet_special(pvar, 1);

    buflen -= len;
    buf += len;
  }
}

int SSH_extract_payload(PTInstVar pvar, unsigned char FAR * dest, int len) {
  int num_bytes = pvar->ssh_state.payload_datalen;

  if (num_bytes > len) {
    num_bytes = len;
  }

  if (!pvar->ssh_state.decompressing) {
    memcpy(dest, pvar->ssh_state.payload + pvar->ssh_state.payload_datastart, num_bytes);
    pvar->ssh_state.payload_datastart += num_bytes;
  } else if (num_bytes > 0) {
    pvar->ssh_state.decompress_stream.next_out = dest;
    pvar->ssh_state.decompress_stream.avail_out = num_bytes;
    
    if (inflate(&pvar->ssh_state.decompress_stream, Z_SYNC_FLUSH) != Z_OK) {
      notify_fatal_error(pvar, "Invalid compressed data in received packet");
      return 0;
    }
  }

  pvar->ssh_state.payload_datalen -= num_bytes;

  return num_bytes;
}

void SSH_get_compression_info(PTInstVar pvar, char FAR * dest, int len) {
  char buf[1024];
  char buf2[1024];

  if (pvar->ssh_state.compressing) {
    unsigned long total_in = pvar->ssh_state.compress_stream.total_in;
    unsigned long total_out = pvar->ssh_state.compress_stream.total_out;

    if (total_out > 0) {
      _snprintf(buf, sizeof(buf), "level %d; ratio %.1f (%ld:%ld)",
        pvar->ssh_state.compression_level, ((double)total_in)/total_out,
        total_in, total_out);
    } else {
      _snprintf(buf, sizeof(buf), "level %d", pvar->ssh_state.compression_level);
    }
  } else {
    strcpy(buf, "none");
  }
  buf[sizeof(buf) - 1] = 0;

  if (pvar->ssh_state.decompressing) {
    unsigned long total_in = pvar->ssh_state.decompress_stream.total_in;
    unsigned long total_out = pvar->ssh_state.decompress_stream.total_out;

    if (total_in > 0) {
      _snprintf(buf2, sizeof(buf2), "level %d; ratio %.1f (%ld:%ld)",
        pvar->ssh_state.compression_level, ((double)total_out)/total_in,
        total_out, total_in);
    } else {
      _snprintf(buf2, sizeof(buf2), "level %d", pvar->ssh_state.compression_level);
    }
  } else {
    strcpy(buf2, "none");
  }
  buf2[sizeof(buf2) - 1] = 0;

  _snprintf(dest, len, "Upstream %s; Downstream %s", buf, buf2);
  dest[len - 1] = 0;
}

void SSH_get_server_ID_info(PTInstVar pvar, char FAR * dest, int len) {
  strncpy(dest, pvar->ssh_state.server_ID == NULL ? "Unknown"
    : pvar->ssh_state.server_ID, len);
  dest[len - 1] = 0;
}

void SSH_get_protocol_version_info(PTInstVar pvar, char FAR * dest, int len) {
  if (pvar->protocol_major == 0) {
    strncpy(dest, "Unknown", len);
  } else {
    _snprintf(dest, len, "%d.%d", pvar->protocol_major, pvar->protocol_minor);
  }
  dest[len - 1] = 0;
}

void SSH_end(PTInstVar pvar) {
  int i;

  for (i = 0; i < 256; i++) {
    SSHPacketHandlerItem FAR * first_item = pvar->ssh_state.packet_handlers[i];

    if (first_item != NULL) {
      SSHPacketHandlerItem FAR * item = first_item;

      do {
        SSHPacketHandlerItem FAR * cur_item = item;

        item = item->next_for_message;
        free(cur_item);
      } while (item != first_item);
    }
    pvar->ssh_state.packet_handlers[i] = NULL;
  }

  free(pvar->ssh_state.hostname);
  pvar->ssh_state.hostname = NULL;
  free(pvar->ssh_state.server_ID);
  pvar->ssh_state.server_ID = NULL;
  buf_destroy(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen);
  buf_destroy(&pvar->ssh_state.precompress_outbuf, &pvar->ssh_state.precompress_outbuflen);
  buf_destroy(&pvar->ssh_state.postdecompress_inbuf, &pvar->ssh_state.postdecompress_inbuflen);

  if (pvar->ssh_state.compressing) {
    deflateEnd(&pvar->ssh_state.compress_stream);
    pvar->ssh_state.compressing = FALSE;
  }
  if (pvar->ssh_state.decompressing) {
    inflateEnd(&pvar->ssh_state.decompress_stream);
    pvar->ssh_state.decompressing = FALSE;
  }
}

/* support for port forwarding */
void SSH_channel_send(PTInstVar pvar, uint32 remote_channel_num,
                      unsigned char FAR * buf, int len) {
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_CHANNEL_DATA, 8 + len);
  
  set_uint32(outmsg, remote_channel_num);
  set_uint32(outmsg + 4, len);
  
  if (pvar->ssh_state.compressing) {
    buf_ensure_size(&pvar->ssh_state.outbuf, &pvar->ssh_state.outbuflen, len + (len >> 6) + 50);
    pvar->ssh_state.compress_stream.next_in = pvar->ssh_state.precompress_outbuf;
    pvar->ssh_state.compress_stream.avail_in = 9;
    pvar->ssh_state.compress_stream.next_out = pvar->ssh_state.outbuf + 12;
    pvar->ssh_state.compress_stream.avail_out = pvar->ssh_state.outbuflen - 12;
    
    if (deflate(&pvar->ssh_state.compress_stream, Z_NO_FLUSH) != Z_OK) {
      notify_fatal_error(pvar, "Error compressing packet data");
      return;
    }
    
    pvar->ssh_state.compress_stream.next_in = (unsigned char FAR *)buf;
    pvar->ssh_state.compress_stream.avail_in = len;
    
    if (deflate(&pvar->ssh_state.compress_stream, Z_SYNC_FLUSH) != Z_OK) {
      notify_fatal_error(pvar, "Error compressing packet data");
      return;
    }
  } else {
    memcpy(outmsg + 8, buf, len);
  }
  
  finish_send_packet_special(pvar, 1);
}

void SSH_fail_channel_open(PTInstVar pvar, uint32 remote_channel_num) {
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_CHANNEL_OPEN_FAILURE, 4);

  set_uint32(outmsg, remote_channel_num);
  finish_send_packet(pvar);
}

void SSH_confirm_channel_open(PTInstVar pvar, uint32 remote_channel_num, uint32 local_channel_num) {
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_CHANNEL_OPEN_CONFIRMATION, 8);

  set_uint32(outmsg, remote_channel_num);
  set_uint32(outmsg + 4, local_channel_num);
  finish_send_packet(pvar);
}

void SSH_channel_output_eof(PTInstVar pvar, uint32 remote_channel_num) {
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_CHANNEL_OUTPUT_CLOSED, 4);

  set_uint32(outmsg, remote_channel_num);
  finish_send_packet(pvar);
}

void SSH_channel_input_eof(PTInstVar pvar, uint32 remote_channel_num) {
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_CHANNEL_INPUT_EOF, 4);

  set_uint32(outmsg, remote_channel_num);
  finish_send_packet(pvar);
}

void SSH_request_forwarding(PTInstVar pvar, int from_server_port,
                            char FAR * to_local_host, int to_local_port) {
  int host_len = strlen(to_local_host);
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_PORT_FORWARD_REQUEST,
    12 + host_len);

  set_uint32(outmsg, from_server_port);
  set_uint32(outmsg + 4, host_len);
  memcpy(outmsg + 8, to_local_host, host_len);
  set_uint32(outmsg + 8 + host_len, to_local_port);
  finish_send_packet(pvar);
  
  enque_forwarding_request_handlers(pvar);
}

void SSH_request_X11_forwarding(PTInstVar pvar,
                                char FAR * auth_protocol, unsigned char FAR * auth_data,
                                int auth_data_len, int screen_num) {
  int protocol_len = strlen(auth_protocol);
  int data_len = auth_data_len*2;
  unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_CMSG_X11_REQUEST_FORWARDING,
    12 + protocol_len + data_len);
  int i;
  char FAR * auth_data_ptr;
   
  set_uint32(outmsg, protocol_len);
  memcpy(outmsg + 4, auth_protocol, protocol_len);
  set_uint32(outmsg + 4 + protocol_len, data_len);
  auth_data_ptr = outmsg + 8 + protocol_len;
  for (i = 0; i < auth_data_len; i++) {
    sprintf(auth_data_ptr + i*2, "%.2x", auth_data[i]);
  }
  set_uint32(outmsg + 8 + protocol_len + data_len, screen_num);

  finish_send_packet(pvar);

  enque_forwarding_request_handlers(pvar);
}

void SSH_open_channel(PTInstVar pvar, uint32 local_channel_num,
                      char FAR * to_remote_host, int to_remote_port, char FAR * originator) {
  static const int msgs[]
    = { SSH_MSG_CHANNEL_OPEN_CONFIRMATION, SSH_MSG_CHANNEL_OPEN_FAILURE };
  static const SSHPacketHandler handlers[]
    = { handle_channel_open_confirmation, handle_channel_open_failure };

  int host_len = strlen(to_remote_host);

  if ((pvar->ssh_state.server_protocol_flags & SSH_PROTOFLAG_HOST_IN_FWD_OPEN) != 0) {
    int originator_len = strlen(originator);
    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_PORT_OPEN,
      16 + host_len + originator_len);
    
    set_uint32(outmsg, local_channel_num);
    set_uint32(outmsg + 4, host_len);
    memcpy(outmsg + 8, to_remote_host, host_len);
    set_uint32(outmsg + 8 + host_len, to_remote_port);
    set_uint32(outmsg + 12 + host_len, originator_len);
    memcpy(outmsg + 16 + host_len, originator, originator_len);
  } else {
    unsigned char FAR * outmsg = begin_send_packet(pvar, SSH_MSG_PORT_OPEN,
      12 + host_len);
    
    set_uint32(outmsg, local_channel_num);
    set_uint32(outmsg + 4, host_len);
    memcpy(outmsg + 8, to_remote_host, host_len);
    set_uint32(outmsg + 8 + host_len, to_remote_port);
  }
  
  finish_send_packet(pvar);

  enque_handlers(pvar, 2, msgs, handlers);
}
