//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "WinSCPFileSystem.h"
#include "WinSCPPlugin.h"
#include "FarTexts.h"
#include "FarConfiguration.h"
#include "farkeys.hpp"
#include <Common.h>
#include <SessionData.h>
#include <ScpMain.h>
#include <Terminal.h>
#include <SysUtils.hpp>
#include <ScpFileSystem.h>
//---------------------------------------------------------------------------
#pragma package(smart_init)
//---------------------------------------------------------------------------
__fastcall TSessionPanelItem::TSessionPanelItem(TSessionData * ASessionData):
  TCustomFarPanelItem()
{
  assert(ASessionData);
  FSessionData = ASessionData;
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    ColumnTitles->Add(FarPlugin->GetMsg(SESSION_NAME_COL_TITLE));
    for (int Index = 0; Index < PANEL_MODES_COUNT; Index++)
    {
      PanelModes->SetPanelMode(Index, "N", "0", ColumnTitles, false, false, false);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsNone, 6, 7);
  KeyBarTitles->SetKeyBarTitle(fsNone, 5, FarPlugin->GetMsg(EXPORT_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
  KeyBarTitles->SetKeyBarTitle(fsShift, 4, FarPlugin->GetMsg(NEW_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 5, 8);
  KeyBarTitles->ClearKeyBarTitle(fsAlt, 6);
  KeyBarTitles->ClearKeyBarTitle(fsCtrl, 4, 11);
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & /*Size*/,
  unsigned long & /*FileAttributes*/,
  TDateTime & /*LastWriteTime*/, TDateTime & /*LastAccess*/,
  unsigned long & /*NumberOfLinks*/, AnsiString & /*Description*/,
  AnsiString & /*Owner*/, void *& UserData, int & /*CustomColumnNumber*/)
{
  FileName = FSessionData->Name;
  UserData = FSessionData;
}
//---------------------------------------------------------------------------
__fastcall TRemoteFilePanelItem::TRemoteFilePanelItem(TRemoteFile * ARemoteFile):
  TCustomFarPanelItem()
{
  assert(ARemoteFile);
  FRemoteFile = ARemoteFile;
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & Size,
  unsigned long & FileAttributes,
  TDateTime & LastWriteTime, TDateTime & LastAccess,
  unsigned long & NumberOfLinks, AnsiString & /*Description*/,
  AnsiString & Owner, void *& UserData, int & CustomColumnNumber)
{
  FileName = FRemoteFile->FileName;
  Size = FRemoteFile->Size;
  FileAttributes =
    (FRemoteFile->IsDirectory ? FILE_ATTRIBUTE_DIRECTORY : 0) |
    (FRemoteFile->IsHidden ? FILE_ATTRIBUTE_HIDDEN : 0) |
    (FRemoteFile->Rights->ReadOnly ? FILE_ATTRIBUTE_READONLY : 0);
  LastWriteTime = FRemoteFile->Modification;
  LastAccess = FRemoteFile->LastAccess;
  NumberOfLinks = FRemoteFile->IsSymLink ? 1 : 0;
  Owner = FRemoteFile->Owner;
  UserData = FRemoteFile;
  CustomColumnNumber = 3;
}
//---------------------------------------------------------------------------
AnsiString __fastcall TRemoteFilePanelItem::CustomColumnData(int Column)
{
  switch (Column) {
    case 0: return FRemoteFile->Group;
    case 1: return FRemoteFile->RightsStr;
    case 2: return FRemoteFile->Rights->Octal;
    default: assert(false); return AnsiString();
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::TranslateColumnTypes(AnsiString & ColumnTypes,
  TStrings * ColumnTitles)
{
  AnsiString AColumnTypes = ColumnTypes;
  ColumnTypes = "";
  AnsiString Column;
  AnsiString Title;
  while (!AColumnTypes.IsEmpty())
  {
    Column = CutToChar(AColumnTypes, ',', false);
    if (Column == "G")
    {
      Column = "C0";
      Title = FarPlugin->GetMsg(GROUP_COL_TITLE);
    }
    else if (Column == "R")
    {
      Column = "C1";
      Title = FarPlugin->GetMsg(RIGHTS_COL_TITLE);
    }
    else if (Column == "RO")
    {
      Column = "C2";
      Title = FarPlugin->GetMsg(RIGHTS_OCTAL_COL_TITLE);
    }
    else
    {
      Title = "";
    }
    ColumnTypes += (ColumnTypes.IsEmpty() ? "" : ",") + Column;
    if (ColumnTitles)
    {
      ColumnTitles->Add(Title);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    if (FarConfiguration->CustomPanelModeDetailed)
    {
      AnsiString ColumnTypes = FarConfiguration->ColumnTypesDetailed;
      AnsiString StatusColumnTypes = FarConfiguration->StatusColumnTypesDetailed;
      
      TranslateColumnTypes(ColumnTypes, ColumnTitles);
      TranslateColumnTypes(StatusColumnTypes, NULL);

      PanelModes->SetPanelMode(5 /*detailed */,
        ColumnTypes, FarConfiguration->ColumnWidthsDetailed,
        ColumnTitles, FarConfiguration->FullScreenDetailed, false, true, false, 
        StatusColumnTypes, FarConfiguration->StatusColumnWidthsDetailed);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::TWinSCPFileSystem(TCustomFarPlugin * APlugin) :
  TCustomFarFileSystem(APlugin)
{
  FDirectoryRead = false;
  FProgressSaveScreenHandle = 0;
  FFileList = NULL;
  FPanelItems = NULL;
  FSavedFindFolder = "";
}
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::~TWinSCPFileSystem()
{
  if (FTerminal)
  {
    SaveSession();
  }
  assert(!FProgressSaveScreenHandle);
  assert(!FFileList);
  assert(!FPanelItems);
  SAFE_DESTROY(FTerminal);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::HandleException(Exception * E, int OpMode)
{
  if (E->InheritsFrom(__classid(EFatal)))
  {
    ShowExtendedException(E, this);
    ClosePlugin();
  }
  else
  {
    TCustomFarFileSystem::HandleException(E, OpMode);
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connected()
{
  return FTerminal != NULL;
}
//---------------------------------------------------------------------------
TWinSCPPlugin * __fastcall TWinSCPFileSystem::WinSCPPlugin()
{
  return ::WinSCPPlugin();
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::GetOpenPluginInfoEx(long unsigned & Flags,
  AnsiString & /*HostFile*/, AnsiString & CurDir, AnsiString & Format,
  AnsiString & PanelTitle, TFarPanelModes * PanelModes, int & /*StartPanelMode*/,
  int & /*StartSortMode*/, bool & /*StartSortOrder*/, TFarKeyBarTitles * KeyBarTitles,
  AnsiString & ShortcutData)
{
  if (Connected())
  {
    Flags = OPIF_USEFILTER | OPIF_USESORTGROUPS | OPIF_USEHIGHLIGHTING |
      OPIF_SHOWPRESERVECASE | OPIF_COMPAREFATTIME;
    CurDir = FTerminal->CurrentDirectory;
    PanelTitle = ::FORMAT(" %s ", (CurDir));
    Format = FTerminal->SessionData->Name;
    ShortcutData = ::FORMAT("%s\1%s", (FTerminal->SessionData->Name, CurDir));

    TRemoteFilePanelItem::SetPanelModes(PanelModes);
    TRemoteFilePanelItem::SetKeyBarTitles(KeyBarTitles);
  }
  else
  {
    Format = "winscp";
    Flags = OPIF_USESORTGROUPS | OPIF_ADDDOTS | OPIF_SHOWNAMESONLY |
      OPIF_SHOWPRESERVECASE;
    PanelTitle = ::FORMAT(" %s ", (GetMsg(STORED_SESSION_TITLE)));

    TSessionPanelItem::SetPanelModes(PanelModes);
    TSessionPanelItem::SetKeyBarTitles(KeyBarTitles);
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::GetFindDataEx(TList * PanelItems)
{
  if (Connected())
  {
    if (!FDirectoryRead && FTerminal->Active)
    {
      FPlugin->ShowConsoleTitle(GetMsg(READING_DIRECTORY_TITLE));
      try
      {
        FTerminal->ReloadDirectory();
      }
      __finally
      {
        FPlugin->ClearConsoleTitle();
      }
    }

    assert(FDirectoryRead);
    FDirectoryRead = false;

    TRemoteFile * File;
    for (int Index = 0; Index < FTerminal->Files->Count; Index++)
    {
      File = FTerminal->Files->Files[Index];
      PanelItems->Add(new TRemoteFilePanelItem(File));
    }
  }
  else
  {
    assert(StoredSessions);
    StoredSessions->Load();
    if (StoredSessions->Count > 0)
    {
      TSessionData * Data;
      for (int Index = 0; Index < StoredSessions->Count; Index++)
      {
        Data = StoredSessions->Sessions[Index];
        PanelItems->Add(new TSessionPanelItem(Data));
      }
    }
    else
    {
      PanelItems->Add(new THintPanelItem(GetMsg(NEW_SESSION_HINT)));
    }
  }
  return true;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::EditConnectSession(TSessionData * Data, bool Edit)
{
  bool NewData = !Data;

  if (NewData)
  {
    Data = new TSessionData("");
  }

  try
  {
    bool ToConnect = false;
    if (Edit || !Data->CanLogin)
    {
      if (SessionDialog(Data, ToConnect))
      {
        int Select = -1;
        if (!NewData || !ToConnect)
        {
          if (NewData)
          {
            AnsiString Name = Data->SessionName;
            if (FPlugin->InputBox(GetMsg(NEW_SESSION_NAME_TITLE),
                  GetMsg(NEW_SESSION_NAME_PROMPT), Name, 0) && !Name.IsEmpty())
            {
              if (StoredSessions->FindByName(Name))
              {
                throw Exception(FORMAT(GetMsg(SESSION_ALREADY_EXISTS_ERROR), (Name)));
              }
              else
              {
                TSessionData * NData = StoredSessions->NewSession(Name, Data);
                Select = StoredSessions->IndexOf(NData);
              }
            }
          }

          StoredSessions->Save();
          UpdatePanel(true);
          if (Select < 0)
          {
            RedrawPanel();
          }
          else
          {
            PanelInfo->FocusedIndex = Select + 1;
          }
        }
      }
    }
    else
    {
      ToConnect = true;
    }

    if (ToConnect && Connect(Data))
    {
      UpdatePanel(true);
      RedrawPanel();
      if (PanelInfo->ItemCount)
      {
        PanelInfo->FocusedIndex = 0;
      }
    }
  }
  __finally
  {
    if (NewData)
    {
      delete Data;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessEventEx(int Event, void * Param)
{
  bool Result = false;
  if (Connected())
  {
    if (Event == FE_COMMAND)
    {
      AnsiString Command = (char *)Param;
      if (Command.SubString(1, 3).LowerCase() == "cd ")
      {
        Result = false;
      }
      else if (FTerminal->IsCapable[fcAnyCommand])
      {
        Result = ExecuteCommand(Command);
      }
      else
      {
        throw Exception(FORMAT(GetMsg(EXECUTE_COMMAND_NOT_SUPPORTED),
          (FTerminal->ProtocolName)));
      }
    }
    else if (Event == FE_IDLE)
    {
      FTerminal->Idle();
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalEchoAddLine(TObject * /*Sender*/,
  const AnsiString AddedLine)
{
  if (!AddedLine.IsEmpty() && (AddedLine[1] == '<' || AddedLine[1] == '!'))
  {
    int ReturnCode;
    AnsiString Line = AddedLine;
    Line.Delete(1, 2);
    if (!TSCPFileSystem::RemoveLastLine(Line, ReturnCode) ||
        !Line.IsEmpty())
    {
      FPlugin->WriteConsole(Line + "\n");
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExecuteCommand(const AnsiString Command)
{
  FTerminal->BeginTransaction();
  try
  {
    FarControl(FCTL_SETCMDLINE, NULL);
    TLogAddLineEvent FOldLogAddLine = FTerminal->Log->OnAddLine;
    FPlugin->ShowConsoleTitle(Command);
    try
    {
      FTerminal->Log->OnAddLine = TerminalEchoAddLine;

      FPlugin->ShowTerminalScreen();

      FTerminal->AnyCommand(Command);
    }
    __finally
    {
      FTerminal->Log->OnAddLine = FOldLogAddLine;
      FPlugin->ScrollTerminalScreen(1);
      FarControl(FCTL_SETUSERSCREEN, NULL);
      FPlugin->ClearConsoleTitle();
    }
  }
  __finally
  {
    if (FTerminal->Active)
    {
      FTerminal->EndTransaction();
      UpdatePanel();
    }
    else
    {
      RedrawPanel();
      RedrawPanel(true);
    }
  }
  return true;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessKeyEx(int Key, unsigned int ControlState)
{
  bool Handled = false;

  if (!Connected())
  {
    TFarPanelItem * Focused = PanelInfo->FocusedItem;
    TSessionData * Data = NULL;

    if (Focused->IsFile && Focused->UserData)
    {
      Data = (TSessionData *)Focused->UserData;
    }

    if ((Key == VK_RETURN && ControlState == 0 && Data) ||
        (Key == VK_F4 && ControlState == 0 && (Data || StoredSessions->Count == 0)) ||
        (Key == VK_F4 && (ControlState & PKF_SHIFT)))
    {
      EditConnectSession(ControlState & PKF_SHIFT ? NULL : Data, Key == VK_F4);
      Handled = true;
    }
  }
  else
  {
    if (Key == 'A' && ControlState & PKF_CONTROL)
    {
      FileProperties();
      Handled = true;
    }
    if (Key == VK_F6 && ControlState & PKF_ALT)
    {
      CreateLink();
      Handled = true;
    }
  }
  return Handled;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CreateLink()
{
  if (FTerminal->IsCapable[fcResolveSymlink] &&
      FTerminal->IsCapable[fcSymbolicLink])
  {
    bool Edit = false;
    TRemoteFile * File = NULL;
    AnsiString FileName;
    AnsiString PointTo;
    bool SymbolicLink = true;

    if (PanelInfo->FocusedItem && PanelInfo->FocusedItem->UserData)
    {
      File = (TRemoteFile *)PanelInfo->FocusedItem->UserData;

      Edit = File->IsSymLink && Terminal->SessionData->ResolveSymlinks;
      if (Edit)
      {
        FileName = File->FileName;
        PointTo = File->LinkTo;
      }
      else
      {
        PointTo = File->FileName;
      }
    }

    if (LinkDialog(FileName, PointTo, SymbolicLink, Edit,
          Terminal->IsCapable[fcHardLink]))
    {
      if (Edit)
      {
        assert(File->FileName == FileName);
        bool Recursive = false;
        Terminal->ExceptionOnFail = true;
        try
        {
          Terminal->DeleteFile("", File, &Recursive);
        }
        __finally
        {
          Terminal->ExceptionOnFail = false;
        }
      }
      Terminal->CreateLink(FileName, PointTo, SymbolicLink);
      UpdatePanel();
      RedrawPanel();
    }
  }
  else
  {
    throw Exception(FORMAT(GetMsg(LINK_UNSUPPORTED),
      (FTerminal->ProtocolName)));
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::FileProperties()
{
  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList)
  {
    assert(!FPanelItems);

    try
    {
      TRemoteProperties CurrentProperties;

      CurrentProperties = TRemoteProperties::CommonProperties(FileList);

      int Flags = 0;
      if (FTerminal->IsCapable[fcModeChanging]) Flags |= cpMode;
      if (FTerminal->IsCapable[fcOwnerChanging]) Flags |= cpOwner;
      if (FTerminal->IsCapable[fcGroupChanging]) Flags |= cpGroup;

      TRemoteProperties NewProperties = CurrentProperties;
      if (PropertiesDialog(FileList, FTerminal->CurrentDirectory,
          FTerminal->UserGroups, &NewProperties, Flags))
      {
        NewProperties = TRemoteProperties::ChangedProperties(CurrentProperties,
          NewProperties);
        try
        {
          FTerminal->ChangeFilesProperties(FileList, &NewProperties);
        }
        __finally
        {
          PanelInfo->ApplySelection();
          UpdatePanel();
          RedrawPanel();
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::SetDirectoryEx(const AnsiString Dir, int OpMode)
{
  assert(Connected());
  // workaround to ignore "change to root directory" command issued by FAR,
  // before file is opened for viewing/editing from "find file" dialog
  // when plugin uses UNIX style paths
  if (OpMode & OPM_FIND && OpMode & OPM_SILENT && Dir == "\\")
  {
    if (FSavedFindFolder.IsEmpty())
    {
      return true;
    }
    else
    {
      bool Result;
      try
      {
        Result = SetDirectoryEx(FSavedFindFolder, OpMode);
      }
      __finally
      {
        FSavedFindFolder = "";
      }
      return Result;
    }
  }
  else
  {
    if (OpMode & OPM_FIND && FSavedFindFolder.IsEmpty())
    {
      FSavedFindFolder = FTerminal->CurrentDirectory;
    }

    FTerminal->ExceptionOnFail = true;
    try
    {
      // OPM_SILENT is used by TWinSCPPlugin::OpenPluginEx(OPEN_COMMANDLINE, ...)
      if (!(OpMode & (OPM_FIND | OPM_SILENT)))
      {
        UpdatePanel();
      }

      if (Dir == "\\")
      {
        FTerminal->ChangeDirectory(ROOTDIRECTORY);
      }
      else if ((Dir == PARENTDIRECTORY) && (FTerminal->CurrentDirectory == ROOTDIRECTORY))
      {
        ClosePlugin();
      }
      else
      {
        FTerminal->ChangeDirectory(Dir);
      }
    }
    __finally
    {
      FTerminal->ExceptionOnFail = false;
    }
    return true;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::MakeDirectoryEx(AnsiString & Name, int OpMode)
{
  if (Connected())
  {
    assert(!(OpMode & OPM_SILENT) || !Name.IsEmpty());

    if ((OpMode & OPM_SILENT) ||
        FPlugin->InputBox(GetMsg(CREATE_FOLDER_TITLE), GetMsg(CREATE_FOLDER_PROMPT),
          Name, 0, "NewFolder"))
    {
      FTerminal->CreateDirectory(Name);
      return 1;
    }
    else
    {
      Name = "";
      return -1;
    }
  }
  else
  {
    Name = "";
    return -1;
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::DeleteFilesEx(TList * PanelItems, int OpMode)
{
  if (Connected())
  {
    AnsiString Query;
    if (PanelItems->Count > 1)
    {
      Query = FORMAT(GetMsg(DELETE_FILES_CONFIRM), (PanelItems->Count));
    }
    else
    {
      Query = FORMAT(GetMsg(DELETE_FILE_CONFIRM),
        (((TFarPanelItem *)PanelItems->Items[0])->FileName));
    }

    if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
      (WinSCPPlugin()->MoreMessageDialog(Query, NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      FFileList = CreateFileList(PanelItems, osRemote);
      try
      {
        FPanelItems = PanelItems;
        FTerminal->DeleteFiles(FFileList);
      }
      __finally
      {
        FPanelItems = NULL;
        SAFE_DESTROY(FFileList);
      }
    }
    return true;
  }
  else
  {
    if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
      (WinSCPPlugin()->MoreMessageDialog(GetMsg(DELETE_SESSIONS_CONFIRM), NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      for (int i = 0; i < PanelItems->Count; i++)
      {
        TFarPanelItem * PanelItem = (TFarPanelItem *)PanelItems->Items[i];
        assert(PanelItem);
        if (PanelItem->IsFile)
        {
          TSessionData * Data = (TSessionData *)PanelItem->UserData;
          assert(Data);
          Data->Remove();
          StoredSessions->Remove(Data);
          PanelItem->Selected = false;
        }
      }
    }
    return true;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::GetFilesEx(TList * PanelItems, bool Move,
  AnsiString & DestPath, int OpMode)
{
  int Result;
  if (Connected())
  {
    FFileList = CreateFileList(PanelItems, osRemote);
    try
    {
      bool Confirmed = (OpMode & OPM_SILENT);
      TCopyParamType CopyParam = Configuration->CopyParam;

      if (!Confirmed)
      {
        Confirmed = CopyDialog(false, Move, FFileList,
          FTerminal->IsCapable[fcTextMode], DestPath, &CopyParam);
        //FPlugin->InputBox(Title, Prompt, DestPath, 0, "Copy");
      }

      if (Confirmed)
      {
        if ((FFileList->Count == 1) && (OpMode & OPM_EDIT))
        {
          FLastEditFolder = DestPath;
          FLastEditFile = FFileList->Strings[0];
        }
        else
        {
          FLastEditFolder = "";
          FLastEditFile = "";
        }

        FPanelItems = PanelItems;
        int Params = (Move ? cpDelete : 0) |
          ((OpMode & (OPM_VIEW | OPM_EDIT | OPM_FIND)) ? cpTemporary : 0);
        FTerminal->CopyToLocal(FFileList, DestPath, &CopyParam, Params);
        Result = 1;
      }
      else
      {
        Result = -1;
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else
  {
    if (StoredSessions->Count > 0 &&
        ExportSessions(PanelItems, Move, DestPath, OpMode))
    {
      Result = 1;
    }
    else
    {
      Result = -1;
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExportSessions(TList * PanelItems, bool /*Move*/,
  AnsiString & DestPath, int OpMode)
{
  AnsiString Title = GetMsg(EXPORT_SESSION_TITLE);
  AnsiString Prompt;
  if (PanelItems->Count == 1)
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSION_PROMPT),
      (((TFarPanelItem *)PanelItems->Items[0])->FileName));
  }
  else
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSIONS_PROMPT), (PanelItems->Count));
  }

  bool Result = (OpMode & OPM_SILENT) ||
    FPlugin->InputBox(Title, Prompt, DestPath, 0, "Copy");

  if (Result)
  {
    TSessionData * Data;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      Data = (TSessionData *)((TFarPanelItem *)PanelItems->Items[i])->UserData;
      THierarchicalStorage * Storage = NULL;
      TSessionData * ExportData = NULL;
      try
      {
        ExportData = new TSessionData(Data->Name);
        ExportData->Assign(Data);
        ExportData->Modified = true;
        Storage = new TIniFileStorage(IncludeTrailingBackslash(DestPath) + ExportData->Name + ".ini");;
        if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, true))
        {
          ExportData->Save(Storage);
        }
      }
      __finally
      {
        delete Storage;
        delete ExportData;
      }

    }
  }
  return Result;
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::PutFilesEx(TList * PanelItems, bool Move, int OpMode)
{
  int Result = 1;
  if (Connected())
  {
    FFileList = CreateFileList(PanelItems, osLocal);
    try
    {
      bool Confirmed = (OpMode & OPM_SILENT);
      bool Ask = !Confirmed;

      // huerictics: do not ask for target directory when uploaded file
      // was downloaded in edit mode
      if (!Confirmed && (GetCurrentDir() == FLastEditFolder) &&
          (FFileList->Count == 1) && (FFileList->Strings[0] == FLastEditFile))
      {
        Confirmed = true;
        FLastEditFolder = "";
        FLastEditFile = "";
      }

      AnsiString DestPath = FTerminal->CurrentDirectory;
      TCopyParamType CopyParam = Configuration->CopyParam;

      if (!Confirmed)
      {
        Confirmed = CopyDialog(true, Move, FFileList,
          FTerminal->IsCapable[fcTextMode], DestPath, &CopyParam);
      }
      
      if (Confirmed)
      {
        FPanelItems = PanelItems;
        int Params = (Move ? cpDelete : 0) | (Ask ? 0 : cpNoConfirmation);
        FTerminal->CopyToRemote(FFileList, DestPath, &CopyParam, Params);
      }
      else
      {
        Result = -1;
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else
  {
    if (!ImportSessions(PanelItems, Move, OpMode))
    {
      return -1;
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ImportSessions(TList * PanelItems, bool /*Move*/,
  int OpMode)
{
  bool Result = (OpMode & OPM_SILENT) ||
    (WinSCPPlugin()->MoreMessageDialog(GetMsg(IMPORT_SESSIONS_PROMPT), NULL,
      qtConfirmation, qaOK | qaCancel) == qaOK);

  if (Result)
  {
    AnsiString FileName;
    TFarPanelItem * PanelItem;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[i];
      bool AnyData = false;
      FileName = PanelItem->FileName;
      if (PanelItem->IsFile)
      {
        THierarchicalStorage * Storage = NULL;
        try
        {
          Storage = new TIniFileStorage(IncludeTrailingBackslash(GetCurrentDir()) + FileName);
          if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, false) &&
              Storage->HasSubKeys())
          {
            AnyData = true;
            StoredSessions->Load(Storage, true);
            StoredSessions->Save();
          }
        }
        __finally
        {
          delete Storage;
        }
      }
      if (!AnyData)
      {
        throw Exception(FORMAT(GetMsg(IMPORT_SESSIONS_EMPTY), (FileName)));
      }
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateSelectedFileList(
  TOperationSide Side)
{
  TStrings * Result;
  if (PanelInfo->SelectedCount > 0)
  {
    Result = CreateFileList(PanelInfo->Items, Side, true);
  }
  else
  {
    TFarPanelItem * PanelItem = PanelInfo->FocusedItem;
    if (PanelItem->IsParentDirectory)
    {
      Result = NULL;
    }
    else
    {
      Result = new TStringList();
      assert(PanelItem->UserData);
      Result->AddObject(PanelItem->FileName, (TObject *)PanelItem->UserData);
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateFileList(TList * PanelItems,
  TOperationSide Side, bool SelectedOnly)
{
  TStrings * FileList = new TStringList();
  try
  {
    AnsiString Name;
    TFarPanelItem * PanelItem;
    TObject * Data = NULL;
    for (int Index = 0; Index < PanelItems->Count; Index++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
      assert(PanelItem);
      if (!SelectedOnly || PanelItem->Selected)
      {
        if (Side == osRemote)
        {
          //File = FTerminal->Files->FindFile(PanelItem->FileName);
          Data = (TRemoteFile *)PanelItem->UserData;
          assert(Data);
        }
        FileList->AddObject(PanelItem->FileName, Data);
      }
    }
  }
  catch (...)
  {
    delete FileList;
    throw;
  }
  return FileList;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::SaveSession()
{
  if (!FTerminal->SessionData->Name.IsEmpty())
  {
    FTerminal->SessionData->RemoteDirectory = FTerminal->CurrentDirectory;
    TSessionData * Data;
    Data = (TSessionData *)StoredSessions->FindByName(FTerminal->SessionData->Name);
    if (Data)
    {
      Data->Assign(FTerminal->SessionData);
      StoredSessions->Save();
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connect(TSessionData * Data)
{
  bool Result = false;
  assert(!FTerminal);
  FTerminal = new TTerminal();
  try
  {
    FTerminal->Configuration = Configuration;
    FTerminal->SessionData = Data;
    FTerminal->OnQueryUser = ((TWinSCPPlugin*)FPlugin)->TerminalQueryUser;
    FTerminal->OnUpdateStatus = TerminalUpdateStatus;
    FTerminal->OnReadDirectory = TerminalReadDirectory;
    FTerminal->OnFinished = OperationFinished;
    FTerminal->OnProgress = OperationProgress;
    try
    {
      FTerminal->Open();
      FTerminal->DoStartup();
    }
    __finally
    {
      FPlugin->ClearProgress();
    }

    Result = true;
  }
  catch(Exception &E)
  {
    ShowExtendedException(&E, NULL);
    SAFE_DESTROY(FTerminal);
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalUpdateStatus(TObject * Sender)
{
  static const int ConnectionStatusStrings[] =
    { STATUS_CLOSED, STATUS_INITWINSOCK, STATUS_LOOKUPHOST, STATUS_CONNECT,
      STATUS_AUTHENTICATE, STATUS_AUTHENTICATED, STATUS_STARTUP,
      STATUS_OPEN_DIRECTORY, STATUS_READY };

  assert(Sender == FTerminal);
  assert((FTerminal->Status >= 0) &&
    (FTerminal->Status < LENOF(ConnectionStatusStrings)));
  FPlugin->ShowProgress(GetMsg((int)ConnectionStatusStrings[FTerminal->Status]));
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalReadDirectory(TObject * /*Sender*/,
  bool /*ReloadOnly*/)
{
  FDirectoryRead = true;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationProgress(
  TFileOperationProgressType & ProgressData, TCancelStatus & /*Cancel*/)
{
  bool First = false;
  if (ProgressData.InProgress && !FProgressSaveScreenHandle)
  {
    FPlugin->SaveScreen(FProgressSaveScreenHandle);
    First = true;
  }
  // operation is finished (or terminated), so we hide progress form
  else if (!ProgressData.InProgress && FProgressSaveScreenHandle)
  {
    FPlugin->RestoreScreen(FProgressSaveScreenHandle);
    FPlugin->ClearConsoleTitle();
  }

  ShowOperationProgress(ProgressData, First);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationFinished(TOperationSide /*Side*/,
  bool /*DragDrop*/, const AnsiString FileName, bool Success,
  bool & /*DisconnectWhenComplete*/)
{
  TFarPanelItem * PanelItem = NULL;;

  if (!FPanelItems)
  {
    TList * PanelItems = PanelInfo->Items;
    for (int Index = 0; Index < PanelItems->Count; Index++)
    {
      if (((TFarPanelItem *)PanelItems->Items[Index])->FileName == FileName)
      {
        PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
        break;
      }
    }
  }
  else
  {
    assert(FFileList);
    assert(FPanelItems->Count == FFileList->Count);
    int Index = FFileList->IndexOf(FileName);
    assert(Index >= 0);
    PanelItem = (TFarPanelItem *)FPanelItems->Items[Index];
  }
  
  assert(PanelItem->FileName == FileName);
  if (Success)
  {
    PanelItem->Selected = false;
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ShowOperationProgress(
  TFileOperationProgressType & ProgressData, bool First)
{
  static unsigned long LastTicks;
  unsigned long Ticks = GetTickCount();
  if (Ticks - LastTicks > 500 || First)
  {
    LastTicks = Ticks;

    static const int ProgressWidth = 46;
    static const int Captions[] = {PROGRESS_COPY, PROGRESS_MOVE, PROGRESS_DELETE, PROGRESS_SETPROPERTIES};
    static AnsiString ProgressFileLabel;
    static AnsiString TargetDirLabel;
    static AnsiString TransferModeLabel;
    static AnsiString ResumeSupportLabel;
    static AnsiString StartTimeLabel;
    static AnsiString TimeElapsedLabel;
    static AnsiString BytesTransferedLabel;
    static AnsiString CPSLabel;

    if (ProgressFileLabel.IsEmpty())
    {
      ProgressFileLabel = GetMsg(PROGRESS_FILE_LABEL);
      TargetDirLabel = GetMsg(TARGET_DIR_LABEL);
      TransferModeLabel = GetMsg(TRANSFER_MODE_LABEL);
      ResumeSupportLabel = GetMsg(RESUME_SUPPORT_LABEL);
      StartTimeLabel = GetMsg(START_TIME_LABEL);
      TimeElapsedLabel = GetMsg(TIME_ELAPSED_LABEL);
      BytesTransferedLabel = GetMsg(BYTES_TRANSFERED_LABEL);
      CPSLabel = GetMsg(CPS_LABEL);
    }

    int Progress;

    bool TransferOperation =
      ((ProgressData.Operation == foCopy) || (ProgressData.Operation == foMove));

    AnsiString Message;
    AnsiString Title = GetMsg(Captions[(int)ProgressData.Operation - 1]);
    AnsiString ProgressBar;
    Progress = ProgressData.OperationProgress();
    ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * Progress / 100);
    ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
    Message = ProgressFileLabel + MinimizeName(ProgressData.FileName,
      ProgressWidth - ProgressFileLabel.Length(), ProgressData.Side == osRemote) + "\n";
    if (TransferOperation && !ProgressData.DragDrop)
    {
      Message += TargetDirLabel + MinimizeName(ProgressData.Directory,
        ProgressWidth - TargetDirLabel.Length(), ProgressData.Side == osLocal) + "\n";
    }
    Message += ProgressBar + "\n";
    if (TransferOperation)
    {
      Message += "\1\n";
      AnsiString StatusLine;
      AnsiString Value;
      Value = GetMsg(ProgressData.AsciiTransfer ? TRANSFER_ASCII : TRANSFER_BINARY);
      StatusLine = TransferModeLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - TransferModeLabel.Length() - Value.Length()) +
        Value + "  ";
      switch (ProgressData.ResumeStatus) {
        case rsEnabled: Value = GetMsg(RESUME_ENABLED); break;
        case rsDisabled: Value = GetMsg(RESUME_DISABLED); break;
        case rsNotAvailable: Value = GetMsg(RESUME_NOT_AVAILABLE); break;
        default: assert(false);
      }
      StatusLine = StatusLine + ResumeSupportLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        ResumeSupportLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = ProgressData.StartTime.TimeString();
      StatusLine = StartTimeLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - StartTimeLabel.Length() - Value.Length()) +
        Value + "  ";
      Value = FormatDateTime(Configuration->TimeFormat, ProgressData.TimeElapsed());
      StatusLine = StatusLine + TimeElapsedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        TimeElapsedLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = FormatBytes(ProgressData.TotalTransfered);
      StatusLine = BytesTransferedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - BytesTransferedLabel.Length() - Value.Length()) +
        Value + "  ";
      Value = FORMAT("%s/s", (FormatBytes(ProgressData.CPS())));
      StatusLine = StatusLine + CPSLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        CPSLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";
      int TransferProgress = ProgressData.TransferProgress();
      if (ProgressData.Count == 1)
      {
        Progress = TransferProgress;
      }
      ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * TransferProgress / 100);
      ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
      Message += ProgressBar + "\n";
    }
    FPlugin->Message(0, Title, Message);

    if (First)
    {
      FPlugin->ShowConsoleTitle(Title);
    }
    FPlugin->UpdateConsoleTitleProgress(Progress);

    if (FPlugin->CheckForEsc())
    {
      CancelConfiguration(ProgressData);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CancelConfiguration(TFileOperationProgressType & ProgressData)
{
  if (!ProgressData.Suspended)
  {
    ProgressData.Suspend();
    try
    {
      TCancelStatus ACancel;
      int Result;
      if (ProgressData.TransferingFile &&
          (ProgressData.TimeExpected() > Configuration->IgnoreCancelBeforeFinish))
      {
        Result = WinSCPPlugin()->MoreMessageDialog(GetMsg(CANCEL_OPERATION_FATAL), NULL,
          qtWarning, qaYes | qaNo | qaCancel);
      }
      else
      {
        Result = WinSCPPlugin()->MoreMessageDialog(GetMsg(CANCEL_OPERATION), NULL,
          qtConfirmation, qaOK | qaCancel);
      }
      switch (Result) {
        case qaYes:
          ACancel = csCancelTransfer; break;
        case qaOK:
        case qaNo:
          ACancel = csCancel; break;
        default:
          ACancel = csContinue; break;
      }

      if (ACancel > ProgressData.Cancel)
      {
        ProgressData.Cancel = ACancel;
      }
    }
    __finally
    {
      ProgressData.Resume();
    }
  }
}
//---------------------------------------------------------------------
AnsiString __fastcall TWinSCPFileSystem::FormatBytes(__int64 Bytes)
{
  static const __int64 FormatBytesAbove = __int64(100*1024);
  AnsiString Result;

  if (Bytes < FormatBytesAbove)
  {
    return FormatFloat("#,##0 \"B\"", Bytes);
  }
  else if (Bytes < __int64(100*1024*1024))
  {
    return FormatFloat("#,##0 \"KB\"", Bytes / 1024);
  }
  else
  {
    return FormatFloat("#,##0 \"MB\"", Bytes / (1024*1024));
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CutFirstDirectory(AnsiString & S, bool Unix)
{
  bool Root;
  int P;
  AnsiString Sep = Unix ? "/" : "\\";
  if (S == Sep)
  {
    S = "";
  }
  else
  {
    if (S[1] == Sep[1])
    {
      Root = true;
      S.Delete(1, 1);
    }
    else
    {
      Root = false;
    }
    if (S[1] == '.')
    {
      S.Delete(1, 4);
    }
    P = S.AnsiPos(Sep[1]);
    if (P)
    {
      S.Delete(1, P);
      S = "..." + Sep + S;
    }
    else
    {
      S = "";
    }
    if (Root)
    {
      S = Sep + S;
    }
  }
}
//---------------------------------------------------------------------------
AnsiString __fastcall TWinSCPFileSystem::MinimizeName(const AnsiString FileName,
  int MaxLen, bool Unix)
{
  AnsiString Drive, Dir, Name, Result;
  AnsiString Sep = Unix ? "/" : "\\";

  Result = FileName;
  if (Unix)
  {
    int P = Result.LastDelimiter("/");
    if (P)
    {
      Dir = Result.SubString(1, P);
      Name = Result.SubString(P + 1, Result.Length() - P);
    }
    else
    {
      Dir = "";
      Name = Result;
    }
  }
  else
  {
    Dir = ExtractFilePath(Result);
    Name = ExtractFileName(Result);

    if (Dir.Length() >= 2 && Dir[2] == ':')
    {
      Drive = Dir.SubString(1, 2);
      Dir.Delete(1, 2);
    }
  }

  while ((!Dir.IsEmpty() || !Drive.IsEmpty()) && (Result.Length() > MaxLen))
  {
    if (Dir == Sep + "..." + Sep)
    {
      Dir == "..." + Sep;
    }
    else if (Dir == "")
    {
      Drive = "";
    }
    else
    {
      CutFirstDirectory(Dir, Unix);
    }
    Result = Drive + Dir + Name;
  }

  if (Result.Length() > MaxLen)
  {
    Result = Result.SubString(1, MaxLen);
  }
  return Result;
}
