//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "WinSCPFileSystem.h"
#include "WinSCPPlugin.h"
#include "FarTexts.h"
#include "FarConfiguration.h"
#include "farkeys.hpp"
#include <Common.h>
#include <SessionData.h>
#include <ScpMain.h>
#include <SysUtils.hpp>
#include <ScpFileSystem.h>
#include <Bookmarks.h>
#include <GUITools.h>
//---------------------------------------------------------------------------
#pragma package(smart_init)
//---------------------------------------------------------------------------
__fastcall TSessionPanelItem::TSessionPanelItem(TSessionData * ASessionData):
  TCustomFarPanelItem()
{
  assert(ASessionData);
  FSessionData = ASessionData;
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    ColumnTitles->Add(FarPlugin->GetMsg(SESSION_NAME_COL_TITLE));
    for (int Index = 0; Index < PANEL_MODES_COUNT; Index++)
    {
      PanelModes->SetPanelMode(Index, "N", "0", ColumnTitles, false, false, false);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsNone, 6, 7);
  KeyBarTitles->SetKeyBarTitle(fsNone, 5, FarPlugin->GetMsg(EXPORT_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
  KeyBarTitles->SetKeyBarTitle(fsShift, 4, FarPlugin->GetMsg(NEW_SESSION_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsShift, 5, FarPlugin->GetMsg(COPY_SESSION_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsShift, 6, FarPlugin->GetMsg(RENAME_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 7, 8);
  KeyBarTitles->ClearKeyBarTitle(fsAlt, 6);
  KeyBarTitles->ClearKeyBarTitle(fsCtrl, 4, 11);
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & /*Size*/,
  unsigned long & /*FileAttributes*/,
  TDateTime & /*LastWriteTime*/, TDateTime & /*LastAccess*/,
  unsigned long & /*NumberOfLinks*/, AnsiString & /*Description*/,
  AnsiString & /*Owner*/, void *& UserData, int & /*CustomColumnNumber*/)
{
  FileName = FSessionData->Name;
  UserData = FSessionData;
}
//---------------------------------------------------------------------------
__fastcall TRemoteFilePanelItem::TRemoteFilePanelItem(TRemoteFile * ARemoteFile):
  TCustomFarPanelItem()
{
  assert(ARemoteFile);
  FRemoteFile = ARemoteFile;
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & Size,
  unsigned long & FileAttributes,
  TDateTime & LastWriteTime, TDateTime & LastAccess,
  unsigned long & NumberOfLinks, AnsiString & /*Description*/,
  AnsiString & Owner, void *& UserData, int & CustomColumnNumber)
{
  FileName = FRemoteFile->FileName;
  Size = FRemoteFile->Size;
  FileAttributes =
    (FRemoteFile->IsDirectory ? FILE_ATTRIBUTE_DIRECTORY : 0) |
    (FRemoteFile->IsHidden ? FILE_ATTRIBUTE_HIDDEN : 0) |
    (FRemoteFile->Rights->ReadOnly ? FILE_ATTRIBUTE_READONLY : 0);
  LastWriteTime = FRemoteFile->Modification;
  LastAccess = FRemoteFile->LastAccess;
  NumberOfLinks = FRemoteFile->IsSymLink ? 1 : 0;
  Owner = FRemoteFile->Owner;
  UserData = FRemoteFile;
  CustomColumnNumber = 3;
}
//---------------------------------------------------------------------------
AnsiString __fastcall TRemoteFilePanelItem::CustomColumnData(int Column)
{
  switch (Column) {
    case 0: return FRemoteFile->Group;
    case 1: return FRemoteFile->RightsStr;
    case 2: return FRemoteFile->Rights->Octal;
    default: assert(false); return AnsiString();
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::TranslateColumnTypes(AnsiString & ColumnTypes,
  TStrings * ColumnTitles)
{
  AnsiString AColumnTypes = ColumnTypes;
  ColumnTypes = "";
  AnsiString Column;
  AnsiString Title;
  while (!AColumnTypes.IsEmpty())
  {
    Column = CutToChar(AColumnTypes, ',', false);
    if (Column == "G")
    {
      Column = "C0";
      Title = FarPlugin->GetMsg(GROUP_COL_TITLE);
    }
    else if (Column == "R")
    {
      Column = "C1";
      Title = FarPlugin->GetMsg(RIGHTS_COL_TITLE);
    }
    else if (Column == "RO")
    {
      Column = "C2";
      Title = FarPlugin->GetMsg(RIGHTS_OCTAL_COL_TITLE);
    }
    else
    {
      Title = "";
    }
    ColumnTypes += (ColumnTypes.IsEmpty() ? "" : ",") + Column;
    if (ColumnTitles)
    {
      ColumnTitles->Add(Title);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    if (FarConfiguration->CustomPanelModeDetailed)
    {
      AnsiString ColumnTypes = FarConfiguration->ColumnTypesDetailed;
      AnsiString StatusColumnTypes = FarConfiguration->StatusColumnTypesDetailed;

      TranslateColumnTypes(ColumnTypes, ColumnTitles);
      TranslateColumnTypes(StatusColumnTypes, NULL);

      PanelModes->SetPanelMode(5 /*detailed */,
        ColumnTypes, FarConfiguration->ColumnWidthsDetailed,
        ColumnTitles, FarConfiguration->FullScreenDetailed, false, true, false,
        StatusColumnTypes, FarConfiguration->StatusColumnWidthsDetailed);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
  KeyBarTitles->SetKeyBarTitle(fsShift, 6, FarPlugin->GetMsg(MOVE_TO_FILE_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsAltShift, 12,
    FarPlugin->GetMsg(OPEN_DIRCTORY_KEYBAR));
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::TWinSCPFileSystem(TCustomFarPlugin * APlugin) :
  TCustomFarFileSystem(APlugin)
{
  FReloadDirectory = false;
  FProgressSaveScreenHandle = 0;
  FSynchronizationSaveScreenHandle = 0;
  FFileList = NULL;
  FPanelItems = NULL;
  FSavedFindFolder = "";
  FDirectoryRead = false;
}
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::~TWinSCPFileSystem()
{
  if (FTerminal)
  {
    SaveSession();
  }
  assert(!FProgressSaveScreenHandle);
  assert(!FSynchronizationSaveScreenHandle);
  assert(!FFileList);
  assert(!FPanelItems);
  SAFE_DESTROY(FTerminal);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::HandleException(Exception * E, int OpMode)
{
  if ((Terminal != NULL) && E->InheritsFrom(__classid(EFatal)))
  {
    Terminal->DoShowExtendedException(E);
  }
  else
  {
    TCustomFarFileSystem::HandleException(E, OpMode);
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connected()
{
  return FTerminal != NULL;
}
//---------------------------------------------------------------------------
TWinSCPPlugin * __fastcall TWinSCPFileSystem::WinSCPPlugin()
{
  return dynamic_cast<TWinSCPPlugin*>(FPlugin);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::GetOpenPluginInfoEx(long unsigned & Flags,
  AnsiString & /*HostFile*/, AnsiString & CurDir, AnsiString & Format,
  AnsiString & PanelTitle, TFarPanelModes * PanelModes, int & /*StartPanelMode*/,
  int & /*StartSortMode*/, bool & /*StartSortOrder*/, TFarKeyBarTitles * KeyBarTitles,
  AnsiString & ShortcutData)
{
  if (Connected())
  {
    Flags = OPIF_USEFILTER | OPIF_USESORTGROUPS | OPIF_USEHIGHLIGHTING |
      OPIF_SHOWPRESERVECASE | OPIF_COMPAREFATTIME;

    // When slash is added to the end of path, windows style paths
    // (vandyke: c:/windows/system) are displayed correctly on command-line, but
    // leaved subdirectory is not focused, when entering parent directory.
    CurDir = FTerminal->CurrentDirectory;
    Format = FTerminal->SessionData->SessionName;
    if (FarConfiguration->HostNameInTitle)
    {
      PanelTitle = ::FORMAT(" %s:%s ", (Format, CurDir));
    }
    else
    {
      PanelTitle = ::FORMAT(" %s ", (CurDir));
    }
    ShortcutData = ::FORMAT("%s\1%s", (FTerminal->SessionData->Name, CurDir));

    TRemoteFilePanelItem::SetPanelModes(PanelModes);
    TRemoteFilePanelItem::SetKeyBarTitles(KeyBarTitles);
  }
  else
  {
    Format = "winscp";
    Flags = OPIF_USESORTGROUPS | OPIF_ADDDOTS | OPIF_SHOWNAMESONLY |
      OPIF_SHOWPRESERVECASE;
    PanelTitle = ::FORMAT(" %s ", (GetMsg(STORED_SESSION_TITLE)));

    TSessionPanelItem::SetPanelModes(PanelModes);
    TSessionPanelItem::SetKeyBarTitles(KeyBarTitles);
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::GetFindDataEx(TList * PanelItems)
{
  if (Connected())
  {
    FDirectoryRead = false;
    if (FReloadDirectory && FTerminal->Active)
    {
      FReloadDirectory = false;
      FTerminal->ReloadDirectory();
    }

    TRemoteFile * File;
    for (int Index = 0; Index < FTerminal->Files->Count; Index++)
    {
      File = FTerminal->Files->Files[Index];
      PanelItems->Add(new TRemoteFilePanelItem(File));
    }
  }
  else
  {
    assert(StoredSessions);
    StoredSessions->Load();
    if (StoredSessions->Count > 0)
    {
      TSessionData * Data;
      for (int Index = 0; Index < StoredSessions->Count; Index++)
      {
        Data = StoredSessions->Sessions[Index];
        PanelItems->Add(new TSessionPanelItem(Data));
      }
    }
    else
    {
      PanelItems->Add(new THintPanelItem(GetMsg(NEW_SESSION_HINT)));
    }
  }
  return true;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::DuplicateRenameSession(TSessionData * Data,
  bool Duplicate)
{
  assert(Data);
  AnsiString Name = Data->Name;
  if (FPlugin->InputBox(GetMsg(Duplicate ? DUPLICATE_SESSION_TITLE : RENAME_SESSION_TITLE),
        GetMsg(Duplicate ? DUPLICATE_SESSION_PROMPT : RENAME_SESSION_PROMPT),
        Name, 0) && !Name.IsEmpty() && (Name != Data->Name))
  {
    if (StoredSessions->FindByName(Name))
    {
      throw Exception(FORMAT(GetMsg(SESSION_ALREADY_EXISTS_ERROR), (Name)));
    }
    else
    {
      TSessionData * NData = StoredSessions->NewSession(Name, Data);

      if (!Duplicate)
      {
        Data->Remove();
        StoredSessions->Remove(Data);
      }

      StoredSessions->Save();

      UpdatePanel();
      RedrawPanel();

      PanelInfo->FocusedIndex = StoredSessions->IndexOf(NData) + 1;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::EditConnectSession(TSessionData * Data, bool Edit)
{
  TSessionData * OrigData = Data;
  bool NewData = !Data;
  bool FillInConnect = !Edit && !Data->CanLogin;

  if (NewData || FillInConnect)
  {
    Data = new TSessionData("");
  }

  try
  {
    if (FillInConnect)
    {
      Data->Assign(OrigData);
      Data->Name = "";
    }

    bool ToConnect = false;
    if (Edit || FillInConnect)
    {
      if (SessionDialog(Data, ToConnect, Edit))
      {
        int Select = -1;
        if ((!NewData && !FillInConnect) || !ToConnect)
        {
          if (NewData)
          {
            AnsiString Name = Data->SessionName;
            if (FPlugin->InputBox(GetMsg(NEW_SESSION_NAME_TITLE),
                  GetMsg(NEW_SESSION_NAME_PROMPT), Name, 0) && !Name.IsEmpty())
            {
              if (StoredSessions->FindByName(Name))
              {
                throw Exception(FORMAT(GetMsg(SESSION_ALREADY_EXISTS_ERROR), (Name)));
              }
              else
              {
                TSessionData * NData = StoredSessions->NewSession(Name, Data);
                Select = StoredSessions->IndexOf(NData);
              }
            }
          }
          else if (FillInConnect)
          {
            AnsiString OrigName = OrigData->Name;
            OrigData->Assign(Data);
            OrigData->Name = OrigName;
          }

          StoredSessions->Save();
          UpdatePanel();
          if (Select < 0)
          {
            RedrawPanel();
          }
          else
          {
            PanelInfo->FocusedIndex = Select + 1;
          }
        }
      }
    }
    else
    {
      ToConnect = true;
    }

    if (ToConnect && Connect(Data))
    {
      UpdatePanel();
      RedrawPanel();
      if (PanelInfo->ItemCount)
      {
        PanelInfo->FocusedIndex = 0;
      }
    }
  }
  __finally
  {
    if (NewData || FillInConnect)
    {
      delete Data;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessEventEx(int Event, void * Param)
{
  bool Result = false;
  if (Connected())
  {
    if (Event == FE_COMMAND)
    {
      AnsiString Command = (char *)Param;
      if (Command.SubString(1, 3).LowerCase() == "cd ")
      {
        Result = false;
      }
      else
      {
        Result = ExecuteCommand(Command);
      }
    }
    else if (Event == FE_IDLE)
    {
      FTerminal->Idle();
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalEchoAddLine(TObject * /*Sender*/,
  const AnsiString AddedLine)
{
  if (!AddedLine.IsEmpty() && (AddedLine[1] == '<' || AddedLine[1] == '!'))
  {
    int ReturnCode;
    AnsiString Line = AddedLine;
    Line.Delete(1, 2);
    if (!TSCPFileSystem::RemoveLastLine(Line, ReturnCode) ||
        !Line.IsEmpty())
    {
      FPlugin->WriteConsole(Line + "\n");
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::RequireCapability(int Capability)
{
  if (!FTerminal->IsCapable[static_cast<TFSCapability>(Capability)])
  {
    throw Exception(FORMAT(GetMsg(OPERATION_NOT_SUPPORTED),
      (FTerminal->ProtocolName)));
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExecuteCommand(const AnsiString Command)
{
  RequireCapability(fcAnyCommand);

  FTerminal->BeginTransaction();
  try
  {
    FarControl(FCTL_SETCMDLINE, NULL);
    TLogAddLineEvent FOldLogAddLine = FTerminal->Log->OnAddLine;
    FPlugin->ShowConsoleTitle(Command);
    try
    {
      FTerminal->Log->OnAddLine = TerminalEchoAddLine;

      FPlugin->ShowTerminalScreen();

      FTerminal->AnyCommand(Command);
    }
    __finally
    {
      FTerminal->Log->OnAddLine = FOldLogAddLine;
      FPlugin->ScrollTerminalScreen(1);
      FarControl(FCTL_SETUSERSCREEN, NULL);
      FPlugin->ClearConsoleTitle();
    }
  }
  __finally
  {
    if (FTerminal->Active)
    {
      FTerminal->EndTransaction();
      UpdatePanel();
    }
    else
    {
      RedrawPanel();
      RedrawPanel(true);
    }
  }
  return true;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessKeyEx(int Key, unsigned int ControlState)
{
  bool Handled = false;

  TFarPanelItem * Focused = PanelInfo->FocusedItem;
  
  if (!Connected())
  {
    TSessionData * Data = NULL;

    if ((Focused != NULL) && Focused->IsFile && Focused->UserData)
    {
      Data = (TSessionData *)Focused->UserData;
    }

    if ((Key == VK_RETURN && (ControlState == 0) && Data) ||
        (Key == VK_F4 && (ControlState == 0) &&
         (Data || StoredSessions->Count == 0)) ||
        (Key == VK_F4 && (ControlState & PKF_SHIFT)))
    {
      EditConnectSession(ControlState & PKF_SHIFT ? NULL : Data, Key == VK_F4);
      Handled = true;
    }

    if (Data && ((Key == VK_F5) || (Key == VK_F6)) &&
      (ControlState & PKF_SHIFT))
    {
      DuplicateRenameSession(Data, Key == VK_F5);
      Handled = true;
    }
  }
  else
  {
    if ((Key == 'F') && (ControlState & PKF_CONTROL))
    {
      InsertFileNameOnCommandLine();
      Handled = true;
    }

    if ((Key == 'R') && (ControlState & PKF_CONTROL))
    {
      FReloadDirectory = true;
    }

    if ((Key == 'A') && (ControlState & PKF_CONTROL))
    {
      FileProperties();
      Handled = true;
    }

    if ((Key == 'G') && (ControlState & PKF_CONTROL))
    {
      ApplyCommand();
      Handled = true;
    }

    if ((Key == VK_F6) && (ControlState & PKF_ALT))
    {
      CreateLink();
      Handled = true;
    }

    if (Focused && (Key == VK_F6) && (ControlState & PKF_SHIFT))
    {
      MoveFiles();
      Handled = true;
    }
    
    if ((Key == VK_F12) && (ControlState & PKF_SHIFT) &&
        (ControlState & PKF_ALT))
    {
      OpenDirectory(false);
      Handled = true;
    }
  }
  return Handled;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CreateLink()
{
  RequireCapability(fcResolveSymlink);
  RequireCapability(fcSymbolicLink);

  bool Edit = false;
  TRemoteFile * File = NULL;
  AnsiString FileName;
  AnsiString PointTo;
  bool SymbolicLink = true;

  if (PanelInfo->FocusedItem && PanelInfo->FocusedItem->UserData)
  {
    File = (TRemoteFile *)PanelInfo->FocusedItem->UserData;

    Edit = File->IsSymLink && Terminal->SessionData->ResolveSymlinks;
    if (Edit)
    {
      FileName = File->FileName;
      PointTo = File->LinkTo;
    }
    else
    {
      PointTo = File->FileName;
    }
  }

  if (LinkDialog(FileName, PointTo, SymbolicLink, Edit,
        Terminal->IsCapable[fcHardLink]))
  {
    if (Edit)
    {
      assert(File->FileName == FileName);
      bool Recursive = false;
      Terminal->ExceptionOnFail = true;
      try
      {
        Terminal->DeleteFile("", File, &Recursive);
      }
      __finally
      {
        Terminal->ExceptionOnFail = false;
      }
    }
    Terminal->CreateLink(FileName, PointTo, SymbolicLink);
    UpdatePanel();
    RedrawPanel();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ApplyCommand()
{
  RequireCapability(FTerminal->IsCapable[fcAnyCommand]);

  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList)
  {
    try
    {
      int Params = 0;
      AnsiString Command;
      if (ApplyCommandDialog(Command, Params))
      {
        Command = TCustomFileSystem::CompleteCustomCommand(
          Command, "", CustomCommandGetParamValue);

        try
        {
          FTerminal->CustomCommandOnFiles(Command, Params, FileList);
        }
        __finally
        {
          PanelInfo->ApplySelection();
          UpdatePanel();
          RedrawPanel();
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::Synchronize(bool Source)
{
  TFarPanelInfo * AnotherPanel = AnotherPanelInfo;
  if (AnotherPanel->IsPlugin || (AnotherPanel->Type != ptFile))
  {
    throw Exception(GetMsg(SYNCHRONIZE_LOCAL_PATH_REQUIRED));
  }

  AnsiString LocalDirectory = AnotherPanel->CurrentDirectory;
  AnsiString RemoteDirectory = FTerminal->CurrentDirectory;

  TTerminal::TSynchronizeMode Mode = Source ? TTerminal::smLocal : TTerminal::smRemote;
  int Params = GUIConfiguration->SynchronizeParams;
  bool SaveSettings = false;

  if (SynchronizeDialog(Mode, Params, LocalDirectory, RemoteDirectory, SaveSettings))
  {
    if (SaveSettings)
    {
      GUIConfiguration->SynchronizeParams = Params;
    }

    FPlugin->SaveScreen(FSynchronizationSaveScreenHandle);
    FPlugin->ShowConsoleTitle(GetMsg(SYNCHRONIZE_PROGRESS_TITLE));
    try
    {
      FSynchronizationStart = Now();
      FTerminal->Synchronize(LocalDirectory, RemoteDirectory,
        Mode, Params, TerminalSynchronizeDirectory);
    }
    __finally
    {
      FPlugin->ClearConsoleTitle();
      FPlugin->RestoreScreen(FSynchronizationSaveScreenHandle);
      UpdatePanel();
      RedrawPanel();
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalSynchronizeDirectory(
  const AnsiString LocalDirectory, const AnsiString RemoteDirectory, bool & Continue)
{
  static unsigned long LastTicks;
  unsigned long Ticks = GetTickCount();
  if ((LastTicks == 0) || (Ticks - LastTicks > 500))
  {
    LastTicks = Ticks;

    static const int ProgressWidth = 46;
    static AnsiString ProgressTitle;
    static AnsiString LocalLabel;
    static AnsiString RemoteLabel;
    static AnsiString StartTimeLabel;
    static AnsiString TimeElapsedLabel;

    if (ProgressTitle.IsEmpty())
    {
      ProgressTitle = GetMsg(SYNCHRONIZE_PROGRESS_TITLE);
      LocalLabel = GetMsg(SYNCHRONIZE_PROGRESS_LOCAL);
      RemoteLabel = GetMsg(SYNCHRONIZE_PROGRESS_REMOTE);
      StartTimeLabel = GetMsg(SYNCHRONIZE_PROGRESS_START_TIME);
      TimeElapsedLabel = GetMsg(SYNCHRONIZE_PROGRESS_ELAPSED);
    }

    AnsiString Message;

    Message = LocalLabel + MinimizeName(LocalDirectory,
      ProgressWidth - LocalLabel.Length(), false);
    Message += AnsiString::StringOfChar(' ', ProgressWidth - Message.Length()) + "\n";
    Message += RemoteLabel + MinimizeName(RemoteDirectory,
      ProgressWidth - RemoteLabel.Length(), true) + "\n";
    Message += StartTimeLabel + FSynchronizationStart.TimeString() + "\n";
    Message += TimeElapsedLabel +
      FormatDateTime(Configuration->TimeFormat, Now() - FSynchronizationStart) + "\n";

    FPlugin->Message(0, ProgressTitle, Message);

    if (FPlugin->CheckForEsc() &&
        (MoreMessageDialog(GetMsg(CANCEL_OPERATION), NULL,
          qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      Continue = false;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CustomCommandGetParamValue(
  const AnsiString AName, AnsiString & Value)
{
  AnsiString Name = AName;
  if (Name.IsEmpty())
  {
    Name = GetMsg(APPLY_COMMAND_PARAM_PROMPT);
  }
  if (!FPlugin->InputBox(GetMsg(APPLY_COMMAND_PARAM_TITLE),
        Name, Value, 0, APPLY_COMMAND_PARAM_HISTORY))
  {
    Abort();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::MoveFiles()
{
  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList)
  {
    assert(!FPanelItems);

    try
    {
      AnsiString Target = FTerminal->CurrentDirectory;
      AnsiString FileMask = "*.*";
      if (RemoteMoveDialog(FileList, Target, FileMask))
      {
        try
        {
          Terminal->MoveFiles(FileList, Target, FileMask);
        }
        __finally
        {
          PanelInfo->ApplySelection();
          UpdatePanel();
          RedrawPanel();
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::FileProperties()
{
  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList)
  {
    assert(!FPanelItems);

    try
    {
      TRemoteProperties CurrentProperties;

      CurrentProperties = TRemoteProperties::CommonProperties(FileList);

      int Flags = 0;
      if (FTerminal->IsCapable[fcModeChanging]) Flags |= cpMode;
      if (FTerminal->IsCapable[fcOwnerChanging]) Flags |= cpOwner;
      if (FTerminal->IsCapable[fcGroupChanging]) Flags |= cpGroup;

      TRemoteProperties NewProperties = CurrentProperties;
      if (PropertiesDialog(FileList, FTerminal->CurrentDirectory,
          FTerminal->Groups, FTerminal->Users, &NewProperties, Flags))
      {
        NewProperties = TRemoteProperties::ChangedProperties(CurrentProperties,
          NewProperties);
        try
        {
          FTerminal->ChangeFilesProperties(FileList, &NewProperties);
        }
        __finally
        {
          PanelInfo->ApplySelection();
          UpdatePanel();
          RedrawPanel();
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::InsertFileNameOnCommandLine()
{
  TFarPanelItem * Focused = PanelInfo->FocusedItem;

  if (Focused != NULL)
  {
    TRemoteFile * File = reinterpret_cast<TRemoteFile *>(Focused->UserData);
    if (File != NULL)
    {
      AnsiString FullName = File->FullFileName;
      if (FullName.Pos(" ") > 0)
      {
        FullName = FORMAT("\"%s\"", (FullName));
      }
      FullName += " ";

      FarControl(FCTL_INSERTCMDLINE, FullName.c_str());
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ShowInformation()
{
  FileSystemInfoDialog();
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::AreCachesEmpty()
{
  assert(Connected());
  return FTerminal->AreCachesEmpty;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ClearCaches()
{
  assert(Connected());
  FTerminal->ClearCaches();
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OpenSessionInPutty()
{
  assert(Connected());
  ::OpenSessionInPutty(FTerminal->SessionData);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OpenDirectory(bool Add)
{
  TBookmarkList * BookmarkList = new TBookmarkList();
  try
  {
    AnsiString Directory = FTerminal->CurrentDirectory;
    AnsiString SessionKey = FTerminal->SessionData->SessionKey;
    TBookmarkList * CurrentBookmarkList;

    CurrentBookmarkList = FarConfiguration->Bookmarks[SessionKey];
    if (CurrentBookmarkList != NULL)
    {
      BookmarkList->Assign(CurrentBookmarkList);
    }

    if (Add)
    {
      TBookmark * Bookmark = new TBookmark;
      Bookmark->Remote = Directory;
      Bookmark->Name = Directory;
      BookmarkList->Add(Bookmark);
      FarConfiguration->Bookmarks[SessionKey] = BookmarkList;
    }

    bool Result = OpenDirectoryDialog(Add, Directory, BookmarkList);

    FarConfiguration->Bookmarks[SessionKey] = BookmarkList;
    
    if (Result)
    {
      FTerminal->ChangeDirectory(Directory);
      UpdatePanel(true);
      RedrawPanel();
    }
  }
  __finally
  {
    delete BookmarkList;
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::SetDirectoryEx(const AnsiString Dir, int OpMode)
{
  assert(Connected());
  // workaround to ignore "change to root directory" command issued by FAR,
  // before file is opened for viewing/editing from "find file" dialog
  // when plugin uses UNIX style paths
  if (OpMode & OPM_FIND && OpMode & OPM_SILENT && Dir == "\\")
  {
    if (FSavedFindFolder.IsEmpty())
    {
      return true;
    }
    else
    {
      bool Result;
      try
      {
        Result = SetDirectoryEx(FSavedFindFolder, OpMode);
      }
      __finally
      {
        FSavedFindFolder = "";
      }
      return Result;
    }
  }
  else
  {
    if (OpMode & OPM_FIND && FSavedFindFolder.IsEmpty())
    {
      FSavedFindFolder = FTerminal->CurrentDirectory;
    }

    if ((OpMode & (OPM_FIND | OPM_SILENT)) == 0)
    {
      FPlugin->ShowConsoleTitle(GetMsg(CHANGING_DIRECTORY_TITLE));
    }
    FTerminal->ExceptionOnFail = true;
    try
    {
      if (Dir == "\\")
      {
        FTerminal->ChangeDirectory(ROOTDIRECTORY);
      }
      else if ((Dir == PARENTDIRECTORY) && (FTerminal->CurrentDirectory == ROOTDIRECTORY))
      {
        ClosePlugin();
      }
      else
      {
        FTerminal->ChangeDirectory(Dir);
      }
    }
    __finally
    {
      FTerminal->ExceptionOnFail = false;
      if ((OpMode & (OPM_FIND | OPM_SILENT)) == 0)
      {
        FPlugin->ClearConsoleTitle();
      }
    }
    return true;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::MakeDirectoryEx(AnsiString & Name, int OpMode)
{
  if (Connected())
  {
    assert(!(OpMode & OPM_SILENT) || !Name.IsEmpty());

    if ((OpMode & OPM_SILENT) ||
        FPlugin->InputBox(GetMsg(CREATE_FOLDER_TITLE), GetMsg(CREATE_FOLDER_PROMPT),
          Name, 0, "NewFolder"))
    {
      FTerminal->CreateDirectory(Name);
      return 1;
    }
    else
    {
      Name = "";
      return -1;
    }
  }
  else
  {
    Name = "";
    return -1;
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::DeleteFilesEx(TList * PanelItems, int OpMode)
{
  if (Connected())
  {
    AnsiString Query;
    if (PanelItems->Count > 1)
    {
      Query = FORMAT(GetMsg(DELETE_FILES_CONFIRM), (PanelItems->Count));
    }
    else
    {
      Query = FORMAT(GetMsg(DELETE_FILE_CONFIRM),
        (((TFarPanelItem *)PanelItems->Items[0])->FileName));
    }

    if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
      (MoreMessageDialog(Query, NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      FFileList = CreateFileList(PanelItems, osRemote);
      try
      {
        FPanelItems = PanelItems;
        FTerminal->DeleteFiles(FFileList);
      }
      __finally
      {
        FPanelItems = NULL;
        SAFE_DESTROY(FFileList);
      }
    }
    return true;
  }
  else
  {
    if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
      (MoreMessageDialog(GetMsg(DELETE_SESSIONS_CONFIRM), NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      for (int i = 0; i < PanelItems->Count; i++)
      {
        TFarPanelItem * PanelItem = (TFarPanelItem *)PanelItems->Items[i];
        assert(PanelItem);
        if (PanelItem->IsFile)
        {
          TSessionData * Data = (TSessionData *)PanelItem->UserData;
          assert(Data);
          Data->Remove();
          StoredSessions->Remove(Data);
          PanelItem->Selected = false;
        }
      }
    }
    return true;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::GetFilesEx(TList * PanelItems, bool Move,
  AnsiString & DestPath, int OpMode)
{
  assert(!FDirectoryRead);
  int Result;
  if (Connected())
  {
    if (OpMode & OPM_FIND)
    {
      throw Exception(GetMsg(VIEW_FROM_FIND_NOT_SUPPORTED));
    }

    FFileList = CreateFileList(PanelItems, osRemote);
    try
    {
      bool EditView = (OpMode & (OPM_EDIT | OPM_VIEW)) != 0;
      bool Confirmed =
        (OpMode & OPM_SILENT) &&
        (!EditView || FarConfiguration->EditorDownloadDefaultMode);
          
      TCopyParamType CopyParam = Configuration->CopyParam;

      if (!Confirmed)
      {
        Confirmed = CopyDialog(false, Move, FFileList,
          FTerminal->IsCapable[fcTextMode], DestPath, &CopyParam, EditView);
      }

      if (Confirmed)
      {
        if ((FFileList->Count == 1) && (OpMode & OPM_EDIT))
        {
          FLastEditFolder = DestPath;
          FLastEditFile = FFileList->Strings[0];
          FLastEditCopyParam = CopyParam;
        }
        else
        {
          FLastEditFolder = "";
          FLastEditFile = "";
        }

        FPanelItems = PanelItems;
        int Params = (Move ? cpDelete : 0) | (EditView ? cpTemporary : 0);
        FTerminal->CopyToLocal(FFileList, DestPath, &CopyParam, Params);
        Result = 1;
      }
      else
      {
        Result = -1;
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else
  {
    if (StoredSessions->Count > 0 &&
        ExportSessions(PanelItems, Move, DestPath, OpMode))
    {
      Result = 1;
    }
    else
    {
      Result = -1;
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExportSessions(TList * PanelItems, bool /*Move*/,
  AnsiString & DestPath, int OpMode)
{
  AnsiString Title = GetMsg(EXPORT_SESSION_TITLE);
  AnsiString Prompt;
  if (PanelItems->Count == 1)
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSION_PROMPT),
      (((TFarPanelItem *)PanelItems->Items[0])->FileName));
  }
  else
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSIONS_PROMPT), (PanelItems->Count));
  }

  bool Result = (OpMode & OPM_SILENT) ||
    FPlugin->InputBox(Title, Prompt, DestPath, 0, "Copy");

  if (Result)
  {
    TSessionData * Data;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      Data = (TSessionData *)((TFarPanelItem *)PanelItems->Items[i])->UserData;
      THierarchicalStorage * Storage = NULL;
      TSessionData * ExportData = NULL;
      try
      {
        ExportData = new TSessionData(Data->Name);
        ExportData->Assign(Data);
        ExportData->Modified = true;
        Storage = new TIniFileStorage(IncludeTrailingBackslash(DestPath) + ExportData->Name + ".ini");;
        if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, true))
        {
          ExportData->Save(Storage);
        }
      }
      __finally
      {
        delete Storage;
        delete ExportData;
      }

    }
  }
  return Result;
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::PutFilesEx(TList * PanelItems, bool Move, int OpMode)
{
  int Result = 1;
  if (Connected())
  {
    FFileList = CreateFileList(PanelItems, osLocal);
    try
    {
      bool Confirmed = (OpMode & OPM_SILENT);
      bool Ask = !Confirmed;

      bool Edit = !Confirmed && (GetCurrentDir() == FLastEditFolder) &&
        (FFileList->Count == 1) && (FFileList->Strings[0] == FLastEditFile);

      TCopyParamType CopyParam;

      // huerictics: do not ask for target directory when uploaded file
      // was downloaded in edit mode
      if (Edit)
      {
        FLastEditFolder = "";
        FLastEditFile = "";
        CopyParam = FLastEditCopyParam;
        Confirmed = FarConfiguration->EditorUploadSameOptions;
        Ask = false;
      }
      else
      {
        CopyParam = Configuration->CopyParam;
      }

      AnsiString DestPath = FTerminal->CurrentDirectory;

      if (!Confirmed)
      {
        Confirmed = CopyDialog(true, Move, FFileList,
          FTerminal->IsCapable[fcTextMode], DestPath, &CopyParam, Edit);
      }

      if (Confirmed)
      {
        FPanelItems = PanelItems;
        int Params = (Move ? cpDelete : 0) | (Ask ? 0 : cpNoConfirmation);
        FTerminal->CopyToRemote(FFileList, DestPath, &CopyParam, Params);
      }
      else
      {
        Result = -1;
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else
  {
    if (!ImportSessions(PanelItems, Move, OpMode))
    {
      return -1;
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ImportSessions(TList * PanelItems, bool /*Move*/,
  int OpMode)
{
  bool Result = (OpMode & OPM_SILENT) ||
    (MoreMessageDialog(GetMsg(IMPORT_SESSIONS_PROMPT), NULL,
      qtConfirmation, qaOK | qaCancel) == qaOK);

  if (Result)
  {
    AnsiString FileName;
    TFarPanelItem * PanelItem;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[i];
      bool AnyData = false;
      FileName = PanelItem->FileName;
      if (PanelItem->IsFile)
      {
        THierarchicalStorage * Storage = NULL;
        try
        {
          Storage = new TIniFileStorage(IncludeTrailingBackslash(GetCurrentDir()) + FileName);
          if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, false) &&
              Storage->HasSubKeys())
          {
            AnyData = true;
            StoredSessions->Load(Storage, true);
            StoredSessions->Save();
          }
        }
        __finally
        {
          delete Storage;
        }
      }
      if (!AnyData)
      {
        throw Exception(FORMAT(GetMsg(IMPORT_SESSIONS_EMPTY), (FileName)));
      }
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateSelectedFileList(
  TOperationSide Side)
{
  TStrings * Result;
  if (PanelInfo->SelectedCount > 0)
  {
    Result = CreateFileList(PanelInfo->Items, Side, true);
  }
  else
  {
    TFarPanelItem * PanelItem = PanelInfo->FocusedItem;
    if (PanelItem->IsParentDirectory)
    {
      Result = NULL;
    }
    else
    {
      Result = new TStringList();
      assert(PanelItem->UserData);
      Result->AddObject(PanelItem->FileName, (TObject *)PanelItem->UserData);
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateFileList(TList * PanelItems,
  TOperationSide Side, bool SelectedOnly)
{
  TStrings * FileList = new TStringList();
  try
  {
    AnsiString Name;
    TFarPanelItem * PanelItem;
    TObject * Data = NULL;
    for (int Index = 0; Index < PanelItems->Count; Index++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
      assert(PanelItem);
      if (!SelectedOnly || PanelItem->Selected)
      {
        if (Side == osRemote)
        {
          Data = (TRemoteFile *)PanelItem->UserData;
          assert(Data);
        }
        FileList->AddObject(PanelItem->FileName, Data);
      }
    }
  }
  catch (...)
  {
    delete FileList;
    throw;
  }
  return FileList;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::SaveSession()
{
  if (!FTerminal->SessionData->Name.IsEmpty())
  {
    FTerminal->SessionData->RemoteDirectory = FTerminal->CurrentDirectory;

    TSessionData * Data;
    Data = (TSessionData *)StoredSessions->FindByName(FTerminal->SessionData->Name);
    if (Data)
    {
      bool Changed = false;
      if (Terminal->SessionData->UpdateDirectories)
      {
        Data->RemoteDirectory = Terminal->SessionData->RemoteDirectory;
        Changed = true;
      }

      if (Changed)
      {
        StoredSessions->Save();
      }
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connect(TSessionData * Data)
{
  bool Result = false;
  assert(!FTerminal);
  FTerminal = new TTerminal();
  try
  {
    FTerminal->Configuration = Configuration;
    FTerminal->SessionData = Data;
    FTerminal->OnQueryUser = TerminalQueryUser;
    FTerminal->OnPromptUser = TerminalPromptUser;
    FTerminal->OnShowExtendedException = TerminalShowExtendedException;
    FTerminal->OnUpdateStatus = TerminalUpdateStatus;
    FTerminal->OnReadDirectory = TerminalReadDirectory;
    FTerminal->OnStartReadDirectory = TerminalStartReadDirectory;
    FTerminal->OnFinished = OperationFinished;
    FTerminal->OnProgress = OperationProgress;
    try
    {
      FTerminal->Open();
      FTerminal->DoStartup();
    }
    __finally
    {
      FPlugin->ClearProgress();
    }

    FTerminal->OnClose = TerminalClose;

    Result = true;
  }
  catch(Exception &E)
  {
    FTerminal->DoShowExtendedException(&E);
    SAFE_DESTROY(FTerminal);
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalClose(TObject * /*Sender*/)
{
  if (!FClosed)
  {
    ClosePlugin();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalUpdateStatus(TObject * /*Sender*/)
{
  static const int ConnectionStatusStrings[] =
    { STATUS_CLOSED, STATUS_INITWINSOCK, STATUS_LOOKUPHOST, STATUS_CONNECT,
      STATUS_AUTHENTICATE, STATUS_AUTHENTICATED, STATUS_STARTUP,
      STATUS_OPEN_DIRECTORY, STATUS_READY };

  assert((FTerminal->Status >= 0) &&
    (FTerminal->Status < LENOF(ConnectionStatusStrings)));
  FPlugin->ShowProgress(GetMsg((int)ConnectionStatusStrings[FTerminal->Status]));
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalStartReadDirectory(TObject * /*Sender*/)
{
  FPlugin->ShowConsoleTitle(GetMsg(READING_DIRECTORY_TITLE));
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalReadDirectory(TObject * /*Sender*/,
  bool /*ReloadOnly*/)
{
  FPlugin->ClearConsoleTitle();
  FDirectoryRead = true;
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::MoreMessageDialog(AnsiString Str,
  TStrings * MoreMessages, TQueryType Type, int Answers, unsigned int Flags)
{
  if ((FProgressSaveScreenHandle != 0) ||
      (FSynchronizationSaveScreenHandle != 0))
  {
    Flags |= FMSG_WARNING;
  }

  return WinSCPPlugin()->MoreMessageDialog(Str, MoreMessages, Type,
    Answers, Flags);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalQueryUser(TObject * /*Sender*/,
  const AnsiString Query, TStrings * MoreMessages, int Answers,
  int Params, int & Answer, TQueryType Type)
{
  AnsiString AQuery = Query;
  if (Params & qpFatalAbort)
  {
    AQuery = FORMAT(GetMsg(WARN_FATAL_ERROR), (AQuery));
  }

  Answer = MoreMessageDialog(AQuery, MoreMessages, Type, Answers);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalPromptUser(TSecureShell * /*SecureShell*/,
  AnsiString Prompt, TPromptKind Kind, AnsiString & Response, bool & Result)
{
  Result = PasswordDialog(Prompt, Kind, Response);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalShowExtendedException(
  TSecureShell * /*SecureShell*/, Exception * E)
{
  WinSCPPlugin()->ShowExtendedException(E);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationProgress(
  TFileOperationProgressType & ProgressData, TCancelStatus & /*Cancel*/)
{
  bool First = false;
  if (ProgressData.InProgress && !FProgressSaveScreenHandle)
  {
    FPlugin->SaveScreen(FProgressSaveScreenHandle);
    First = true;
  }

  // operation is finished (or terminated), so we hide progress form
  if (!ProgressData.InProgress && FProgressSaveScreenHandle)
  {
    FPlugin->RestoreScreen(FProgressSaveScreenHandle);
    FPlugin->ClearConsoleTitle();
  }
  else
  {
    ShowOperationProgress(ProgressData, First);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationFinished(TFileOperation Operation,
  TOperationSide /*Side*/, bool /*DragDrop*/, const AnsiString FileName, bool Success,
  bool & /*DisconnectWhenComplete*/)
{
  if ((Operation != foCalculateSize) &&
      (FSynchronizationSaveScreenHandle == 0))
  {
    TFarPanelItem * PanelItem = NULL;;

    if (!FPanelItems)
    {
      TList * PanelItems = PanelInfo->Items;
      for (int Index = 0; Index < PanelItems->Count; Index++)
      {
        if (((TFarPanelItem *)PanelItems->Items[Index])->FileName == FileName)
        {
          PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
          break;
        }
      }
    }
    else
    {
      assert(FFileList);
      assert(FPanelItems->Count == FFileList->Count);
      int Index = FFileList->IndexOf(FileName);
      assert(Index >= 0);
      PanelItem = (TFarPanelItem *)FPanelItems->Items[Index];
    }

    assert(PanelItem->FileName == FileName);
    if (Success)
    {
      PanelItem->Selected = false;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ShowOperationProgress(
  TFileOperationProgressType & ProgressData, bool First)
{
  static unsigned long LastTicks;
  unsigned long Ticks = GetTickCount();
  if (Ticks - LastTicks > 500 || First)
  {
    LastTicks = Ticks;

    static const int ProgressWidth = 46;
    static const int Captions[] = {PROGRESS_COPY, PROGRESS_MOVE, PROGRESS_DELETE,
      PROGRESS_SETPROPERTIES, 0, 0, PROGRESS_CALCULATE_SIZE,
      PROGRESS_REMOTE_MOVE };
    static AnsiString ProgressFileLabel;
    static AnsiString TargetDirLabel;
    static AnsiString TransferModeLabel;
    static AnsiString ResumeSupportLabel;
    static AnsiString StartTimeLabel;
    static AnsiString TimeElapsedLabel;
    static AnsiString BytesTransferedLabel;
    static AnsiString CPSLabel;
    static AnsiString TimeEstimatedLabel;

    if (ProgressFileLabel.IsEmpty())
    {
      ProgressFileLabel = GetMsg(PROGRESS_FILE_LABEL);
      TargetDirLabel = GetMsg(TARGET_DIR_LABEL);
      TransferModeLabel = GetMsg(TRANSFER_MODE_LABEL);
      ResumeSupportLabel = GetMsg(RESUME_SUPPORT_LABEL);
      StartTimeLabel = GetMsg(START_TIME_LABEL);
      TimeElapsedLabel = GetMsg(TIME_ELAPSED_LABEL);
      BytesTransferedLabel = GetMsg(BYTES_TRANSFERED_LABEL);
      CPSLabel = GetMsg(CPS_LABEL);
      TimeEstimatedLabel = GetMsg(TIME_ESTIMATED_LABEL);
    }

    int Progress;

    bool TransferOperation =
      ((ProgressData.Operation == foCopy) || (ProgressData.Operation == foMove));

    AnsiString Message;
    AnsiString Title = GetMsg(Captions[(int)ProgressData.Operation - 1]);
    AnsiString ProgressBar;
    Progress = ProgressData.OverallProgress();
    ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * Progress / 100);
    ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
    Message = ProgressFileLabel + MinimizeName(ProgressData.FileName,
      ProgressWidth - ProgressFileLabel.Length(), ProgressData.Side == osRemote) + "\n";
    if (TransferOperation && !ProgressData.DragDrop)
    {
      Message += TargetDirLabel + MinimizeName(ProgressData.Directory,
        ProgressWidth - TargetDirLabel.Length(), ProgressData.Side == osLocal) + "\n";
    }
    Message += ProgressBar + "\n";
    if (TransferOperation)
    {
      Message += "\1\n";
      AnsiString StatusLine;
      AnsiString Value;
      Value = GetMsg(ProgressData.AsciiTransfer ? TRANSFER_ASCII : TRANSFER_BINARY);
      StatusLine = TransferModeLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - TransferModeLabel.Length() - Value.Length()) +
        Value + "  ";
      switch (ProgressData.ResumeStatus) {
        case rsEnabled: Value = GetMsg(RESUME_ENABLED); break;
        case rsDisabled: Value = GetMsg(RESUME_DISABLED); break;
        case rsNotAvailable: Value = GetMsg(RESUME_NOT_AVAILABLE); break;
        default: assert(false);
      }
      StatusLine = StatusLine + ResumeSupportLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        ResumeSupportLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = FormatDateTime(Configuration->TimeFormat, ProgressData.TimeElapsed());
      StatusLine = TimeElapsedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - TimeElapsedLabel.Length() - Value.Length()) +
        Value + "  ";

      AnsiString LabelText;
      if (ProgressData.TotalSizeSet)
      {
        Value = FormatDateTime(Configuration->TimeFormat, ProgressData.TotalTimeExpected());
        LabelText = TimeEstimatedLabel;
      }
      else
      {
        Value = ProgressData.StartTime.TimeString();
        LabelText = StartTimeLabel;
      }
      StatusLine = StatusLine + LabelText +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        LabelText.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = FormatBytes(ProgressData.TotalTransfered);
      StatusLine = BytesTransferedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - BytesTransferedLabel.Length() - Value.Length()) +
        Value + "  ";
      Value = FORMAT("%s/s", (FormatBytes(ProgressData.CPS())));
      StatusLine = StatusLine + CPSLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        CPSLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";
      int TransferProgress = ProgressData.TransferProgress();
      ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * TransferProgress / 100);
      ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
      Message += ProgressBar + "\n";
    }
    FPlugin->Message(0, Title, Message);

    if (First)
    {
      FPlugin->ShowConsoleTitle(Title);
    }
    FPlugin->UpdateConsoleTitleProgress(ProgressData.OverallProgress());

    if (FPlugin->CheckForEsc())
    {
      CancelConfiguration(ProgressData);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CancelConfiguration(TFileOperationProgressType & ProgressData)
{
  if (!ProgressData.Suspended)
  {
    ProgressData.Suspend();
    try
    {
      TCancelStatus ACancel;
      int Result;
      if (ProgressData.TransferingFile &&
          (ProgressData.TimeExpected() > GUIConfiguration->IgnoreCancelBeforeFinish))
      {
        Result = MoreMessageDialog(GetMsg(CANCEL_OPERATION_FATAL), NULL,
          qtWarning, qaYes | qaNo | qaCancel);
      }
      else
      {
        Result = MoreMessageDialog(GetMsg(CANCEL_OPERATION), NULL,
          qtConfirmation, qaOK | qaCancel);
      }
      switch (Result) {
        case qaYes:
          ACancel = csCancelTransfer; break;
        case qaOK:
        case qaNo:
          ACancel = csCancel; break;
        default:
          ACancel = csContinue; break;
      }

      if (ACancel > ProgressData.Cancel)
      {
        ProgressData.Cancel = ACancel;
      }
    }
    __finally
    {
      ProgressData.Resume();
    }
  }
}
//---------------------------------------------------------------------
AnsiString __fastcall TWinSCPFileSystem::FormatBytes(__int64 Bytes)
{
  static const __int64 FormatBytesAbove = __int64(100*1024);
  AnsiString Result;

  if (Bytes < FormatBytesAbove)
  {
    return FormatFloat("#,##0 \"B\"", Bytes);
  }
  else if (Bytes < __int64(100*1024*1024))
  {
    return FormatFloat("#,##0 \"KB\"", Bytes / 1024);
  }
  else
  {
    return FormatFloat("#,##0 \"MB\"", Bytes / (1024*1024));
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CutFirstDirectory(AnsiString & S, bool Unix)
{
  bool Root;
  int P;
  AnsiString Sep = Unix ? "/" : "\\";
  if (S == Sep)
  {
    S = "";
  }
  else
  {
    if (S[1] == Sep[1])
    {
      Root = true;
      S.Delete(1, 1);
    }
    else
    {
      Root = false;
    }
    if (S[1] == '.')
    {
      S.Delete(1, 4);
    }
    P = S.AnsiPos(Sep[1]);
    if (P)
    {
      S.Delete(1, P);
      S = "..." + Sep + S;
    }
    else
    {
      S = "";
    }
    if (Root)
    {
      S = Sep + S;
    }
  }
}
//---------------------------------------------------------------------------
AnsiString __fastcall TWinSCPFileSystem::MinimizeName(const AnsiString FileName,
  int MaxLen, bool Unix)
{
  AnsiString Drive, Dir, Name, Result;
  AnsiString Sep = Unix ? "/" : "\\";

  Result = FileName;
  if (Unix)
  {
    int P = Result.LastDelimiter("/");
    if (P)
    {
      Dir = Result.SubString(1, P);
      Name = Result.SubString(P + 1, Result.Length() - P);
    }
    else
    {
      Dir = "";
      Name = Result;
    }
  }
  else
  {
    Dir = ExtractFilePath(Result);
    Name = ExtractFileName(Result);

    if (Dir.Length() >= 2 && Dir[2] == ':')
    {
      Drive = Dir.SubString(1, 2);
      Dir.Delete(1, 2);
    }
  }

  while ((!Dir.IsEmpty() || !Drive.IsEmpty()) && (Result.Length() > MaxLen))
  {
    if (Dir == Sep + "..." + Sep)
    {
      Dir = "..." + Sep;
    }
    else if (Dir == "")
    {
      Drive = "";
    }
    else
    {
      CutFirstDirectory(Dir, Unix);
    }
    Result = Drive + Dir + Name;
  }

  if (Result.Length() > MaxLen)
  {
    Result = Result.SubString(1, MaxLen);
  }
  return Result;
}
