//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include <StrUtils.hpp>
#include "WinSCPFileSystem.h"
#include "WinSCPPlugin.h"
#include "FarTexts.h"
#include "FarConfiguration.h"
#include "farkeys.hpp"
#include <Common.h>
#include <SessionData.h>
#include <ScpMain.h>
#include <SysUtils.hpp>
#include <ScpFileSystem.h>
#include <Queue.h>
#include <Bookmarks.h>
#include <GUITools.h>
#include <CompThread.hpp>
// FAR WORKAROUND
//---------------------------------------------------------------------------
#pragma package(smart_init)
//---------------------------------------------------------------------------
__fastcall TSessionPanelItem::TSessionPanelItem(TSessionData * ASessionData):
  TCustomFarPanelItem()
{
  assert(ASessionData);
  FSessionData = ASessionData;
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    ColumnTitles->Add(FarPlugin->GetMsg(SESSION_NAME_COL_TITLE));
    for (int Index = 0; Index < PANEL_MODES_COUNT; Index++)
    {
      PanelModes->SetPanelMode(Index, "N", "0", ColumnTitles, false, false, false);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsNone, 6, 7);
  KeyBarTitles->SetKeyBarTitle(fsNone, 5, FarPlugin->GetMsg(EXPORT_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
  KeyBarTitles->SetKeyBarTitle(fsShift, 4, FarPlugin->GetMsg(NEW_SESSION_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsShift, 5, FarPlugin->GetMsg(COPY_SESSION_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsShift, 6, FarPlugin->GetMsg(RENAME_SESSION_KEYBAR));
  KeyBarTitles->ClearKeyBarTitle(fsShift, 7, 8);
  KeyBarTitles->ClearKeyBarTitle(fsAlt, 6);
  KeyBarTitles->ClearKeyBarTitle(fsCtrl, 4, 11);
}
//---------------------------------------------------------------------------
void __fastcall TSessionPanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & /*Size*/,
  unsigned long & /*FileAttributes*/,
  TDateTime & /*LastWriteTime*/, TDateTime & /*LastAccess*/,
  unsigned long & /*NumberOfLinks*/, AnsiString & /*Description*/,
  AnsiString & /*Owner*/, void *& UserData, int & /*CustomColumnNumber*/)
{
  FileName = FSessionData->Name;
  UserData = FSessionData;
}
//---------------------------------------------------------------------------
__fastcall TRemoteFilePanelItem::TRemoteFilePanelItem(TRemoteFile * ARemoteFile):
  TCustomFarPanelItem()
{
  assert(ARemoteFile);
  FRemoteFile = ARemoteFile;
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::GetData(
  unsigned long & /*Flags*/, AnsiString & FileName, __int64 & Size,
  unsigned long & FileAttributes,
  TDateTime & LastWriteTime, TDateTime & LastAccess,
  unsigned long & /*NumberOfLinks*/, AnsiString & /*Description*/,
  AnsiString & Owner, void *& UserData, int & CustomColumnNumber)
{
  FileName = FRemoteFile->FileName;
  Size = FRemoteFile->Size;
  FileAttributes =
    FLAGMASK(FRemoteFile->IsDirectory, FILE_ATTRIBUTE_DIRECTORY) |
    FLAGMASK(FRemoteFile->IsHidden, FILE_ATTRIBUTE_HIDDEN) |
    FLAGMASK(FRemoteFile->Rights->ReadOnly, FILE_ATTRIBUTE_READONLY) |
    FLAGMASK(FRemoteFile->IsSymLink, FILE_ATTRIBUTE_REPARSE_POINT);
  LastWriteTime = FRemoteFile->Modification;
  LastAccess = FRemoteFile->LastAccess;
  Owner = FRemoteFile->Owner;
  UserData = FRemoteFile;
  CustomColumnNumber = 4;
}
//---------------------------------------------------------------------------
AnsiString __fastcall TRemoteFilePanelItem::CustomColumnData(int Column)
{
  switch (Column) {
    case 0: return FRemoteFile->Group;
    case 1: return FRemoteFile->RightsStr;
    case 2: return FRemoteFile->Rights->Octal;
    case 3: return FRemoteFile->LinkTo;
    default: assert(false); return AnsiString();
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::TranslateColumnTypes(AnsiString & ColumnTypes,
  TStrings * ColumnTitles)
{
  AnsiString AColumnTypes = ColumnTypes;
  ColumnTypes = "";
  AnsiString Column;
  AnsiString Title;
  while (!AColumnTypes.IsEmpty())
  {
    Column = CutToChar(AColumnTypes, ',', false);
    if (Column == "G")
    {
      Column = "C0";
      Title = FarPlugin->GetMsg(GROUP_COL_TITLE);
    }
    else if (Column == "R")
    {
      Column = "C1";
      Title = FarPlugin->GetMsg(RIGHTS_COL_TITLE);
    }
    else if (Column == "RO")
    {
      Column = "C2";
      Title = FarPlugin->GetMsg(RIGHTS_OCTAL_COL_TITLE);
    }
    else if (Column == "L")
    {
      Column = "C3";
      Title = FarPlugin->GetMsg(LINK_TO_COL_TITLE);
    }
    else
    {
      Title = "";
    }
    ColumnTypes += (ColumnTypes.IsEmpty() ? "" : ",") + Column;
    if (ColumnTitles)
    {
      ColumnTitles->Add(Title);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetPanelModes(TFarPanelModes * PanelModes)
{
  assert(FarPlugin);
  TStrings * ColumnTitles = new TStringList();
  try
  {
    if (FarConfiguration->CustomPanelModeDetailed)
    {
      AnsiString ColumnTypes = FarConfiguration->ColumnTypesDetailed;
      AnsiString StatusColumnTypes = FarConfiguration->StatusColumnTypesDetailed;

      TranslateColumnTypes(ColumnTypes, ColumnTitles);
      TranslateColumnTypes(StatusColumnTypes, NULL);

      PanelModes->SetPanelMode(5 /*detailed */,
        ColumnTypes, FarConfiguration->ColumnWidthsDetailed,
        ColumnTitles, FarConfiguration->FullScreenDetailed, false, true, false,
        StatusColumnTypes, FarConfiguration->StatusColumnWidthsDetailed);
    }
  }
  __finally
  {
    delete ColumnTitles;
  }
}
//---------------------------------------------------------------------------
void __fastcall TRemoteFilePanelItem::SetKeyBarTitles(TFarKeyBarTitles * KeyBarTitles)
{
  KeyBarTitles->ClearKeyBarTitle(fsShift, 1, 3);
  KeyBarTitles->SetKeyBarTitle(fsShift, 5, FarPlugin->GetMsg(COPY_TO_FILE_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsShift, 6, FarPlugin->GetMsg(MOVE_TO_FILE_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsAltShift, 12,
    FarPlugin->GetMsg(OPEN_DIRECTORY_KEYBAR));
  KeyBarTitles->SetKeyBarTitle(fsAltShift, 6,
    FarPlugin->GetMsg(RENAME_FILE_KEYBAR));
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
class TFarInteractiveCustomCommand : public TInteractiveCustomCommand
{
public:
  TFarInteractiveCustomCommand(TCustomFarPlugin * Plugin,
    TCustomCommand * ChildCustomCommand);

protected:
  virtual void __fastcall Prompt(int Index, const AnsiString & Prompt,
    AnsiString & Value);

private:
  TCustomFarPlugin * FPlugin;
};
//---------------------------------------------------------------------------
TFarInteractiveCustomCommand::TFarInteractiveCustomCommand(
  TCustomFarPlugin * Plugin, TCustomCommand * ChildCustomCommand) :
  TInteractiveCustomCommand(ChildCustomCommand)
{
  FPlugin = Plugin;
}
//---------------------------------------------------------------------------
void __fastcall TFarInteractiveCustomCommand::Prompt(int /*Index*/,
  const AnsiString & Prompt, AnsiString & Value)
{
  AnsiString APrompt = Prompt;
  if (APrompt.IsEmpty())
  {
    APrompt = FPlugin->GetMsg(APPLY_COMMAND_PARAM_PROMPT);
  }
  if (!FPlugin->InputBox(FPlugin->GetMsg(APPLY_COMMAND_PARAM_TITLE),
        APrompt, Value, 0, APPLY_COMMAND_PARAM_HISTORY))
  {
    Abort();
  }
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Attempt to allow keepalives from background thread.
// Not finished nor used.
class TKeepaliveThread : public TCompThread
{
public:
  __fastcall TKeepaliveThread(TWinSCPFileSystem * FileSystem, TDateTime Interval);
  virtual void __fastcall Execute();
  virtual void __fastcall Terminate();

private:
  TWinSCPFileSystem * FFileSystem;
  TDateTime FInterval;
  HANDLE FEvent;
};
//---------------------------------------------------------------------------
__fastcall TKeepaliveThread::TKeepaliveThread(TWinSCPFileSystem * FileSystem,
  TDateTime Interval) :
  TCompThread(true)
{
  FEvent = CreateEvent(NULL, false, false, NULL);

  FFileSystem = FileSystem;
  FInterval = Interval;
  Resume();
}
//---------------------------------------------------------------------------
void __fastcall TKeepaliveThread::Terminate()
{
  TCompThread::Terminate();
  SetEvent(FEvent);
}
//---------------------------------------------------------------------------
void __fastcall TKeepaliveThread::Execute()
{
  while (!Terminated)
  {
    static long MillisecondsPerDay = 24 * 60 * 60 * 1000;
    if ((WaitForSingleObject(FEvent, double(FInterval) * MillisecondsPerDay) != WAIT_FAILED) &&
        !Terminated)
    {
      FFileSystem->KeepaliveThreadCallback();
    }
  }
  CloseHandle(FEvent);
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::TWinSCPFileSystem(TCustomFarPlugin * APlugin) :
  TCustomFarFileSystem(APlugin)
{
  FReloadDirectory = false;
  FProgressSaveScreenHandle = 0;
  FSynchronizationSaveScreenHandle = 0;
  FAuthenticationSaveScreenHandle = 0;
  FFileList = NULL;
  FPanelItems = NULL;
  FSavedFindFolder = "";
  FTerminal = NULL;
  FQueue = NULL;
  FQueueStatus = NULL;
  FQueueStatusSection = new TCriticalSection();
  FQueueStatusInvalidated = false;
  FQueueItemInvalidated = false;
  FRefreshLocalDirectory = false;
  FRefreshRemoteDirectory = false;
  FNoProgress = false;
  FNoProgressFinish = false;
  FKeepaliveThread = NULL;
  FSynchronisingBrowse = false;
  FSynchronizeController = NULL;
  FCapturedLog = NULL;
  FAuthenticationLog = NULL;
}
//---------------------------------------------------------------------------
__fastcall TWinSCPFileSystem::~TWinSCPFileSystem()
{
  if (FTerminal)
  {
    SaveSession();
  }
  assert(FSynchronizeController == NULL);
  assert(!FAuthenticationSaveScreenHandle);
  assert(!FProgressSaveScreenHandle);
  assert(!FSynchronizationSaveScreenHandle);
  assert(!FFileList);
  assert(!FPanelItems);
  delete FQueue;
  FQueue = NULL;
  delete FQueueStatus;
  FQueueStatus = NULL;
  delete FQueueStatusSection;
  FQueueStatusSection = NULL;
  if (FTerminal != NULL)
  {
    GUIConfiguration->SynchronizeBrowsing = FSynchronisingBrowse;
  }
  SAFE_DESTROY(FTerminal);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::HandleException(Exception * E, int OpMode)
{
  if ((Terminal != NULL) && E->InheritsFrom(__classid(EFatal)))
  {
    if (!FClosed)
    {
      ClosePlugin();
    }
    Terminal->DoShowExtendedException(E);
  }
  else
  {
    TCustomFarFileSystem::HandleException(E, OpMode);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::KeepaliveThreadCallback()
{
  TGuard Guard(FCriticalSection);

  if (Connected())
  {
    FTerminal->Idle();
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::SessionList()
{
  return (FTerminal == NULL);
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connected()
{
  // Check for active added to avoid "disconnected" message popup repeatedly
  // from "idle"
  return !SessionList() && FTerminal->Active;
}
//---------------------------------------------------------------------------
TWinSCPPlugin * __fastcall TWinSCPFileSystem::WinSCPPlugin()
{
  return dynamic_cast<TWinSCPPlugin*>(FPlugin);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::Close()
{
  try
  {
    SAFE_DESTROY(FKeepaliveThread);

    if (Connected())
    {
      assert(FQueue != NULL);
      if (!FQueue->IsEmpty &&
          (MoreMessageDialog(GetMsg(PENDING_QUEUE_ITEMS), NULL, qtWarning,
             qaOK | qaCancel) == qaOK))
      {
        QueueShow(true);
      }
    }
  }
  __finally
  {
    TCustomFarFileSystem::Close();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::GetOpenPluginInfoEx(long unsigned & Flags,
  AnsiString & /*HostFile*/, AnsiString & CurDir, AnsiString & Format,
  AnsiString & PanelTitle, TFarPanelModes * PanelModes, int & /*StartPanelMode*/,
  int & /*StartSortMode*/, bool & /*StartSortOrder*/, TFarKeyBarTitles * KeyBarTitles,
  AnsiString & ShortcutData)
{
  if (!SessionList())
  {
    Flags = OPIF_USEFILTER | OPIF_USESORTGROUPS | OPIF_USEHIGHLIGHTING |
      OPIF_SHOWPRESERVECASE | OPIF_COMPAREFATTIME;

    // When slash is added to the end of path, windows style paths
    // (vandyke: c:/windows/system) are displayed correctly on command-line, but
    // leaved subdirectory is not focused, when entering parent directory.
    CurDir = FTerminal->CurrentDirectory;
    Format = FTerminal->SessionData->SessionName;
    if (FarConfiguration->HostNameInTitle)
    {
      PanelTitle = ::FORMAT(" %s:%s ", (Format, CurDir));
    }
    else
    {
      PanelTitle = ::FORMAT(" %s ", (CurDir));
    }
    ShortcutData = ::FORMAT("%s\1%s", (FTerminal->SessionData->Name, CurDir));

    TRemoteFilePanelItem::SetPanelModes(PanelModes);
    TRemoteFilePanelItem::SetKeyBarTitles(KeyBarTitles);
  }
  else
  {
    Format = "winscp";
    Flags = OPIF_USESORTGROUPS | OPIF_ADDDOTS | OPIF_SHOWNAMESONLY |
      OPIF_SHOWPRESERVECASE;
    PanelTitle = ::FORMAT(" %s ", (GetMsg(STORED_SESSION_TITLE)));

    TSessionPanelItem::SetPanelModes(PanelModes);
    TSessionPanelItem::SetKeyBarTitles(KeyBarTitles);
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::GetFindDataEx(TList * PanelItems, int OpMode)
{
  bool Result;
  if (Connected())
  {
    assert(!FNoProgress);
    // OPM_FIND is used also for calculation of directory size (F3, quick view).
    // However directory is usually read from SetDirectory, so FNoProgress
    // seems to have no effect here.
    // Do not know if OPM_SILENT is even used.
    FNoProgress = FLAGSET(OpMode, OPM_FIND) || FLAGSET(OpMode, OPM_SILENT);
    try
    {
      if (FReloadDirectory && FTerminal->Active)
      {
        FReloadDirectory = false;
        FTerminal->ReloadDirectory();
      }

      TRemoteFile * File;
      for (int Index = 0; Index < FTerminal->Files->Count; Index++)
      {
        File = FTerminal->Files->Files[Index];
        PanelItems->Add(new TRemoteFilePanelItem(File));
      }
    }
    __finally
    {
      FNoProgress = false;
    }
    Result = true;
  }
  else if (SessionList())
  {
    Result = true;
    assert(StoredSessions);
    StoredSessions->Load();
    if (StoredSessions->Count > 0)
    {
      TSessionData * Data;
      for (int Index = 0; Index < StoredSessions->Count; Index++)
      {
        Data = StoredSessions->Sessions[Index];
        PanelItems->Add(new TSessionPanelItem(Data));
      }
    }
    else
    {
      PanelItems->Add(new THintPanelItem(GetMsg(NEW_SESSION_HINT)));
    }
  }
  else
  {
    Result = false;
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::DuplicateRenameSession(TSessionData * Data,
  bool Duplicate)
{
  assert(Data);
  AnsiString Name = Data->Name;
  if (FPlugin->InputBox(GetMsg(Duplicate ? DUPLICATE_SESSION_TITLE : RENAME_SESSION_TITLE),
        GetMsg(Duplicate ? DUPLICATE_SESSION_PROMPT : RENAME_SESSION_PROMPT),
        Name, 0, "", 255, SessionNameValidate) &&
      !Name.IsEmpty() && (Name != Data->Name))
  {
    if (StoredSessions->FindByName(Name))
    {
      throw Exception(FORMAT(GetMsg(SESSION_ALREADY_EXISTS_ERROR), (Name)));
    }
    else
    {
      TSessionData * NData = StoredSessions->NewSession(Name, Data);

      if (!Duplicate)
      {
        Data->Remove();
        StoredSessions->Remove(Data);
      }

      StoredSessions->Save();

      if (UpdatePanel())
      {
        RedrawPanel();

        PanelInfo->FocusedIndex = StoredSessions->IndexOf(NData) + 1;
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::SessionNameValidate(AnsiString & SessionName)
{
  TSessionData::ValidateName(SessionName);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::EditConnectSession(TSessionData * Data, bool Edit)
{
  TSessionData * OrigData = Data;
  bool NewData = !Data;
  bool FillInConnect = !Edit && !Data->CanLogin;

  if (NewData || FillInConnect)
  {
    Data = new TSessionData("");
  }

  try
  {
    if (FillInConnect)
    {
      Data->Assign(OrigData);
      Data->Name = "";
    }

    TSessionAction Action;
    if (Edit || FillInConnect)
    {
      Action = (FillInConnect ? saConnect : (OrigData == NULL ? saAdd : saEdit));
      if (SessionDialog(Data, Action))
      {
        int Select = -1;
        if ((!NewData && !FillInConnect) || (Action != saConnect))
        {
          if (NewData)
          {
            AnsiString Name = Data->SessionName;
            if (FPlugin->InputBox(GetMsg(NEW_SESSION_NAME_TITLE),
                  GetMsg(NEW_SESSION_NAME_PROMPT), Name, 0, "", 255,
                  SessionNameValidate) &&
                !Name.IsEmpty())
            {
              if (StoredSessions->FindByName(Name))
              {
                throw Exception(FORMAT(GetMsg(SESSION_ALREADY_EXISTS_ERROR), (Name)));
              }
              else
              {
                TSessionData * NData = StoredSessions->NewSession(Name, Data);
                Select = StoredSessions->IndexOf(NData);
              }
            }
          }
          else if (FillInConnect)
          {
            AnsiString OrigName = OrigData->Name;
            OrigData->Assign(Data);
            OrigData->Name = OrigName;
          }

          StoredSessions->Save();
          if (UpdatePanel())
          {
            if (Select < 0)
            {
              RedrawPanel();
            }
            else
            {
              PanelInfo->FocusedIndex = Select + 1;
            }
          }
        }
      }
    }
    else
    {
      Action = saConnect;
    }

    if ((Action == saConnect) && Connect(Data))
    {
      if (UpdatePanel())
      {
        RedrawPanel();
        if (PanelInfo->ItemCount)
        {
          PanelInfo->FocusedIndex = 0;
        }
      }
    }
  }
  __finally
  {
    if (NewData || FillInConnect)
    {
      delete Data;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessEventEx(int Event, void * Param)
{
  bool Result = false;
  if (Connected())
  {
    if (Event == FE_COMMAND)
    {
      AnsiString Command = (char *)Param;
      if (!Command.Trim().IsEmpty() &&
          (Command.SubString(1, 3).LowerCase() != "cd "))
      {
        Result = ExecuteCommand(Command);
      }
    }
    else if (Event == FE_IDLE)
    {
      // FAR WORKAROUND
      // Control(FCTL_CLOSEPLUGIN) does not seem to close plugin when called from
      // ProcessEvent(FE_IDLE). So if TTerminal::Idle() causes session to close
      // we must count on having ProcessEvent(FE_IDLE) called again.
      FTerminal->Idle();
      if (FQueue != NULL)
      {
        FQueue->Idle();
      }
      ProcessQueue();
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalCaptureLog(TObject * /*Sender*/,
  TLogLineType Type, const AnsiString AddedLine)
{
  if ((Type == llOutput) || (Type == llStdError))
  {
    if (FOutputLog)
    {
      FPlugin->WriteConsole(AddedLine + "\n");
    }
    if (FCapturedLog != NULL)
    {
      FCapturedLog->Add(AddedLine);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::RequireLocalPanel(TFarPanelInfo * Panel, AnsiString Message)
{
  if (Panel->IsPlugin || (Panel->Type != ptFile))
  {
    throw Exception(Message);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::RequireCapability(int Capability)
{
  if (!FTerminal->IsCapable[static_cast<TFSCapability>(Capability)])
  {
    throw Exception(FORMAT(GetMsg(OPERATION_NOT_SUPPORTED),
      (FTerminal->ProtocolName)));
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::EnsureCommandSessionFallback(TFSCapability Capability)
{
  bool Result = FTerminal->IsCapable[Capability] ||
    FTerminal->CommandSessionOpened;

  if (!Result)
  {
    if (!GUIConfiguration->ConfirmCommandSession)
    {
      Result = true;
    }
    else
    {
      TMessageParams Params;
      Params.Params = qpNeverAskAgainCheck;
      int Answer = MoreMessageDialog(FORMAT(GetMsg(PERFORM_ON_COMMAND_SESSION),
        (FTerminal->ProtocolName, FTerminal->ProtocolName)), NULL,
        qtConfirmation, qaOK | qaCancel, &Params);
      if (Answer == qaNeverAskAgain)
      {
        GUIConfiguration->ConfirmCommandSession = false;
        Result = true;
      }
      else
      {
        Result = (Answer == qaOK);
      }
    }

    if (Result)
    {
      ConnectTerminal(FTerminal->CommandSession);
    }
  }

  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExecuteCommand(const AnsiString Command)
{
  if (FTerminal->AllowedAnyCommand(Command) &&
      EnsureCommandSessionFallback(fcAnyCommand))
  {
    FTerminal->BeginTransaction();
    try
    {
      FarControl(FCTL_SETCMDLINE, NULL);
      FPlugin->ShowConsoleTitle(Command);
      try
      {
        FPlugin->ShowTerminalScreen();

        FOutputLog = true;
        FTerminal->AnyCommand(Command, TerminalCaptureLog);
      }
      __finally
      {
        FPlugin->ScrollTerminalScreen(1);
        FPlugin->SaveTerminalScreen();
        FPlugin->ClearConsoleTitle();
      }
    }
    __finally
    {
      if (FTerminal->Active)
      {
        FTerminal->EndTransaction();
        UpdatePanel();
      }
      else
      {
        RedrawPanel();
        RedrawPanel(true);
      }
    }
  }
  return true;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ProcessKeyEx(int Key, unsigned int ControlState)
{
  bool Handled = false;

  TFarPanelItem * Focused = PanelInfo->FocusedItem;

  if ((Key == 'W') && (ControlState & PKF_SHIFT) &&
        (ControlState & PKF_ALT))
  {
    WinSCPPlugin()->CommandsMenu();
    Handled = true;
  }
  else if (SessionList())
  {
    TSessionData * Data = NULL;

    if ((Focused != NULL) && Focused->IsFile && Focused->UserData)
    {
      Data = (TSessionData *)Focused->UserData;
    }

    if ((Key == VK_RETURN && (ControlState == 0) && Data) ||
        (Key == VK_F4 && (ControlState == 0) &&
         (Data || StoredSessions->Count == 0)) ||
        (Key == VK_F4 && (ControlState & PKF_SHIFT)))
    {
      EditConnectSession(ControlState & PKF_SHIFT ? NULL : Data, Key == VK_F4);
      Handled = true;
    }

    if (Data && ((Key == VK_F5) || (Key == VK_F6)) &&
      (ControlState & PKF_SHIFT))
    {
      DuplicateRenameSession(Data, Key == VK_F5);
      Handled = true;
    }
  }
  else if (Connected())
  {
    if ((Key == 'F') && (ControlState & PKF_CONTROL))
    {
      InsertFileNameOnCommandLine();
      Handled = true;
    }

    if ((Key == 'R') && (ControlState & PKF_CONTROL))
    {
      FReloadDirectory = true;
    }

    if ((Key == 'A') && (ControlState & PKF_CONTROL))
    {
      FileProperties();
      Handled = true;
    }

    if ((Key == 'G') && (ControlState & PKF_CONTROL))
    {
      ApplyCommand();
      Handled = true;
    }

    if ((Key == 'Q') && (ControlState & PKF_SHIFT) &&
          (ControlState & PKF_ALT))
    {
      QueueShow(false);
      Handled = true;
    }

    if ((Key == 'B') && (ControlState & PKF_CONTROL) &&
          (ControlState & PKF_ALT))
    {
      ToggleSynchronizeBrowsing();
      Handled = true;
    }

    if ((Key == VK_F6) && ((ControlState & (PKF_ALT | PKF_SHIFT)) == PKF_ALT))
    {
      CreateLink();
      Handled = true;
    }

    if (Focused && ((Key == VK_F5) || (Key == VK_F6)) &&
        ((ControlState & (PKF_ALT | PKF_SHIFT)) == PKF_SHIFT))
    {
      TransferFiles((Key == VK_F6));
      Handled = true;
    }

    if (Focused && (Key == VK_F6) &&
        ((ControlState & (PKF_ALT | PKF_SHIFT)) == (PKF_SHIFT | PKF_ALT)))
    {
      RenameFile();
      Handled = true;
    }

    if ((Key == VK_F12) && (ControlState & PKF_SHIFT) &&
        (ControlState & PKF_ALT))
    {
      OpenDirectory(false);
      Handled = true;
    }
  }
  return Handled;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CreateLink()
{
  RequireCapability(fcResolveSymlink);
  RequireCapability(fcSymbolicLink);

  bool Edit = false;
  TRemoteFile * File = NULL;
  AnsiString FileName;
  AnsiString PointTo;
  bool SymbolicLink = true;

  if (PanelInfo->FocusedItem && PanelInfo->FocusedItem->UserData)
  {
    File = (TRemoteFile *)PanelInfo->FocusedItem->UserData;

    Edit = File->IsSymLink && Terminal->SessionData->ResolveSymlinks;
    if (Edit)
    {
      FileName = File->FileName;
      PointTo = File->LinkTo;
    }
    else
    {
      PointTo = File->FileName;
    }
  }

  if (LinkDialog(FileName, PointTo, SymbolicLink, Edit,
        Terminal->IsCapable[fcHardLink]))
  {
    if (Edit)
    {
      assert(File->FileName == FileName);
      bool Recursive = false;
      Terminal->ExceptionOnFail = true;
      try
      {
        Terminal->DeleteFile("", File, &Recursive);
      }
      __finally
      {
        Terminal->ExceptionOnFail = false;
      }
    }
    Terminal->CreateLink(FileName, PointTo, SymbolicLink);
    if (UpdatePanel())
    {
      RedrawPanel();
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TemporarilyDownloadFiles(
  TStrings * FileList, bool ForceText, AnsiString & TempDir)
{
  TCopyParamType CopyParam = GUIConfiguration->DefaultCopyParam;
  if (ForceText)
  {
    CopyParam.TransferMode = tmAscii;
  }
  CopyParam.FileNameCase = ncNoChange;
  CopyParam.PreserveReadOnly = false;
  CopyParam.ResumeSupport = rsOff;

  TempDir = FPlugin->TemporaryDir();
  if (TempDir.IsEmpty() || !ForceDirectories(TempDir))
  {
    throw Exception(FMTLOAD(CREATE_TEMP_DIR_ERROR, (TempDir)));
  }

  FTerminal->ExceptionOnFail = true;
  try
  {
    try
    {
      FTerminal->CopyToLocal(FileList, TempDir, &CopyParam, cpTemporary);
    }
    catch(...)
    {
      try
      {
        RecursiveDeleteFile(ExcludeTrailingBackslash(TempDir), false);
      }
      catch(...)
      {
      }
      throw;
    }
  }
  __finally
  {
    FTerminal->ExceptionOnFail = false;
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ApplyCommand()
{
  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList != NULL)
  {
    try
    {
      int Params = FarConfiguration->ApplyCommandParams;
      AnsiString Command = FarConfiguration->ApplyCommandCommand;
      if (ApplyCommandDialog(Command, Params))
      {
        FarConfiguration->ApplyCommandParams = Params;
        FarConfiguration->ApplyCommandCommand = Command;
        if (FLAGCLEAR(Params, ccLocal))
        {
          if (EnsureCommandSessionFallback(fcAnyCommand))
          {
            TRemoteCustomCommand RemoteCustomCommand;
            TFarInteractiveCustomCommand InteractiveCustomCommand(
              FPlugin, &RemoteCustomCommand);

            Command = InteractiveCustomCommand.Complete(Command, false);

            try
            {
              TLogAddLineEvent OutputEvent = NULL;
              FOutputLog = false;
              if (FLAGSET(Params, ccShowResults))
              {
                assert(!FNoProgress);
                FNoProgress = true;
                FOutputLog = true;
                OutputEvent = TerminalCaptureLog;
              }

              if (FLAGSET(Params, ccCopyResults))
              {
                assert(FCapturedLog == NULL);
                FCapturedLog = new TStringList();
                OutputEvent = TerminalCaptureLog;
              }

              try
              {
                if (FLAGSET(Params, ccShowResults))
                {
                  FPlugin->ShowTerminalScreen();
                }

                FTerminal->CustomCommandOnFiles(Command, Params, FileList, OutputEvent);
              }
              __finally
              {
                if (FLAGSET(Params, ccShowResults))
                {
                  FNoProgress = false;
                  FPlugin->ScrollTerminalScreen(1);
                  FPlugin->SaveTerminalScreen();
                }

                if (FLAGSET(Params, ccCopyResults))
                {
                  FPlugin->FarCopyToClipboard(FCapturedLog);
                  SAFE_DESTROY(FCapturedLog);
                }
              }
            }
            __finally
            {
              PanelInfo->ApplySelection();
              if (UpdatePanel())
              {
                RedrawPanel();
              }
            }
          }
        }
        else
        {
          TLocalCustomCommand LocalCustomCommand;
          TFarInteractiveCustomCommand InteractiveCustomCommand(FPlugin,
            &LocalCustomCommand);

          Command = InteractiveCustomCommand.Complete(Command, false);

          TStrings * LocalFileList = NULL;
          TStrings * RemoteFileList = NULL;
          try
          {
            bool FileListCommand = LocalCustomCommand.IsFileListCommand(Command);
            bool LocalFileCommand = LocalCustomCommand.HasLocalFileName(Command);

            if (LocalFileCommand)
            {
              TFarPanelInfo * AnotherPanel = AnotherPanelInfo;
              RequireLocalPanel(AnotherPanel, GetMsg(APPLY_COMMAND_LOCAL_PATH_REQUIRED));

              LocalFileList = CreateSelectedFileList(osLocal, AnotherPanel);

              if (FileListCommand)
              {
                if ((LocalFileList == NULL) || (LocalFileList->Count != 1))
                {
                  throw Exception(GetMsg(CUSTOM_COMMAND_SELECTED_UNMATCH1));
                }
              }
              else
              {
                if ((LocalFileList == NULL) ||
                    ((LocalFileList->Count != 1) &&
                     (FileList->Count != 1) &&
                     (LocalFileList->Count != FileList->Count)))
                {
                  throw Exception(GetMsg(CUSTOM_COMMAND_SELECTED_UNMATCH));
                }
              }
            }

            AnsiString TempDir;

            TemporarilyDownloadFiles(FileList, false, TempDir);

            try
            {
              RemoteFileList = new TStringList();

              TMakeLocalFileListParams MakeFileListParam;
              MakeFileListParam.FileList = RemoteFileList;
              MakeFileListParam.IncludeDirs = FLAGSET(Params, ccApplyToDirectories);
              MakeFileListParam.Recursive =
                FLAGSET(Params, ccRecursive) && !FileListCommand;

              ProcessLocalDirectory(TempDir, &FTerminal->MakeLocalFileList, &MakeFileListParam);

              TFileOperationProgressType Progress(&OperationProgress, &OperationFinished);

              Progress.Start(foCustomCommand, osRemote, FileListCommand ? 1 : FileList->Count);

              try
              {
                if (FileListCommand)
                {
                  AnsiString LocalFile;
                  AnsiString FileList = MakeFileList(RemoteFileList);

                  if (LocalFileCommand)
                  {
                    assert(LocalFileList->Count == 1);
                    LocalFile = LocalFileList->Strings[0];
                  }

                  TLocalCustomCommand CustomCommand("", LocalFile, FileList);
                  ExecuteShellAndWait(FPlugin->Handle, CustomCommand.Complete(Command, true),
                    TProcessMessagesEvent(NULL));
                }
                else if (LocalFileCommand)
                {
                  if (LocalFileList->Count == 1)
                  {
                    AnsiString LocalFile = LocalFileList->Strings[0];

                    for (int Index = 0; Index < RemoteFileList->Count; Index++)
                    {
                      AnsiString FileName = RemoteFileList->Strings[Index];
                      TLocalCustomCommand CustomCommand
                        (FileName, LocalFile, "");
                      ExecuteShellAndWait(FPlugin->Handle,
                        CustomCommand.Complete(Command, true), TProcessMessagesEvent(NULL));
                    }
                  }
                  else if (RemoteFileList->Count == 1)
                  {
                    AnsiString FileName = RemoteFileList->Strings[0];

                    for (int Index = 0; Index < LocalFileList->Count; Index++)
                    {
                      TLocalCustomCommand CustomCommand
                        (FileName, LocalFileList->Strings[Index], "");
                      ExecuteShellAndWait(FPlugin->Handle,
                        CustomCommand.Complete(Command, true), TProcessMessagesEvent(NULL));
                    }
                  }
                  else
                  {
                    if (LocalFileList->Count != RemoteFileList->Count)
                    {
                      throw Exception(GetMsg(CUSTOM_COMMAND_PAIRS_DOWNLOAD_FAILED));
                    }

                    for (int Index = 0; Index < LocalFileList->Count; Index++)
                    {
                      AnsiString FileName = RemoteFileList->Strings[Index];
                      TLocalCustomCommand CustomCommand
                        (FileName, LocalFileList->Strings[Index], "");
                      ExecuteShellAndWait(FPlugin->Handle,
                        CustomCommand.Complete(Command, true), TProcessMessagesEvent(NULL));
                    }
                  }
                }
                else
                {
                  for (int Index = 0; Index < RemoteFileList->Count; Index++)
                  {
                    TLocalCustomCommand CustomCommand
                      (RemoteFileList->Strings[Index], "", "");
                    ExecuteShellAndWait(FPlugin->Handle,
                      CustomCommand.Complete(Command, true), TProcessMessagesEvent(NULL));
                  }
                }
              }
              __finally
              {
                Progress.Stop();
              }
            }
            __finally
            {
              RecursiveDeleteFile(ExcludeTrailingBackslash(TempDir), false);
            }
          }
          __finally
          {
            delete RemoteFileList;
            delete LocalFileList;
          }
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::Synchronize(const AnsiString LocalDirectory,
  const AnsiString RemoteDirectory, TTerminal::TSynchronizeMode Mode,
  const TCopyParamType & CopyParam, int Params, TSynchronizeChecklist ** Checklist,
  TSynchronizeOptions * Options)
{
  TSynchronizeChecklist * AChecklist = NULL;
  try
  {
    FPlugin->SaveScreen(FSynchronizationSaveScreenHandle);
    FPlugin->ShowConsoleTitle(GetMsg(SYNCHRONIZE_PROGRESS_COMPARE_TITLE));
    FSynchronizationStart = Now();
    FSynchronizationCompare = true;
    try
    {
      AChecklist = FTerminal->SynchronizeCollect(LocalDirectory, RemoteDirectory,
        Mode, &CopyParam, Params | TTerminal::spNoConfirmation,
        TerminalSynchronizeDirectory, Options);
    }
    __finally
    {
      FPlugin->ClearConsoleTitle();
      FPlugin->RestoreScreen(FSynchronizationSaveScreenHandle);
    }  

    FPlugin->SaveScreen(FSynchronizationSaveScreenHandle);
    FPlugin->ShowConsoleTitle(GetMsg(SYNCHRONIZE_PROGRESS_TITLE));
    FSynchronizationStart = Now();
    FSynchronizationCompare = false;
    try
    {
      FTerminal->SynchronizeApply(AChecklist, LocalDirectory, RemoteDirectory,
        &CopyParam, Params | TTerminal::spNoConfirmation,
        TerminalSynchronizeDirectory);
    }
    __finally
    {
      FPlugin->ClearConsoleTitle();
      FPlugin->RestoreScreen(FSynchronizationSaveScreenHandle);
    }
  }
  __finally
  {
    if (Checklist == NULL)
    {
      delete AChecklist;
    }
    else
    {
      *Checklist = AChecklist;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::SynchronizeAllowSelectedOnly()
{
  return
    (PanelInfo->SelectedCount > 0) ||
    (AnotherPanelInfo->SelectedCount > 0);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::GetSynchronizeOptions(
  int Params, TSynchronizeOptions & Options)
{
  if (FLAGSET(Params, spSelectedOnly) && SynchronizeAllowSelectedOnly())
  {
    Options.Filter = new TStringList();
    Options.Filter->CaseSensitive = false;
    Options.Filter->Duplicates = dupAccept;

    if (PanelInfo->SelectedCount > 0)
    {
      CreateFileList(PanelInfo->Items, osRemote, true, "", true, Options.Filter);
    }
    if (AnotherPanelInfo->SelectedCount > 0)
    {
      CreateFileList(AnotherPanelInfo->Items, osLocal, true, "", true, Options.Filter);
    }
    Options.Filter->Sort();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::FullSynchronize(bool Source)
{
  TFarPanelInfo * AnotherPanel = AnotherPanelInfo;
  RequireLocalPanel(AnotherPanel, GetMsg(SYNCHRONIZE_LOCAL_PATH_REQUIRED));

  AnsiString LocalDirectory = AnotherPanel->CurrentDirectory;
  AnsiString RemoteDirectory = FTerminal->CurrentDirectory;

  bool SaveMode = !(GUIConfiguration->SynchronizeModeAuto < 0);
  TTerminal::TSynchronizeMode Mode =
    (SaveMode ? (TTerminal::TSynchronizeMode)GUIConfiguration->SynchronizeModeAuto :
      (Source ? TTerminal::smLocal : TTerminal::smRemote));
  int Params = GUIConfiguration->SynchronizeParams;
  bool SaveSettings = false;

  TCopyParamType CopyParam = GUIConfiguration->DefaultCopyParam;
  TUsableCopyParamAttrs CopyParamAttrs = Terminal->UsableCopyParamAttrs(0);
  int Options =
    FLAGMASK(!FTerminal->IsCapable[fcTimestampChanging], fsoDisableTimestamp) |
    FLAGMASK(SynchronizeAllowSelectedOnly(), fsoAllowSelectedOnly);
  if (FullSynchronizeDialog(Mode, Params, LocalDirectory, RemoteDirectory,
        &CopyParam, SaveSettings, SaveMode, Options, CopyParamAttrs))
  {
    TSynchronizeOptions SynchronizeOptions;
    GetSynchronizeOptions(Params, SynchronizeOptions);

    if (SaveSettings)
    {
      GUIConfiguration->SynchronizeParams = Params;
      if (SaveMode)
      {
        GUIConfiguration->SynchronizeModeAuto = Mode;
      }
    }

    TSynchronizeChecklist * Checklist = NULL;
    try
    {
      FPlugin->SaveScreen(FSynchronizationSaveScreenHandle);
      FPlugin->ShowConsoleTitle(GetMsg(SYNCHRONIZE_PROGRESS_COMPARE_TITLE));
      FSynchronizationStart = Now();
      FSynchronizationCompare = true;
      try
      {
        Checklist = FTerminal->SynchronizeCollect(LocalDirectory, RemoteDirectory,
          Mode, &CopyParam, Params | TTerminal::spNoConfirmation,
          TerminalSynchronizeDirectory, &SynchronizeOptions);
      }
      __finally
      {
        FPlugin->ClearConsoleTitle();
        FPlugin->RestoreScreen(FSynchronizationSaveScreenHandle);
      }

      if (Checklist->Count == 0)
      {
        MoreMessageDialog(GetMsg(COMPARE_NO_DIFFERENCES), NULL,
           qtInformation, qaOK);
      }
      else if (FLAGCLEAR(Params, TTerminal::spPreviewChanges) ||
               SynchronizeChecklistDialog(Checklist, Mode, Params,
                 LocalDirectory, RemoteDirectory))
      {
        if (FLAGSET(Params, TTerminal::spPreviewChanges))
        {
          FSynchronizationStart = Now();
        }
        FPlugin->SaveScreen(FSynchronizationSaveScreenHandle);
        FPlugin->ShowConsoleTitle(GetMsg(SYNCHRONIZE_PROGRESS_TITLE));
        FSynchronizationStart = Now();
        FSynchronizationCompare = false;
        try
        {
          FTerminal->SynchronizeApply(Checklist, LocalDirectory, RemoteDirectory,
            &CopyParam, Params | TTerminal::spNoConfirmation,
            TerminalSynchronizeDirectory);
        }
        __finally
        {
          FPlugin->ClearConsoleTitle();
          FPlugin->RestoreScreen(FSynchronizationSaveScreenHandle);
        }
      }
    }
    __finally
    {
      delete Checklist;
      if (UpdatePanel())
      {
        RedrawPanel();
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalSynchronizeDirectory(
  const AnsiString LocalDirectory, const AnsiString RemoteDirectory,
  bool & Continue, bool Collect)
{
  static unsigned long LastTicks;
  unsigned long Ticks = GetTickCount();
  if ((LastTicks == 0) || (Ticks - LastTicks > 500))
  {
    LastTicks = Ticks;

    static const int ProgressWidth = 46;
    static AnsiString ProgressTitle;
    static AnsiString ProgressTitleCompare;
    static AnsiString LocalLabel;
    static AnsiString RemoteLabel;
    static AnsiString StartTimeLabel;
    static AnsiString TimeElapsedLabel;

    if (ProgressTitle.IsEmpty())
    {
      ProgressTitle = GetMsg(SYNCHRONIZE_PROGRESS_TITLE);
      ProgressTitleCompare = GetMsg(SYNCHRONIZE_PROGRESS_COMPARE_TITLE);
      LocalLabel = GetMsg(SYNCHRONIZE_PROGRESS_LOCAL);
      RemoteLabel = GetMsg(SYNCHRONIZE_PROGRESS_REMOTE);
      StartTimeLabel = GetMsg(SYNCHRONIZE_PROGRESS_START_TIME);
      TimeElapsedLabel = GetMsg(SYNCHRONIZE_PROGRESS_ELAPSED);
    }

    AnsiString Message;

    Message = LocalLabel + MinimizeName(LocalDirectory,
      ProgressWidth - LocalLabel.Length(), false);
    Message += AnsiString::StringOfChar(' ', ProgressWidth - Message.Length()) + "\n";
    Message += RemoteLabel + MinimizeName(RemoteDirectory,
      ProgressWidth - RemoteLabel.Length(), true) + "\n";
    Message += StartTimeLabel + FSynchronizationStart.TimeString() + "\n";
    Message += TimeElapsedLabel +
      FormatDateTimeSpan(Configuration->TimeFormat, Now() - FSynchronizationStart) + "\n";

    FPlugin->Message(0, (Collect ? ProgressTitleCompare : ProgressTitle), Message);

    if (FPlugin->CheckForEsc() &&
        (MoreMessageDialog(GetMsg(CANCEL_OPERATION), NULL,
          qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      Continue = false;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::Synchronize()
{
  TFarPanelInfo * AnotherPanel = AnotherPanelInfo;
  RequireLocalPanel(AnotherPanel, GetMsg(SYNCHRONIZE_LOCAL_PATH_REQUIRED));

  TSynchronizeParamType Params;
  Params.LocalDirectory = AnotherPanel->CurrentDirectory;
  Params.RemoteDirectory = FTerminal->CurrentDirectory;
  int UnusedParams = (GUIConfiguration->SynchronizeParams &
    (TTerminal::spPreviewChanges | TTerminal::spTimestamp |
     TTerminal::spNotByTime | TTerminal::spBySize));
  Params.Params = GUIConfiguration->SynchronizeParams & ~UnusedParams;
  Params.Options = GUIConfiguration->SynchronizeOptions;
  bool SaveSettings = false;
  TSynchronizeController Controller(&DoSynchronize, &DoSynchronizeInvalid,
    &DoSynchronizeTooManyDirectories);
  assert(FSynchronizeController == NULL);
  FSynchronizeController = &Controller;

  try
  {
    TCopyParamType CopyParam = GUIConfiguration->DefaultCopyParam;
    int CopyParamAttrs = Terminal->UsableCopyParamAttrs(0).Upload;
    int Options =
      FLAGMASK(SynchronizeAllowSelectedOnly(), soAllowSelectedOnly);
    if (SynchronizeDialog(Params, &CopyParam, Controller.StartStop,
          SaveSettings, Options, CopyParamAttrs, GetSynchronizeOptions) &&
        SaveSettings)
    {
      GUIConfiguration->SynchronizeParams = Params.Params | UnusedParams;
      GUIConfiguration->SynchronizeOptions = Params.Options;
    }
  }
  __finally
  {
    FSynchronizeController = NULL;
    // plugin might have been closed during some synchronisation already
    if (!FClosed)
    {
      if (UpdatePanel())
      {
        RedrawPanel();
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::DoSynchronize(
  TSynchronizeController * /*Sender*/, const AnsiString LocalDirectory,
  const AnsiString RemoteDirectory, const TCopyParamType & CopyParam,
  const TSynchronizeParamType & Params, TSynchronizeChecklist ** Checklist,
  TSynchronizeOptions * Options, bool Full)
{
  try
  {
    int PParams = Params.Params;
    if (!Full)
    {
      PParams |= TTerminal::spNoRecurse | TTerminal::spUseCache |
        TTerminal::spDelayProgress | TTerminal::spSubDirs;
    }
    Synchronize(LocalDirectory, RemoteDirectory, TTerminal::smRemote, CopyParam,
      PParams, Checklist, Options);
  }
  catch(Exception & E)
  {
    HandleException(&E);
    throw;
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::DoSynchronizeInvalid(
  TSynchronizeController * /*Sender*/, const AnsiString Directory,
  const AnsiString ErrorStr)
{
  AnsiString Message;
  if (!Directory.IsEmpty())
  {
    Message = FORMAT(GetMsg(WATCH_ERROR_DIRECTORY), (Directory));
  }
  else
  {
    Message = GetMsg(WATCH_ERROR_GENERAL);
  }

  MoreMessageDialog(Message, NULL, qtError, qaOK);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::DoSynchronizeTooManyDirectories(
  TSynchronizeController * /*Sender*/, int & MaxDirectories)
{
  if (MaxDirectories < GUIConfiguration->MaxWatchDirectories)
  {
    MaxDirectories = GUIConfiguration->MaxWatchDirectories;
  }
  else
  {
    TMessageParams Params;
    Params.Params = qpNeverAskAgainCheck;
    int Result = MoreMessageDialog(
      FORMAT(GetMsg(TOO_MANY_WATCH_DIRECTORIES), (MaxDirectories, MaxDirectories)), NULL,
      qtConfirmation, qaYes | qaNo, &Params);

    if ((Result == qaYes) || (Result == qaNeverAskAgain))
    {
      MaxDirectories *= 2;
      if (Result == qaNeverAskAgain)
      {
        GUIConfiguration->MaxWatchDirectories = MaxDirectories;
      }
    }
    else
    {
      Abort();
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CustomCommandGetParamValue(
  const AnsiString AName, AnsiString & Value)
{
  AnsiString Name = AName;
  if (Name.IsEmpty())
  {
    Name = GetMsg(APPLY_COMMAND_PARAM_PROMPT);
  }
  if (!FPlugin->InputBox(GetMsg(APPLY_COMMAND_PARAM_TITLE),
        Name, Value, 0, APPLY_COMMAND_PARAM_HISTORY))
  {
    Abort();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TransferFiles(bool Move)
{
  if (Move)
  {
    RequireCapability(fcRemoteMove);
  }

  if (Move || EnsureCommandSessionFallback(fcRemoteCopy))
  {
    TStrings * FileList = CreateSelectedFileList(osRemote);
    if (FileList)
    {
      assert(!FPanelItems);

      try
      {
        AnsiString Target = FTerminal->CurrentDirectory;
        AnsiString FileMask = "*.*";
        if (RemoteTransferDialog(FileList, Target, FileMask, Move))
        {
          try
          {
            if (Move)
            {
              Terminal->MoveFiles(FileList, Target, FileMask);
            }
            else
            {
              Terminal->CopyFiles(FileList, Target, FileMask);
            }
          }
          __finally
          {
            PanelInfo->ApplySelection();
            if (UpdatePanel())
            {
              RedrawPanel();
            }
          }
        }
      }
      __finally
      {
        delete FileList;
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::RenameFile()
{
  TFarPanelItem * PanelItem = PanelInfo->FocusedItem;
  assert(PanelItem != NULL);

  if (!PanelItem->IsParentDirectory)
  {
    RequireCapability(fcRename);

    TRemoteFile * File = static_cast<TRemoteFile *>(PanelItem->UserData);
    AnsiString NewName = File->FileName;
    if (RenameFileDialog(File, NewName))
    {
      try
      {
        Terminal->RenameFile(File, NewName, true);
      }
      __finally
      {
        if (UpdatePanel())
        {
          RedrawPanel();
        }
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::FileProperties()
{
  TStrings * FileList = CreateSelectedFileList(osRemote);
  if (FileList)
  {
    assert(!FPanelItems);

    try
    {
      TRemoteProperties CurrentProperties;

      CurrentProperties = TRemoteProperties::CommonProperties(FileList);

      int Flags = 0;
      if (FTerminal->IsCapable[fcModeChanging]) Flags |= cpMode;
      if (FTerminal->IsCapable[fcOwnerChanging]) Flags |= cpOwner;
      if (FTerminal->IsCapable[fcGroupChanging]) Flags |= cpGroup;

      TRemoteProperties NewProperties = CurrentProperties;
      if (PropertiesDialog(FileList, FTerminal->CurrentDirectory,
          FTerminal->Groups, FTerminal->Users, &NewProperties, Flags))
      {
        NewProperties = TRemoteProperties::ChangedProperties(CurrentProperties,
          NewProperties);
        try
        {
          FTerminal->ChangeFilesProperties(FileList, &NewProperties);
        }
        __finally
        {
          PanelInfo->ApplySelection();
          if (UpdatePanel())
          {
            RedrawPanel();
          }
        }
      }
    }
    __finally
    {
      delete FileList;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::InsertFileNameOnCommandLine()
{
  TFarPanelItem * Focused = PanelInfo->FocusedItem;

  if (Focused != NULL)
  {
    TRemoteFile * File = reinterpret_cast<TRemoteFile *>(Focused->UserData);
    if (File != NULL)
    {
      AnsiString FullName = File->FullFileName;
      if (FullName.Pos(" ") > 0)
      {
        FullName = FORMAT("\"%s\"", (FullName));
      }
      FullName += " ";

      FarControl(FCTL_INSERTCMDLINE, FullName.c_str());
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::GetSpaceAvailable(const AnsiString Path,
  TSpaceAvailable & ASpaceAvailable, bool & Close)
{
  // terminal can be already closed (e.g. dropped connection)
  if ((Terminal != NULL) && Terminal->IsCapable[fcCheckingSpaceAvailable])
  {
    try
    {
      Terminal->SpaceAvailable(Path, ASpaceAvailable);
    }
    catch(Exception & E)
    {
      if (!Terminal->Active)
      {
        Close = true;
      }
      HandleException(&E);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ShowInformation()
{
  TFileSystemInfo FileSystemInfo;
  TGetSpaceAvailable OnGetSpaceAvailable = NULL;
  Terminal->FileSystemInfo(FileSystemInfo);
  if (Terminal->IsCapable[fcCheckingSpaceAvailable])
  {
    OnGetSpaceAvailable = GetSpaceAvailable;
  }
  FileSystemInfoDialog(FileSystemInfo, Terminal->CurrentDirectory, OnGetSpaceAvailable);
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::AreCachesEmpty()
{
  assert(Connected());
  return FTerminal->AreCachesEmpty;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ClearCaches()
{
  assert(Connected());
  FTerminal->ClearCaches();
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OpenSessionInPutty()
{
  assert(Connected());
  ::OpenSessionInPutty(GUIConfiguration->PuttyPath, FTerminal->SessionData,
    GUIConfiguration->PuttyPassword ? Terminal->Password : AnsiString());
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::QueueShow(bool ClosingPlugin)
{
  assert(Connected());
  assert(FQueueStatus != NULL);
  QueueDialog(FQueueStatus, ClosingPlugin);
  ProcessQueue();
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OpenDirectory(bool Add)
{
  TBookmarkList * BookmarkList = new TBookmarkList();
  try
  {
    AnsiString Directory = FTerminal->CurrentDirectory;
    AnsiString SessionKey = FTerminal->SessionData->SessionKey;
    TBookmarkList * CurrentBookmarkList;

    CurrentBookmarkList = FarConfiguration->Bookmarks[SessionKey];
    if (CurrentBookmarkList != NULL)
    {
      BookmarkList->Assign(CurrentBookmarkList);
    }

    if (Add)
    {
      TBookmark * Bookmark = new TBookmark;
      Bookmark->Remote = Directory;
      Bookmark->Name = Directory;
      BookmarkList->Add(Bookmark);
      FarConfiguration->Bookmarks[SessionKey] = BookmarkList;
    }

    bool Result = OpenDirectoryDialog(Add, Directory, BookmarkList);

    FarConfiguration->Bookmarks[SessionKey] = BookmarkList;

    if (Result)
    {
      FTerminal->ChangeDirectory(Directory);
      if (UpdatePanel(true))
      {
        RedrawPanel();
      }
    }
  }
  __finally
  {
    delete BookmarkList;
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::HomeDirectory()
{
  FTerminal->HomeDirectory();
  if (UpdatePanel(true))
  {
    RedrawPanel();
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::IsSynchronizedBrowsing()
{
  return FSynchronisingBrowse;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ToggleSynchronizeBrowsing()
{
  FSynchronisingBrowse = !FSynchronisingBrowse;

  if (FarConfiguration->ConfirmSynchronizedBrowsing)
  {
    AnsiString Message = FSynchronisingBrowse ?
      GetMsg(SYNCHRONIZE_BROWSING_ON) : GetMsg(SYNCHRONIZE_BROWSING_OFF);
    TMessageParams Params;
    Params.Params = qpNeverAskAgainCheck;
    if (MoreMessageDialog(Message, NULL, qtInformation, qaOK, &Params) ==
          qaNeverAskAgain)
    {
      FarConfiguration->ConfirmSynchronizedBrowsing = false;
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::SetDirectoryEx(const AnsiString Dir, int OpMode)
{
  if (!Connected())
  {
    return false;
  }
  // FAR WORKAROUND
  // workaround to ignore "change to root directory" command issued by FAR,
  // before file is opened for viewing/editing from "find file" dialog
  // when plugin uses UNIX style paths
  else if (OpMode & OPM_FIND && OpMode & OPM_SILENT && Dir == "\\")
  {
    if (FSavedFindFolder.IsEmpty())
    {
      return true;
    }
    else
    {
      bool Result;
      try
      {
        Result = SetDirectoryEx(FSavedFindFolder, OpMode);
      }
      __finally
      {
        FSavedFindFolder = "";
      }
      return Result;
    }
  }
  else
  {
    if (OpMode & OPM_FIND && FSavedFindFolder.IsEmpty())
    {
      FSavedFindFolder = FTerminal->CurrentDirectory;
    }

    assert(!FNoProgress);
    bool Normal = FLAGCLEAR(OpMode, OPM_FIND | OPM_SILENT);
    AnsiString PrevPath = FTerminal->CurrentDirectory;
    FNoProgress = !Normal;
    if (!FNoProgress)
    {
      FPlugin->ShowConsoleTitle(GetMsg(CHANGING_DIRECTORY_TITLE));
    }
    FTerminal->ExceptionOnFail = true;
    try
    {
      if (Dir == "\\")
      {
        FTerminal->ChangeDirectory(ROOTDIRECTORY);
      }
      else if ((Dir == PARENTDIRECTORY) && (FTerminal->CurrentDirectory == ROOTDIRECTORY))
      {
        ClosePlugin();
      }
      else
      {
        FTerminal->ChangeDirectory(Dir);
      }
    }
    __finally
    {
      FTerminal->ExceptionOnFail = false;
      if (!FNoProgress)
      {
        FPlugin->ClearConsoleTitle();
      }
      FNoProgress = false;
    }

    if (Normal && FSynchronisingBrowse &&
        (PrevPath != FTerminal->CurrentDirectory))
    {
      TFarPanelInfo * AnotherPanel = AnotherPanelInfo;
      if (AnotherPanel->IsPlugin || (AnotherPanel->Type != ptFile))
      {
        MoreMessageDialog(GetMsg(SYNCHRONIZE_LOCAL_PATH_REQUIRED), NULL, qtError, qaOK);
      }
      else
      {
        try
        {
          AnsiString RemotePath = UnixIncludeTrailingBackslash(FTerminal->CurrentDirectory);
          AnsiString FullPrevPath = UnixIncludeTrailingBackslash(PrevPath);
          AnsiString ALocalPath;
          if (RemotePath.SubString(1, FullPrevPath.Length()) == FullPrevPath)
          {
            ALocalPath = IncludeTrailingBackslash(AnotherPanel->CurrentDirectory) +
              FromUnixPath(RemotePath.SubString(FullPrevPath.Length() + 1,
                RemotePath.Length() - FullPrevPath.Length()));
          }
          else if (FullPrevPath.SubString(1, RemotePath.Length()) == RemotePath)
          {
            AnsiString NewLocalPath;
            ALocalPath = ExcludeTrailingBackslash(AnotherPanel->CurrentDirectory);
            while (!UnixComparePaths(FullPrevPath, RemotePath))
            {
              NewLocalPath = ExcludeTrailingBackslash(ExtractFileDir(ALocalPath));
              if (NewLocalPath == ALocalPath)
              {
                Abort();
              }
              ALocalPath = NewLocalPath;
              FullPrevPath = UnixExtractFilePath(UnixExcludeTrailingBackslash(FullPrevPath));
            }
          }
          else
          {
            Abort();
          }

          // IncludeTrailingBackslash to expand C: to C:\.
          if (!FarControl(FCTL_SETANOTHERPANELDIR,
                IncludeTrailingBackslash(ALocalPath).c_str()))
          {
            Abort();
          }

          ResetCachedInfo();
          AnotherPanel = AnotherPanelInfo;
          if (!ComparePaths(AnotherPanel->CurrentDirectory, ALocalPath))
          {
            Abort();
          }
          else
          {
            RedrawPanel(true);
          }
        }
        catch(Exception & E)
        {
          FSynchronisingBrowse = false;
          WinSCPPlugin()->ShowExtendedException(&E);
          MoreMessageDialog(GetMsg(SYNC_DIR_BROWSE_ERROR), NULL, qtInformation, qaOK);
        }
      }
    }

    return true;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::MakeDirectoryEx(AnsiString & Name, int OpMode)
{
  if (Connected())
  {
    assert(!(OpMode & OPM_SILENT) || !Name.IsEmpty());

    TRemoteProperties Properties = GUIConfiguration->NewDirectoryProperties;
    bool SaveSettings = false;

    if ((OpMode & OPM_SILENT) ||
        CreateDirectoryDialog(Name, &Properties, SaveSettings))
    {
      if (SaveSettings)
      {
        GUIConfiguration->NewDirectoryProperties = Properties;
      }
      FTerminal->CreateDirectory(Name, &Properties);
      return 1;
    }
    else
    {
      Name = "";
      return -1;
    }
  }
  else
  {
    Name = "";
    return -1;
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::DeleteFilesEx(TList * PanelItems, int OpMode)
{
  if (Connected())
  {
    FFileList = CreateFileList(PanelItems, osRemote);
    FPanelItems = PanelItems;
    try
    {
      AnsiString Query;
      bool Recycle = FTerminal->SessionData->DeleteToRecycleBin &&
        !FTerminal->IsRecycledFile(FFileList->Strings[0]);
      if (PanelItems->Count > 1)
      {
        Query = FORMAT(GetMsg(Recycle ? RECYCLE_FILES_CONFIRM : DELETE_FILES_CONFIRM),
          (PanelItems->Count));
      }
      else
      {
        Query = FORMAT(GetMsg(Recycle ? RECYCLE_FILE_CONFIRM : DELETE_FILE_CONFIRM),
          (((TFarPanelItem *)PanelItems->Items[0])->FileName));
      }

      if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
        (MoreMessageDialog(Query, NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
      {
        FTerminal->DeleteFiles(FFileList);
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
    return true;
  }
  else if (SessionList())
  {
    if ((OpMode & OPM_SILENT) || !FarConfiguration->ConfirmDeleting ||
      (MoreMessageDialog(GetMsg(DELETE_SESSIONS_CONFIRM), NULL, qtConfirmation, qaOK | qaCancel) == qaOK))
    {
      for (int i = 0; i < PanelItems->Count; i++)
      {
        TFarPanelItem * PanelItem = (TFarPanelItem *)PanelItems->Items[i];
        assert(PanelItem);
        if (PanelItem->IsFile)
        {
          TSessionData * Data = (TSessionData *)PanelItem->UserData;
          assert(Data);
          Data->Remove();
          StoredSessions->Remove(Data);
          PanelItem->Selected = false;
        }
      }
    }
    return true;
  }
  else
  {
    return false;
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::GetFilesEx(TList * PanelItems, bool Move,
  AnsiString & DestPath, int OpMode)
{
  int Result;
  if (Connected())
  {
    // FAR WORKAROUND
    // is it?
    // Probable reason was that search result window displays files from several
    // directories and the plusin can hold data for one directory only
    if (OpMode & OPM_FIND)
    {
      throw Exception(GetMsg(VIEW_FROM_FIND_NOT_SUPPORTED));
    }

    FFileList = CreateFileList(PanelItems, osRemote);
    try
    {
      bool EditView = (OpMode & (OPM_EDIT | OPM_VIEW)) != 0;
      bool Confirmed =
        (OpMode & OPM_SILENT) &&
        (!EditView || FarConfiguration->EditorDownloadDefaultMode);

      TGUICopyParamType CopyParam = GUIConfiguration->DefaultCopyParam;
      if (EditView)
      {
        CopyParam.FileNameCase = ncNoChange;
        CopyParam.PreserveReadOnly = false;
        CopyParam.ResumeSupport = rsOff;
        // we have no way to give FAR back the modified filename, so make sure we
        // fail downloading file not valid on windows
        CopyParam.ReplaceInvalidChars = false;
        CopyParam.FileMask = "";
        CopyParam.ExcludeFileMask = TFileMasks();
      }

      // these parameters are known in advance
      int Params =
        FLAGMASK(Move, cpDelete);

      if (!Confirmed)
      {
        int CopyParamAttrs = Terminal->UsableCopyParamAttrs(Params).Download;
        int Options =
          FLAGMASK(EditView, coTempTransfer | coDisableNewerOnly);
        Confirmed = CopyDialog(false, Move, FFileList, DestPath,
          &CopyParam, Options, CopyParamAttrs);

        if (Confirmed && !EditView && CopyParam.Queue)
        {
          // these parameters are known only after transfer dialog
          Params |=
            FLAGMASK(CopyParam.QueueNoConfirmation, cpNoConfirmation) |
            FLAGMASK(CopyParam.NewerOnly, cpNewerOnly);
          FQueue->AddItem(new TDownloadQueueItem(FTerminal, FFileList,
            DestPath, &CopyParam, Params));
          Confirmed = false;
        }
      }

      if (Confirmed)
      {
        if ((FFileList->Count == 1) && (OpMode & OPM_EDIT))
        {
          FOriginalEditFile = IncludeTrailingBackslash(DestPath) +
            UnixExtractFileName(FFileList->Strings[0]);
          FLastEditFile = FOriginalEditFile;
          FLastEditCopyParam = CopyParam;
          FLastEditorID = -1;
        }
        else
        {
          FOriginalEditFile = "";
          FLastEditFile = "";
          FLastEditorID = -1;
        }

        FPanelItems = PanelItems;
        // these parameters are known only after transfer dialog
        Params |=
          FLAGMASK(EditView, cpTemporary) |
          FLAGMASK(CopyParam.NewerOnly, cpNewerOnly);
        FTerminal->CopyToLocal(FFileList, DestPath, &CopyParam, Params);
        Result = 1;
      }
      else
      {
        Result = -1;
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else if (SessionList())
  {
    if (StoredSessions->Count > 0 &&
        ExportSessions(PanelItems, Move, DestPath, OpMode))
    {
      Result = 1;
    }
    else
    {
      Result = -1;
    }
  }
  else
  {
    Result = -1;
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ExportSessions(TList * PanelItems, bool /*Move*/,
  AnsiString & DestPath, int OpMode)
{
  AnsiString Title = GetMsg(EXPORT_SESSION_TITLE);
  AnsiString Prompt;
  if (PanelItems->Count == 1)
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSION_PROMPT),
      (((TFarPanelItem *)PanelItems->Items[0])->FileName));
  }
  else
  {
    Prompt = FORMAT(GetMsg(EXPORT_SESSIONS_PROMPT), (PanelItems->Count));
  }

  bool Result = (OpMode & OPM_SILENT) ||
    FPlugin->InputBox(Title, Prompt, DestPath, 0, "Copy");

  if (Result)
  {
    TSessionData * Data;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      Data = (TSessionData *)((TFarPanelItem *)PanelItems->Items[i])->UserData;
      THierarchicalStorage * Storage = NULL;
      TSessionData * ExportData = NULL;
      try
      {
        ExportData = new TSessionData(Data->Name);
        ExportData->Assign(Data);
        ExportData->Modified = true;
        Storage = new TIniFileStorage(IncludeTrailingBackslash(DestPath) + ExportData->Name + ".ini");;
        if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, true))
        {
          ExportData->Save(Storage, false);
        }
      }
      __finally
      {
        delete Storage;
        delete ExportData;
      }

    }
  }
  return Result;
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::UploadFiles(bool Move, int OpMode, bool Edit)
{
  int Result = 1;
  bool Confirmed = (OpMode & OPM_SILENT);
  bool Ask = !Confirmed;

  TGUICopyParamType CopyParam;

  if (Edit)
  {
    CopyParam = FLastEditCopyParam;
    Confirmed = FarConfiguration->EditorUploadSameOptions;
    Ask = false;
  }
  else
  {
    CopyParam = GUIConfiguration->DefaultCopyParam;
  }

  AnsiString DestPath = FTerminal->CurrentDirectory;

  // these parameters are known in advance
  int Params =
    FLAGMASK(Move, cpDelete);

  if (!Confirmed)
  {
    int CopyParamAttrs = Terminal->UsableCopyParamAttrs(Params).Upload;
    // huerictics: do not ask for target directory when uploaded file
    // was downloaded in edit mode
    int Options =
      FLAGMASK(Edit, coTempTransfer) |
      FLAGMASK(Edit || !Terminal->IsCapable[fcNewerOnlyUpload], coDisableNewerOnly);
    Confirmed = CopyDialog(true, Move, FFileList, DestPath,
      &CopyParam, Options, CopyParamAttrs);

    if (Confirmed && !Edit && CopyParam.Queue)
    {
      // these parameters are known only after transfer dialog
      Params |=
        FLAGMASK(CopyParam.QueueNoConfirmation, cpNoConfirmation) |
        FLAGMASK(CopyParam.NewerOnly, cpNewerOnly);
      FQueue->AddItem(new TUploadQueueItem(FTerminal, FFileList,
        DestPath, &CopyParam, Params));
      Confirmed = false;
    }
  }

  if (Confirmed)
  {
    assert(!FNoProgressFinish);
    // it does not make sense to unselect file being uploaded from editor,
    // moreover we may upload the file under name that does not exist in
    // remote panel
    FNoProgressFinish = Edit;
    try
    {
      // these parameters are known only after transfer dialog
      Params |=
        FLAGMASK(!Ask, cpNoConfirmation) |
        FLAGMASK(Edit, cpTemporary) |
        FLAGMASK(CopyParam.NewerOnly, cpNewerOnly);
      FTerminal->CopyToRemote(FFileList, DestPath, &CopyParam, Params);
    }
    __finally
    {
      FNoProgressFinish = false;
    }
  }
  else
  {
    Result = -1;
  }
  return Result;
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::PutFilesEx(TList * PanelItems, bool Move, int OpMode)
{
  int Result;
  if (Connected())
  {
    FFileList = CreateFileList(PanelItems, osLocal);
    try
    {
      FPanelItems = PanelItems;

      // if file is saved under different name, FAR tries to upload original file,
      // but let's be robust and check for new name, in case it changes.
      // OMP_EDIT is set since 1.70 final, only.
      // When comparing, beware that one path may be long path and the other short
      // (since 1.70 alpha 6, DestPath in GetFiles is short path,
      // while current path in PutFiles is long path)
      if (FLAGCLEAR(OpMode, OPM_SILENT) && (FFileList->Count == 1) &&
          (CompareFileName(FFileList->Strings[0], FOriginalEditFile) ||
           CompareFileName(FFileList->Strings[0], FLastEditFile)))
      {
        // editor should be closed already
        assert(FLastEditorID < 0);

        if (FarConfiguration->EditorUploadOnSave)
        {
          // already uploaded from EE_SAVE
          Result = -1;
        }
        else
        {
          // just in case file was saved under different name
          FFileList->Strings[0] = FLastEditFile;

          FOriginalEditFile = "";
          FLastEditFile = "";

          Result = UploadFiles(Move, OpMode, true);
        }
      }
      else
      {
        Result = UploadFiles(Move, OpMode, false);
      }
    }
    __finally
    {
      FPanelItems = NULL;
      SAFE_DESTROY(FFileList);
    }
  }
  else if (SessionList())
  {
    if (!ImportSessions(PanelItems, Move, OpMode))
    {
      Result = -1;
    }
    else
    {
      Result = 1;
    }
  }
  else
  {
    Result = -1;
  }
  return Result;
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::ImportSessions(TList * PanelItems, bool /*Move*/,
  int OpMode)
{
  bool Result = (OpMode & OPM_SILENT) ||
    (MoreMessageDialog(GetMsg(IMPORT_SESSIONS_PROMPT), NULL,
      qtConfirmation, qaOK | qaCancel) == qaOK);

  if (Result)
  {
    AnsiString FileName;
    TFarPanelItem * PanelItem;
    for (int i = 0; i < PanelItems->Count; i++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[i];
      bool AnyData = false;
      FileName = PanelItem->FileName;
      if (PanelItem->IsFile)
      {
        THierarchicalStorage * Storage = NULL;
        try
        {
          Storage = new TIniFileStorage(IncludeTrailingBackslash(GetCurrentDir()) + FileName);
          if (Storage->OpenSubKey(Configuration->StoredSessionsSubKey, false) &&
              Storage->HasSubKeys())
          {
            AnyData = true;
            StoredSessions->Load(Storage, true);
            StoredSessions->Save();
          }
        }
        __finally
        {
          delete Storage;
        }
      }
      if (!AnyData)
      {
        throw Exception(FORMAT(GetMsg(IMPORT_SESSIONS_EMPTY), (FileName)));
      }
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateSelectedFileList(
  TOperationSide Side, TFarPanelInfo * PanelInfo)
{
  if (PanelInfo == NULL)
  {
    PanelInfo = this->PanelInfo;
  }

  TStrings * Result;
  if (PanelInfo->SelectedCount > 0)
  {
    Result = CreateFileList(PanelInfo->Items, Side, true,
      PanelInfo->CurrentDirectory);
  }
  else
  {
    TFarPanelItem * PanelItem = PanelInfo->FocusedItem;
    if (PanelItem->IsParentDirectory)
    {
      Result = NULL;
    }
    else
    {
      Result = new TStringList();
      assert((Side == osLocal) || PanelItem->UserData);
      AnsiString FileName = PanelItem->FileName;
      if (Side == osLocal)
      {
        FileName = IncludeTrailingBackslash(PanelInfo->CurrentDirectory) + FileName;
      }
      Result->AddObject(FileName, (TObject *)PanelItem->UserData);
    }
  }
  return Result;
}
//---------------------------------------------------------------------------
TStrings * __fastcall TWinSCPFileSystem::CreateFileList(TList * PanelItems,
  TOperationSide Side, bool SelectedOnly, AnsiString Directory, bool FileNameOnly,
  TStrings * AFileList)
{
  TStrings * FileList = (AFileList == NULL ? new TStringList() : AFileList);
  try
  {
    AnsiString FileName;
    TFarPanelItem * PanelItem;
    TObject * Data = NULL;
    for (int Index = 0; Index < PanelItems->Count; Index++)
    {
      PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
      assert(PanelItem);
      if (!SelectedOnly || PanelItem->Selected)
      {
        FileName = PanelItem->FileName;
        if (Side == osRemote)
        {
          Data = (TRemoteFile *)PanelItem->UserData;
          assert(Data);
        }
        if (Side == osLocal)
        {
          if (ExtractFilePath(FileName).IsEmpty())
          {
            if (!FileNameOnly)
            {
              if (Directory.IsEmpty())
              {
                Directory = GetCurrentDir();
              }
              FileName = IncludeTrailingBackslash(Directory) + FileName;
            }
          }
          else
          {
            if (FileNameOnly)
            {
              FileName = ExtractFileName(FileName);
            }
          }
        }
        FileList->AddObject(FileName, Data);
      }
    }
  }
  catch (...)
  {
    if (AFileList == NULL)
    {
      delete FileList;
    }
    throw;
  }
  return FileList;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::SaveSession()
{
  if (!FTerminal->SessionData->Name.IsEmpty())
  {
    FTerminal->SessionData->RemoteDirectory = FTerminal->CurrentDirectory;

    TSessionData * Data;
    Data = (TSessionData *)StoredSessions->FindByName(FTerminal->SessionData->Name);
    if (Data)
    {
      bool Changed = false;
      if (Terminal->SessionData->UpdateDirectories)
      {
        Data->RemoteDirectory = Terminal->SessionData->RemoteDirectory;
        Changed = true;
      }

      if (Changed)
      {
        StoredSessions->Save();
      }
    }
  }
}
//---------------------------------------------------------------------------
bool __fastcall TWinSCPFileSystem::Connect(TSessionData * Data)
{
  bool Result = false;
  assert(!FTerminal);
  FTerminal = new TTerminal();
  try
  {
    FTerminal->Configuration = Configuration;
    FTerminal->SessionData = Data;
    FTerminal->OnUpdateStatus = TerminalUpdateStatus;
    FTerminal->OnQueryUser = TerminalQueryUser;
    FTerminal->OnPromptUser = TerminalPromptUser;
    FTerminal->OnDisplayBanner = TerminalDisplayBanner;
    FTerminal->OnShowExtendedException = TerminalShowExtendedException;
    FTerminal->OnReadDirectory = TerminalReadDirectory;
    FTerminal->OnStartReadDirectory = TerminalStartReadDirectory;
    FTerminal->OnReadDirectoryProgress = TerminalReadDirectoryProgress;
    FTerminal->OnStdError = TerminalOnStdError;
    FTerminal->OnFinished = OperationFinished;
    FTerminal->OnProgress = OperationProgress;
    FTerminal->OnDeleteLocalFile = TerminalDeleteLocalFile;
    ConnectTerminal(FTerminal);

    FTerminal->OnClose = TerminalClose;

    assert(FQueue == NULL);
    FQueue = new TTerminalQueue(FTerminal, Configuration);
    FQueue->TransfersLimit = GUIConfiguration->QueueTransfersLimit;
    FQueue->OnQueryUser = TerminalQueryUser;
    FQueue->OnPromptUser = TerminalPromptUser;
    FQueue->OnShowExtendedException = TerminalShowExtendedException;
    FQueue->OnListUpdate = QueueListUpdate;
    FQueue->OnQueueItemUpdate = QueueItemUpdate;

    assert(FQueueStatus == NULL);
    FQueueStatus = FQueue->CreateStatus(NULL);

    // TODO: Create instance of TKeepaliveThread here, once its implementation
    // is complete

    Result = true;
  }
  catch(Exception &E)
  {
    FTerminal->DoShowExtendedException(&E);
    SAFE_DESTROY(FTerminal);
    delete FQueue;
    FQueue = NULL;
  }

  FSynchronisingBrowse = GUIConfiguration->SynchronizeBrowsing;

  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ConnectTerminal(TTerminal * Terminal)
{
  Terminal->Open();
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalClose(TObject * /*Sender*/)
{
  // noop
  // Plugin closure is now invoked from HandleException
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::LogAuthentication(
  TSecureShell * SecureShell, AnsiString Msg)
{
  assert(FAuthenticationLog != NULL);
  FAuthenticationLog->Add(Msg);
  TStringList * AuthenticationLogLines = new TStringList();
  try
  {
    int Width = 42;
    int Height = 11;
    FarWrapText(FAuthenticationLog->Text.TrimRight(), AuthenticationLogLines, Width);
    int Count;
    AnsiString Message;
    if (AuthenticationLogLines->Count == 0)
    {
      Message = AnsiString::StringOfChar(' ', Width) + "\n";
      Count = 1;
    }
    else
    {
      while (AuthenticationLogLines->Count > Height)
      {
        AuthenticationLogLines->Delete(0);
      }
      AuthenticationLogLines->Strings[0] =
        AuthenticationLogLines->Strings[0] + 
          AnsiString::StringOfChar(' ', Width - AuthenticationLogLines->Strings[0].Length());
      Message = AnsiReplaceStr(AuthenticationLogLines->Text, "\r", "");
      Count = AuthenticationLogLines->Count;
    }

    Message += AnsiString::StringOfChar('\n', Height - Count);
    
    FPlugin->Message(0, SecureShell->SessionData->SessionName, Message);
  }
  __finally
  {
    delete AuthenticationLogLines;
  }  
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalUpdateStatus(
  TSecureShell * SecureShell, bool Active)
{
  if (Active)
  {
    if (FAuthenticationLog == NULL)
    {
      FAuthenticationLog = new TStringList();
      FPlugin->SaveScreen(FAuthenticationSaveScreenHandle);
      FPlugin->ShowConsoleTitle(Terminal->SessionData->SessionName);
    }

    static const int ConnectionStatusStrings[] =
      { STATUS_CLOSED, STATUS_INITWINSOCK, STATUS_LOOKUPHOST, STATUS_CONNECT,
        STATUS_AUTHENTICATE, STATUS_AUTHENTICATED, STATUS_STARTUP,
        STATUS_OPEN_DIRECTORY, STATUS_READY };
    assert((SecureShell->Status >= 0) &&
      (SecureShell->Status < LENOF(ConnectionStatusStrings)));

    AnsiString Str = GetMsg(
      static_cast<int>(ConnectionStatusStrings[SecureShell->Status]));
    LogAuthentication(SecureShell, Str);
    FPlugin->UpdateConsoleTitle(Str);
  }
  else
  {
    if (FAuthenticationLog != NULL)
    {
      FPlugin->ClearConsoleTitle();
      FPlugin->RestoreScreen(FAuthenticationSaveScreenHandle);
      SAFE_DESTROY(FAuthenticationLog);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalOnStdError(TObject * Sender,
  TLogLineType /*Type*/, const AnsiString AddedLine)
{
  TTerminal * Terminal = dynamic_cast<TTerminal *>(Sender);
  assert(Terminal != NULL);
  if (Terminal->Status == sshAuthenticate)
  {
    LogAuthentication(Terminal, AddedLine);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalStartReadDirectory(TObject * /*Sender*/)
{
  if (!FNoProgress)
  {
    FPlugin->ShowConsoleTitle(GetMsg(READING_DIRECTORY_TITLE));
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalReadDirectoryProgress(
  TObject * /*Sender*/, int Progress, bool & Cancel)
{
  if (Progress < 0)
  {
    if (!FNoProgress && (Progress == -2))
    {
      MoreMessageDialog(GetMsg(DIRECTORY_READING_CANCELLED), NULL,
         qtWarning, qaOK);
    }
  }
  else
  {
    if (FPlugin->CheckForEsc())
    {
      Cancel = true;
    }

    if (!FNoProgress)
    {
      FPlugin->UpdateConsoleTitle(
        FORMAT("%s (%d)", (GetMsg(READING_DIRECTORY_TITLE), Progress)));
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalReadDirectory(TObject * /*Sender*/,
  bool /*ReloadOnly*/)
{
  if (!FNoProgress)
  {
    FPlugin->ClearConsoleTitle();
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalDeleteLocalFile(const AnsiString FileName)
{
  if (!RecursiveDeleteFile(FileName,
        FLAGSET(FPlugin->FarSystemSettings(), FSS_DELETETORECYCLEBIN)))
  {
    throw Exception(FORMAT(GetMsg(DELETE_LOCAL_FILE_ERROR), (FileName)));
  }
}
//---------------------------------------------------------------------------
int __fastcall TWinSCPFileSystem::MoreMessageDialog(AnsiString Str,
  TStrings * MoreMessages, TQueryType Type, int Answers, const TMessageParams * Params)
{
  TMessageParams AParams;

  if ((FProgressSaveScreenHandle != 0) ||
      (FSynchronizationSaveScreenHandle != 0))
  {
    if (Params != NULL)
    {
      AParams = *Params;
    }
    Params = &AParams;
    AParams.Flags |= FMSG_WARNING;
  }

  return WinSCPPlugin()->MoreMessageDialog(Str, MoreMessages, Type,
    Answers, Params);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalQueryUser(TObject * /*Sender*/,
  const AnsiString Query, TStrings * MoreMessages, int Answers,
  const TQueryParams * Params, int & Answer, TQueryType Type, void * /*Arg*/)
{
  TMessageParams AParams;
  AnsiString AQuery = Query;

  if (Params != NULL)
  {
    if (Params->Params & qpFatalAbort)
    {
      AQuery = FORMAT(GetMsg(WARN_FATAL_ERROR), (AQuery));
    }

    AParams.Aliases = Params->Aliases;
    AParams.AliasesCount = Params->AliasesCount;
    AParams.Params = Params->Params & (qpNeverAskAgainCheck | qpAllowContinueOnError);
    AParams.Timeout = Params->Timeout;
    AParams.TimeoutAnswer = Params->TimeoutAnswer;
  }

  Answer = MoreMessageDialog(AQuery, MoreMessages, Type, Answers, &AParams);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalPromptUser(TSecureShell * /*SecureShell*/,
  AnsiString Prompt, TPromptKind Kind, AnsiString & Response, bool & Result,
  void * /*Arg*/)
{
  if (Kind == pkPrompt)
  {
    AnsiString Caption = ::CutToChar(Prompt, '|', true);
    if (Prompt.IsEmpty())
    {
      Prompt = Caption;
      Caption = "";
    }

    Result = FPlugin->InputBox(Caption, Prompt, Response, 0);
  }
  else
  {
    Result = PasswordDialog(Prompt, Kind, Response);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalDisplayBanner(
  TSecureShell * /*SecureShell*/, AnsiString SessionName,
  const AnsiString & Banner, bool & NeverShowAgain, int Options)
{
  BannerDialog(SessionName, Banner, NeverShowAgain, Options);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::TerminalShowExtendedException(
  TSecureShell * /*SecureShell*/, Exception * E, void * /*Arg*/)
{
  WinSCPPlugin()->ShowExtendedException(E);
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationProgress(
  TFileOperationProgressType & ProgressData, TCancelStatus & /*Cancel*/)
{
  if (FNoProgress)
  {
    return;
  }

  bool First = false;
  if (ProgressData.InProgress && !FProgressSaveScreenHandle)
  {
    FPlugin->SaveScreen(FProgressSaveScreenHandle);
    First = true;
  }

  // operation is finished (or terminated), so we hide progress form
  if (!ProgressData.InProgress && FProgressSaveScreenHandle)
  {
    FPlugin->RestoreScreen(FProgressSaveScreenHandle);
    FPlugin->ClearConsoleTitle();
  }
  else
  {
    ShowOperationProgress(ProgressData, First);
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::OperationFinished(TFileOperation Operation,
  TOperationSide Side, bool /*Temp*/, const AnsiString FileName, bool Success,
  bool & /*DisconnectWhenComplete*/)
{
  USEDPARAM(Side);

  if ((Operation != foCalculateSize) &&
      (FSynchronizationSaveScreenHandle == 0) &&
      !FNoProgress && !FNoProgressFinish)
  {
    TFarPanelItem * PanelItem = NULL;;

    if (!FPanelItems)
    {
      TList * PanelItems = PanelInfo->Items;
      for (int Index = 0; Index < PanelItems->Count; Index++)
      {
        if (((TFarPanelItem *)PanelItems->Items[Index])->FileName == FileName)
        {
          PanelItem = (TFarPanelItem *)PanelItems->Items[Index];
          break;
        }
      }
    }
    else
    {
      assert(FFileList);
      assert(FPanelItems->Count == FFileList->Count);
      int Index = FFileList->IndexOf(FileName);
      assert(Index >= 0);
      PanelItem = (TFarPanelItem *)FPanelItems->Items[Index];
    }

    assert(PanelItem->FileName ==
      ((Side == osLocal) ? ExtractFileName(FileName) : FileName));
    if (Success)
    {
      PanelItem->Selected = false;
    }
  }

  if (Success && (FSynchronizeController != NULL))
  {
    if (Operation == foCopy)
    {
      assert(Side == osLocal);
      FSynchronizeController->LogOperation(soUpload, FileName);
    }
    else if (Operation == foDelete)
    {
      assert(Side == osRemote);
      FSynchronizeController->LogOperation(soDelete, FileName);
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ShowOperationProgress(
  TFileOperationProgressType & ProgressData, bool First)
{
  static unsigned long LastTicks;
  unsigned long Ticks = GetTickCount();
  if (Ticks - LastTicks > 500 || First)
  {
    LastTicks = Ticks;

    static const int ProgressWidth = 46;
    static const int Captions[] = {PROGRESS_COPY, PROGRESS_MOVE, PROGRESS_DELETE,
      PROGRESS_SETPROPERTIES, 0, 0, PROGRESS_CALCULATE_SIZE,
      PROGRESS_REMOTE_MOVE, PROGRESS_REMOTE_COPY };
    static AnsiString ProgressFileLabel;
    static AnsiString TargetDirLabel;
    static AnsiString TransferModeLabel;
    static AnsiString ResumeSupportLabel;
    static AnsiString StartTimeLabel;
    static AnsiString TimeElapsedLabel;
    static AnsiString BytesTransferedLabel;
    static AnsiString CPSLabel;
    static AnsiString TimeLeftLabel;

    if (ProgressFileLabel.IsEmpty())
    {
      ProgressFileLabel = GetMsg(PROGRESS_FILE_LABEL);
      TargetDirLabel = GetMsg(TARGET_DIR_LABEL);
      TransferModeLabel = GetMsg(TRANSFER_MODE_LABEL);
      ResumeSupportLabel = GetMsg(RESUME_SUPPORT_LABEL);
      StartTimeLabel = GetMsg(START_TIME_LABEL);
      TimeElapsedLabel = GetMsg(TIME_ELAPSED_LABEL);
      BytesTransferedLabel = GetMsg(BYTES_TRANSFERED_LABEL);
      CPSLabel = GetMsg(CPS_LABEL);
      TimeLeftLabel = GetMsg(TIME_LEFT_LABEL);
    }

    int Progress;

    bool TransferOperation =
      ((ProgressData.Operation == foCopy) || (ProgressData.Operation == foMove));

    AnsiString Message;
    AnsiString Title = GetMsg(Captions[(int)ProgressData.Operation - 1]);
    AnsiString ProgressBar;
    Progress = ProgressData.OverallProgress();
    ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * Progress / 100);
    ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
    AnsiString FileName = ProgressData.FileName;
    // for upload from temporary directory,
    // do not show source directory
    if (TransferOperation && (ProgressData.Side == osLocal) && ProgressData.Temp)
    {
      FileName = ExtractFileName(FileName);
    }
    Message = ProgressFileLabel + MinimizeName(FileName,
      ProgressWidth - ProgressFileLabel.Length(), ProgressData.Side == osRemote) + "\n";
    // for downloads to temporary directory,
    // do not show target directory
    if (TransferOperation && !((ProgressData.Side == osRemote) && ProgressData.Temp))
    {
      Message += TargetDirLabel + MinimizeName(ProgressData.Directory,
        ProgressWidth - TargetDirLabel.Length(), ProgressData.Side == osLocal) + "\n";
    }
    Message += ProgressBar + "\n";
    if (TransferOperation)
    {
      Message += "\1\n";
      AnsiString StatusLine;
      AnsiString Value;
      Value = GetMsg(ProgressData.AsciiTransfer ? TRANSFER_ASCII : TRANSFER_BINARY);
      StatusLine = TransferModeLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - TransferModeLabel.Length() - Value.Length()) +
        Value + "  ";
      switch (ProgressData.ResumeStatus) {
        case rsEnabled: Value = GetMsg(RESUME_ENABLED); break;
        case rsDisabled: Value = GetMsg(RESUME_DISABLED); break;
        case rsNotAvailable: Value = GetMsg(RESUME_NOT_AVAILABLE); break;
        default: assert(false);
      }
      StatusLine = StatusLine + ResumeSupportLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        ResumeSupportLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = FormatDateTimeSpan(Configuration->TimeFormat, ProgressData.TimeElapsed());
      StatusLine = TimeElapsedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - TimeElapsedLabel.Length() - Value.Length()) +
        Value + "  ";

      AnsiString LabelText;
      if (ProgressData.TotalSizeSet)
      {
        Value = FormatDateTimeSpan(Configuration->TimeFormat, ProgressData.TotalTimeLeft());
        LabelText = TimeLeftLabel;
      }
      else
      {
        Value = ProgressData.StartTime.TimeString();
        LabelText = StartTimeLabel;
      }
      StatusLine = StatusLine + LabelText +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        LabelText.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";

      Value = FormatBytes(ProgressData.TotalTransfered);
      StatusLine = BytesTransferedLabel +
        AnsiString::StringOfChar(' ', ProgressWidth / 2 - 1 - BytesTransferedLabel.Length() - Value.Length()) +
        Value + "  ";
      Value = FORMAT("%s/s", (FormatBytes(ProgressData.CPS())));
      StatusLine = StatusLine + CPSLabel +
        AnsiString::StringOfChar(' ', ProgressWidth - StatusLine.Length() -
        CPSLabel.Length() - Value.Length()) +
        Value;
      Message += StatusLine + "\n";
      int TransferProgress = ProgressData.TransferProgress();
      ProgressBar = AnsiString::StringOfChar('\xDB', ProgressWidth * TransferProgress / 100);
      ProgressBar += AnsiString::StringOfChar('\xB0', ProgressWidth - ProgressBar.Length());
      Message += ProgressBar + "\n";
    }
    FPlugin->Message(0, Title, Message);

    if (First)
    {
      FPlugin->ShowConsoleTitle(Title);
    }
    FPlugin->UpdateConsoleTitleProgress((short)ProgressData.OverallProgress());

    if (FPlugin->CheckForEsc())
    {
      CancelConfiguration(ProgressData);
    }
  }
}
//---------------------------------------------------------------------------
TTerminalQueueStatus * __fastcall TWinSCPFileSystem::ProcessQueue()
{
  TTerminalQueueStatus * Result = NULL;
  assert(FQueueStatus != NULL);

  if (FQueueStatusInvalidated || FQueueItemInvalidated)
  {
    if (FQueueStatusInvalidated)
    {
      TGuard Guard(FQueueStatusSection);

      FQueueStatusInvalidated = false;

      assert(FQueue != NULL);
      FQueueStatus = FQueue->CreateStatus(FQueueStatus);
      Result = FQueueStatus;
    }

    FQueueItemInvalidated = false;

    TQueueItemProxy * QueueItem;
    for (int Index = 0; Index < FQueueStatus->ActiveCount; Index++)
    {
      QueueItem = FQueueStatus->Items[Index];
      if ((bool)QueueItem->UserData)
      {
        QueueItem->Update();
        Result = FQueueStatus;
      }

      if (GUIConfiguration->QueueAutoPopup &&
          TQueueItem::IsUserActionStatus(QueueItem->Status))
      {
        QueueItem->ProcessUserAction();
      }
    }
  }

  if (FRefreshRemoteDirectory)
  {
    if ((Terminal != NULL) && Terminal->Active)
    {
      Terminal->RefreshDirectory();
      if (UpdatePanel())
      {
        RedrawPanel();
      }
    }
    FRefreshRemoteDirectory = false;
  }
  if (FRefreshLocalDirectory)
  {
    if (FPlugin->GetPanelFileSystem(true) == NULL)
    {
      if (UpdatePanel(false, true))
      {
        RedrawPanel(true);
      }
    }
    FRefreshLocalDirectory = false;
  }

  return Result;
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::QueueListUpdate(TTerminalQueue * Queue)
{
  if (FQueue == Queue)
  {
    FQueueStatusInvalidated = true;
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::QueueItemUpdate(TTerminalQueue * Queue,
  TQueueItem * Item)
{
  if (FQueue == Queue)
  {
    TGuard Guard(FQueueStatusSection);

    assert(FQueueStatus != NULL);

    TQueueItemProxy * QueueItem = FQueueStatus->FindByQueueItem(Item);

    if ((Item->Status == TQueueItem::qsDone) && (Terminal != NULL))
    {
      FRefreshLocalDirectory = (QueueItem == NULL) ||
        (!QueueItem->Info->ModifiedLocal.IsEmpty());
      FRefreshRemoteDirectory = (QueueItem == NULL) ||
        (!QueueItem->Info->ModifiedRemote.IsEmpty());
    }

    if (QueueItem != NULL)
    {
      QueueItem->UserData = (void*)true;
      FQueueItemInvalidated = true;
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::CancelConfiguration(TFileOperationProgressType & ProgressData)
{
  if (!ProgressData.Suspended)
  {
    ProgressData.Suspend();
    try
    {
      TCancelStatus ACancel;
      int Result;
      if (ProgressData.TransferingFile &&
          (ProgressData.TimeExpected() > GUIConfiguration->IgnoreCancelBeforeFinish))
      {
        Result = MoreMessageDialog(GetMsg(CANCEL_OPERATION_FATAL), NULL,
          qtWarning, qaYes | qaNo | qaCancel);
      }
      else
      {
        Result = MoreMessageDialog(GetMsg(CANCEL_OPERATION), NULL,
          qtConfirmation, qaOK | qaCancel);
      }
      switch (Result) {
        case qaYes:
          ACancel = csCancelTransfer; break;
        case qaOK:
        case qaNo:
          ACancel = csCancel; break;
        default:
          ACancel = csContinue; break;
      }

      if (ACancel > ProgressData.Cancel)
      {
        ProgressData.Cancel = ACancel;
      }
    }
    __finally
    {
      ProgressData.Resume();
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::UploadOnSave()
{
  if ((FLastEditorID < 0) || !FarConfiguration->EditorUploadOnSave)
  {
    assert(FLastEditorID >= 0);
    assert(FarConfiguration->EditorUploadOnSave);
  }
  else
  {
    EditorInfo Info;
    if (FPlugin->FarEditorControl(ECTL_GETINFO, &Info))
    {
      if ((FLastEditorID != Info.EditorID) ||
          FLastEditFile.IsEmpty())
      {
        assert(false);
      }
      else
      {
        // make sure this is reset before any dialog is shownn as it may cause recursion
        FEditorPendingSave = false;

        assert(FFileList == NULL);
        FFileList = new TStringList();
        try
        {
          // always upload under the most recent name
          FFileList->Add(FLastEditFile);
          UploadFiles(false, 0, true);
        }
        __finally
        {
          SAFE_DESTROY(FFileList);
        }
      }
    }
  }
}
//---------------------------------------------------------------------------
void __fastcall TWinSCPFileSystem::ProcessEditorEvent(int Event, void * /*Param*/)
{
  // EE_REDRAW is the first for optimalisation
  if (Event == EE_REDRAW)
  {
    if (FEditorPendingSave)
    {
      UploadOnSave();
    }
  }
  else if (Event == EE_READ)
  {
    if (!FLastEditFile.IsEmpty())
    {
      EditorInfo Info;
      if (FPlugin->FarEditorControl(ECTL_GETINFO, &Info) &&
          AnsiSameText(FLastEditFile, Info.FileName))
      {
        FLastEditorID = Info.EditorID;
        FEditorPendingSave = false;
      }
    }
  }
  else if (Event == EE_CLOSE)
  {
    if (FEditorPendingSave)
    {
      assert(false); // should not happen, but let's be robust
      UploadOnSave();
    }

    EditorInfo Info;
    if (FPlugin->FarEditorControl(ECTL_GETINFO, &Info) &&
        (FLastEditorID == Info.EditorID))
    {
      FLastEditorID = -1;
    }
  }
  else if (Event == EE_SAVE)
  {
    if (FLastEditorID >= 0)
    {
      EditorInfo Info;
      if (FPlugin->FarEditorControl(ECTL_GETINFO, &Info) &&
          (FLastEditorID == Info.EditorID))
      {
        // if the file is saved under different name ("save as"), we upload
        // the file back under that name
        FLastEditFile = Info.FileName;

        if (FarConfiguration->EditorUploadOnSave)
        {
          FEditorPendingSave = true;
        }
      }
    }
  }
}
