/*
 * pass.c - password and other goodies for ppdd utilities 
 *
 * Copyright 1998,1999,2000 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <sys/termios.h>

#include "ppddmount.h"


int check_access(char const *file, int fd, int fail) 
{
	char	*progname = "check_access";
	struct stat stats;
	int mode;
	int ok = 1;
	char *mess;
	char message[80];

	if (fail) {
		mess = "Error";
	} else {
		mess = "Warning";
	}

	if (fstat(fd,&stats)) {
		PPDDERROR(150)
		ok = 0;
	}

	if ((mode = stats.st_mode & 07777)) {
		sprintf(message,
	"%s: permissions on %s are %04o should be 0000\n",mess,file,mode);
		PPDDEMESS(151,message)
		ok = 0;
	}

	if (stats.st_uid) {
		sprintf(message,
	"%s: %s should be owned by root\n",mess,file);
		PPDDEMESS(152,message)
		ok = 0;
	}

	if (stats.st_gid) {
		sprintf(message,
	"%s: %s should be group root\n",file,file);
		PPDDEMESS(153,message)
		ok = 0;
	}
	return (ok);
}


void Blowfish_Encrypt_ecb(Blowfish_Key key, unsigned char* inbuf,
                                          unsigned char* outbuf, int length)
{
        int i;
        for (i = 0; i < length; i += 8)
        {
            Blowfish_Encrypt(inbuf+i,outbuf+i,key);
        }
}


void Blowfish_Decrypt_ecb(Blowfish_Key key, unsigned char* inbuf,
                                          unsigned char* outbuf, int length)
{
        int i;
        for (i = 0; i < length; i += 8)
        {
            Blowfish_Decrypt(inbuf+i,outbuf+i,key);
        }
}

int read_control_block(int fd, void* buffer)
{
	char	*progname = "read_control_block";

	lseek (fd, 0, SEEK_SET);
	if (read (fd, buffer, 1024) != 1024) {
		PPDDERROR(207)
		return(1);
	}
	return(0);
}

int write_md5_block(int fd, void* buffer)
{
	char	*progname = "write_md5_block";

	lseek (fd, CB5SEEK, SEEK_SET);
	if (write (fd, (buffer + CB5SEEK), CB5LENGTH) != CB5LENGTH) {
		PPDDERROR(208)
		return(1);
	}
	return(0);
}

int write_control_block(int fd, void* buffer)
{
	char	*progname = "write_control_block";

	lseek (fd, 0, SEEK_SET);
	if (write (fd, buffer, 1024) != 1024) {
		PPDDERROR(209)
		return(1);
	}
	return(0);
}

int make_crc (struct crypt_control_block *cblock, int check)
{
	int		ok = 1;
        unsigned long 	crc;

	crc = crc32((unsigned char*)cblock, CBXLENGTH);

	if (check == PPDD_CRC_CHECK) {
		ok = memcmp(cblock->crc, &crc, 4);
	} else {
		memcpy(cblock->crc, &crc, 4);
	}
	crc = 0;
	return(ok);
}


static void readoldpass(const char* ppfile, 
                        int *len1, unsigned char *passphrase1,
		  	int *len2, unsigned char *passphrase2)
{
	char	*progname = "readoldpass";
	FILE	*pfile;

	pfile = fopen (ppfile, "r");
	
	if (pfile == NULL) {
		PPDDERROR(551)
		*len1 = 0;
		*len2 = 0;
		return;
	}

	memset(passphrase1+8,0,120);
	memset(passphrase2+8,0,120);
	
	if (fgets(passphrase1+8,120,pfile)) {
		*len1 = strlen(passphrase1+8) - 1;
	} else {
		*len1 = 0;
	}

	if (fgets(passphrase2+8,120,pfile)) {
		*len2 = strlen(passphrase2+8) - 1;
	} else {
		*len2 = 0;
	}

	fclose(pfile);
}

void getoldpass(int *len1, unsigned char *passphrase1,
		  	int *len2, unsigned char *passphrase2)
{
	char	*pass;

	pass = getpass("Enter first pass phrase: ");
	*len1 = strlen(pass);
	if (*len1 > 120)
		*len1 = 120;
	memcpy(passphrase1+8, pass, *len1);
	memset(pass, 0, strlen(pass));
	pass = getpass("Enter second pass phrase: ");
	*len2 = strlen(pass);
	if (*len2 > 120)
		*len2 = 120;
	memcpy(passphrase2+8, pass, *len2);
	memset(pass, 0, strlen(pass));
}

static int getnewpass(int master, int *len1, unsigned char *passphrase1,
				  int *len2, unsigned char *passphrase2)
{
	int 	len;
	char	*pass;

	if (master) {
		pass = getpass("New first master pass phrase: ");
	} else {
		pass = getpass("New first working pass phrase: ");
	}
	len = strlen(pass);
	if (len == 0) 
		return -1;
	if (len < 4) {
	        fprintf(stderr, "Error: Pass phrases too short.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	if (len < 24) {
	        fprintf(stderr, "Warning: Pass phrase should be at least 24 characters long.\n");
	}
	if (len > 120) {
	        fprintf(stderr, "Warning: Pass phrase truncated.\n");
		len = 120;
	}
	memcpy(passphrase1+8, pass, len);
	memset(pass, 0, strlen(pass));
	pass = getpass("Reenter pass phrase: ");
	*len1 = strlen(pass);
	if (len1 == 0) 
		return -1;
	if (*len1 > 120) {
		*len1 = 120;
	}
	if (len != *len1) {
	        fprintf(stderr, "Error: Pass phrases are not the same.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	if (memcmp(passphrase1+8, pass, len)) {
	        fprintf(stderr, "Error: Pass phrases are not the same.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	if (master) {
		pass = getpass("New second master pass phrase: ");
	} else {
		pass = getpass("New second working pass phrase: ");
	}
	len = strlen(pass);
	if (len == 0) 
		return -1;
	if (len < 4) {
	        fprintf(stderr, "Error: Pass phrases too short.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	if (len < 24) {
	        fprintf(stderr, "Warning: Pass phrase should be at least 24 characters long.\n");
	}
	if (len > 120) {
	        fprintf(stderr, "Warning: Pass phrase truncated.\n");
		len = 120;
	}
	memcpy(passphrase2+8, pass, len);
	memset(pass, 0, strlen(pass));
	pass = getpass("Reenter pass phrase: ");
	*len2 = strlen(pass);
	if (len2 == 0) 
		return -1;
	if (*len2 > 120) {
		*len2 = 120;
	}
	if (len != *len2) {
	        fprintf(stderr, "Error: Pass phrases are not the same.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	if (memcmp(passphrase2+8, pass, len)) {
	        fprintf(stderr, "Error: Pass phrases are not the same.\n");
		memset(pass, 0, strlen(pass));
		return 0;
	}
	return 1;
}


static void make_key_16(unsigned char *key, unsigned char *iv,
					int *len1, unsigned char *passphrase1,
					int *len2, unsigned char *passphrase2)
{
	Blowfish_Key 		bkey;
	int			len,i,j,k;

/* copy the seed into the first part of the first pass phrase */

	memcpy(passphrase1, iv, 8);

	len = *len1 + 8;

/* limit first pass phrase to 112 chars */

	if (len > 112) 
		len = 112;

/* if it is more than 56 chars (max for blowfish) fold it */

	if (len > 56) {
		for(i = 56; i < len; i++) {
			passphrase1[i - 56] ^= passphrase1[i];
		}
		len = 56;
	}

/* make a key from the first pass phrase */

	Blowfish_ExpandUserKey(passphrase1, len, bkey);

/* encrypt the original seed as an iv for the cbc encrypt of pass phrase 2 */

	Blowfish_Encrypt(iv, passphrase2, bkey);

	len = *len2 + 8;

/* if pass phrase 2 is too short pad it with the original seed */

	if (len < 16) {
		memcpy(&passphrase2[len], iv, 8);
		len = 16;
	}

/* ensure that there is trailing padding too */

	memcpy(&passphrase2[len], iv, 8);

/* cbc encrypt pass phrase 2 */

	for(j = 8; j < len; j += 8)
	{
	    i = j - 8;
	    for(k = 0; k < 8; k++)
	    {
		passphrase2[j + k] ^= passphrase2[i + k];
	    }
	    Blowfish_Encrypt(&passphrase2[j], &passphrase2[j], bkey);
	}

/* use the last two encrypted cipherblocks as the final result key */

	memcpy(key, &passphrase2[len - 16], 16);

/* clear memory */

	memset(passphrase1, 0, sizeof(passphrase1));
	memset(passphrase2, 0, sizeof(passphrase2));
	memset(bkey, 0, sizeof(bkey));
	*len1 = 0;
	*len2 = 0;
	len = 0;
	i = 0;
	j = 0;
	k = 0;
}


void make_key(unsigned char *key, unsigned char *iv,
					int *len1, unsigned char *passphrase1,
					int *len2, unsigned char *passphrase2)
{
	char p1[120];
	char p2[120];
	unsigned char k1[16];
	unsigned char k2[16];
	int l1;
	int l2;

	l1 = *len1;
	l2 = *len2;
	memcpy(p1,passphrase1,120);
	memcpy(p2,passphrase2,120);
	make_key_16(k1,iv,&l1,p1,&l2,p2);	

	l1 = *len1;
	l2 = *len2;
	memcpy(p1,passphrase1,120);
	memcpy(p2,passphrase2,120);
	make_key_16(k2,iv,&l2,p2,&l1,p1);	

	memcpy(key,k1,16);
	memcpy(key+16,k2,16);

	memset(p1,0,120);
	memset(p2,0,120);
	memset(k1,0,16);
	memset(k2,0,16);
}


int set_new_keys(int fd, int ffd, struct ppdd_info *info, 
					struct crypt_control_block *cblock)
{
	char	*progname = "set_new_keys";
	int 	ok;

	ok = 1;
	memcpy(&info->file_version, &cblock->file_version,
						sizeof(unsigned short));
	memcpy (&info->keys, cblock->keys, sizeof(info->keys));
	if (ok) {
	        if (ioctl(fd, PPDD_SET_STATUS, info) < 0) {
		    PPDDERROR(202)
		    ioctl(fd, PPDD_CLR_FD, 0);
		    ok = 0;
		}
	}
	return (ok);
}

int checkpass(int ffd, 
		const char* ppfile, 
		const char* pass1, 
		const char* pass2, 
		struct ppdd_info* info,
		struct crypt_control_block* cblock,
		unsigned char* mkey)
{
	char		*progname = "checkpass";
	int		ok, len1, len2;
	unsigned char	passphrase1[136];
	unsigned char	passphrase2[136];
	unsigned char	iv[8];
	Blowfish_Key 	bkey;
	unsigned char	temp[PPDDMKSIZE];
	unsigned char	key[PPDDKEYSIZE];

	if (read_control_block(ffd, cblock)) {
		PPDDERROR(211)
		return(0);
	}

	memcpy (iv, cblock, 8);

	if (ppfile) {
		readoldpass(ppfile, &len1, passphrase1, &len2, passphrase2);
	} else {
		if (pass1) {
			memcpy(passphrase1+8,pass1,120);
			memcpy(passphrase2+8,pass2,120);
			len1 = strlen(pass1);
			len2 = strlen(pass2);
		} else {
			getoldpass(&len1, passphrase1, &len2, passphrase2);
		}
	}

	make_key(key, iv, &len1, passphrase1, &len2, passphrase2);
	Blowfish_ExpandUserKey(key, PPDDKEYSIZE, bkey);
	Blowfish_Decrypt_ecb(bkey, cblock->mkey, cblock->mkey, CBMLENGTH);

	ok = 1;
	if (make_crc(cblock,PPDD_CRC_CHECK)) {
	    memcpy(temp, cblock->mkey, PPDDMKSIZE);
	    Blowfish_ExpandUserKey(&cblock->mkey[4], PPDDKEYSIZE, bkey);
	    if (read_control_block(ffd, cblock)) {
		PPDDERROR(217)
		ok = 0;
	    } else {
		Blowfish_Decrypt_ecb(bkey, cblock->mkey, cblock->mkey, 
								CBMLENGTH);
		if (make_crc(cblock,PPDD_CRC_CHECK)) {
			fprintf(stderr,
		"Wrong pass phrase or uninitialised crypto file system.\n");
			ok = 0;
		} else {
		    if (mkey) memcpy(mkey, &temp[4], PPDDKEYSIZE);
		}
	    }
	} else {
	    if (mkey) memcpy(mkey, key, PPDDKEYSIZE);
	}
	memset(temp, 0, PPDDMKSIZE);
	memset(key, 0, PPDDKEYSIZE);
	memset(bkey, 0, sizeof(bkey));
	memset(passphrase1, 0, sizeof(passphrase1));
	memset(passphrase2, 0, sizeof(passphrase2));
        len1 = len2 = 0;
	return (ok);
}


int newpass(unsigned char* ukey,
	struct crypt_control_block* cblock, int master)
{
	int		ok, len1, len2;
	unsigned char	passphrase1[136];
	unsigned char	passphrase2[136];
	Blowfish_Key 	bkey;
	unsigned char	temp1[PPDDMKSIZE];
	unsigned char	temp2[PPDDMKSIZE];
	unsigned char	key[PPDDKEYSIZE];

	if (master < 0){
	    ok = 1;
	} else {
	    do {
		ok = getnewpass(master, &len1, passphrase1, &len2, passphrase2);
	    } while (ok == 0);
	    if (ok == -1) 
		ok = 0;
	}
	if (ok) {
		printf("Generating keys - please wait\n");
	}
	if (ok) {
	    if (master) {
		if (master > 0) {
	    	    make_key(key, cblock->iv, &len1, passphrase1, 
						&len2, passphrase2);
		    memcpy(ukey, key, PPDDKEYSIZE);
		}

		ok = random_fill_fast (cblock->mkey, PPDDMKSIZE);

	    } else {
		make_key(key, cblock->iv, &len1, passphrase1, 
						&len2, passphrase2);
		memcpy(ukey, &cblock->mkey[4], PPDDKEYSIZE);
		Blowfish_ExpandUserKey(key, PPDDKEYSIZE, bkey);
		ok = random_fill_fast (temp1, PPDDMKSIZE);
		memcpy(&temp1[4], &cblock->mkey[4], PPDDKEYSIZE);
		Blowfish_Encrypt_ecb(bkey, temp1, temp2, PPDDMKSIZE);
		Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
		Blowfish_Decrypt_ecb(bkey, temp2, cblock->mkey, PPDDMKSIZE);
	    }
	    ok = random_fill_fast(cblock->x4,sizeof(cblock->x4));
	    make_crc(cblock,PPDD_CRC_MAKE);
	}
	memset(temp1, 0, PPDDMKSIZE);
	memset(temp2, 0, PPDDMKSIZE);
	memset(bkey, 0, sizeof(bkey));
	memset(key, 0, PPDDKEYSIZE);
	memset(passphrase1, 0, sizeof(passphrase1));
	memset(passphrase2, 0, sizeof(passphrase2));
        len1 = len2 = 0;
	return (ok);
}


int random_fill_fast (void *block, int size)
{
	char		*progname = "random_fill_fast";
	int 		i, urfd;
	Blowfish_Key 	bkey;
	unsigned char 	seed[512];
	unsigned char 	ckey[3];
	time_t		key;
	unsigned char 	*pk;
	unsigned long 	xs[3];
 	fd_set 		rfds;
 	struct timeval 	tv;

/* open the fast random file */

	if ((urfd = open ("/dev/urandom", O_RDWR)) < 0) {
		PPDDERROR(201)
		return 0;
	}

	if (size & 7) {
	    if (read (urfd, block, size) != size) {
		PPDDERROR(203)
		close (urfd);
		return 0;
	    }
	    close (urfd);
	    return 1;
	}

/* get 512 bytes as seed */

	if (read (urfd, seed, 512) != 512) {
		PPDDERROR(204)
		close (urfd);
		return 0;
	}

/* this adds an element of unpredictability */

	key = time(NULL);

	Blowfish_ExpandUserKey(&key, sizeof(key), bkey);
	Blowfish_Encrypt_ecb(bkey, seed, seed, 512);

/* reset 512 bytes as seed */

	if (write (urfd, seed, 512) != 512) {
		PPDDERROR(205)
	        close (urfd);
		return 0;
	}

	pk = malloc(size);
	if (pk == NULL) return 1;

/* fill the scramble seed with random bytes */

	if (read (urfd, xs, sizeof(xs)) != sizeof(xs)) {
		PPDDERROR(206)
	        close (urfd);
		return 0;
	}

/* fill the block with random bytes */

	if (read (urfd, block, size) != size) {
		PPDDERROR(210)
	        close (urfd);
		return 0;
	}

/* scramble the block into *pk */

	ppdd_scramble((size >> 2), (unsigned long*)(block), (unsigned long*)pk,
				xs[0], xs[1], xs[2]); 


/* fill the block with random bytes again */

	if (read (urfd, block, size) != size) {
		PPDDERROR(211)
	        close (urfd);
		return 0;
	}

	close (urfd);

/* now some real random, but only a few bytes in case it stalls */

	if ((urfd = open ("/dev/random", O_RDONLY)) < 0) {
		PPDDERROR(212)
		return 0;
	}

	for(i=0;i<sizeof(ckey);) {

/* the real random device often waits before giving us a character
   so lets use this delay to mash up the random data we already have */

		key = time(NULL);
		tv.tv_usec = 10000 * (key % 100);
		tv.tv_sec = 0;

  		FD_ZERO(&rfds);
  		FD_SET(urfd, &rfds);
  		select(urfd+1, &rfds, NULL, NULL, &tv);
  		if (FD_ISSET(urfd, &rfds))
  		{
			if (read (urfd, ckey+i, 1) != 1) {
				PPDDERROR(213)
				return 0;
			}
			i++;
			Blowfish_ExpandUserKey(&tv.tv_usec,
						sizeof(tv.tv_usec), bkey);
		} else {
			Blowfish_Encrypt_ecb(bkey, block, block, size);
		}
	}

	close (urfd);

	Blowfish_ExpandUserKey(&key, sizeof(key), bkey);
	Blowfish_Encrypt_ecb(bkey, block, block, size);

/* add the results */

	for(i=0;i<size;i++) {
		((unsigned char*)block)[i] ^= ~pk[i];
	}

	memset(pk, 0, size);
	free(pk);

	memset(seed, 0, 512);
	memset(&key, 0, sizeof(key));
	memset(ckey, 0, sizeof(ckey));
	memset(bkey, 0, sizeof(bkey));

	return 1;
}


int random_fill_slow (void *block, int size)
{
	char		*progname = "random_fill_slow";
	int 		ok,done,i, rfd;
	unsigned char 	k,r,s;
	unsigned char 	*pk;
	fd_set 		rfds;
	struct termios 	otios;
	struct termios 	ntios;
	struct timeval 	tv;
	unsigned long 		xs[6];

/* we want to turn off echo so get the terminal flags */

	tcgetattr(0, &otios);
	tcgetattr(0, &ntios);

	ntios.c_lflag &= ~(ECHO|ICANON);

/* echo now turned off */

	tcsetattr(0, TCSANOW, &ntios);

/* we are going to use the Linux real random device  */

	if ((rfd = open ("/dev/random", O_RDONLY)) < 0) {
		PPDDERROR(214)
		return 0;
	}

/* whatever we were given in the block now gets scrambled */

	ok = random_fill_fast (xs, sizeof(xs));
	if (!ok) return 0;

	pk = malloc(size);
	if (pk == NULL) return 0;

	ppdd_scramble((size >> 2), (unsigned long*)(block), (unsigned long*)pk,
				xs[0], xs[1], xs[2]); 

	ppdd_scramble((size >> 2), (unsigned long*)pk, (unsigned long*)(block),
				xs[3], xs[4], xs[5]); 

	memset(pk, 0, size);
	memset(xs, 0, sizeof(xs));

	free(pk);

	printf("Generating keys: %d bytes needed.\n\n", size);
	printf("    Enter random characters on the keyboard.\n");
	printf("    If you get bored turn caps lock on and off a few times.\n\n");
	printf("    Don't hold the keys down.\n\n");

	pk = block;
	s = 0;
	done = 0;

	for (i=size;i > 0;) {
 	    FD_ZERO(&rfds);
	    FD_SET(0, &rfds);
	    FD_SET(rfd, &rfds);
	    select(rfd+1, &rfds, NULL, NULL, NULL);
	    if (FD_ISSET(rfd, &rfds)) {
	        if (read (rfd, &k, 1) != 1) {
		    PPDDERROR(215)
		    return 0;
		}

/* add the random byte we get on top of what we already have */

	       	*pk ^= ~k;
		done = 1;
	    }
	    if (FD_ISSET(0, &rfds)) {
	        if (read (0, &r, 1) != 1) {
		    PPDDERROR(216)
		    return 0;
		}

/* if the user holds a key down or if we haven't had a random
   byte generated in this time then don't step along */

		if (r != s) {
		    s = r;

/* step along the block - one step per character entered
   there will usually be several random characters generated
   in this time - if not then we don't step along yet */

		    if (done) {

/* use the input character itself to rotate the random byte */

			r &= 7;
			*pk = ((*pk << r) | (*pk >> (8 - r)));

		        pk++;
		        done = 0;
		    	if ((i & 63) == 0) {
                	    printf("    Keep going: only %d left to generate!\n",i);
		    	}
			i--;
		    }
		}
	    }
        }

        close(rfd);

        printf("\nDone! You can rest your fingers now.\n\n");

/* we now want to flush any typed ahead characters */

	tv.tv_sec = 2;
	tv.tv_usec = 0;

	done = 0;

	while (!done) {
		FD_ZERO(&rfds);
		FD_SET(0, &rfds);
		select(1, &rfds, NULL, NULL, &tv);
		done = 1;
		if (FD_ISSET(0, &rfds)) {
		    if (read (0, &r, 1) != 1) {
		        PPDDERROR(218)
			return 0;
		    }
		    done = 0;
		}
	}

/* and now reset the terminal as it was before */

	tcsetattr(0, TCSANOW, &otios);
	return 1;
}
