/*
 * ppddcsum.c - checks or generates checksums for ppdd
 *
 * Copyright 1999 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "ppddmount.h"

static char *progname;


static int ppddcsroot(int generate, const char *device, const char *file) 
{
	struct ppdd_info           	info;
	struct crypt_control_block 	cblock;
	Blowfish_Key               	bkey;
	int                        	fd, ffd, ok;


	sync();

	if ((fd = open (device, O_RDONLY)) < 0) {
		PPDDERROR(311)
		return 1;
	}

	if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(312)
		close(fd);
		return 1;
	}

	if (generate) {
		if ((ffd = open (info.name, O_RDWR)) < 0) {
			PPDDERROR(313)
			close(fd);
			return 1;
		}
		if (read(ffd,&cblock,1024) != 1024) {
			PPDDERROR(314)
			close(fd);
			close(ffd);
			return 1;
		}
	} else {
		if ((ffd = open (file, O_RDONLY)) < 0) {
			PPDDERROR(315)
			close(fd);
			return 1;
		}
		if (read(ffd,&cblock,1024) != 1024) {
			PPDDERROR(316)
			close(fd);
			close(ffd);
			return 1;
		}
		close(ffd);
		if ((ffd = open (info.name, O_RDWR)) < 0) {
			PPDDERROR(317)
			close(fd);
			return 1;
		}
	}

	ok = 1;

	if (ioctl(fd, PPDD_GET_MD5_KEY, &info) < 0) {
		PPDDERROR(318)
		ok = 0;
	}

	sync();

        close(fd);

	if (ok) {
		memcpy(cblock.keys, &info.keys, PPDDKEYSIZE);
		memset(&info,0,sizeof(info));
        	Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
        	Blowfish_Decrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);

		if (generate) {
			random_fill_fast(&cblock.x5,sizeof(cblock.x5));
			random_fill_fast(&cblock.x6,sizeof(cblock.x6));
			ok = checkcsum(0, 1, ffd, &cblock);
                	sync();
		} else {
	    		ok = checkcsum(0, 0, ffd, &cblock); 
		}
	}

	memset(&cblock,0,sizeof(cblock));
	memset(&info, 0, sizeof(info));
	memset(&bkey,0,sizeof(bkey));
        close(ffd);
	return 0;
}

static int ppddcsum(int plain, int generate, const char *file) 
{
	struct ppdd_info           	info;
	struct crypt_control_block 	cblock;
	Blowfish_Key               	bkey;
	int                        	fd, ok;
	unsigned int			mode;
	char				*genchk, *pltext;

	if (generate) {
		genchk = "Generating";
	} else {
		genchk = "Checking";
	}

	if (plain) {
		pltext = "plain and ciphertext checksums";
	} else {
		pltext = "ciphertext checksum";
	}

	printf("%s %s on %s\n",genchk,pltext,file);

	memset(&info, 0, sizeof(info));
	strncpy(info.name, file, PP_NAME_SIZE);
	info.name[PP_NAME_SIZE-1] = 0;

	mode = O_RDONLY;
	if (generate) mode = O_RDWR;

	if ((fd = open (file, mode)) < 0) {
		PPDDERROR(319)
		return 1;
	}

    	ok = checkpass(fd, NULL, NULL, NULL, &info, &cblock, NULL);

	if (ok) {
            Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
            Blowfish_Decrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);
	    ok = checkcsum(plain, generate, fd, &cblock); 
	}
	memset(&bkey, 0, sizeof(bkey));
	memset(&info, 0, sizeof(info));
	memset(&cblock, 0, sizeof(cblock));
	close(fd);
	return (1 - ok);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-g -p -gp] file|device\n\
  %s -r  ppdd_device file|device\n\
  %s -gr ppdd_device\n\n" ,progname, progname, progname);
	fprintf(stderr, "      -g generate new checksums\n");
	fprintf(stderr, "         (default = check)\n");
	fprintf(stderr, "      -p check/generate plaintext and ciphertext\n");
	fprintf(stderr, "         (default = ciphertext only\n");
	fprintf(stderr, "      -r use to check ppdd root filesystem\n\n");
	fprintf(stderr, "      Use this to check for unexpected changes\n");
	fprintf(stderr, "      in a ppdd filesystem.\n\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int generate = 0;
	int plain = 0;
	int root = 0;

	progname = "ppddcsum";
	ppdd_intro(progname);

	while ((c = getopt(argc,argv,"rpg")) != EOF) {
		switch (c) {
			case 'g':
                                generate = 1;
				break;
			case 'p':
                                plain = 1;
				break;
			case 'r':
                                root = 1;
				break;
			default:
				usage();
		}
	}

	if (root) {
 		if (plain) usage();
 		if (generate) {
			if (argc != (3)) usage();
			return (ppddcsroot(1,argv[argc-1],NULL));
		} else {
			if (argc != (4)) usage();
			return (ppddcsroot(0,argv[argc-2],argv[argc-1]));
		}
	} else {
		c = plain|generate;
		if (argc != (2 + c)) usage();
		return (ppddcsum(plain,generate,argv[argc-1]));
	}
	exit (1); 
}

