/*
 * setppdd.c - setup and control ppdd devices
 *
 * Copyright 1998,9 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/mount.h>
#include <sys/ioctl.h>

#include "ppddmount.h"


static int csum_all(int plain, int fd, struct ppdd_keys *keys, 
			unsigned char *pmd5, unsigned char *cmd5)
{
	char		*progname = "csum_all";
	unsigned char 	iblock[32768];
	int 		rlength, count, reply, iseek;
	long		dlength, mlength;
	struct MD5Context pmd5C;
	struct MD5Context cmd5C;

	printf("Starting calculating checksums\n");

	MD5Init(&pmd5C);
	MD5Init(&cmd5C);

	iseek = 1024;

	lseek (fd,iseek,SEEK_SET);

	count = 1024;
	dlength = 1024;
	mlength = 0;

	while (1)
	{
	        rlength = read (fd, iblock, sizeof(iblock));
		if (rlength == 0) {
		    MD5Final(pmd5, &pmd5C);
		    MD5Final(cmd5, &cmd5C);
		    return 1;
		}
		if (rlength < 0) {
			PPDDERROR(140)
			return 0;
		}
		MD5Update(&cmd5C, iblock, rlength);
		if (plain) {
	            reply = transfer_bf(keys, PPDDDECRYPT, iblock,
					iblock, rlength >> 9,
					(dlength >> 9));
		    if (reply) {
			PPDDERROR(141)
			return 0;
		    }
		    MD5Update(&pmd5C, iblock, rlength);
		}
		dlength += rlength;
		count--;
		if (count == 0) {
                        mlength += dlength >> 20;
                        dlength &= 0x0fffff;
			printf("%ld Mb checked so far\n", mlength);
			count = 1024;
		}
	}
	return 1;
}


int checkcsum(int plain, int generate, int fd, 
				struct crypt_control_block *cblock) 
{
	struct ppdd_keys 		keys;
	unsigned char			cmd5[16];
	unsigned char			pmd5[16];
	int                        	i, ok, reply;
	Blowfish_Key               	bkey;

	ok = 1;
	reply = 0;

	if (!generate){
	    if (!(cblock->flags & 2)) {
	        fprintf(stderr, "No ciphertext checksum in control block.\n");
	        ok = 0;
	    }
	    if (plain && (!(cblock->flags & 1))) {
	        fprintf(stderr, "No plaintext checksum in control block.\n");
	        ok = 0;
	    }
	    if (ok) {
	        fprintf(stderr, "Checksum this device? (Y/N): ");
	        do {	
		    reply = getc(stdin);
	        } while ((reply != 'Y') && (reply != 'y')
		    && (reply != 'N') && (reply != 'n'));
	        if ((reply == 'N') || (reply == 'n')) {
		    reply = 'Y';
	            ok = 0;
	        }
	    } else {
	        fprintf(stderr, "Ignore missing checksum? (Y/N): ");
	        do {	
		    reply = getc(stdin);
	        } while ((reply != 'Y') && (reply != 'y')
		    && (reply != 'N') && (reply != 'n'));
	    }	
	}
	if (ok) {
	    setup_bf((struct ppdd_ukeys*)cblock->keys, &keys);
	    ok = csum_all (plain, fd, &keys, pmd5, cmd5);
	}
	if (ok) {
	    if (generate) {
	        if (plain) { 
		    memcpy(cblock->pmd5,pmd5,16);
		} else {
		    random_fill_fast(&cblock->flags,20);
		}
		memcpy(cblock->cmd5,cmd5,16);
		for(i=0;i<16;i++) cblock->cmd5[i] ^= cblock->pmd5[i];
	        cblock->flags &= 0xf0;
	        cblock->flags |= 0x02;
	        cblock->flags |= 0x08;
	        cblock->flags |= plain;
	        cblock->flags |= plain << 2;
	        Blowfish_ExpandUserKey(cblock->keys, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, &cblock->flags,
						&cblock->flags, CB5LENGTH);
	        write_md5_block(fd, cblock);
                sync();
	    } else {
	        for(i=0;i<16;i++) cblock->cmd5[i] ^= cblock->pmd5[i];
	        if (memcmp(cblock->cmd5,cmd5,16)) {
	       	    fprintf(stderr, "Error: ciphertext checksum failed.\n");
	            ok = 0;
	        } else {
	       	    fprintf(stderr, "Ciphertext checksum OK\n");
		}
	        if (plain) { 
                    if (memcmp(cblock->pmd5,pmd5,16)) {
	       	        fprintf(stderr, "Error: plaintext checksum failed.\n");
	                ok = 0;
	            } else {
	       	        fprintf(stderr, "Plaintext checksum OK\n");
		    }
		}
		if (!ok) {
	            fprintf(stderr, "Ignore checksum error? (Y/N): ");
	    	    do {	
			reply = getc(stdin);
	    	    } while ((reply != 'Y') && (reply != 'y')
			&& (reply != 'N') && (reply != 'n'));
		}
	    }
	}
	if (!ok && (reply == 'y' || reply == 'Y')) ok = 1;
	memset(&keys, 0, sizeof(keys));
	memset(&bkey, 0, sizeof(bkey));
	return (ok);
}


int find_ppdd_entry(const char *input, char *device, char *file, 
		char *pass1, char *pass2, int *check)
{
	char	*progname = "find_ppdd_entry";
	char 	line[257];
	char 	*linep;
	char 	dev[32];
	char 	fil[PP_NAME_SIZE];
        char 	*nl;
        char 	*sp;
	int 	f,len;
	FILE 	*fd;

	f = open("/etc/ppddtab",O_RDONLY);
	if (f < 0) {
		PPDDERROR(142)
		return 1;
	}

	check_access("/etc/ppddtab",f,0);

	fd = fdopen(f,"r");
	if (fd == NULL) {
		PPDDERROR(143)
		return 1;
	}

	dev[0] = 0;
	fil[0] = 0;

	for(;((input[0] == 0)||((strcmp(dev,input))&&(strcmp(fil,input))));) {
		nl = fgets(line,256,fd);	
		if (nl == NULL) {
			fclose(fd);
			return 1;
		}
		line[255] = 0;
		len = strlen(line);
		line[len-1] = ' ';
		line[len] = 0;

		linep = line;
		*check = 0;
		if (line[0] == 'Y' || line[0] == 'y') {
			*check = 1;
			linep++;
		}
		if (!(sp = strsep(&linep,"/"))) {
			fclose(fd);
			return 1;
		}

		if (!(sp = strsep(&linep," "))) {
			fclose(fd);
			return 1;
		}
		strncpy(dev+1,sp,32);
		dev[0] = '/';
		dev[31] = 0;
		if (!(sp = strsep(&linep,"/"))) {
			fclose(fd);
			return 1;
		}
		if (!(sp = strsep(&linep," "))) {
			fclose(fd);
			return 1;
		}
		strncpy(fil+1,sp,63);
		fil[0] = '/';
		fil[63] = 0;
		strcpy(device,dev);
		strcpy(file,fil);
		if (!(sp = strsep(&linep,"\""))) {
			pass1[0] = 0;
			pass2[0] = 0;
		} else {
			if (!(sp = strsep(&linep,"\""))) {
				pass1[0] = 0;
				pass2[0] = 0;
			} else {
				strncpy(pass1,sp,120);
				pass1[119] = 0;
				if (!(sp = strsep(&linep,"\""))) {
					pass2[0] = 0;
				} else {
					if (!(sp = strsep(&linep,"\""))) {
						pass2[0] = 0;
					} else {
						strncpy(pass2,sp,120);
						pass2[119] = 0;
					}
				}
			}	
			if ((input[0] == 0) && (pass1[0]) && (pass2[0])) {
				printf("Connecting %s to %s\n",dev,file);
				set_ppdd(dev,file,pass1,pass2,NULL,*check,0);
			}
		}
	}

	memset(line, 0, sizeof(line));
	fclose(fd);
	return(0);
}

int set_ppdd(const char *device, 
		const char *file, 
		const char *pass1, 
		const char *pass2, 
		const char *ppfile,
		int check,
		int blocksize)
{
	char				*progname = "set_ppdd";
	struct ppdd_info 		info;
	struct crypt_control_block 	cblock;
	Blowfish_Key               	bkey;
	int				fd, ffd, ok;
	char				filename[PP_NAME_SIZE];
        
 
	if (file[0] == '/') {
		strcpy (filename, file);
	} else {
		if (!getcwd (filename, PP_NAME_SIZE-1)) {
			PPDDERROR(148)
			return 1;
		}
		strcat (filename, "/");
		strcat (filename, file);
	}
	if ((ffd = open (filename, O_RDONLY)) < 0) {
		PPDDERROR(144)
		return 1;
	}

	check_access(filename, ffd, 0);

	if ((fd = open (device, O_RDONLY)) < 0) {
		PPDDERROR(145)
		close (ffd);
		return 1;
	}

	check_access(device, fd, 0);

        if (ioctl(fd, PPDD_SET_FD, ffd) < 0) {
		PPDDERROR(146)
		close (ffd);
		close (fd);
	  	return 1;
	}

        if (blocksize) {
            if (ioctl(fd, PPDD_SET_BSIZE, blocksize) < 0) {
		PPDDERROR(149)
		ioctl(fd, PPDD_CLR_FD, 0);
		close (ffd);
		close (fd);
	  	return 1;
	    }
	}
	if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(147)
		ioctl(fd, PPDD_CLR_FD, 0);
		close (ffd);
		close (fd);
	  	return 1;
	}

	if (info.driver_version != PP_VERSION) {
		fprintf(stderr,"Warning: kernel driver is version %d \n\
       but I am version %d !!!\n\n", info.driver_version, PP_VERSION);
	}

	memset(&info, 0, sizeof(info));
	strncpy(info.name, filename, PP_NAME_SIZE);
	info.name[PP_NAME_SIZE-1] = 0;

	ok = checkpass(ffd, ppfile, pass1, pass2, &info, &cblock, NULL);

	if (ok && check) {
            Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
            Blowfish_Decrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);
 	    ok = checkcsum(0, 0, ffd, &cblock); 
	}
	if (ok) ok = set_new_keys(fd, ffd, &info, &cblock);

	close(ffd);

	if (ok)  {
	    if ((ffd = open(filename, O_RDWR)) > -1) {
	       	cblock.flags &= 0xf0;
	       	cblock.flags |= check << 3;
	        Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);
	        write_md5_block(ffd, &cblock);
                sync();
		close(ffd);
	    }
	}
	if (!ok) {
		ioctl(fd, PPDD_CLR_FD, 0);
	}
	memset(&info, 0, sizeof(info));
	memset(&cblock, 0, sizeof(cblock));
	memset(&bkey, 0, sizeof(bkey));
	close(fd);
        sync();
	return (1 - ok);
}
