/*
 * ppddcsum.c - checks or generates checksums for ppdd
 *
 * Copyright 1999,2002 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "ppddmount.h"

static char *progname;


static int ppddcsum(int plain, int generate, int tab, const char *file)
{
	struct crypt_control_block 	cblock;
	Blowfish_Key               	bkey;
	int                        	pfd, fd, ok;
	unsigned int			mode;
	char				*genchk, *pltext;
        char                            pass1[128];
        char                            pass2[128];

	if (generate) {
		genchk = "Generating";
	} else {
		genchk = "Verifying";
	}

	if (plain) {
		pltext = "plain and ciphertext checksums";
	} else {
		pltext = "ciphertext checksum";
	}

	printf("%s %s on %s\n",genchk,pltext,file);

	mode = O_RDONLY;
	if (generate) mode = O_RDWR;

	if ((fd = open (file, mode)) < 0) {
		PPDDERROR(319)
		return 1;
	}
        if (tab) {
	   if ((pfd = open ("/etc/ppddtab", O_RDONLY)) < 0) {
		PPDDERROR(320)
		return 1;
	   }
           ok = find_ppdd_entry(pfd, NULL, file, pass1, pass2, NULL);
           if (ok) {
              ok = checkpass(fd, NULL, pass1, pass2, &cblock, NULL, NULL);
           }
           memset(pass1,0,128);
           memset(pass2,0,128);
        } else {
           ok = checkpass(fd, NULL, NULL, NULL, &cblock, NULL, NULL);
        }
	if (ok) {
            Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
            Blowfish_Decrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);
	    ok = checkcsum(plain, generate, fd, tab, &cblock, file); 
	}
	memset(&bkey, 0, sizeof(bkey));
	memset(&cblock, 0, sizeof(cblock));
	close(fd);
	return (1 - ok);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-g -p -gp -gt -pt -gpt] file|device\n\n" , progname);
	fprintf(stderr, "      -g generate new checksums\n");
	fprintf(stderr, "         (default = check)\n");
	fprintf(stderr, "      -p check/generate plaintext and ciphertext\n");
	fprintf(stderr, "         (default = ciphertext only\n");
	fprintf(stderr, "      -t take pass phrase from /etc/ppddtab\n");
	fprintf(stderr, "      Use this to check for unexpected changes\n");
	fprintf(stderr, "      in a ppdd filesystem.\n\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int generate = 0;
	int plain = 0;
	int tab = 0;

	progname = "ppddcsum";
	ppdd_intro(progname);

	while ((c = getopt(argc,argv,"pgt")) != EOF) {
		switch (c) {
			case 'g':
                                generate = 1;
				break;
			case 'p':
                                plain = 1;
				break;
			case 't':
                                tab = 1;
				break;
			default:
				usage();
		}
	}

	if (argc != (optind + 1)) usage();
	return (ppddcsum(plain,generate,tab,argv[argc-1]));
	exit (1); 
}

