/*
 * ppdecrypt.c - decrypt a ppdd filesystem without kernel support
 *
 * Copyright 1999,2002 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "ppddmount.h"

static char *progname;

static int decrypt_all(int pgp, int convent, int fd, int ffd, const char *fname,
			struct ppdd_keys *keys, unsigned char *opmd5,
			unsigned char *ocmd5, int checked)
{
	unsigned char 	iblock[32768];
	unsigned char 	oblock[32768];
	int 		i, rlength, wlength, count, reply, iseek, oseek;
	int 		check_p, check_c;
        unsigned long   size, dlength, mlength, block, tblocks;
        time_t          now, then;
	struct MD5Context pmd5C;
	struct MD5Context cmd5C;
	unsigned char	pmd5[16];
	unsigned char	cmd5[16];

        tblocks = getfilesize(fd);
        size = tblocks >> 11;
        printf("Decrypting %s (%ldMb)\n", fname, size);

        then = time(NULL);

	check_p = checked & 1;
	check_c = checked & 2;

	MD5Init(&pmd5C);
	MD5Init(&cmd5C);

	iseek = (1 + pgp) * 1024;
	oseek = (!convent) * 1024;

	lseek (fd,iseek,SEEK_SET);
	lseek (ffd,oseek,SEEK_SET);

	count = 1024;
	dlength = 1024;
	block = 2;
	mlength = 0;

	while (1)
	{
                if ((tblocks - block) < 65) {
                    rlength = read (fd, iblock, 512);
                } else {
                    rlength = read (fd, iblock, sizeof(iblock));
                }
		if (rlength == 0) {
		    if (checked) {
		        MD5Final(pmd5, &pmd5C);
		        MD5Final(cmd5, &cmd5C);
			for (i=0;i<16;i++) cmd5[i] ^= opmd5[i];
			i = 1;
		        if (memcmp(cmd5,ocmd5,16) && check_c) {
			    fprintf(stderr,
		"Error: ciphertext checksum failure\n");
			    i = 0;
			}
		        if (memcmp(pmd5,opmd5,16) && check_p) {
			    fprintf(stderr,
		"Error: plaintext checksum failure\n");
			    i = 0;
		        } 
		        mlength += dlength >> 20;
		        printf("Finished: %ld Mb decrypted\n", mlength);
			return (i);
		    } else {
			fprintf(stderr,"Warning: no checksum available\n");
		    }
		    mlength += dlength >> 20;
		    printf("Finished: %ld Mb decrypted\n", mlength);
		    return 1;
		}
		if (rlength < 0) {
			PPDDERROR(160)
			return 0;
		}
		if (check_c) MD5Update(&cmd5C, iblock, rlength);
	        reply = transfer_bf(keys, PPDDDECRYPT, iblock,
					oblock, rlength >> 9, block);
		if (reply) {
			PPDDERROR(161)
			return 0;
		}
		if (check_p) MD5Update(&pmd5C, oblock, rlength);
		wlength = write (ffd, oblock, rlength);
		if (wlength != rlength) {
			PPDDERROR(162)
			return 0;
		}
		dlength += wlength;
		block += wlength >> 9;
		count--;
		if (count == 0) {
                        count = 1024;
                        now = time(NULL);
                        if (now > (then + 59)) {
                                then = now;
                                mlength += dlength >> 20;
                                dlength &= 0x0fffff;
                                printf("%s: %ld of %ld Mb so far\n",
                                        fname, mlength, size);
				fflush(stdout);
                        }
		}
	}
	return 1;
}

static int pgp_decrypt(int size, int fd, void *iblock, void *oblock)
{
	int 		apipe[2];
	int 		bpipe[2];
	int 		cstatus, reply, max, len, wlen, rlen, left;
	pid_t 		pid;
 	fd_set 		rfds, wfds, *prfds, *pwfds;
 	struct timeval 	tv;
	char		phrase[128], *pass, message[80], pass2[128];

	if (pipe (apipe)) {
		PPDDERROR(163)
		return 0;
	}
	if (pipe (bpipe)) {
		PPDDERROR(164)
		return 0;
	}

	if (fd < 0) {
	    pass = getpass ("Enter the passphrase for \"backup\": ");
	    len = strlen(pass);
	    if (len == 0) return 0;
	    strcpy (phrase, pass);
	    memset(pass,0,len);
	} else {
	    pass = phrase;
            memset(phrase,0,128);
            memset(pass2,0,128);
            find_ppdd_entry(fd, "/backup", "/backup", phrase, pass2, NULL);
            memset(pass2,0,128);
	    len = strlen(phrase);
	    if (len == 0) return 0;
	}
	strcat (phrase, "\n");
	len++;

	setenv("PGPPASSFD","0",1);

	if ((pid = fork())) {
	    if (pid < 0) {
		PPDDERROR(165)
		return 0;
	    }
	    close (apipe[0]);
	    close (bpipe[1]);
	    max = apipe[1];
	    if (max < bpipe[0]) max = bpipe[0];
	    max++;
	    prfds = &rfds;
	    pwfds = &wfds;
	    left = size;
	    memcpy (&wlen,iblock,sizeof(wlen)); 
	    for(;;) {
		tv.tv_usec = 0;
		tv.tv_sec = 1;
  		FD_ZERO(&rfds);
  		FD_ZERO(&wfds);
  		FD_SET(bpipe[0], &rfds);
		FD_SET(apipe[1], &wfds);
		select(max, prfds, pwfds, NULL, &tv);
  		if (pwfds && (FD_ISSET(apipe[1], &wfds))) {
		    if (pass) {
		        if (len != write (apipe[1], phrase, len)) {
			    PPDDERROR(166)
			    return 0;
	    	        }
			memset(phrase,0,len);
		        pass = NULL;
		    } else {
		        if (wlen != write (apipe[1], iblock+4, wlen)) {
			    PPDDERROR(167)
			    return 0;
	    	        }
	    		close (apipe[1]);
		        pwfds = NULL;
		    }
		}
  		if (prfds && (FD_ISSET(bpipe[0], &rfds))) {
	    	    rlen = read (bpipe[0], oblock, left);
	    	    if (rlen < 0) {
			PPDDERROR(168)
		  	return 0;
	            }
		    left -= rlen;
		    if ((left < 1) || (rlen == 0)) {
	    	        close (bpipe[0]);
		        prfds = NULL;
		    }
		}
	    	if (waitpid (pid, &cstatus, WNOHANG)) {
	    	    reply = WEXITSTATUS(cstatus);
		    if (reply) {
			sprintf(message,"reply (%d) from pgp\n",reply);
			PPDDEMESS(169,message)
		    }
	    	    return (!reply);
		}
	    }
	} else {
	    close (apipe[1]);
	    close (bpipe[0]);
	    dup2(apipe[0],0);
	    close (apipe[0]);
	    dup2(bpipe[1],1);
	    close (bpipe[1]);
	    execlp("pgpv","pgpv","-f","+batchmode=1",NULL);
	    PPDDERROR(176)
	    exit (1);
	}
	return 0;
}

static int ppdecrypt(int pgp, int convent, int fd, const char *ifile, 
						   const char *ofile )
{
	struct crypt_control_block 	cblock;
	struct ppdd_keys 		keys;
	char                            pgpblock[2048];
	char                            pass1[128];
	char                            pass2[128];
	Blowfish_Key		   	bkey;
	int                        	ifd, ofd, ok;

	ofd = -1;

	if ((ifd = open (ifile, O_RDONLY)) < 0) {
		PPDDERROR(170)
		return 1;
	}

	check_access (ifile, ifd, 0);

	if (pgp) {
	    if (sizeof(pgpblock) != read (ifd, pgpblock, sizeof(pgpblock))) {
		PPDDERROR(171)
		return 1;
	    }
	    ok = pgp_decrypt(1024, fd, pgpblock, &cblock);
	} else {
	    if (fd < 0) {
	        ok = checkpass(ifd, NULL, NULL, NULL, &cblock, NULL, NULL);
	    } else {
                memset(pass1,0,128);
                memset(pass2,0,128);
                find_ppdd_entry(fd, NULL, ifile, pass1, pass2, NULL);
                ok = checkpass(ifd, NULL, pass1, pass2, &cblock, NULL, NULL);
                memset(pass1,0,128);
                memset(pass2,0,128);
            }
	}
	if (ok) {
	    setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
	    if (ofile) {
	        if ((ofd = open (ofile, O_WRONLY|O_TRUNC|O_CREAT, 0)) < 0) {
		    PPDDERROR(172)
	            ok = 0;
	        }
	    } else {
	        if ((ofd = open (ifile, O_WRONLY)) < 0) {
		    PPDDERROR(173)
	            ok = 0;
	        }
	    }
	}
	if (ok) {
	    if ((fchown (ofd, 0, 0))) {
		PPDDERROR(174)
	        ok = 0;
	    }
	}
	if (ok) {
	    if ((fchmod (ofd, 0))) {
		PPDDERROR(175)
	        ok = 0;
	    }
	}
	if (ok) {
	        Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
	        Blowfish_Decrypt_ecb(bkey, &cblock.flags, &cblock.flags,
								CB5LENGTH);
		ok = decrypt_all (pgp, convent, ifd, ofd, ifile, &keys,
				cblock.pmd5, cblock.cmd5, cblock.flags);
	}
	memset(&cblock, 0, sizeof(cblock));
	memset(&keys, 0, sizeof(keys));
	close(ifd);
	close(ofd);
	return (1 - ok);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-c] [ -f file_descriptor] input-file|input-device output-file|output-device \n\
  %s -p[c] [ -f file_descriptor] input-file|input-device output-file\n\
  %s -o [ -f file_descriptor] file|device\n\n", progname, progname,progname);
	fprintf(stderr, "      -c conventional decryption\n");
	fprintf(stderr, "         (first 1024 bytes preserved)\n");
	fprintf(stderr, "      -p use pgp decryption\n");
	fprintf(stderr, "      -f get pass phrase from an open file descriptor\n");
	fprintf(stderr, "      -o overwrite existing\n\n");
	fprintf(stderr, "      Use this to decrypt a ppdd filesystem\n");
	fprintf(stderr, "      without having ppdd support in the kernel\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int pgp = 0;
	int convent = 0;
	int overwrite = 0;
	int fd = -1;

	progname = "ppdecrypt";
	if (ppdd_intro(progname)) usage();

	while ((c = getopt(argc,argv,"copf:")) != EOF) {
		switch (c) {
			case 'c':
                                convent = 1;
				break;
			case 'o':
                                overwrite = 1;
				break;
			case 'p':
                                pgp = 1;
				break;
			case 'f':
                                if (optarg[1]) usage();
                                if (optarg[0] > '9') usage();
                                if (optarg[0] < '0') usage();
                                fd = optarg[0] - '0';
				break;
			default:
				usage();
		}
	}

	if (convent && overwrite) usage();
	if (pgp && overwrite) usage();
	if (overwrite) {
		if (argc != (optind + 1)) usage();
		return (ppdecrypt(pgp,convent,fd,argv[argc-1],NULL));
	} else {
		if (argc != (optind + 2)) usage();
		return (ppdecrypt(pgp,convent,fd,argv[argc-2],argv[argc-1]));
	}
	return 1;
}

