/*
  Licensed under the GNU Public License.
  Copyright (C) 1998-2000 by Thomas M. Vier, Jr. All Rights Reserved.

  wipe v1.2.2
  by Tom Vier <thomassr@erols.com> -- Tortured Souls Software

  http://users.erols.com/thomassr/zero/tortured_souls.html

  wipe is free software.
  See LICENSE for more information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define VERSION "1.2.2"
#define COPYRIGHT "Copyright (C) 1998-2000 Thomas M. Vier, Jr. All Rights Reserved."

#define EMAIL "thomassr@erols.com"

#define URL "http://wipe.sourceforge.net/"
#define FRESHMEAT "http://freshmeat.net/appindex/1999/03/09/920974872.html"

/* includes */
#include <config.h>
#include <require.h>

/* for GNU includes */
#define __USE_BSD
#define _BSD_SOURCE
#define __USE_POSIX
#define _POSIX_SOURCE

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <limits.h>
#include <time.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>

#include <errno.h>

/* conditional includes are in includes.h to same scroll time */
/* conditional defines are in defines.h */

#include <debug.h>

#include <includes.h>
#include <defines.h>
#include <type.h>
#include <struct.h>
#include <wipe.h>

#include <tiger.h>

/* defines */

/* exit codes */
#define OK 0
#define LOCK_FAILED 32

#define SUCCESS 0
#define FAILED 1
#define BAD_USAGE 2

/* 25K */
#define PERCENT_ENABLE_SIZE (25 << 10)

/* size of buffer for data to be hash */
#define HASHIN_SIZE 256

/* default sector size */
#define SECTOR_SIZE 512

/* default block dev buffer size -- 4MB */
#define CHUNKBUF_SIZE (4 << 20)

/* global vars */

const char version[] = VERSION;
const char email[] = EMAIL;
const char copyright[] = COPYRIGHT;

int exit_code = 0;           /* final exit code for main() */
char *program_name;          /* argv[0], what we're being called as */

extern int errno;

char prompt[4];              /* interactive prompt */

int fd;                      /* file descriptor */
int random_source = -1;      /* /dev/urandom */

unsigned char custom_byte;   /* custom overwrite byte */

char *hashin = NULL;         /* hash input buffer */
size_t hashin_size = HASHIN_SIZE;

char *buffer = NULL;         /* file buffer */
size_t buffer_size;          /* i hope sizeof(size_t) == sizeof(off_t) */

size_t fdoutsize;            /* intended file length or block dev length */
size_t chunk_size = CHUNKBUF_SIZE;

long sectors = 0;            /* block device sector count */
size_t sector_size = 0;

long int loop;               /* loop = devsize / bufsize */
size_t mod;                  /* mod = devsize % bufsize */

char pathname[PATH_MAX+1],   /* original pathname */
real_pathname[PATH_MAX+1];   /* current filename -- path is the same */
size_t oldpathlen = 0;       /* length of pathname[] for last file */

struct stat filestat;        /* file status */
mode_t mode;                 /* original file mode */

struct opt options;          /* wipe options */
struct complete percent;     /* stuff for showing completion percentage */

#ifdef BSDBLKDEV
struct disklabel *dlabel;     /* partition disklabel */
#endif

/* functions */

void help(int code, FILE *out)
{
  fprintf(out, ""
  "Wipe v%s\n"
  "by Tom Vier <%s>\n\n"

  "Usage is %s [options] [file-list]\n\n"

  "Options:         Default: %s -ZedVM -S512 -C%d -l1 -t%d -x4 -p1\n\n"

  "-h          --   help - display this screen\n"
  "-u          --   usage\n"
  "-c          --   show copyright and license\n"
  "-w          --   show warranty information\n"
  "-i          --   interactive - confirm each file\n"
  "-I          --   disable interaction\n"
  "-f          --   force file wiping and override interaction\n"
  "-r  and  -R --   recursion - traverse subdirectories\n"
  "-s          --   silent - disable percentage reporting\n"
  "-v          --   force verbose - always show percentage\n"
  "-V          --   verbose - show percentage if file is >= %dK\n"
  "-d  and  -D --   delete (-d) or keep (-D) after wiping\n"
  "-m  and  -M --   enable (-m) or disable (-M) memory mapped files\n"
  "-e          --   enable entropy - faster, maybe less secure\n"
  "-E          --   raw mode - use /dev/urandom directly\n"
  "-z          --   zero-out file - single pass of zeroes\n"
  "-Z          --   perform normal wipe passes\n"
  "-B(count)   --   block device sector count\n"
  "-S(size)    --   block device sector size\n"
  "-C(size)    --   chunk size - maximum file buffer size in kilobytes\n"
  "-t(size)    --   sets tiger hash input buffer size\n"
  "-l[0-3]     --   sets wipe secure level\n"
  "-x[1-32] -X --   sets number of random passes or disables\n"
  "-p(1-32)    --   wipe file x number of times\n"
  "-b(0-255)   --   overwrite file with this value byte\n"
  "", version, email, program_name, program_name, 
      (CHUNKBUF_SIZE >> 10), HASHIN_SIZE, (PERCENT_ENABLE_SIZE >> 10));
  exit(code);
}

void badopt(int c)
{
  fprintf(stderr, ""
	  "error: bad option: %c\n"
	  "Type \'%s -h\' for help.\n"
	  "", c, program_name);
  exit(BAD_USAGE);
}

void usage(int code, FILE *out)
{
  fprintf(out, ""
  "Wipe v%s\n"
  "by Tom Vier <%s>\n\n"
  "", version, email);

  fprintf(out, ""
  "usage: "
  "%s [-ucwsiIhHfFdDvVmMzZeErR] "
  "[-B(count)] [-S(size)] [-C(size)] [-t(size)] [-l[0-3]] [-x[1-32] -X] "
  "[-p(1-32)] [-b(0-255)] [files]\n"
  "", program_name);

  exit(code);
}

void show_copy(void)
{
  printf(""
  "Wipe v%s by Tom Vier <%s>\n"
  "%s\n\n"

  "Wipe homepage: %s\n"
  "Freshmeat Appindex: %s\n\n"

  "Licensed under the GNU Public License.\n"
  "Wipe comes with ABSOLUTELY NO WARRANTY; for details type \"%s -w\".\n"
  "", version, email, copyright, URL, FRESHMEAT, program_name);
}

void show_war(void)
{
  printf(""
  "Wipe v%s\n"
  "%s\n\n"

  "This program is free software; you can redistribute it and/or modify\n"
  "it under the terms of the GNU General Public License as published by\n"
  "the Free Software Foundation; either version 2 of the License, or\n"
  "(at your option) any later version.\n\n"

  "This program is distributed in the hope that it will be useful,\n"
  "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
  "GNU General Public License for more details.\n\n"

  "You should have received a copy of the GNU General Public License\n"
  "along with this program; if not, write to the Free Software\n"
  "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
  "", version, copyright);
}

/*
  percent_init -- initialize percent data
*/

void percent_init(void)
{
  int total_passes = 27;

  if (!options.verbose)
    return;

  if (options.random)
    /* two random loops per wipe loop, so multiply it by 2 */
    total_passes += (options.random_loop + 1) * 2;

  percent.complete = 0; /* initialize */

  if (options.verbose == 1 || buffer_size >= PERCENT_ENABLE_SIZE)
    percent.display = 1; /* enable percentage reporting */

  if (options.verbose == 2) /* force percentage reporting */
    {
      percent.display = 1; percent.complete = 0;
    }

  percent.increment = (100.0 / total_passes) / (options.wipe_multiply + 1);

  if (loop)
    {
      /* percent_update() should be called after each loop */
      percent.increment /= loop;
      /* percent reporting will have an error margin of + or - chunk_size */
    }

  percent_line_clear();
  printf(" \r%s: 0%%", pathname); fflush(stdout); /* display */
}

/*
  percent_update -- update and display completion
*/

void percent_update(void)
{
  if (!options.verbose)
    return;

#ifdef PAUSE
  /* this is a better place, since percent_update is always called */
  printf("Hit a key:");
  fgetc(stdin);
#endif

  if (!percent.display) return; /* display disabled */
  //  if (!percent.complete) percent_line_clear(); /* new file, clear line */

  percent.complete += percent.increment;

#ifdef SANITY
  if (percent.complete > 120)
    {
      fprintf(stderr, "\rpercent.complete == %d\n", (int) percent.complete);
      abort();
    }
#endif

  printf(" \r%s: %d%%", pathname, (int) (percent.complete)); fflush(stdout); /* display */
}

void percent_line_clear(void)
{
  char spaces[PATH_MAX+1];

  if (oldpathlen)
    {
      //printf("pathlen == %d\noldpathlen == %d\n", strnlen(pathname, PATH_MAX), oldpathlen);
      memset(spaces, (char) 0x20, sizeof(spaces));

      spaces[oldpathlen+1] = (char) 0x00;
      printf("\r%s       \r", spaces);
    }
  /*
  else
    printf("\r       \r");
  */

  oldpathlen = strnlen(pathname, PATH_MAX);
}

/*
  gettrn_char -- return random character
*/

char gettrn_char(void)
{
  char entropy; /* a yummy byte of entropy */

#ifdef SANITY
  if (random_source < 0)
    {
      printf("\rerror: gettrn_char called while random_source not available\n");
      abort();
    }
#endif

  read(random_source, &entropy, 1); /* mmmm....entropy */

  return(entropy);
}

/*
  init_hashin -- initializes the hashin buffer
		 call this before hashout_update()
*/

int init_hashin(void)
{
  static word hash_seed; /* secret xor seed */

  if (hashin == NULL)
    {
      if ((hashin = realloc(hashin, hashin_size)) == NULL)
	  return(errno);
    }

  if (!options.seclevel)
    /*
      get the seed first; it's important for it to be
      stronger than any given chunk of the hash input
      buffer, since it modulates the hash pool via xor.
    */
    read(random_source, &hash_seed, sizeof(hash_seed));

  read(random_source, hashin, hashin_size);

  return(0);
}

/*
  hashin_update -- updates input hash buffer using xor
                   only called by hashout_update()
*/

void hashin_update(void)
{
  static word hash_seed; /* secret xor seed */
  word *buf; /* pointer to hashin */
  int i; off_t hashin_off;

  buf = (word *) hashin;
  i = hashin_off = 0;

  if (options.seclevel)
    read(random_source, &hash_seed, sizeof(hash_seed)); /* get a new seed */
  else
    /*
      increment hash_seed by the modulus of itself and the value in an offset
      in the hash input buffer, based on the current hash_seed

      hopefully, for most files this won't repeat. other than that, the only
      issue i can think of is, if the hash input buffer where to be a certain
      size and hash_seed was initialized to a certain value, such that they
      would "resonate" (for lack of a better term) and lock onto a single 
      offset, thereby cycling between two sets of values for the whole hash
      input buffer, which would result in large repeating patterns in the
      random passes. though, i haven't thought about it too much.

      if resonance is a problem, incrementing the value located at the offset
      in the hash pool may be enough to avoid it.
    */
    hash_seed += hash_seed % buf[(int) (((float) hashin_size - 1.0) * hash_seed / (WORD_MAX+1.0))];

  /* xor a (processor) word at a time */
  while (i < (hashin_size / sizeof(hash_seed)))
    buf[i++] ^= hash_seed;
}

/*
  hashout_update -- this is where all the random pass dirty work is done
                    this will call hashin_update

		    this will fill the file buffer w/ random data
*/

void hashout_update(void)
{
  short int rbytes;
  char *rres, *rbuf;
  __u64 res[3], *buf, *bufend;

  buf = (__u64 *) buffer;
  bufend = buf + (buffer_size / sizeof(__u64));

  while (buf < bufend)
  {
    if ((buf+3) >= bufend) /* don't go too far */
      {
	rbytes = (int) ((void *) bufend - (void *) buf); /* the (void *) prevents scaling */
	break; /* not enough room */
      }

    hashin_update();
    tiger(hashin, hashin_size, buf); /* zero copy */

    buf += 3; /* tiger outputs three 64bit values */
  }

  if (rbytes)
    {
      rbuf = (char *) buf; rres = (char *) res;

      hashin_update();
      tiger(hashin, hashin_size, res);

      while (rbytes--)
	*rbuf++ = *rres++; /* yes, that increments the pointers */
    }
}

/*
  random_pass -- fills file with random bytes

  int passes  -- number of passes to perform
*/

void random_pass(int passes)
{
  int i;

#ifdef SANITY
  if (random_source < 0)
    {
      printf("\rerror: called random_pass(%d) while random_source == %d", passes, random_source);
      abort();
    }
#endif

  if (options.seclevel == 2)
    {
      if (init_hashin() != 0)
	{
	  fprintf(stderr, "\r%s: %s: cannot initialize hash buffer\n",
		  program_name, strerror(errno), errno);
	  exit(errno);
	}
    }

  while (passes--)
    {
      if (options.seclevel >= 3)
	{
	  if (init_hashin() != 0)
	    {
	      fprintf(stderr, "\r%s: %s: cannot initialize hash buffer\n",
		      program_name, strerror(errno));
	      exit(errno);
	    }
	}

      if (options.mmap && !S_ISBLK(mode))
	{
	  hashout_update(); /* fills buffer */
	  msync(buffer, buffer_size, MS_SYNC); /* sync data to disk */

#ifdef SYNC
	  fdatasync(fd); /* sync data to disk */
#endif
	  percent_update();
	}
      else
	{
	  for (i=0; i < loop; ++i)
	    {
	      hashout_update(); /* fill buffer */
	      write(fd, buffer, buffer_size);

	      percent_update();
	    }

	  if (mod)
	    {
	      hashout_update();
	      write(fd, buffer, mod);
	    }

#ifdef SYNC
	  fdatasync(fd); /* sync */
#endif
	  /* that's one pass */
	  lseek(fd, 0, SEEK_SET);
	}
    }
}

/*
  random_pass_raw -- fills file with random bytes
                     read directly from /dev/{u,}random
*/

void random_pass_raw(int passes)
{
  int i;

#ifdef SANITY
  if (random_source < 0)
    {
      printf("\rerror: called random_pass_raw(%d) while random_source == %d", passes, random_source);
      abort();
    }
#endif

 while (passes--)
   {
     if (options.mmap && !S_ISBLK(mode))
       {
	 read(random_source, buffer, buffer_size);
	 msync(buffer, buffer_size, MS_SYNC);

#ifdef SYNC
	 fdatasync(fd);
#endif
	 percent_update();
       }
     else
       {
	 for (i=0; i < loop; ++i)
	   {
	     read(random_source, buffer, buffer_size);
	     write(fd, buffer, buffer_size);
	     percent_update();
	   }

	 if (mod)
	   {
	     read(random_source, buffer, mod);
	     write(fd, buffer, mod);
	   }

#ifdef SYNC
	 fdatasync(fd);
#endif
	 lseek(fd, 0, SEEK_SET); /* reset file offset */
       }
   }
}

/*
  write_pass -- fill file with given byte

  char wipe_byte -- byte to fill file with
*/

void write_pass(const unsigned char wipe_byte)
{
  int i;

  memset(buffer, wipe_byte, buffer_size);

  if (options.mmap && !S_ISBLK(mode))
    {
      msync(buffer, buffer_size, MS_SYNC); /* sync data to disk */

#ifdef SYNC
      fdatasync(fd); /* sync data to disk */
#endif
      percent_update();
    }
  else
    {
      for (i=0; i < loop; ++i)
	{
	  write(fd, buffer, buffer_size);
	  percent_update();
	}

      if (mod)
	write(fd, buffer, mod);

#ifdef SYNC
      fdatasync(fd);
#endif

      lseek(fd, 0, SEEK_SET); /* reset read/write-offset to 0 */
    }
}

/*
  write3_pass            -- over writes file with given bytes

  char wipe_byte1        -- first byte
  char wipe_byte2        -- second byte
  char wipe_byte3        -- third byte
*/

void write3_pass(const unsigned char wipe_byte1, const unsigned char wipe_byte2, const unsigned char wipe_byte3)
{
  size_t offset = 0, length;
  unsigned char wipe_stream[3];
  int remaining, i;

  /* hopefully, smart compilers will just alias wipe_stream[] to wipe_byte1-3 */

  wipe_stream[0] = wipe_byte1;
  wipe_stream[1] = wipe_byte2;
  wipe_stream[2] = wipe_byte3;

  length = buffer_size;
  while (length % 3) --length;

  while (offset < length)
    {memcpy((buffer + offset), &wipe_stream, 3); offset += 3;}

  remaining = buffer_size % 3; /* see if anything's left */

  switch (remaining)
    {
    case 1:
      memcpy((buffer + offset), &wipe_stream, 1);
    case 2:
      memcpy((buffer + offset), &wipe_stream, 2);
    }

  if (options.mmap && !S_ISBLK(mode))
    {
      msync(buffer, buffer_size, MS_SYNC); /* sync data to disk */

#ifdef SYNC
      fdatasync(fd); /* sync data to disk */
#endif
      percent_update();
    }
  else
    {
      for (i=0; i < loop; ++i)
	{
	  write(fd, buffer, buffer_size);
	  percent_update();
	}

      if (mod)
	write(fd, buffer, mod);

#ifdef SYNC
      fdatasync(fd);
#endif

      lseek(fd, 0, SEEK_SET); /* rewind */
    }
}

/*
  rand_safe_char -- return a safe low-ASCII char
*/

char rand_safe_char(void)
{
  int i;

  /* Colin Plumb's string - taken from his program, sterilize */
  /* Possible file name characters - a very safe set. */
  const char nameset[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_+=%@#.";

  /*
    see the man page for rand(3) for more info on this algorithm.

    it was originally adapted from:

    Numerical Recipes in C: The Art of Scientific Computing
    (William  H.  Press, Brian P. Flannery, Saul A. Teukolsky,
    William T.  Vetterling;  New  York:  Cambridge  University
    Press,  1990 (1st ed, p. 207))
  */

  /* now pick a random char from nameset */
  /* gettrn_char must return unsigned char! */
  i = (int) ((float) (sizeof(nameset) - 1.0) * gettrn_char() / (UCHAR_MAX+1.0));

  return(nameset[i]);
}

/*
  rename_str -- fill str[] of size len with getrn_char()
*/

void rename_str(char str[], size_t len)
{
  int i=0;

  while (i < len)
    str[i++] = rand_safe_char();
}

/*
  wipename -- rename to random characters

  there's not much sense in doing the wipe passes
  on the filename. let's just give it a random name.
*/

int wipename(void)
{
  size_t len = NAME_MAX, path_len;
  char *base, dest_pathname[PATH_MAX+1];

  /* if they don't want random passes, we'll skip the rename */
  if (!options.random) return(0);

  /* make sure it's NUL-terminated */
  dest_pathname[sizeof(dest_pathname)-1] = 0;

  /* copy the path to the file */
  strncpy(dest_pathname, pathname, PATH_MAX);

  /* point *base at just after the path */
  if ((base = strrchr(dest_pathname, '/')) == NULL)
    base = dest_pathname;
  else
    ++base;

  /* truncate the path to get the length */
  memset(base, (char) 0x00, 1);
  path_len = strlen(dest_pathname); /* so we don't have to call strlen() > once */

  /*
    we try to use as long a filename as possible,
    but not longer than PATH_MAX.
  */
  while (len + path_len > PATH_MAX) --len;

  rename_str(base, len); /* get a random filename */

#ifdef NORENAME
  fprintf(stderr, "\rrenamed to %s\n", base);
#else
  if (rename(real_pathname, dest_pathname) == 0)
    {
      /* update global pathname */
      strncpy(real_pathname, dest_pathname, PATH_MAX);
    }
  else
    {
      if (!options.force && options.verbose)
	{
	  prompt[0] = 0; /* clear prompt */
	  while (prompt[0] != 'y' && prompt[0] != 'n')
	    {
	      printf("\r%s: %s: can't rename \'%s\'"
		     "\nContinue? ", program_name, strerror(errno), pathname);
	      fgets(prompt, sizeof(prompt), stdin);

	      if (prompt[0] == 'y') {exit_code=errno; return(errno);}
	      if (prompt[0] == 'n') abort();
	    }
	}
    }
#endif
  return(0);
}

/*
  destroy -- destroy the file
             calls subroutines wipe() and zero()
*/

int destroy(char file[])
{
  int code=0;

#ifdef SANITY
  if (strncmp(file, real_pathname, PATH_MAX) != 0)
    {
      printf("\r%s: destroy(): file[] != real_pathname[]\n"
	      "file[] == %s\nreal_pathname == %s\n",
	      program_name, file, real_pathname);
      abort();
    }

  if (!S_ISREG(mode))
    {
      printf("\r%s: destory(): not a regular file: %s\n",
	      program_name, file);
      abort();
    }
#endif

#ifdef DEBUG
  fprintf(stderr, "filename: %s\n", file);
#endif

#ifdef MMAP
  if ((fd = open(file, O_RDWR)) < 0)
#else
  if ((fd = open(file, O_WRONLY)) < 0)
#endif
    {
      fprintf(stderr, "\r%s: %s: cannot open file: %s\n", 
	      program_name, strerror(errno), file);
      exit_code = errno; return(errno);
    }

#ifndef SYNC
# ifndef IRIX
#  ifdef HAVE_FCNTL
  fcntl(fd, F_SETFL, O_SYNC);
#  endif
# endif
#endif

#ifdef LOCKABLE
# if (!(defined(DEBUG))||((defined(DEBUG)&&(defined(LOCKTEST)))))
  if (lock()) {close(fd); return(LOCK_FAILED);} /* lock file */
# endif
#endif

  if (options.delete)
    wipename(); /** rename **/

  code = wipe();  /**** wipe file ****/

  /* destroy file references, unless this is debug build */
  if (options.delete)
    {
#ifndef DEBUG
# ifdef HAVE_FTRUNCATE
      ftruncate(fd, 0);
# endif
#endif

      if (options.delete)
	wipename(); /** rename **/

#ifndef DEBUG
      if (unlink(real_pathname) != 0)
	{
	  fprintf(stderr, 
	    "\r%s: %s: Unable to unlink %s: file wiped, but not deleted!\n", 
		  program_name, strerror(errno), pathname);
	  exit_code = 1; close(fd); return(1);
	}
#endif
    }

  if (!options.delete)
    {
      /* restore file mode */
      fchmod(fd, (mode & 07777));
    }

#ifdef HAVE_FSYNC
  fsync(fd);
#endif

  close(fd);
  return(code);
}

/*
  destroy_blkdev -- destroy the device's data
                    calls subroutines wipeblkdev() and zero()
*/

int destroy_blkdev(char file[])
{
  int code=0;

  if ((fd = open(file, O_WRONLY)) < 0)
    {
      fprintf(stderr, "\r%s: %s: cannot open block device: %s\n", 
	      program_name, strerror(errno), file);
      exit_code = errno; return(errno);
    }

#ifndef SYNC
# ifndef IRIX
#  ifdef HAVE_FCNTL
  fcntl(fd, F_SETFL, O_SYNC);
#  endif
# endif
#endif

#ifdef BSDBLKDEV
  if (ioctl(fd, DIOCGDINFO, dlabel) != 0)
    {
      fprintf(stderr, "\r%s: %s: ioctl failed: can't get disklabel: %s\n",
	      program_name, strerror(errno), pathname);
      exit_code = errno; return(errno);
    }
#endif

  if (sectors == 0)
    {
#ifdef LINUXBLKDEV
      if (ioctl(fd, BLKGETSIZE, &sectors) != 0)
	{
	  fprintf(stderr, "\r%s: %s: ioctl failed: can't get sector count: %s\n", 
		  program_name, strerror(errno), pathname);
	  exit_code = errno; return(errno);
	}
#elif BSDBLKDEV
      sectors == dlabel->p_size;
#endif
    }

  if (sector_size == 0)
    {
#ifdef BSDBLKDEV
      sector_size = dlabel->d_secsize;
#else
      sector_size = SECTOR_SIZE;
#endif
    }

  if (sectors * sector_size == 0)
    {
      fprintf(stderr, "\r%s: %s: block device is zero length (sector size %ld, "
	      "sector count %ld)\n", program_name, pathname, sector_size, sectors);
      return(1);
    }

  fdoutsize = sectors * sector_size;

  if (fdoutsize > chunk_size)
    {
      loop = fdoutsize / chunk_size;
      mod = fdoutsize % chunk_size;

      buffer_size = chunk_size;
    }
  else
    {
      loop = 1; mod = 0;
      buffer_size = fdoutsize;
    }

#ifdef DEBUG
  fprintf(stderr, "\rzeroblkdev: fdoutsize == %ld\n", fdoutsize);
#endif

#ifdef SANITY
  if (strncmp(file, real_pathname, PATH_MAX) != 0)
    {
      printf("\r%s: destroy(): file[] != real_pathname[]\n"
	      "file[] == %s\nreal_pathname == %s\n",
	      program_name, file, real_pathname);
      abort();
    }

  if (!S_ISBLK(mode))
    {
      printf("\r%s: destory(): not a block device: %s\n",
	      program_name, file);
      abort();
    }
#endif

  code = wipeblkdev(); /**** wipe block device ****/

#ifdef HAVE_FSYNC
  fsync(fd);
#endif

  close(fd);
  return(code);
}

/*
  getfilebuf -- allocate file buffer - heap or mmaped
*/

void getfilebuf(void)
{
  if (options.mmap)
    {
      errno = 0; /* just in case */
      /* mem map shared, synchronous */
      buffer = mmap(NULL, buffer_size, PROT_READ|PROT_WRITE, MAP_SHARED|MS_SYNC, fd, 0);
      if (errno != 0 || buffer == NULL || buffer == MAP_FAILED)
	{
	  fprintf(stderr, "\r%s: %s: %s: cannot mmap %ld bytes",
		  program_name, pathname, strerror(errno), buffer_size);
	  exit_code = errno; //return(errno);
	}
#ifdef HAVE_MADVISE
      madvise(buffer, buffer_size, MADV_SEQUENTIAL);
#endif
    }
  else
    {
      if ((buffer = realloc(buffer, buffer_size)) == NULL)
	{
	  fprintf(stderr, "\r%s: %s: %s: cannot allocate %ld bytes for file buffer\n", 
		  program_name, pathname, strerror(errno), buffer_size);
	  exit_code = errno; //return(errno);
	}
    }
}

/*
  zeroblkdev -- zeroes out a block dev
*/

int zeroblkdev(void)
{
  int i;

  if ((buffer = realloc(buffer, buffer_size)) == NULL)
    {
      fprintf(stderr, "\r%s: %s: %s: cannot allocate %ld bytes for block device "
	      "buffer\n", program_name, pathname, strerror(errno), chunk_size);
      exit_code = errno; return(errno);
    }

  if (options.custom)
    {
#ifdef DEBUG
      fprintf(stderr, "\rusing custom byte: %d\n", custom_byte);
#endif
      memset(buffer, custom_byte, buffer_size);
    }
  else
    bzero(buffer, buffer_size);

  percent_init();

  /*
    eg
    fdoutsize = 5000
    chunk_size = 4096
    devsize / bufsize = 1
    devsize % bufsize = 904
   */

  for (i=0; i < loop; ++i)
    {
      write(fd, buffer, buffer_size);
      percent_update();
    }

  if (mod)
    write(fd, buffer, mod);

#ifdef SYNC
  fdatasync(fd); /* sync data to disk */
#endif

  if (percent.complete)
    {printf("\r%s: 100%%", pathname); fflush(stdout); percent.reported = 1;}

  return(0);
}

/*
  zero -- zeroes out a file. if options.custom, fills w/ custom byte
*/

int zero(void)
{
  int i;

  fdoutsize = filestat.st_size;

  if (fdoutsize == 0) return(0);

  if (!options.mmap)
    {
      if (fdoutsize > chunk_size)
	{
	  loop = fdoutsize / chunk_size;
	  mod = fdoutsize % chunk_size;

	  buffer_size = chunk_size;
	}
      else
	{
	  loop = 1; mod = 0;
	  buffer_size = fdoutsize;
	}
    }
  else
      loop = mod = 0;

  getfilebuf();

  if (options.custom)
    {
#ifdef DEBUG
      fprintf(stderr, "\rusing custom byte: %d\n", custom_byte);
#endif
      memset(buffer, custom_byte, buffer_size);
    }
  else
    bzero(buffer, buffer_size);

  if (options.mmap)
    {
#ifdef MMAP
      msync(buffer, buffer_size, MS_SYNC); /* sync data to disk */
#else
      write(fd, buffer, buffer_size);
#endif

#ifdef SYNC
      fdatasync(fd); /* sync data to disk */
#endif

#ifdef MMAP
      munmap(buffer, buffer_size);
#endif
    }
  else
    {
      percent_init();

      for (i=0; i < loop; ++i)
	{
	  write(fd, buffer, buffer_size);
	  percent_update();
	}

      if (mod)
	write(fd, buffer, mod);

#ifdef SYNC
      fdatasync(fd);
#endif
      lseek(fd, 0, SEEK_SET);
    }

  if (percent.complete)
    {printf("\r%s: 100%%", pathname); fflush(stdout); percent.reported = 1;}

  return(0);
}

/*
  wipepasses -- runs the actual passes
*/

void wipepasses(void)
{
  int i; /* for write pass 0x11 increments */
  int wipe_loop;

  wipe_loop = options.wipe_multiply + 1;

  while (wipe_loop--)
    {
      if (options.random)
	{
	  /* run four random passes */
	  if (options.hash)
	    random_pass(options.random_loop + 1);
	  else
	    random_pass_raw(options.random_loop + 1);
	}

      /*
	these patterns where taking from Peter Gutmann's 1996 USENIX article,
	Secure Deletion of Data from Magnetic and Solid-State Memory

	http://www.cs.auckland.ac.nz/~pgut001/secure_del.html

	thanx, peter!
      */

      /*
	comment format: 
	pass number -- binary pattern -- target encoding scheme
      */

      /* fifth pass -- 01 -- RLL(1,7) and MFM */
      write_pass(0x55);

      /* sixth pass -- 10 -- same */
      write_pass(0xaa);

      /* seventh pass -- 10010010 01001001 00100100 -- RLL(2,7) and MFM */
      write3_pass(0x92, 0x49, 0x24);

      /* eighth pass  -- 01001001 00100100 10010010 -- same */
      write3_pass(0x49, 0x24, 0x92);

      /* ninth pass   -- 00100100 10010010 01001001 -- same */
      write3_pass(0x24, 0x92, 0x49);

      /* tenth pass -- start 0x11 increment passes */
      for (i = 0x00; i <= 0xff; i += 0x11) write_pass(i);

      /* 26 -- RLL(2,7) and MFM passes, again */
      write3_pass(0x92, 0x49, 0x24);
      write3_pass(0x49, 0x24, 0x92);
      write3_pass(0x24, 0x92, 0x49);

      /* 29 -- 01101101 10110110 11011011 -- RLL(2,7) */
      write3_pass(0x6d, 0xb6, 0xdb);
      /* 10110110 11011011 01101101 */
      write3_pass(0xb6, 0xdb, 0x6d);
      /* 11011011 01101101 10110110 */
      write3_pass(0xdb, 0x6d, 0xb6);

      if (options.random)
	{
	  /* run four random passes */
	  if (options.hash)
	    random_pass(options.random_loop + 1);
	  else
	    random_pass_raw(options.random_loop + 1);
	}
    }

  /* done with passes */
}

/*
  wipe -- runs wipe passes on a given file
*/

int wipe(void)
{
  if (filestat.st_size == 0)
    return(0); /* no need to write anything */

  if (options.zero) return(zero()); /* cute, eh? */
  if (options.custom) return(zero()); /* zero() also does custom byte wipes */

  /*
     increase buffer size till we have enough space for 3 byte passes,
     so it doesn't have to truncate three byte passes
  */
  fdoutsize = filestat.st_size; /* init */

  /* make sure we cover all allocated blocks, first */
  while (fdoutsize % filestat.st_blksize) ++fdoutsize;
  while (fdoutsize % 3) ++fdoutsize; /* beast algorithm =P */

  if (!options.mmap)
    {
      if (fdoutsize > chunk_size)
	{
	  loop = fdoutsize / chunk_size;
	  mod = fdoutsize % chunk_size;

	  buffer_size = chunk_size;
	}
      else
	{
	  loop = 1; mod = 0;
	  buffer_size = fdoutsize;
	}
    }
  else
    loop = mod = 0;

  /* ******** YOU ARE HERE ******** */

  getfilebuf();

  percent_init();

  wipepasses(); /**** run the passes ****/

  /*
     sometimes a float isn't enough decimals to accurately hold
     percent.increment, so we explicitly print 100% completion.
     otherwise we might end up with (int) percent.complete < 100;
     probably 98 or 99.
  */

  if (percent.complete)
    {printf("\r%s: 100%%", pathname); fflush(stdout); percent.reported = 1;}

#ifdef MMAP
  if (options.mmap)
    munmap(buffer, buffer_size);
#endif

  return(0);
}

int wipeblkdev(void)
{
  if (options.zero) return(zeroblkdev());
  if (options.custom) return(zeroblkdev());

  getfilebuf();

  percent_init();

  wipepasses(); /**** run the passes ****/

  if (percent.complete)
    {printf("\r%s: 100%%", pathname); percent.reported = 1;}

  return(0);
}

/*
  lock -- create as good a lock as possible
          what a mess
*/

#ifdef LOCKABLE
int lock(void)
{
# ifndef HAVE_FCNTL
#  ifdef HAVE_LOCKF
  off_t lock_size;
#  endif
# endif

# ifdef HAVE_FCNTL
  struct flock file_lock;

  file_lock.l_type = F_WRLCK;
  file_lock.l_whence = 0;
  file_lock.l_start = 0;
  file_lock.l_len = 0;
# endif

# ifdef LOCKTEST
  fprintf(stderr, "attempting lock on %s...", pathname);
# endif

# ifdef HAVE_FCHMOD
  /* try for a mandatory lock */
  if (fchmod(fd, 02600) != 0)
    {
#  ifdef LOCKTEST
      fprintf(stderr, "mandatory lock failed\ntrying for advisory...");
#  endif
    }
# endif

# ifdef HAVE_FCNTL
  if (fcntl(fd, F_SETLKW, &file_lock) == 0)
    {
#  ifdef LOCKTEST
      fprintf(stderr, "got it\n");
#  endif
      return(0); /* lock successful */
    }

# elif HAVE_LOCKF
  lseek(fd, 0, SEEK_SET);
  lock_size = filestat.st_size;
  while (lock_size % filestat.st_blksize) ++lock_size;
  while (lock_size % 3) ++lock_size;

  if (lockf(fd, F_LOCK, lock_size) == 0)
    {
#  ifdef LOCKTEST
      fprintf(stderr, "got it\n");
#  endif
      return(0); /* lock successful */
    }

# elif HAVE_FLOCK
  if (flock(fd, LOCK_EX) == 0)
    {
#  ifdef LOCKTEST
      fprintf(stderr, "got it\n");
#  endif
      return(0); /* lock successful */
    }
# endif

  /* lock failed */
  if (options.force && !options.verbose)
    return(OK);

  prompt[0] = 0; /* clear prompt */
  while (prompt[0] != 'y' && prompt[0] != 'n')
    {
      printf("\r%s: lock failed: \'%s\'. Wipe anyway? ", program_name, pathname);
      fgets(prompt, sizeof(prompt), stdin);

      if (prompt[0] == 'y') return(OK);
      if (prompt[0] == 'n') return(LOCK_FAILED);
    }

  return(0);
}
#endif

int getfiledata(char *file)
{
  if (strlen(file) > PATH_MAX)
    return(ENAMETOOLONG);

  /* this is to make sure the pathname strings are NUL-terminated */
  pathname[sizeof(pathname)-1] = 0;
  real_pathname[sizeof(real_pathname)-1] = 0;

  /*
   * save the original pathname, so we can use it
   * when interacting with the user
   */

  /* copy pathname */
  strncpy(pathname, file, PATH_MAX);
  strncpy(real_pathname, file, PATH_MAX);

  /* get file inode data */
  /* we'll use lstat() so that it shows symlinks,
     instead of what they point to */
  if (lstat(file, &filestat) != 0)
    {
      fprintf(stderr, "\r%s: %s: %s\n", 
	      program_name, strerror(errno), file);
      exit_code = errno; return(errno);
    }

  memcpy(&mode, &filestat.st_mode, sizeof(mode));

  return(0);
}

/*
  drilldown -- drill down through a directory

  char str[]  --  directory
*/

int drilldown(const char str[])
{
  DIR *dir;
  struct dirent *entry;
  char *base, dirname[PATH_MAX+1];
  int len;

  /* copy str to dirname, so we can write to it */
  strncpy(dirname, str, sizeof(dirname));

#ifdef DEBUG
  fprintf(stderr, "checking %s\n", dirname);
#endif

  if ((dir = opendir(dirname)) == NULL)
    {
      fprintf(stderr, "\r%s: %s: cannot open directory\n", 
	      program_name, strerror(errno), dirname);
      exit_code = errno; return(errno);
    }

  if (chdir(dirname) != 0)
    {
      fprintf(stderr, "\r%s: %s: cannot enter directory\n", 
	      program_name, strerror(errno), dirname);
      closedir(dir); exit_code = errno; return(errno);
    }

  //  readdir(dir);  /* skip "."  */
  //  readdir(dir);  /* skip ".." */

  //  while ((entry = readdir(dir)) != NULL) checkfile(entry->d_name);

  len = sizeof(entry->d_name);

  while ((entry = readdir(dir)) != NULL)
    {
      if (strncmp(entry->d_name, ".", len) && strncmp(entry->d_name, "..", len) != 0)
	checkfile(entry->d_name);
    }

  if (chdir("..") != 0)
    {
      fprintf(stderr, "\r%s: %s: cannot enter directory: ..\n"
	              "\nWe're trapped!\n"
	              "terminating process...\n", program_name, strerror(errno));
      abort();
    }

  closedir(dir);

#ifdef DEBUG
  if (options.delete)
    fprintf(stderr, "\rwould have removed: %s\n", dirname);
#endif

#ifndef DEBUG
  if (options.delete)
    {
      /* make sure dirname doesn't end in a slash,
	 otherwise base will point to \0 */

      len = strlen(dirname); --len;
      base = (char *) ((dirname + len)); /* find the last char */

      /* loop it, in case of multiple slashes */
      while (strncmp((char *) base, "/", 1) == 0)
	{
	  dirname[len] = (char) 0;

	  len = strlen(dirname); --len;
	  base = (char *) ((dirname + len));
	}

      /* point *base at just after the path to the directory */
      if ((base = strrchr(dirname, '/')) == NULL)
	base = dirname;
      else
	++base;

      if (options.interactive)
	{
	  prompt[0] = 0; /* clear prompt */
	  while (prompt[0] != 'y' && prompt[0] != 'n')
	    {
	      printf("\r%s: remove directory \'%s\'? ",
		     program_name, dirname);
	      fgets(prompt, sizeof(prompt), stdin);

	      if (prompt[0] == 'y')
		{
		  if (rmdir(base) != 0)
		    {
		      fprintf(stderr, "\r%s: %s: unable to remove: %s\n",
			      program_name, strerror(errno), base);
		      exit_code = 1; return(1);
		    }
		}
	      if (prompt[0] == 'n') return(0); /* skip to next file */

	      /* else, incorrect response -- re-prompt */
	    } /* while prompt */
	}
      else
	{
	  if (rmdir(base) != 0)
	    {
	      fprintf(stderr, "\r%s: %s: unable to remove directory: %s\n",
		      program_name, strerror(errno), base);
	      exit_code = 1; return(1);
	    }
	}
    }
#endif

  return(0);
}

/*
  checkfile -- check access and destroy(), if ok
*/

int checkfile(char file[])
{
  int code;

  if ((code = getfiledata(file)) != 0)
    {exit_code = code; return(code);}

  if (options.delete)
    {
      /* char dev */
      if (S_ISCHR(mode))
	{
#ifndef DEBUG
	  if (options.verbose)
	    {
	      if (!options.interactive)
		fprintf(stderr, "\r%s: %s is a character device -- skipping\n");
	      else
		{
		  prompt[0] = 0; /* clear prompt */
		  while (prompt[0] != 'y' && prompt[0] != 'n')
		    {
		      printf("\r%s: remove char device file \'%s\'? ",
			     program_name, file);
		      fgets(prompt, sizeof(prompt), stdin);

		      if (prompt[0] == 'y')
			{
			  if (unlink(file) != 0)
			    {
			      fprintf(stderr, "\r%s: %s: cannot remove: %s\n",
				      program_name, strerror(errno), file);
			    }
			}
		      if (prompt[0] == 'n') return(0); /* skip to next file */

		      /* else, incorrect response -- re-prompt */
		    } /* while prompt */
		} /* else - options.interactive */
	    } /* if options.verbose */
#endif
	  return(0);
	} /* if ischr */

      /* fifo */
      if (S_ISFIFO(mode))
	{
#ifndef DEBUG
	  if (options.interactive)
	    {
	      prompt[0] = 0; /* clear prompt */
	      while (prompt[0] != 'y' && prompt[0] != 'n')
		{
		  printf("\r%s: remove fifo \'%s\'? ",
			 program_name, file);
		  fgets(prompt, sizeof(prompt), stdin);

		  if (prompt[0] == 'y')
		    {
		      if (unlink(file) != 0)
			{
			  fprintf(stderr, "\r%s: %s: cannot remove: %s\n",
				  program_name, strerror(errno), file);
			}
		    }
		  if (prompt[0] == 'n') return(0); /* skip to next file */

		  /* else, incorrect response -- re-prompt */
		} /* while prompt */
	    }
	  else
	    {
	      if (unlink(file) != 0)
		{
		  fprintf(stderr, "\r%s: %s: cannot remove fifo: %s\n",
			  program_name, strerror(errno), file);
		}
	    }
#endif
	  return(0);
	}

      /* socket */
      if (S_ISSOCK(mode))
	{
#ifndef DEBUG
	  if (options.interactive)
	    {
	      prompt[0] = 0; /* clear prompt */
	      while (prompt[0] != 'y' && prompt[0] != 'n')
		{
		  printf("\r%s: remove socket \'%s\'? ",
			 program_name, file);
		  fgets(prompt, sizeof(prompt), stdin);

		  if (prompt[0] == 'y')
		    {
		      if (unlink(file) != 0)
			{
			  fprintf(stderr, "\r%s: %s: cannot remove: %s\n",
				  program_name, strerror(errno), file);
			}
		    }
		  if (prompt[0] == 'n') return(0); /* skip to next file */

		  /* else, incorrect response -- re-prompt */
		} /* while prompt */
	    }
	  else
	    {
	      if (unlink(file) != 0)
		{
		  fprintf(stderr, "\r%s: %s: cannot remove socket: %s\n",
			  program_name, strerror(errno), file);
		}
	    }
#endif
	  return(0);
	}

      /* see if it's a sym link */
      if (S_ISLNK(mode))
	{
#ifndef DEBUG
	  if (options.interactive)
	    {
	      prompt[0] = 0; /* clear prompt */
	      while (prompt[0] != 'y' && prompt[0] != 'n')
		{
		  printf("\r%s: remove sym link \'%s\'? ",
			 program_name, file);
		  fgets(prompt, sizeof(prompt), stdin);

		  if (prompt[0] == 'y')
		    {
		      if (unlink(file) != 0)
			{
			  fprintf(stderr, "\r%s: %s: cannot remove: %s\n",
				  program_name, strerror(errno), file);
			}
		    }
		  if (prompt[0] == 'n') return(0); /* skip to next file */

		  /* else, incorrect response -- re-prompt */
		} /* while prompt */
	    }
	  else
	    {
	      if (unlink(file) != 0)
		{
		  fprintf(stderr, "\r%s: %s: cannot remove sym link: %s\n",
			  program_name, strerror(errno), file);
		}
	    }
#endif
	  return(0);
	}

    } /* if options.delete */
  else
    {
      switch (mode & S_IFMT)
	{
	case S_IFCHR:
	case S_IFLNK:
	case S_IFIFO:
	case S_IFSOCK:
	  return(0);
	}
    }

  /* block dev */
  if (S_ISBLK(mode))
    {
      /*  force overrides interaction  */
      if (options.interactive)
	{
	  prompt[0] = 0; /* clear prompt */
	  while (prompt[0] != 'y' && prompt[0] != 'n')
	    {
	      printf("\r%s: destroy block device \'%s\'? ", program_name, file);
	      fgets(prompt, sizeof(prompt), stdin);

	      if (prompt[0] == 'y')
		{destroy_blkdev(file); return(0);}
	      if (prompt[0] == 'n') return(0); /* skip to next file */

	      /* else, incorrect response -- re-prompt */
	    }
	}

      if (options.force)
	{
	  destroy_blkdev(file); return(0);
	}

      /* we want to respect file modes, so explicitly test permissions */
      else
	{
	  if (access(file, W_OK) == 0)
	    {
	      destroy_blkdev(file); return(0);
	    }
	  else
	    {
	      prompt[0] = 0; /* clear prompt */
	      while (prompt[0] != 'y' && prompt[0] != 'n')
		{
		  printf("\r%s: destroy block device \'%s\', "
			 "overriding mode %04o? ",
			 program_name, file, mode & 07777);

		  fgets(prompt, sizeof(prompt), stdin);

		  if (prompt[0] == 'y')
		    {
		      destroy_blkdev(file); return(0);
		    }
		  if (prompt[0] == 'n') return(0); /* skip to next file */

		  /* else, invalid response, re-prompt */
		}
	    }
	}

      return(0);
    } /* if isblk */

  /* see if it's a directory */
  if (S_ISDIR(mode))
    {
      if (options.recursion)
	{
	  /*  force overrides interaction  */
	  if (options.interactive)
	    {
	      prompt[0] = 0; /* clear prompt */
	      while (prompt[0] != 'y' && prompt[0] != 'n')
		{
		  printf("\r%s: destroy files in \'%s\'? ",
			 program_name, file);
		  fgets(prompt, sizeof(prompt), stdin);

		  if (prompt[0] == 'y') {drilldown(file); return(0);}
		  if (prompt[0] == 'n') return(0); /* skip to next file */

		  /* else, incorrect response -- re-prompt */
		}
	    }

	  if (options.force)
	    {
	      drilldown(file); return(0);
	    }

          /* we want to respect file modes, so explicitly test permissions */
	  else
	    {
	      if (access(file, R_OK|W_OK|X_OK) == 0)
		{
		  drilldown(file); return(0);
		}
	      else
		{
		  prompt[0] = 0; /* clear prompt */
		  while (prompt[0] != 'y' && prompt[0] != 'n')
		    {
		      lstat(file, &filestat);

		      printf("\r%s: destroy \'%s\', overriding mode %04o?", 
		      program_name, file, mode & 07777);

		      fgets(prompt, sizeof(prompt), stdin);

		      if (prompt[0] == 'y')
			{drilldown(file); return(0);}
		      if (prompt[0] == 'n') return(0); /* skip to next file */

		      /* else, invalid response, re-prompt */
		    }
		}
	    }
	}
      else
	{
	  /* if !options.recursion */
	  fprintf(stderr, "%s: %s is a directory -- skipping\n",
		  program_name, file);
	  return(0);
	}
    }

  /*  force overrides interaction  */
  if (options.interactive)
    {
      prompt[0] = 0; /* clear prompt */
      while (prompt[0] != 'y' && prompt[0] != 'n')
	{
	  printf("\r%s: destroy \'%s\'? ", program_name, file);
	  fgets(prompt, sizeof(prompt), stdin);

	  if (prompt[0] == 'y')
	    {destroy(file); return(0);}
	  if (prompt[0] == 'n') return(0); /* skip to next file */

	  /* else, incorrect response -- re-prompt */
	}
    }

  if (options.force)
    {
      destroy(file); return(0);
    }

  /* we want to respect file modes, so explicitly test permissions */

  else
    {
      if (access(file, W_OK) == 0)
	{
	  destroy(file); return(0);
	}
      else
	{
	  prompt[0] = 0; /* clear prompt */
	  while (prompt[0] != 'y' && prompt[0] != 'n')
	    {
	      printf("\r%s: destroy \'%s\', overriding mode %04o? ", 
		     program_name, file, mode & 07777);

	      fgets(prompt, sizeof(prompt), stdin);

	      if (prompt[0] == 'y')
		{
		  destroy(file); return(0);
		}
	      if (prompt[0] == 'n') return(0); /* skip to next file */

	      /* else, invalid response, re-prompt */
	    }
	}
    }

  return(0);
}

/* yea! it's main()! */

int main(int argc, char **argv)
{
  int opt;                        /* option character                      */
  long long tmp; int tmpd;        /* to check input range                  */
  extern int optopt;              /* getopt() stuff                        */
  extern char *optarg;            /* getopt() stuff                        */
  extern int optind, opterr;      /* getopt() stuff                        */

  opterr = 0;                     /* we'll handle bad options              */

  program_name = argv[0];

  errno = 0;
  custom_byte = 0x00;

  /* set defaults */
  percent.display           = 0;  /* initialize                             */
  percent.reported          = 0;  /* initialize                             */
  options.mmap              = 0;  /* mmaped disabled -- msync not reliable? */
  options.hash              = 1;  /* use /dev/urandom hash algorithm        */
  options.verbose           = 1;  /* show percent if >= PERCENT_ENABLE_SIZE */
  options.recursion         = 0;  /* do not traverse directories            */
  options.zero              = 0;  /* don't just zero-out the file           */
  options.force             = 0;  /* respect file permissions               */
  options.delete            = 1;  /* remove targets                         */
  options.custom            = 0;  /* don't use a custom byte                */
  options.random            = 1;  /* perform random passes                  */
  options.seclevel          = 1;  /* fast/secure mode                       */
  options.interactive       = 0;  /* don't confirm each file                */
  options.random_loop   = 4 - 1;  /* default to four (p)random passes       */
  options.wipe_multiply = 1 - 1;  /* perform wipe loop once                 */

  prompt[0] = 0; /* clear prompt */

#ifdef SANITY
  /* sanity checks */
  if (sizeof(size_t) != sizeof(off_t))
    {
      printf("sizeof(size_t) != sizeof(off_t): file offsets are screwed!\n");
      abort();
    }
#endif

  while ((opt = getopt(argc, argv, "S:T:C:B:t:p:b:l::x::XucwsiIhHfFdDvVzZeErRmM")) != -1)
    {
      switch (opt)
	{
	case 'B': /* sector count */
	  sscanf(optarg, "%lld", &tmp);

	  if (tmp < 1)
	    {
	      fprintf(stderr, "%s: error: bad option: sector count < 1\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  sectors = tmp;
	  break;

	case 'S': /* sector size */
	  sscanf(optarg, "%lld", &tmp);

	  if (tmp < 1)
	    {
	      fprintf(stderr, "%s: error: bad option: sector size < 1\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  sector_size = tmp;
	  break;

	case 'C': /* chunk size -- max buf size */
	case 'T': /* old -- block device buffer size */
	  sscanf(optarg, "%lld", &tmp);

	  if (tmp < 1)
	    {
	      fprintf(stderr, "%s: error: bad option: block device buffer size < 1\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  chunk_size = tmp << 10;
	  break;

	case 't': /* hash input buffer size */
	  sscanf(optarg, "%lld", &tmp);

	  if (tmp < 1)
	    {
	      fprintf(stderr, "%s: error: bad option: hash input buffer size < 1\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  hashin_size = tmp;
	  break;

	case 'p': /* wipe multiply */
	  sscanf(optarg, "%lld", &tmp);
	
	  if (tmp < 1)
	    {
	      fprintf(stderr, "%s: error: bad option: wipe multiply < 1\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  if (tmp > 32)
	    {
	      fprintf(stderr, "%s: error: bad option: wipe multiply > 32\n",
		      program_name);
	      exit(BAD_USAGE);
	    }

	  options.wipe_multiply = tmp - 1;
	  break;

	case 'b':  /* overwrite file with byte */
	  sscanf(optarg, "%i", &tmpd);

	  if (tmpd < 0)
	    {
	      fprintf(stderr, "%s: error: bad option: wipe byte < 0\n", program_name);
	      exit(BAD_USAGE);
	    }

	  if (tmpd > 255)
	    {
	      fprintf(stderr, "%s: error: bad option: wipe byte > 255\n", program_name);
	      exit(BAD_USAGE);
	    }

	  options.custom = 1; custom_byte = tmpd;
	  break;

	case 'l':  /* set wipe secure level */
	  if (optarg == 0)
	    options.seclevel = 1;
	  else
	    {
	      sscanf(optarg, "%lld", &tmp);

	      if (tmp < 0 || tmp > 3)
		{
		  fprintf(stderr, "%s: error: bad option: secure level < 0 or > 4\n",
			  program_name);
		  exit(BAD_USAGE);
		}

	      options.seclevel = tmp;
	    }
	  break;

	case 'x':  /* perform random passes */
	  if (optarg == 0)
	    options.random = 1;
	  else
	    {
	      tmp = atoi(optarg);

	      if (tmp < 0)
		{
		  fprintf(stderr, "%s: error: bad option: random loop < 0\n",
			  program_name);
		  exit(BAD_USAGE);
		}

	      if (tmp > 32)
		{
		  fprintf(stderr, "%s: error: bad option: random loop > 32\n",
			  program_name);
		  exit(BAD_USAGE);
		}

	      if (tmp == 0)
		options.random = 0;
	      else
		options.random_loop = tmp - 1;
	    }
	  break;

	case 'X':  /* don't perform random passes */
	  options.random = 0;
	  break;

	case 'm':  /* memory mapped files */
#ifdef MMAP
	  options.mmap = 1;
#else
	  fprintf(stderr, "\r%s: mmap support was not compiled in; option ignored\n",
		  program_name);
#endif
	  break;

	case 'M':  /* do not memory map files */
	  options.mmap = 0;
	  break;

	case 'e':  /* enable entropy */
	  options.hash = 1; /* use /dev/urandom hash */
	  break;

	case 'E':  /* raw mode */
	  options.hash = 0;
	  break;

	case 'r':  /* recursion */
	case 'R':  /* some people are used to '-R' */
	  options.recursion = 1; /* enable recursion */
	  break;

	case 'i':  /* interactive -- disables force */
	  options.force = 0;
	  options.interactive = 1;
	  break;

	case 'I':  /* non-interactive */
	  options.interactive = 0;
	  break;

	case 'f':  /* force -- ignore permissions and override interaction */
	  options.force = 1;
	  options.interactive = 0;
	  break;

	case 'F':  /* disable force */
	  options.force = 0;
	  break;

	case 'd':  /* delete targets */
	  options.delete = 1;
	  break;

	case 'D':  /* don't remove targets */
	  options.delete = 0;
	  break;

	case 'c':  /* copyright */
	  show_copy(); exit(0);
	  break;

	case 'w':  /* warranty */
	  show_war(); exit(0);
	  break;

	case 'u':  /* usage */
	  usage(0, stdout);
	  break;

	case 'h':  /* help */
	case 'H':
	  help(0, stdout);
	  break;

	case 'v':  /* force verbose */
	  options.verbose = 2;
	  break;

	case 'V':  /* verbose */
	  options.verbose = 1;
	  break;

	case 's':  /* silent */
	  options.verbose = 0;
	  break;

	case 'z':  /* zero */
	  options.zero = 1;
	  break;

	case 'Z':  /* don't just zero */
	  options.zero = 0;
	  break;

	default: /* bad option */
	  badopt(optopt);
	  break;
	}
    }

#ifdef OPTIONTEST
  printf("options are:\n");
  printf("mmap              = %d\n", options.mmap);
  printf("hash              = %d\n", options.hash);
  printf("verbose           = %d\n", options.verbose);
  printf("recursion         = %d\n", options.recursion);
  printf("zero              = %d\n", options.zero);
  printf("force             = %d\n", options.force);
  printf("delete            = %d\n", options.delete);
  printf("custom            = %d\n", options.custom);
  printf("custom_byte       = 0x%x\n", custom_byte);
  printf("random            = %d\n", options.random);
  printf("seclevel          = %d\n", options.seclevel);
  printf("interactive       = %d\n", options.interactive);
  printf("random_loop       = %d\n", options.random_loop + 1);
  printf("wipe_multiply     = %d\n\n", options.wipe_multiply + 1);

  exit(0);
#endif

#ifdef FILETEST
  fprintf(stderr, "getopt() parsed %d args\n", optind - 1);
#endif

  if (optind == argc)
    {
      show_copy();
      fprintf(stderr, "\nType \'%s -u\' for usage.\n",
	      program_name);
      exit(BAD_USAGE);
    }

  if ((random_source = open("/dev/urandom", O_RDONLY)) < 0)
    {
      if ((random_source = open("/dev/random", O_RDONLY)) < 0)
	{
	  if (options.verbose)
	    {
	      fprintf(stderr, 
		      "\r%s: warning: cannot open random device; "
		      "random passes disabled\n");
	    }
	  options.random = 0; options.random_loop = 0;
	}
      else
	{
	  if (options.verbose)
	    {
	      fprintf(stderr, 
		      "\r%s: %s: cannot open /dev/urandom; "
		      "using /dev/random instead\n",
		      program_name, strerror(errno));
	    }
	}
    }

  if (options.random)
    {
      if (options.seclevel <= 1)
	{
	  /* initialize hash buffer */
	  if (init_hashin() != 0)
	    {
	      fprintf(stderr, "\r%s: %s: cannot initialize hash buffer\n",
		      program_name, strerror(errno));
	      exit(errno);
	    }
	} /* else it's called per pass or per pass group */
    }

  /* check access and wipe if ok */
  while (optind < argc) checkfile(argv[optind++]); /***** destory *****/

  if (random_source) close(random_source); /* not really needed, but who cares */

#ifdef FILETEST
  fprintf(stderr, "\n");
#endif

  if (percent.reported)
    {
      oldpathlen = strnlen(pathname, PATH_MAX);
      percent_line_clear(); printf("\r");
    }

  return(exit_code);
}
